/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.types.tables;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import uk.ac.imperial.doc.kenya.errors.KenyaInternalError;
import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.passes.Util;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.util.SourceCodeWarningFactory;
import uk.ac.imperial.doc.kenya.types.KArrayType;
import uk.ac.imperial.doc.kenya.types.KBasicType;
import uk.ac.imperial.doc.kenya.types.KBoundClassType;
import uk.ac.imperial.doc.kenya.types.KClassType;
import uk.ac.imperial.doc.kenya.types.KEnumType;
import uk.ac.imperial.doc.kenya.types.KFunction;
import uk.ac.imperial.doc.kenya.types.KParamType;
import uk.ac.imperial.doc.kenya.types.KType;
import uk.ac.imperial.doc.kenya.types.tables.util.PairKFunctionReturnType;

final class FuncComp implements Comparator<KFunction>{
    
    /**
     * Compare's two KFunctions to see if they are the same for a SortedSet
     * ordering.
     * @throws ClassCastException if you do not give it 2 KFunctions
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    public int compare(KFunction f1,KFunction f2){        
        
        /* Sort order on a Function:
         *  Lexicographic on name.
         *  Numeric on number of params.
         *  Lexicographic on each parameter.
         */
        int res;
        
        res = f1.getName().compareTo(f2.getName() );
        if( res != 0){ return res; }
        
        List<KType> f1A = f1.getArgs();
        List<KType> f2A = f2.getArgs();
        
        res  = f1A.size() - f2A.size(); 
        if( res != 0){ return res; }
        

        Iterator<KType> i1 = f1A.iterator();
        Iterator<KType> i2 = f2A.iterator();
        
        while( i1.hasNext() ){
            KType kt1 = i1.next();
            KType kt2 = i2.next();
            
            res = compareTypes(kt1, kt2);
            if( res != 0 ){ return res; }
        }
        
        // It got this far. They must be the same.
        return 0;
    }
    
    
    /**
     * Assuming two correctly typed types, checks to see if they are
     * type matches for each other & returns an ordering if they are not.
     * 
     * The semantics of this compare are:
     *  void < boolean < char < int < double < String < Enum < Enum lexico
     *  Class < Class name lexico < Bound Class < Bound Class name lexico 
     * < Bound Class params lexico on compareTypes < Array 
     * < Array lexico compareTypes on subtypes < Template (all templates are equal)
     * Pre req's that everything here is correctly / validly typed.
     * @param kt1 The "left hand" KType
     * @param kt2 The "right hand" KType
     * @return An int representing the relavate value of the type match
     * @throws IllegalArgumentException if kt1 is not a recognized KType. (kt2 being valid is a pre-cond.
     * @throws NullPointerException if either argument is null.
     */
    static int compareTypes(KType kt1, KType kt2){
        if( kt1 == null || kt2 == null ){
            throw new NullPointerException("Cannot have null arguments!!!! kt1:" + kt1.toString() + " kt2:" + kt2.toString() );
        }
        
        /* Low hanging fruit, but use it non-the-less. */
        if( kt1 == kt2 ){return 0;}
        
        //This is one big if statement. Couldn't think of a nicer fast way to do it.
        if( kt1 instanceof KBasicType ){
            if( kt2 instanceof KBasicType ){
                KBasicType kb1 = (KBasicType) kt1;
                KBasicType kb2 = (KBasicType) kt2;
                
                if( kb1 == KBasicType.getVoid() ){
                    if( kb2 == KBasicType.getVoid()){
                        return 0;
                    }else{
                        return -1;
                    }
                }else if( kb1 == KBasicType.getBoolean() ){
                    if( kb2 == KBasicType.getVoid()){
                        return 1;
                    }else if( kb2 == KBasicType.getBoolean()){
                        return 0;
                    }else{
                        return -1;
                    }
                }else if( kb1 == KBasicType.getChar() ){
                    if( kb2 == KBasicType.getVoid() || kb2 == KBasicType.getBoolean()){
                        return 1;
                    }else if( kb2 == KBasicType.getChar()){
                        return 0;
                    }else{
                        return -1;
                    }
                }else if( kb1 == KBasicType.getInt()){
                    //Note reversal here
                    if( kb2 == KBasicType.getDouble() || kb2 == KBasicType.getString()){
                        return -1;
                    }else if( kb2 == KBasicType.getInt()){
                        return 0;
                    }else{
                        return 1;
                    }
                }else if( kb1 == KBasicType.getDouble()){
                    if( kb2 == KBasicType.getString()){
                        return -1;
                    }else if( kb2 == KBasicType.getDouble()){
                        return 0;
                    }else{
                        return 1;
                    }
                }else if( kb1 == KBasicType.getString()){
                    if( kb2 == KBasicType.getString()){
                        return 0;
                    }else{
                        return 1;
                    }
                }else{
                    throw new IllegalArgumentException("Got an unknown basic type.");
                }
            }else{
                return -1;
            }
        }else if( kt1 instanceof KEnumType ){
            if( kt2 instanceof KBasicType ){
                return 1;
            }else if( kt2 instanceof KEnumType){
                return( kt1.getName().compareTo(kt2.getName()));
            }else{
                return -1;
            }
        }else if( kt1 instanceof KClassType){
            if( kt2 instanceof KBasicType || kt2 instanceof KEnumType){
                return 1;
            }else if( kt2 instanceof KClassType){
                return( kt1.getName().compareTo(kt2.getName()));
            }else{
                return -1;
            }
        }else if( kt1 instanceof KBoundClassType){
            //Note reversal in comparison order below
            if( kt2 instanceof KArrayType || kt2 instanceof KParamType ){
                return -1;
            }else if( kt2 instanceof KBoundClassType){
                /* The more interesting of cases.
                 * 
                 * */
                KBoundClassType kbct1 = (KBoundClassType) kt1;
                KBoundClassType kbct2 = (KBoundClassType) kt2;
                
                String n1 = kbct1.getBase().getName();
                String n2 = kbct2.getBase().getName();
                
                int res = n1.compareTo(n2);
                if( res != 0){ return res; }
                
                /* So we have two classes, with the same base class. */
                Iterator<KParamType> it = kbct1.getBase().getTypeParamList().iterator();
                
                while(it.hasNext()){
                    KParamType kpt = (KParamType) it.next();
                    
                    KType type1 = kbct1.getBindings().lookup(kpt);
                    KType type2 = kbct2.getBindings().lookup(kpt);
                    
                    res = compareTypes(type1,type2);
                    if( res != 0 ){ return res; }
                }
                
                /* So we got that far and everything matches. Guess they're the same. */
                return 0;
                
            }else{
                return 1;
            }
        }else if( kt1 instanceof KArrayType) {
            if( kt2 instanceof KParamType ){
                return -1;
            }else if(kt2 instanceof KArrayType){
                KArrayType kat1 = (KArrayType) kt1;
                KArrayType kat2 = (KArrayType) kt2;
                
                /* Horribly recursive. Could be done on the stack, 
                 * but no1 creates a huge nested array...do they? */
                return compareTypes(kat1.getChildType(), kat2.getChildType());
                
            }else{
                return 1;
            }
        }else if( kt1 instanceof KParamType){
            if( kt2 instanceof KParamType){
                return 0; // All KParamTypes are thee same.
            }else{
                return 1;
            }
        }else{
            throw new IllegalArgumentException("Unknown type for kt1:" + kt1.toString() + " " + kt2.toString());
        }
        
    }
    
        
}

/**
 * Holds the KFunctions for the current file.  Can resolve the correct function
 * given an argument list (in the case of overloaded functions) and also
 * checks that illegal duplicates arn't defined. 
 * @author toa02
 *
 */
public class FunctionTable {

    private Map<String,SortedSet<KFunction>> _lvlOneLookup; // Turns a String name for function into a SortedSet of KFunctions that
    							// match that name.
    
    private final FuncComp _comparator; // Reuse the same comparator all through.
    /**
     * Constructor, sets everything up.
     */
    public FunctionTable(){
        _lvlOneLookup = new HashMap<String,SortedSet<KFunction>>();
        _comparator = new FuncComp();
    }
    
    /**
     * Places a new KFunction into this function table, and throws appropriate exceptions if its already there.
     * @param kf KFunction to be added to the function table.
     * @param builtin boolean, should be true if this is a builtin method ( affects the error messages thrown ).
     * @throws SourceCodeException Thrown if a duplicate function definition is found.
     * @throws AnonymousSourceCodeException Thrown by some methods this calls.
     */
    public void add(KFunction kf, boolean builtin) throws SourceCodeException{
        SortedSet<KFunction> s;
        if( _lvlOneLookup.containsKey(kf.getName())){
            s = _lvlOneLookup.get(kf.getName());

        }else{
            s = new TreeSet<KFunction>(_comparator);
            _lvlOneLookup.put(kf.getName(),s);
        }

        if(!s.add(kf)){
            if( !kf.isBuiltin() ){
                
                int ln = kf.getNode().getIdentifier().getLine();
                int pos = kf.getNode().getIdentifier().getPos();
                int len = kf.getNode().getIdentifier().getText().trim().length();
                
                KFunction conf = (KFunction) s.tailSet(kf).first(); 
                if( !conf.isBuiltin()){
                    int confLn = conf.getNode().getIdentifier().getLine();
                    int confPos = conf.getNode().getIdentifier().getPos();
                    int confLen = conf.getNode().getIdentifier().getText().trim().length();
            
                    int[][] lnkPos = { { confLn, confPos, confLen } };
                    SourceCodeException.throwDuplicateFunction(ln, pos, len, lnkPos,conf.getName() + paramsToString(kf.getArgs()),Util.GLOBAL);
                }else{
                    SourceCodeException.throwDuplicateFunction_Builtin(ln, pos, len, conf.getName());
                }
            }else{
                KFunction conf = (KFunction) s.tailSet(kf).first(); 
                if( conf.isBuiltin() ){
                    throw new KenyaInternalError("Conflicting Builtin Functions detected. Please check Kenya installation.");
                }else{
                    //So I kf is a builtin, conf is normal
                    int ln = conf.getNode().getIdentifier().getLine();
                    int pos = conf.getNode().getIdentifier().getPos();
                    int len = conf.getNode().getIdentifier().getText().trim().length();
                    SourceCodeException.throwDuplicateFunction_Builtin(ln, pos, len, conf.getName());
                }    
            }
        }

    }
    
    
    /**
     * Gets the KType return type for the given function name and parameters. It will throw an
     * AnonymousSourceCodeException with a useful error message if there are duplicate's that will
     * match. (Since we can have casting issues with ints the least number of int/double casts
     * will win in those situations aswell. [ I'm nice like that :) ] ).
     * @param name String name of the method.
     * @param paramTypes List of KTypes that are the parameters you want to apply to this method.
     * @return A PairKFunctionReturnType representing what the bound return type of this function is and the KFunction itself.
     * @throws AnonymousSourceCodeException with a useful error message if a function with that name 
     * is not found, with that arity doesn't exist, etc. etc.
     */
    public PairKFunctionReturnType getFuncAndRetType ( String name, List<KType> paramTypes, int ln, int pos, int len, List<ISourceCodeWarning> warnings, String scope ){
        if( !_lvlOneLookup.containsKey(name)){
            String args = paramsToString(paramTypes);
            SourceCodeException.throwLost_Function(ln, pos, len, args, name, scope);
        }
        
        SortedSet<KFunction> fns = _lvlOneLookup.get(name);
//      List errors = new LinkedList();
        Iterator<KFunction> it = fns.iterator();
        boolean foundMatch = false;
        KFunction bestMatch = null;
        KType retType = null;
        int numCasts = -1;
        List<ISourceCodeWarning> localWarnings = new LinkedList<ISourceCodeWarning>();
        
        
        funcList: while(it.hasNext()){
            KFunction kf = (KFunction) it.next();

            if(kf.getArgs().size() != paramTypes.size()){
                continue;
            }

            
            // So both have the same name, and the same arity.
            List<KType> theirArgs = kf.getArgs();
            TypeTable tt = new TypeTable();
            
            Iterator<KType> theirs = theirArgs.iterator();
            Iterator<KType> mine = paramTypes.iterator();
            int tmpNumCasts = 0;
            
            while( theirs.hasNext()){
                
                KType thisMe = (KType) mine.next();
                KType thisHim = (KType) theirs.next();
                
                thisHim.compareAndBind(thisMe, tt); // do it once
                									// to setup the tt final
                									// bindings
            }
            
            theirs = theirArgs.iterator();
            mine = paramTypes.iterator();
            
            while( theirs.hasNext() ){
            	
            	KType thisMe = (KType) mine.next();
                KType thisHim = (KType) theirs.next();
            	
                int res = thisHim.compareAndBind(thisMe, tt);
                if( res < 0 ){
                    if( res == -2 ){
                        if( kf.isBuiltin() ){
                            /* Ignore it, its probably System.arraycopy :s */
                            //warnings.add(SourceCodeWarningFactory.attemptedBasicTypeArrayUsage(ln, pos, len, lnkPos ));
                        }else{
	                        int[][] lnkPos = { { 
	        	                kf.getNode().getIdentifier().getLine(),
	        	                kf.getNode().getIdentifier().getPos(),
	        	                kf.getNode().getIdentifier().getText().length()
	        	            				}};
	                        localWarnings.add(SourceCodeWarningFactory.attemptedBasicTypeArrayUsage(ln, pos, len, lnkPos ));
                        }
                    }
                    continue funcList; 
                 } // Didn't match.
                
                tmpNumCasts += res;
            }//inner while loop
            
	        if(foundMatch && (tmpNumCasts < numCasts)){
	            bestMatch = kf;
	            numCasts = tmpNumCasts;
	        }else if( foundMatch && (tmpNumCasts == numCasts) ){
	            int[][] lnkPos = null;

	            if( !bestMatch.isBuiltin() && !kf.isBuiltin()){
	            lnkPos = new int[][] { { 
	                kf.getNode().getIdentifier().getLine(),
	                kf.getNode().getIdentifier().getPos(),
	                kf.getNode().getIdentifier().getText().length()
	            				},
	            				{
   	                bestMatch.getNode().getIdentifier().getLine(),
  	                bestMatch.getNode().getIdentifier().getPos(),
  	                bestMatch.getNode().getIdentifier().getText().length()
	            				}
	            				};
	            }else if( !bestMatch.isBuiltin() ){
	                lnkPos = new int[][] { { 
		                bestMatch.getNode().getIdentifier().getLine(),
		                bestMatch.getNode().getIdentifier().getPos(),
		                bestMatch.getNode().getIdentifier().getText().length()
		            				}};
	            }else if( !kf.isBuiltin()){
	                lnkPos = new int[][] { { 
		                kf.getNode().getIdentifier().getLine(),
		                kf.getNode().getIdentifier().getPos(),
		                kf.getNode().getIdentifier().getText().length()
		            				}};
	            }
	            
	            warnings.addAll(localWarnings);
	            String args1 = paramsToString(bestMatch.getArgs());
	            String args2 = paramsToString(theirArgs);
	            SourceCodeException.throwAmbiguous_Function(ln, pos, len, lnkPos, args1, args2, Util.GLOBAL, name);
	        }
	            
	        if( !foundMatch ){
	            foundMatch = true;
	            bestMatch = kf;
	            retType = kf.getReturnType().bind(tt);
	            numCasts = tmpNumCasts;
	        }
            
        }//funcList while loop
        
        
        
        if( foundMatch){
            /*
             * Have to do a check to ensure that we are not returning an
             * array of basic type that has been bound ( not allowed )
             */
            KType normalReturnType = bestMatch.getReturnType();
            if( normalReturnType instanceof KArrayType ){
                KArrayType kat = (KArrayType) normalReturnType;
                if( kat.getBaseType() instanceof KParamType){
                    KArrayType actRT = (KArrayType) retType;
                    KType bbase = actRT.getBaseType();
                    
                    if( bbase instanceof KBasicType && bbase != KBasicType.getString() 
                            && actRT.getDepth() == kat.getDepth() ){
                        
                        int depth = ((KArrayType) retType).getDepth();
                        
                        KType outRT = new KBoundClassType( (((KBasicType)bbase).getClassType()),
                                new TypeTable());
                        //((KClassType)outRT).setChildren(new SymbolTable());
                        for(int i = 0; i < depth ; i++){
                            outRT = new KArrayType(outRT);
                        }
                        retType = outRT;
                    }
                }
                     
            }
            
            return new PairKFunctionReturnType(bestMatch,retType);
        }else{
            String args = paramsToString(paramTypes);

            Iterator<KFunction> errit = fns.iterator();
            int i = 0;
            while(errit.hasNext()){
                KFunction kf = errit.next();
                if( !kf.isBuiltin()){
                    i++;
                }
            }
            
            int[][] lnkPos = new int[i][3];            
            
            i= 0;
            errit = fns.iterator();
            
            while(errit.hasNext()){
                KFunction kf = (KFunction) errit.next();
                if(!kf.isBuiltin()){
                    lnkPos[i][0] = kf.getNode().getIdentifier().getLine();
                    lnkPos[i][1] = kf.getNode().getIdentifier().getPos();
                    lnkPos[i][2] = kf.getNode().getIdentifier().getText().length();
                    i++;
                }
            }
            warnings.addAll(localWarnings);
            SourceCodeException.throwLost_Function_Possible(ln, pos, len, lnkPos, args, name, scope );
            return null; // unreachable
        }
        
        
    }
    

    /**
     * Quick utility method to turn a List of KTypes that are paramters to a method into a String.
     * @param l List of KTypes.
     * @return String representing that list of paramters surrounded by "("'s.
     */
    public static String paramsToString(List<KType> l){
        StringBuffer sb = new StringBuffer();
        
        sb.append("(");
        Iterator<KType> it = l.iterator();
        
        while(it.hasNext()){
            KType kt = it.next();
            sb.append(kt.getName());
            if( it.hasNext()){
                sb.append(", ");
            }
        }
        
        sb.append(")");
        return sb.toString();
    }
    
    public Set<String> getNames() {
        return _lvlOneLookup.keySet();
    }
    
    public SortedSet<KFunction> getKFunctions(String name) {
        return _lvlOneLookup.get(name);
    }
    
    /**
     * Returns a string representation of all the functions stored in this table.
     * @see java.lang.Object#toString()
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        
        sb.append("Functions:");
        sb.append(NewLineClass.NEWLINE);
        
        Iterator<String> it = _lvlOneLookup.keySet().iterator();

        while(it.hasNext()){

            Iterator<KFunction> it2 = ((Set<KFunction>)_lvlOneLookup.get(it.next())).iterator();
            while(it2.hasNext()){
                sb.append("\t");
                sb.append( it2.next().toString());
                if( it2.hasNext()){
                    sb.append(NewLineClass.NEWLINE);
                }
            }
            
            if( it.hasNext()){
                sb.append(NewLineClass.NEWLINE);
            }
        }
        
        
        return sb.toString();
    }
}