/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.types.tables;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import uk.ac.imperial.doc.kenya.types.KParamType;
import uk.ac.imperial.doc.kenya.types.KType;


/**
 * Type table, holds a stackable system of Type-Parameters to other bindings.
 * @author toa02
 *
 */
public class TypeTable{

    private final Map<KParamType,KType> _initial;
    
    /**
     * 
     * @param initialCapacity
     */
    public TypeTable( Map<KParamType,KType> initialMappings) {
        _initial = initialMappings;
    }
    
    public TypeTable( TypeTable clone ){
    	_initial = new HashMap<KParamType,KType>();
    	Iterator<KParamType> it = clone._initial.keySet().iterator();
    	
    	while( it.hasNext() ){
    		KParamType o = it.next();
    		_initial.put(o, clone._initial.get(o));
    	}
    }
    
    public TypeTable(){
        _initial = new HashMap<KParamType,KType>();
    }
    
    /**
     * Puts a new mapping into this TypeTable.
     * @param src The KParamType which is the source.
     * @param target The KType which is the target in the binding.
     * @return true iff the binding has been put in. False if its 
     * already mapped (no change is made).
     */
    public boolean put( KParamType src, KType target){
        if( _initial.containsKey(src)){
            return false;
        }
        
        _initial.put(src, target);
        return true;
    }
    
    public boolean remove(KParamType src){
    	if( _initial.containsKey(src)){
    		_initial.remove(src);
    		return true;
    	}
    	
    	return false;
    	
    }
    
    /**
     * Returns an Iterator to the KParamTypes which are the keys in 
     * this Map.
     * @return Iterator An iterator to the KParamTypes which are the keys in the TypeTable Map.
     */
    public Set<KParamType> getKeys(){
        return _initial.keySet();
    }
    

    /**
     * Returns the KType from performing the lookup from this KParamType.
     * Throws a NoSuchElementException if it doesn't exist.
     * @param KParamType source The KParamType to use for the lookup.
     * @return KType what the KParamType maps to.
     */
    public KType lookup(KParamType source) throws NoSuchElementException{
        if(_initial.containsKey(source)){
            return (KType) _initial.get(source);
        }
        
        throw new NoSuchElementException("Couldn't find a mapping for the parameter.");
        
    }

    /**
     * Lets you check if this TypeTable contains a mapping for a 
     * given KParamType.
     * @param source KParamType to see if there is a mapping for it.
     * @return true iff this KParamType exists in this Map.
     */
    public boolean containsKey(KParamType source){
        return (_initial.containsKey(source));
    }

    /**
     * Forces a binding to take place, regardless if one is there already.
     * @param source KParamType Source or the key for the mapping.
     * @param target KType target for the mapping.
     */
    public void force(KParamType source, KType target){
        if(_initial.containsKey(source)){
            _initial.remove(source);
        }
        
        _initial.put(source, target);
    }
    
    /**
     * Returns a nice String represntation of this TypeTable
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        sb.append("{");
            Iterator<KParamType> it2 = _initial.keySet().iterator();
            while(it2.hasNext()){
                KParamType kpt = (KParamType) it2.next();
                KType kt = (KType) _initial.get(kpt);
                
                sb.append(kpt);
                sb.append("=>");
                sb.append(kt.getName());
                
                if(it2.hasNext())
                    sb.append(",");
            }
            
        sb.append("}");
        return sb.toString();
    }

}
