/*
 * Created on 17-Nov-2004
 */
package kenya;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import kenya.errors.SourceCodeException;
import kenya.interpreter.util.InterpreterLastPointReachedCatcher;
import mediator.ICheckedCode;
import mediator.IJavaCode;
import mediator.Mediator;
import mediator.stackMachine.IStackMachineInformationProvider;
import mediator.util.StreamHandler;

import org.wellquite.kenya.stackMachine.ClosureMiscHelper;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.misc.AbstractJob;
import org.wellquite.kenya.stackMachine.misc.IJob;
import org.wellquite.kenya.stackMachine.misc.JobDispatch;
import org.wellquite.kenya.stackMachine.scope.ClosureScope;
import org.wellquite.kenya.stackMachine.scope.IClosureScope;
import org.wellquite.kenya.stackMachine.types.ArrayTypeFactory;
import org.wellquite.kenya.stackMachine.types.StringTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IStringType;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * @author Thomas Timbul
 */
public class CommandLineKenya {
	
	public static final String VERSION = "4.2";
	
	private static boolean exiting = false;
	
	static final List exitListeners = new ArrayList();
	
	public static Getopt getOpts(String[] args) {
		return new Getopt("Kenya", args, "hdpfeltyscin", new LongOpt[] {
        new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
        new LongOpt("debug", LongOpt.NO_ARGUMENT, null, 'd'),
        new LongOpt("parse", LongOpt.NO_ARGUMENT, null, 'p'),
        new LongOpt("typefind", LongOpt.NO_ARGUMENT, null, 'f'),
        new LongOpt("definitionfind", LongOpt.NO_ARGUMENT, null, 'e'),
        new LongOpt("builtinload", LongOpt.NO_ARGUMENT, null, 'l'),
        new LongOpt("typecheck", LongOpt.NO_ARGUMENT, null, 't'),
        new LongOpt("structcheck", LongOpt.NO_ARGUMENT, null, 'y'),
        new LongOpt("translate", LongOpt.NO_ARGUMENT, null, 's'),
        new LongOpt("execute", LongOpt.REQUIRED_ARGUMENT, null, 'c'),
        new LongOpt("interpret", LongOpt.NO_ARGUMENT, null, 'i'),
        new LongOpt("license", LongOpt.NO_ARGUMENT, null, 'n') });
	}
	
	private static final AbstractJob exitJob = new AbstractJob() {
		
		public void execute() {
			List exiters = new ArrayList(exitListeners);
			for (int idx = 0; idx < exiters.size(); idx++) {
				AbstractJob exitingJob = (AbstractJob) exiters.get(idx);
				if (exitingJob != null) exitingJob.execute();
			}
			JobDispatch.shutDownAllJobs();
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
				/* silently fail */
			}
			System.exit(0);
		}
	};
	
	public static void main(String[] args) {
		Getopt op = getOpts(args);
		
		int opt;
		
		int debugOptions = 0;
		boolean translate = false; // Should we translate?
		boolean execute = false; // Should we excute?
		String path = null; // Where should we compile to?
		
		boolean interpret = false; // interpret the thing [ takes precedence ]
		while ((opt = op.getopt()) != -1) {
			switch (opt) {
				case 'h':
					/* Help */
					printHelp();
					System.exit(0);
					break;
				case 'n':
					KenyaCore.printLicense();
					System.exit(0);
					break;
				case 'd':
					/* Debug */
					debugOptions |= Mediator.DBG_MESSAGES;
					break;
				case 'p':
					/* Parse */
					debugOptions |= Mediator.DBG_PARSE;
					break;
				case 'f':
					/* TypeFind */
					debugOptions |= Mediator.DBG_TYPEFIND;
					break;
				case 'e':
					/* Definiton Find */
					debugOptions |= Mediator.DBG_DEFINITIONFIND;
					break;
				case 'l':
					/* Builtin Loader */
					debugOptions |= Mediator.DBG_BUILTINLOAD;
					break;
				case 't':
					/* TypeCheck */
					debugOptions |= Mediator.DBG_TYPECHECK;
					break;
				case 'y':
					/* Structual Check */
					debugOptions |= Mediator.DBG_STRUCTURECHECK;
					break;
				case 's':
					/* Translate */
					translate = true;
					break;
				case 'c':
					/* Compile */
					execute = true;
					path = op.getOptarg();
					break;
				case 'i':
					/* Interpret */
					interpret = true;
					break;
				case '?':
					/* Error message from getopt */
				default:
					/* Not recognised */
					printHelp();
				System.exit(1);
				break;
			}
		}
		
		int argIndex = op.getOptind();
    if (argIndex < args.length) {
    	/* There are file(s) to process */
    	String[] argsLeft = new String[args.length - argIndex];
      System.arraycopy(args, argIndex, argsLeft, 0, argsLeft.length);
      doCommandLine(argsLeft, debugOptions, translate, execute,
              interpret, path);
    } else {
    	/* There are no files to process */
    	KenyaCore.printError("Command Line Kenya needs files to operate upon.");
    }
		
	}
	
	public static void doCommandLine(String[] files, int dbg,
      boolean translate, boolean execute, boolean interpret,
      String filePath) {

  final StackMachine sm = new StackMachine();

  try {
      FileReader fis = new FileReader(files[0]);
      ICheckedCode icc = Mediator.check(fis, dbg, System.out);

      if (icc.isErroredCode()) {
          Iterator it = icc.getErrors().iterator();
          while (it.hasNext()) {
              SourceCodeException sce = (SourceCodeException) it.next();
              System.err.println(sce.getMessage());
          }
          System.out.println(files[0] + ": INVALID CODE.");
      } else {
          System.out.println(files[0] + ": VALID CODE.");

          if (interpret) {
              IStackMachineInformationProvider ismp = icc
                      .getBaseStackMachine();

              InterpreterLastPointReachedCatcher ilprc = new InterpreterLastPointReachedCatcher(
                      sm);
              sm.addPositionReachedListener(ilprc);
              sm.setStepMode(false);

              try {
                  IClosureScope scope = new ClosureScope();
                  ClosureMiscHelper.executeClosureInScope(scope, ismp
                          .getPreInitClosure(), sm);

                  IType[] array = new IType[files.length - 1];
                  for (int idx = 1; idx < files.length; idx++) {
                      array[idx - 1] = StringTypeFactory
                              .createStringType(files[idx]);
                  }
                  sm.push(ArrayTypeFactory.createArrayType(array, IStringType.TYPE_NAME));
                  sm.invokeMethod(ismp.getEntryPointClass(), ismp
                          .getEntryPoint());
              } catch (Throwable t) {
                  sm.getErr().println(
                          t + " at " + ilprc.getLastPointData());
              } finally {
                  exit();
              }
          } else if (translate) {
              BufferedReader r = new BufferedReader(icc.translate()
                      .getPlaceHeldCode());

              String s;
              while ((s = r.readLine()) != null) {
                  System.out.println(s);
              }
          } else if (execute) {
              long v = 0;
              String s = "_Command";
              IJavaCode ijc = icc.translate();
              while (!ijc.isValidClassName(s + v)) {
                  v++;
              }

              String[] args = new String[files.length - 1];
              System.arraycopy(files, 1, args, 0, args.length);

              final PipedOutputStream stdInPOS = new PipedOutputStream();
              PipedInputStream stdIn = new PipedInputStream(stdInPOS);

              BufferedWriter stdInWriter = new BufferedWriter(
                      new OutputStreamWriter(stdInPOS));
              BufferedWriter stdOutWriter = new BufferedWriter(
                      new OutputStreamWriter(System.out));
              BufferedWriter stdErrWriter = new BufferedWriter(
                      new OutputStreamWriter(System.err));

              final BufferedReader stdInReader = new BufferedReader(
                      new InputStreamReader(stdIn));

              final StreamHandler sh = new StreamHandler(
                      new BufferedReader(new InputStreamReader(System.in)),
                      stdInWriter);

              JobDispatch.enqueueJob(sh);

              JobDispatch.enqueueJob(new AbstractJob() {

                  public void execute() {
                      try {
                          synchronized (sh) {
                              while (!sh.isFinished()) {
                                  sh.wait();
                              }
                          }
                          if (KenyaCore.DEBUG_OUTPUT) {
                              System.out
                                      .println("stream handler B finished");
                          }
                          stdInPOS.close();
                          if (KenyaCore.DEBUG_OUTPUT) {
                              System.out
                                      .println("stream handler BB finished");
                          }
                      } catch (InterruptedException ie) {
                          ie.printStackTrace();
                      } catch (IOException ioe) {
                          ioe.printStackTrace();
                      }

                  }
              });

              if (KenyaCore.DEBUG_OUTPUT) {
                  System.out.println("about to execute");
              }

              boolean res = icc.translate().execute(filePath, s + v,
                      stdInReader, stdInWriter, stdOutWriter,
                      stdErrWriter, args);

              if (res == false && KenyaCore.DEBUG_OUTPUT) {
                  System.err.println("Translate & Execute failed");
              }

              System.in.close();
              System.out.flush();
              System.err.flush();

              exit();
          }
      }

  } catch (FileNotFoundException e) {
      System.err.println("File Not Found: " + files[0]);
  } catch (IOException ioe) {
      System.err.println("IOException on: " + files[0]);
  }

  exit();
}
	
	public synchronized static void exit() {
		if (!exiting) {
			exiting = true;
			JobDispatch.enqueueJob(exitJob);
		}
	}
	
	public synchronized static boolean isExiting() {
		return exiting;
	}
	
	public synchronized static void addExitListener(IJob listener) {
		if (listener == null) {
			System.err.println("Attempt made to add a null exit listener.");
			throw new NullPointerException();
		} else
			exitListeners.add(listener);
	}
	
	public synchronized static void removeExitListener(AbstractJob listener) {
		exitListeners.remove(listener);
	}
	
	
	public static void printHelp() {
	    System.out.println(cmdhelp);
	}
	
	public static final String cmdhelp =
		KenyaCore.commonhelp1
		+ "        kenya (options) kenyaFile [arguments]"
		+ KenyaCore.NEWLINE
		+ "At least one option needs to be specified, valid options are:"
		+ KenyaCore.optionshelp;
		

}
