/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 02-Jul-2004
 */
package kenya.builtIns.builtInMethods;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import kenya.builtIns.BuiltInMethod;
import kenya.builtIns.IBuiltInMethodFactory;
import kenya.builtIns.IGetMethodCode;
import kenya.builtIns.SMLibraryContainer;
import kenya.types.KArrayType;
import kenya.types.KBasicType;
import kenya.types.KParamType;
import kenya.types.KType;

import org.wellquite.kenya.stackMachine.InterpretedMethod;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IArrayType;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class ArrayCopy implements IBuiltInMethodFactory {

    private static final Set emptySet = Collections
            .unmodifiableSet(new HashSet());

    private static final Map emptyMap = Collections
            .unmodifiableMap(new HashMap());

    private static final String name = "arraycopy";

    private static final List templateParameters = new LinkedList();

    private static final List booleanParameters = new LinkedList();

    private static final List charParameters = new LinkedList();

    private static final List intParameters = new LinkedList();

    private static final List doubleParameters = new LinkedList();

    private static final List stringParameters = new LinkedList();
    
    private static final Map templateMap = new HashMap();

    public static final Set reservedClasses = new HashSet();

    static {
        KType arrayType = new KParamType("AT");
        templateParameters.add(new KArrayType(arrayType));
        templateParameters.add(new KArrayType(arrayType));
        templateMap.put("AT", arrayType);

        booleanParameters.add(new KArrayType(KBasicType.getBoolean()));
        booleanParameters.add(new KArrayType(KBasicType.getBoolean()));

        charParameters.add(new KArrayType(KBasicType.getChar()));
        charParameters.add(new KArrayType(KBasicType.getChar()));

        intParameters.add(new KArrayType(KBasicType.getInt()));
        intParameters.add(new KArrayType(KBasicType.getInt()));

        doubleParameters.add(new KArrayType(KBasicType.getDouble()));
        doubleParameters.add(new KArrayType(KBasicType.getDouble()));

        stringParameters.add(new KArrayType(KBasicType.getString()));
        stringParameters.add(new KArrayType(KBasicType.getString()));
        
        reservedClasses.add("System");
    }

    private static final KType returnType = KBasicType.getVoid();

    public String getName() {
        return name;
    }

    private static final IGetMethodCode javaCode = new IGetMethodCode() {

        public String[] getCode(List parameterValues, List parameterTypes) {
            if (parameterValues.size() != parameterTypes.size())
                    throw new RuntimeException("I was expecting "
                            + parameterTypes.size()
                            + " templateParameters and received "
                            + parameterValues.size() + " templateParameters");

            String[] lho = (String[]) parameterValues.get(0);
            String[] rho = (String[]) parameterValues.get(1);

            String[] lhsA = { "System.", "arraycopy", "(", " " };
            String[] midA = lho;
            String[] rhsA = { ",", " ", "0", ",", " " };

            String[] lhsGen = BuiltInMethod.concatenate(lhsA, midA, rhsA);
            String[] midGen = rho;
            String[] rhsGen = { ",", " ", "0", ",", " ", "Math.", "min", "(",
                    " " };

            String[] lhsOverall = BuiltInMethod.concatenate(lhsGen, midGen,
                    rhsGen);

            String[] lhsB = lho;
            String[] midB = { ".length", ",", " " };
            String[] rhsB = rho;

            String[] midOverall = BuiltInMethod.concatenate(lhsB, midB, rhsB);

            String[] rhsOverall = { ".length", " ", ")", " ", ")" };

            return BuiltInMethod
                    .concatenate(lhsOverall, midOverall, rhsOverall);
            //return( "System.arraycopy( " + lhs + ", 0, " + rhs + ", 0,
            // Math.min( " + lhs + ", " + rhs + ") )" );
        }
    };

    private static final AbstractAtomicClosure execute = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType[] ary0 = ((IArrayType) sm.pop()).getValue();
            IArrayType ary1 = (IArrayType) sm.pop();
            for (int idx = 0; idx < ary0.length && idx < ary1.getLength(); idx++) {
                ary1.set(idx, ary0[idx]);
            }
            sm.push(ary1);
        }

        public String toString() {
            return name;
        }
    };

    public Set buildMethods() {
        Set methods = new HashSet();

        SMLibraryContainer smLibrary = SMLibraryContainer.getLibrary();
        IInterpretedMethod method = new InterpretedMethod(getName(), smLibrary
                .getLibraryClass(), true, true);
        method.setMethodBody(execute);
        smLibrary.addStaticMethod(method);

        BuiltInMethod bim = new BuiltInMethod();
        bim.setCodeHook(javaCode);
        bim.setImports(emptySet);
        bim.setName(getName());
        bim.setMethodParameters(templateParameters);
        bim.hasReturnType(returnType);
        bim.setTemplateParameters(templateMap);
        bim.setReservedClasses(reservedClasses);
        bim.setInterpretedMethod(method);
        methods.add(bim);

        BuiltInMethod bimBoolean = new BuiltInMethod();
        bimBoolean.setCodeHook(javaCode);
        bimBoolean.setImports(emptySet);
        bimBoolean.setName(getName());
        bimBoolean.setMethodParameters(booleanParameters);
        bimBoolean.hasReturnType(returnType);
        bimBoolean.setTemplateParameters(emptyMap);
        bimBoolean.setReservedClasses(reservedClasses);
        bimBoolean.setInterpretedMethod(method);
        methods.add(bimBoolean);

        BuiltInMethod bimChar = new BuiltInMethod();
        bimChar.setCodeHook(javaCode);
        bimChar.setImports(emptySet);
        bimChar.setName(getName());
        bimChar.setMethodParameters(charParameters);
        bimChar.hasReturnType(returnType);
        bimChar.setTemplateParameters(emptyMap);
        bimChar.setReservedClasses(reservedClasses);
        bimChar.setInterpretedMethod(method);
        methods.add(bimChar);

        BuiltInMethod bimInt = new BuiltInMethod();
        bimInt.setCodeHook(javaCode);
        bimInt.setImports(emptySet);
        bimInt.setName(getName());
        bimInt.setMethodParameters(intParameters);
        bimInt.hasReturnType(returnType);
        bimInt.setTemplateParameters(emptyMap);
        bimInt.setReservedClasses(reservedClasses);
        bimInt.setInterpretedMethod(method);
        methods.add(bimInt);

        BuiltInMethod bimDouble = new BuiltInMethod();
        bimDouble.setCodeHook(javaCode);
        bimDouble.setImports(emptySet);
        bimDouble.setName(getName());
        bimDouble.setMethodParameters(doubleParameters);
        bimDouble.hasReturnType(returnType);
        bimDouble.setTemplateParameters(emptyMap);
        bimDouble.setReservedClasses(reservedClasses);
        bimDouble.setInterpretedMethod(method);
        methods.add(bimDouble);

        BuiltInMethod bimString = new BuiltInMethod();
        bimString.setCodeHook(javaCode);
        bimString.setImports(emptySet);
        bimString.setName(getName());
        bimString.setMethodParameters(stringParameters);
        bimString.hasReturnType(returnType);
        bimString.setTemplateParameters(emptyMap);
        bimString.setReservedClasses(reservedClasses);
        bimString.setInterpretedMethod(method);
        methods.add(bimString);
        
        return methods;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return getName();
    }
}