/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

package kenya.errors;

import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import kenya.types.KClassType;
import kenya.types.KType;
import kenya.types.KVariable;
import mediator.IJavaCode;
import minijava.lexer.LexerException;
import minijava.parser.ParserException;

/**
 * The new and improved SourceCodeException class. DO NOT DELETE THE LINES
 * MARKER BEGIN INJECT and MARKER END INJECT They
 * are referenced by the script. [with the // immediately before the marker's]
 * See the SourceCodeException target in build.xml
 * @author toa02
 *  
 */
public class SourceCodeException extends RuntimeException implements
        ISourceCodeError {

    private static final long serialVersionUID = 3449404;

    public static final int JAVAC_MESSAGE = 0;

    public static final int HUMAN_MESSAGE = 1;

    public static final int LINE_NUM = 0;

    public static final int CHAR_NUM = 1;

    public static final int TOKEN_LEN = 2;

    /**
     * Wraps up a ParserException into a SourceCodeException.
     * 
     * @param pe
     * @return
     */
    public static SourceCodeException wrapParserException(ParserException pe) {
        int ln = pe.getToken().getLine();
        int pos = pe.getToken().getPos();

        Pattern p = Pattern.compile("\\[(\\d+),(\\d+)\\]");

        Matcher m = p.matcher(pe.getMessage());
        if (m.find()) {
            ln = Integer.parseInt(m.group(1));
            pos = Integer.parseInt(m.group(2));
        } else {
            throw new KenyaInternalError("Didn't match!" + pe.getMessage());
        }

        int len = pe.getToken().getText().trim().length();
        if (len == 0) {
            len = 1;
        }
        String entire = m.group(0);

        String mess = pe.getMessage().substring(entire.length());

        // This is horribly tied to the grammar, but what can I do?
        mess = mess.replaceAll(":", " one of:");
        mess = mess.replaceAll("'", "");
        mess = mess.replaceAll(" and", " &&");
        mess = mess.replaceAll(" or", " ||");
        mess = mess.replaceAll(" xor", " ^");
        mess = mess.replaceAll(" not", " !");
        mess = mess.replaceAll(" identifier", " <identifier>");
        mess = mess.replaceAll(" stringliteral", " <string>");
        mess = mess.replaceAll(" charliteral", " <character>");
        mess = mess.replaceAll(" bracket pair", " [ ]");
        mess = mess.replaceAll(" intnumber", " <integer>");
        mess = mess.replaceAll(" dpnumber", " <double>");

        String[] mess2 = pe.getToken().getText().trim().length() > 0 ? new String[] {
                mess,
                "You have invalid syntax with this text: "
                        + pe.getToken().getText().trim() }
                : new String[] { mess, "You have invalid syntax" };
        return new SourceCodeException(mess2, new int[] { ln, pos, len }, null,
                -1);
    }

    /**
     * Wraps up a LexerException ino a SourceCodeException.
     * 
     * @param le
     * @return
     */
    public static SourceCodeException wrapLexerException(LexerException le) {
        int ln;
        int pos;
        Pattern p = Pattern.compile("\\[(\\d+),(\\d+)\\]");

        Matcher m = p.matcher(le.getMessage());
        if (m.find()) {
            ln = Integer.parseInt(m.group(1));
            pos = Integer.parseInt(m.group(2));
        } else {
            throw new KenyaInternalError("m didn't match! " + le.getMessage());
        }

        String entire = m.group(0);

        String mess = le.getMessage().substring(entire.length());
        String[] mess2 = { mess, "You have invalid syntax" };
        return new SourceCodeException(mess2, new int[] { ln, pos, 1 }, null,
                -1);
    }

    public static SourceCodeException noValidMainFunction() {

        String[] mess = {
                "Exception in thread \"main\" java.lang.NoSuchMethodError: main",
                "You need either a method void main(); or void main(String[] args)" };

        return new SourceCodeException(mess, new int[] { 0, 0, 1 }, null, -2);
    }

    public static SourceCodeException badClassName(ISourceCodeLocation pos,
            String name) {

        String[] mess = {
                "<none>",
                "The class name "
                        + name
                        + " is so commonly used in java that using it here will create a lot of confusion" };

        return new SourceCodeException(mess, new int[] { pos.getLineNumber(),
                pos.getColumnNumber(), pos.getTokenLength() }, null, -3);
    }

    public static SourceCodeException badClassBuiltinName(
            ISourceCodeLocation pos, String name) {

        String[] mess = {
                "<none>",
                "The class name "
                        + name
                        + " is is a reserved class name from one of the built-in methods you are using" };

        return new SourceCodeException(mess, new int[] { pos.getLineNumber(),
                pos.getColumnNumber(), pos.getTokenLength() }, null, -4);
    }

    public static SourceCodeException badEnumName(ISourceCodeLocation pos,
            String name) {

        String[] mess = {
                "<none>",
                "The enum name "
                        + name
                        + " is so commonly used in java that using it here will create a lot of confusion" };

        return new SourceCodeException(mess, new int[] { pos.getLineNumber(),
                pos.getColumnNumber(), pos.getTokenLength() }, null, -5);
    }

    public static SourceCodeException badEnumBuiltinName(
            ISourceCodeLocation pos, String name) {

        String[] mess = {
                "<none>",
                "The enum name "
                        + name
                        + " is a reserved class name from one of the built-in methods you are using" };

        return new SourceCodeException(mess, new int[] { pos.getLineNumber(),
                pos.getColumnNumber(), pos.getTokenLength() }, null, -6);
    }

    //MARKER BEGIN INJECT
	/** AutoGenerated Method, errID = 1 */
	static final int ClassConflictBasicType = 1;
	public static void throwClassConflictBasicType(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("<identifier> expected");
		
		humanSB.append("You have tried to define a Class Type with a name which is reserved for Kenya's Basic Type ");
		humanSB.append(type);
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, ClassConflictBasicType);
	}

	/** AutoGenerated Method, errID = 2 */
	static final int DuplicateClassName = 2;
	public static void throwDuplicateClassName(int ln, int pos, int len, int[][] linkPos, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("duplicate class: ");
		javacSB.append(type);
		
		humanSB.append("You have tried to define a Class Type with a name that is already defined as a Class Type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateClassName);
	}

	/** AutoGenerated Method, errID = 3 */
	static final int DuplicateClassEnumName = 3;
	public static void throwDuplicateClassEnumName(int ln, int pos, int len, int[][] linkPos, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("duplicate class: ");
		javacSB.append(type);
		
		humanSB.append("You have tried to define a Class Type with a name that is already defined as an Enum Type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateClassEnumName);
	}

	/** AutoGenerated Method, errID = 4 */
	static final int EnumConflictBasicType = 4;
	public static void throwEnumConflictBasicType(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("<identifier> expected");
		
		humanSB.append("You have tried to define an Enum Type with a name which is reserved for Kenya's Basic Type ");
		humanSB.append(type);
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, EnumConflictBasicType);
	}

	/** AutoGenerated Method, errID = 5 */
	static final int DuplicateEnumName = 5;
	public static void throwDuplicateEnumName(int ln, int pos, int len, int[][] linkPos, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("duplicate class: ");
		javacSB.append(type);
		
		humanSB.append("You have tried to define an Enum Type with a name that is already defined as an Enum Type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateEnumName);
	}

	/** AutoGenerated Method, errID = 6 */
	static final int DuplicateEnumClassName = 6;
	public static void throwDuplicateEnumClassName(int ln, int pos, int len, int[][] linkPos, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("duplicate class: ");
		javacSB.append(type);
		
		humanSB.append("You have tried to define an Enum Type with a name that is already defined as a Class Type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateEnumClassName);
	}

	/** AutoGenerated Method, errID = 7 */
	static final int VoidConstant = 7;
	public static void throwVoidConstant(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'(' expected.");
		
		humanSB.append("You have tried to create a constant of type 'void'");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, VoidConstant);
	}

	/** AutoGenerated Method, errID = 8 */
	static final int VoidParamType = 8;
	public static void throwVoidParamType(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("illegal start of type");
		
		humanSB.append("You have tried to create a class using 'void' as one of its parameter types");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, VoidParamType);
	}

	/** AutoGenerated Method, errID = 9 */
	static final int DuplicateVariable = 9;
	public static void throwDuplicateVariable(int ln, int pos, int len, int[][] linkPos, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append(name);
		javacSB.append(" is already defined in ");
		javacSB.append(scope);
		
		humanSB.append("You have tried to create a variable with the the name ( ");
		humanSB.append(name);
		humanSB.append(" ), but there is already a variable in this scope ( ");
		humanSB.append(scope);
		humanSB.append(" ) with that name");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateVariable);
	}

	/** AutoGenerated Method, errID = 10 */
	static final int DuplicateEnumChild = 10;
	public static void throwDuplicateEnumChild(int ln, int pos, int len, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append(name);
		javacSB.append(" is already defined in ");
		javacSB.append(scope);
		
		humanSB.append("You have tried to create an enum constant with the same name as another enum constant in this enum declaration");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateEnumChild);
	}

	/** AutoGenerated Method, errID = 11 */
	static final int DuplicateFunction = 11;
	public static void throwDuplicateFunction(int ln, int pos, int len, int[][] linkPos, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append(name);
		javacSB.append(" is already defined in ");
		javacSB.append(scope);
		
		humanSB.append("You have tried to declare a function which cannot be told apart from an existing function");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateFunction);
	}

	/** AutoGenerated Method, errID = 12 */
	static final int DuplicateFunction_Builtin = 12;
	public static void throwDuplicateFunction_Builtin(int ln, int pos, int len, String name){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append(name);
		javacSB.append(" is already defined in <**Kenya Builtin Methods**>");
		
		humanSB.append("You have tried to declare a function which cannot be told apart from a Kenya built-in function");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, DuplicateFunction_Builtin);
	}

	/** AutoGenerated Method, errID = 13 */
	static final int ClassConstantError = 13;
	public static void throwClassConstantError(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("<none>");
		
		humanSB.append("You have tried to declare a constant of Class Type, but Kenya only supports constants of Basic Type (boolean, char, int, double, String)");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, ClassConstantError);
	}

	/** AutoGenerated Method, errID = 14 */
	static final int IncompatibleTypes_ArrayIndex = 14;
	public static void throwIncompatibleTypes_ArrayIndex(int ln, int pos, int len, KType rhs){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(rhs);
		javacSB.append("" + IJavaCode.NEWLINE + "required: int");
		
		humanSB.append("You are trying to use a ");
		humanSB.append(rhs);
		humanSB.append(" type as an index into an array, however you can only index into an array with an expression of type int");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_ArrayIndex);
	}

	/** AutoGenerated Method, errID = 15 */
	static final int IncompatibleTypes_Assignment = 15;
	public static void throwIncompatibleTypes_Assignment(int ln, int pos, int len, KType lhs, KType rhs){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(rhs);
		javacSB.append("" + IJavaCode.NEWLINE + "required: ");
		javacSB.append(lhs);
		
		humanSB.append("You are attempting an assignment operation, but have tried to assign an expression of type ");
		humanSB.append(rhs);
		humanSB.append(" to an expression of type ");
		humanSB.append(lhs);
		humanSB.append(". These two types are not 'assignment compatible'");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_Assignment);
	}

	/** AutoGenerated Method, errID = 16 */
	static final int IncompatibleTypes_IfBoolean = 16;
	public static void throwIncompatibleTypes_IfBoolean(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(type);
		javacSB.append("" + IJavaCode.NEWLINE + "required: boolean");
		
		humanSB.append("You are attempting to use a ");
		humanSB.append(type);
		humanSB.append(" as the test in an if statement, however if statements can only test boolean expressions");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_IfBoolean);
	}

	/** AutoGenerated Method, errID = 17 */
	static final int IncompatibleTypes_WhileBoolean = 17;
	public static void throwIncompatibleTypes_WhileBoolean(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(type);
		javacSB.append("" + IJavaCode.NEWLINE + "required: boolean");
		
		humanSB.append("You are attempting to use a ");
		humanSB.append(type);
		humanSB.append(" as the test in a while loop, however while loops can only test boolean expressions");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_WhileBoolean);
	}

	/** AutoGenerated Method, errID = 18 */
	static final int IncompatibleTypes_ForBoolean = 18;
	public static void throwIncompatibleTypes_ForBoolean(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(type);
		javacSB.append("" + IJavaCode.NEWLINE + "required: boolean");
		
		humanSB.append("You are attempting to use a ");
		humanSB.append(type);
		humanSB.append(" as the test in a for loop, however for loops can only test boolean expressions");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_ForBoolean);
	}

	/** AutoGenerated Method, errID = 19 */
	static final int IncompatibleTypes_AssertBoolean = 19;
	public static void throwIncompatibleTypes_AssertBoolean(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(type);
		javacSB.append("" + IJavaCode.NEWLINE + "required: boolean");
		
		humanSB.append("You are attempting to use a ");
		humanSB.append(type);
		humanSB.append(" as the test in an assert statement, however assertions can only test boolean expressions");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_AssertBoolean);
	}

	/** AutoGenerated Method, errID = 20 */
	static final int IncompatibleTypes_SwitchTopLevel = 20;
	public static void throwIncompatibleTypes_SwitchTopLevel(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(type);
		javacSB.append("" + IJavaCode.NEWLINE + "required: int");
		
		humanSB.append("You are attempting to switch on a ");
		humanSB.append(type);
		humanSB.append(", however in Kenya you can only switch on char, int, or enum type expressions");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_SwitchTopLevel);
	}

	/** AutoGenerated Method, errID = 21 */
	static final int IncompatibleTypes_SwitchCaseLevel = 21;
	public static void throwIncompatibleTypes_SwitchCaseLevel(int ln, int pos, int len, KType swType, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(type);
		javacSB.append("" + IJavaCode.NEWLINE + "required: int");
		
		humanSB.append("You have a switch case of type ");
		humanSB.append(type);
		humanSB.append(" when the main switch itself is of type ");
		humanSB.append(swType);
		humanSB.append(", these are not compatible");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_SwitchCaseLevel);
	}

	/** AutoGenerated Method, errID = 22 */
	static final int IncompatibleTypes_Return = 22;
	public static void throwIncompatibleTypes_Return(int ln, int pos, int len, KType returnValue, KType signatureValue){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incompatible types" + IJavaCode.NEWLINE + "found: ");
		javacSB.append(returnValue);
		javacSB.append("" + IJavaCode.NEWLINE + "required: ");
		javacSB.append(signatureValue);
		
		humanSB.append("You are attempting to return a type of ");
		humanSB.append(returnValue);
		humanSB.append(" in a function that you have declared to be of type ");
		humanSB.append(signatureValue);
		humanSB.append(", these are not compatible");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IncompatibleTypes_Return);
	}

	/** AutoGenerated Method, errID = 23 */
	static final int Return_MissingReturnValue = 23;
	public static void throwReturn_MissingReturnValue(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("missing return value");
		
		humanSB.append("You are attempting to return from a function of type ");
		humanSB.append(type);
		humanSB.append(", but are not returning something of that type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Return_MissingReturnValue);
	}

	/** AutoGenerated Method, errID = 24 */
	static final int Return_ValueInVoid = 24;
	public static void throwReturn_ValueInVoid(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot return a value from a method whose return type is void");
		
		humanSB.append("You are attempting to return a value (of type ");
		humanSB.append(type);
		humanSB.append(") from a method that you have declared to be void (i.e. does not return anything)");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Return_ValueInVoid);
	}

	/** AutoGenerated Method, errID = 25 */
	static final int Switch_EnumCaseExpected = 25;
	public static void throwSwitch_EnumCaseExpected(int ln, int pos, int len, KType swType){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("an enum switch case label must be the unqualified name of an enumeration constant");
		
		humanSB.append("You are switching on the enum type ");
		humanSB.append(swType);
		humanSB.append(", therefore all the case statements of this switch must be just the names of members of this enum");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Switch_EnumCaseExpected);
	}

	/** AutoGenerated Method, errID = 26 */
	static final int Switch_DuplicateDefault = 26;
	public static void throwSwitch_DuplicateDefault(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("duplicate default label");
		
		humanSB.append("You have created a switch statement with multiple default labels, however a switch statement can only have one default label");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Switch_DuplicateDefault);
	}

	/** AutoGenerated Method, errID = 27 */
	static final int FunctionApplication_NoDots = 27;
	public static void throwFunctionApplication_NoDots(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("<none>");
		
		humanSB.append("Since Kenya does not support functions inside classes, there will never be a case where you will need a '.' to find them");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, FunctionApplication_NoDots);
	}

	/** AutoGenerated Method, errID = 28 */
	static final int FunctionApplication_Void = 28;
	public static void throwFunctionApplication_Void(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'void' type not allowed here");
		
		humanSB.append("You are attempting to call a function, but you are passing a void type in as one of the parameters");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, FunctionApplication_Void);
	}

	/** AutoGenerated Method, errID = 29 */
	static final int VariableDeclaration_Void = 29;
	public static void throwVariableDeclaration_Void(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("illegal start of expression");
		
		humanSB.append("You are attempting to create a variable of type 'void' ( what are you planning to do with it? :) )");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, VariableDeclaration_Void);
	}

	/** AutoGenerated Method, errID = 30 */
	static final int ArrayDeclaration_TooManyBraces = 30;
	public static void throwArrayDeclaration_TooManyBraces(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("illegal initialiser for ");
		javacSB.append(type);
		
		humanSB.append("You have braces in an array declaration going to too high a depth");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, ArrayDeclaration_TooManyBraces);
	}

	/** AutoGenerated Method, errID = 31 */
	static final int Binop_InvalidApplication = 31;
	public static void throwBinop_InvalidApplication(int ln, int pos, int len, KType lhs, String name, KType rhs){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("operator ");
		javacSB.append(name);
		javacSB.append(" cannot be applied to ");
		javacSB.append(lhs);
		javacSB.append(",");
		javacSB.append(rhs);
		
		humanSB.append("You have tried to use the operator ");
		humanSB.append(name);
		humanSB.append(" in the form '");
		humanSB.append(lhs);
		humanSB.append(" ");
		humanSB.append(name);
		humanSB.append(" ");
		humanSB.append(rhs);
		humanSB.append("', which it cannot be used for");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Binop_InvalidApplication);
	}

	/** AutoGenerated Method, errID = 32 */
	static final int Unop_InvalidApplication = 32;
	public static void throwUnop_InvalidApplication(int ln, int pos, int len, String name, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("operator ");
		javacSB.append(name);
		javacSB.append(" cannot be applied to ");
		javacSB.append(type);
		
		humanSB.append("You have tried to use the unary operator ");
		humanSB.append(name);
		humanSB.append(" on a ");
		humanSB.append(type);
		humanSB.append(", this cannot be done");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Unop_InvalidApplication);
	}

	/** AutoGenerated Method, errID = 33 */
	static final int Incomparable = 33;
	public static void throwIncomparable(int ln, int pos, int len, KType lhs, String name, KType rhs){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("incomparable types: ");
		javacSB.append(lhs);
		javacSB.append(" and ");
		javacSB.append(rhs);
		
		humanSB.append("You have tried to compare a ");
		humanSB.append(lhs);
		humanSB.append(" with a ");
		humanSB.append(rhs);
		humanSB.append(", this cannot be done");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Incomparable);
	}

	/** AutoGenerated Method, errID = 34 */
	static final int Array_TooManyAccesses = 34;
	public static void throwArray_TooManyAccesses(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("array required, but ");
		javacSB.append(type);
		javacSB.append(" found");
		
		humanSB.append("You have indexed into an array too many times");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Array_TooManyAccesses);
	}

	/** AutoGenerated Method, errID = 35 */
	static final int Lost_Variable = 35;
	public static void throwLost_Variable(int ln, int pos, int len, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot find symbol" + IJavaCode.NEWLINE + "symbol: ");
		javacSB.append(name);
		javacSB.append("  " + IJavaCode.NEWLINE + "location: ");
		javacSB.append(scope);
		
		humanSB.append("You are trying to reference a variable that isn't valid ( not declared / wrong scope / bad initialiser? )");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Lost_Variable);
	}

	/** AutoGenerated Method, errID = 36 */
	static final int Lost_Class = 36;
	public static void throwLost_Class(int ln, int pos, int len, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot find symbol" + IJavaCode.NEWLINE + "symbol: ");
		javacSB.append(name);
		javacSB.append("  " + IJavaCode.NEWLINE + "location: ");
		javacSB.append(scope);
		
		humanSB.append("You are trying to reference a Class Type that hasn't been declared");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Lost_Class);
	}

	/** AutoGenerated Method, errID = 37 */
	static final int Lost_Function = 37;
	public static void throwLost_Function(int ln, int pos, int len, String args, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot find symbol" + IJavaCode.NEWLINE + "symbol: method ");
		javacSB.append(name);
		javacSB.append(args);
		javacSB.append(" " + IJavaCode.NEWLINE + "location: ");
		javacSB.append(scope);
		
		humanSB.append("You are trying to call a function that you have not declared, or is not a built in function");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Lost_Function);
	}

	/** AutoGenerated Method, errID = 38 */
	static final int Lost_Function_Possible = 38;
	public static void throwLost_Function_Possible(int ln, int pos, int len, int[][] linkPos, String args, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot find symbol" + IJavaCode.NEWLINE + "symbol: method ");
		javacSB.append(name);
		javacSB.append(args);
		javacSB.append(" " + IJavaCode.NEWLINE + "location: ");
		javacSB.append(scope);
		
		humanSB.append("You are trying to call a function that you have not declared, though it is possible you have the wrong types of parameters to this function call");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Lost_Function_Possible);
	}

	/** AutoGenerated Method, errID = 39 */
	static final int BasicType_Dereference = 39;
	public static void throwBasicType_Dereference(int ln, int pos, int len, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append(type);
		javacSB.append(" cannot be dereferenced.");
		
		humanSB.append("You are attempting to apply a '.' to a ");
		humanSB.append(type);
		humanSB.append(", this cannot be done as Basic Types have no members");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, BasicType_Dereference);
	}

	/** AutoGenerated Method, errID = 40 */
	static final int Array_Dereference = 40;
	public static void throwArray_Dereference(int ln, int pos, int len, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot find symbol" + IJavaCode.NEWLINE + "symbol: variable ");
		javacSB.append(name);
		javacSB.append(" " + IJavaCode.NEWLINE + "location: ");
		javacSB.append(scope);
		
		humanSB.append("You can only use a '.' on an array to access its length (with .length). You cannot use '.' on an array for any other purpose in Kenya");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Array_Dereference);
	}

	/** AutoGenerated Method, errID = 41 */
	static final int TypeParam_Dereference = 41;
	public static void throwTypeParam_Dereference(int ln, int pos, int len, String name){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot find symbol" + IJavaCode.NEWLINE + "symbol: variable ");
		javacSB.append(name);
		javacSB.append(" " + IJavaCode.NEWLINE + "location: class java.lang.Object");
		
		humanSB.append("You have tried to dereference a parameterised typed, but you cannot know (in Kenya) what their members could be");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_Dereference);
	}

	/** AutoGenerated Method, errID = 42 */
	static final int TypeParam_WithTypeParam = 42;
	public static void throwTypeParam_WithTypeParam(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'>' expected");
		
		humanSB.append("A Type Parameter cannot have Type Parameters");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_WithTypeParam);
	}

	/** AutoGenerated Method, errID = 43 */
	static final int TypeParam_MisMatch = 43;
	public static void throwTypeParam_MisMatch(int ln, int pos, int len, KClassType klass, int num){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("<possible> wrong number of type arguments; required ");
		javacSB.append(num);
		
		humanSB.append("This is not a valid reference to class ");
		humanSB.append(klass);
		humanSB.append(" as there is a mismatch with its Type Parameters");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_MisMatch);
	}

	/** AutoGenerated Method, errID = 44 */
	static final int TypeParam_OnEnum = 44;
	public static void throwTypeParam_OnEnum(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'{' expected");
		
		humanSB.append("An enum declaration cannot have Type Parameters");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_OnEnum);
	}

	/** AutoGenerated Method, errID = 45 */
	static final int TypeParam_WithArrayBrackets = 45;
	public static void throwTypeParam_WithArrayBrackets(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'>' expected");
		
		humanSB.append("A Type Parameter cannot have array index brackets attached to it");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_WithArrayBrackets);
	}

	/** AutoGenerated Method, errID = 46 */
	static final int TypeParam_Duplicate = 46;
	public static void throwTypeParam_Duplicate(int ln, int pos, int len, String name, String scope){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append(name);
		javacSB.append(" is already defined in ");
		javacSB.append(scope);
		
		humanSB.append("You have duplicate Type Parameters with the name ");
		humanSB.append(name);
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_Duplicate);
	}

	/** AutoGenerated Method, errID = 47 */
	static final int TypeParam_BasicType = 47;
	public static void throwTypeParam_BasicType(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("< identifier > expected");
		
		humanSB.append("You cannot use the name of a Basic Type as a Type Parameter");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParam_BasicType);
	}

	/** AutoGenerated Method, errID = 48 */
	static final int VoidArrayType = 48;
	public static void throwVoidArrayType(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'void' type not allowed here");
		
		humanSB.append("You are trying to make a void array, what are you going to store in it? :)");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, VoidArrayType);
	}

	/** AutoGenerated Method, errID = 49 */
	static final int Ambiguous_Function = 49;
	public static void throwAmbiguous_Function(int ln, int pos, int len, int[][] linkPos, String argType1, String argType2, String global, String name){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("reference to ");
		javacSB.append(name);
		javacSB.append(" is ambiguous, both method ");
		javacSB.append(name);
		javacSB.append(argType1);
		javacSB.append(" in ");
		javacSB.append(global);
		javacSB.append(" and method ");
		javacSB.append(name);
		javacSB.append(argType2);
		javacSB.append(" in ");
		javacSB.append(global);
		javacSB.append(" match");
		
		humanSB.append("You are calling a function, however there are several that match and Kenya isn't sure which one you mean");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Ambiguous_Function);
	}

	/** AutoGenerated Method, errID = 50 */
	static final int Generic_Array_Creation = 50;
	public static void throwGeneric_Array_Creation(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("generic array creation");
		
		humanSB.append("You cannot create arrays which are of a Parameterized Type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Generic_Array_Creation);
	}

	/** AutoGenerated Method, errID = 51 */
	static final int AssignToValue = 51;
	public static void throwAssignToValue(int ln, int pos, int len, String val){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("unexpected type" + IJavaCode.NEWLINE + "required: variable " + IJavaCode.NEWLINE + "found: value");
		
		humanSB.append("You have tried to assign a value ( ");
		humanSB.append(val);
		humanSB.append(" ) to something new");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, AssignToValue);
	}

	/** AutoGenerated Method, errID = 52 */
	static final int AssignToConstant = 52;
	public static void throwAssignToConstant(int ln, int pos, int len, int[][] linkPos, KVariable name){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot assign a value to final variable ");
		javacSB.append(name);
		
		humanSB.append("You have tried to assign a constant variable a new value ( think about it... )");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, AssignToConstant);
	}

	/** AutoGenerated Method, errID = 53 */
	static final int UnopValuePP = 53;
	public static void throwUnopValuePP(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("unexpected type" + IJavaCode.NEWLINE + "required: variable " + IJavaCode.NEWLINE + "found: value");
		
		humanSB.append("You cannot apply ++ to a non-variable.");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, UnopValuePP);
	}

	/** AutoGenerated Method, errID = 54 */
	static final int UnopValueMM = 54;
	public static void throwUnopValueMM(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("unexpected type" + IJavaCode.NEWLINE + "required: variable " + IJavaCode.NEWLINE + "found: value");
		
		humanSB.append("You cannot apply -- to a non-variable.");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, UnopValueMM);
	}

	/** AutoGenerated Method, errID = 55 */
	static final int UninitialisedVariable = 55;
	public static void throwUninitialisedVariable(int ln, int pos, int len, int[][] linkPos, String name){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("variable ");
		javacSB.append(name);
		javacSB.append(" might not have been initialised");
		
		humanSB.append("You are trying to use a variable that it is possible you have not given a value to yet");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, UninitialisedVariable);
	}

	/** AutoGenerated Method, errID = 56 */
	static final int Unreachable = 56;
	public static void throwUnreachable(int ln, int pos, int len, int[][] linkPos){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("unreachable statement");
		
		humanSB.append("This statement cannot be executed or reached");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, Unreachable);
	}

	/** AutoGenerated Method, errID = 57 */
	static final int SwitchNeedsConstant = 57;
	public static void throwSwitchNeedsConstant(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("constant expression required");
		
		humanSB.append("The cases for a switch statement all have to be constants at compile time, this expression is not constant");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, SwitchNeedsConstant);
	}

	/** AutoGenerated Method, errID = 58 */
	static final int IllegalBreakPlace = 58;
	public static void throwIllegalBreakPlace(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("break outside switch or loop");
		
		humanSB.append("You can only place break statements inside switch statements and while or for loops");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, IllegalBreakPlace);
	}

	/** AutoGenerated Method, errID = 59 */
	static final int MissingReturnStatement = 59;
	public static void throwMissingReturnStatement(int ln, int pos, int len, int[][] linkPos, KType type){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("missing return statement");
		
		humanSB.append("This function needs to return a variable of type ");
		humanSB.append(type);
		humanSB.append(", but not all paths through the function return a variable of that type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, MissingReturnStatement);
	}

	/** AutoGenerated Method, errID = 60 */
	static final int BadInt = 60;
	public static void throwBadInt(int ln, int pos, int len, String num){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("integer number too large: ");
		javacSB.append(num);
		
		humanSB.append("This is not a valid number for an int. ( ints can be between -2147483648 and 2147483647 )");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, BadInt);
	}

	/** AutoGenerated Method, errID = 61 */
	static final int BadDouble = 61;
	public static void throwBadDouble(int ln, int pos, int len, String num){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("floating point number too large: ");
		javacSB.append(num);
		
		humanSB.append("This is not a valid number for a double. ( doubles can be between roughly +/- 1.7976931348623157E308 )");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, BadDouble);
	}

	/** AutoGenerated Method, errID = 62 */
	static final int InvalidEnumAssig = 62;
	public static void throwInvalidEnumAssig(int ln, int pos, int len, String name){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot assign a value to final variable ");
		javacSB.append(name);
		
		humanSB.append("You cannot assign to an enum constant");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, InvalidEnumAssig);
	}

	/** AutoGenerated Method, errID = 63 */
	static final int InvalidArrayLengthAssig = 63;
	public static void throwInvalidArrayLengthAssig(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("cannot assign a value to final variable length");
		
		humanSB.append("You cannot assign to the length of an array.");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, InvalidArrayLengthAssig);
	}

	/** AutoGenerated Method, errID = 64 */
	static final int SwitchDuplicateLabel = 64;
	public static void throwSwitchDuplicateLabel(int ln, int pos, int len, int[][] linkPos){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("duplicate case label");
		
		humanSB.append("You have defined a switch case that is the same as another switch case");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, SwitchDuplicateLabel);
	}

	/** AutoGenerated Method, errID = 65 */
	static final int PossibleLossOfPrecisionCharInt = 65;
	public static void throwPossibleLossOfPrecisionCharInt(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("possible loss of precision" + IJavaCode.NEWLINE + "required: char " + IJavaCode.NEWLINE + "found: int");
		
		humanSB.append("You are assigning a char variable with something that *could* be of int type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, PossibleLossOfPrecisionCharInt);
	}

	/** AutoGenerated Method, errID = 66 */
	static final int TypeParamNoBindingFunction = 66;
	public static void throwTypeParamNoBindingFunction(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("<none applicable>");
		
		humanSB.append("You are defining a function where its return type is never specified by the parameters to this function");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, TypeParamNoBindingFunction);
	}

	/** AutoGenerated Method, errID = 67 */
	static final int VoidInAssert = 67;
	public static void throwVoidInAssert(int ln, int pos, int len){
		StringBuffer javacSB = new StringBuffer();
		StringBuffer humanSB = new StringBuffer();

		javacSB.append("'void' type not allowed here");
		
		humanSB.append("An assert message cannot be of void type");
		
		String[] out = { javacSB.toString(), humanSB.toString() };
		int[][] linkPos = null;
		int[] errPos = { ln, pos, len };
		
		throw new SourceCodeException(out, errPos, linkPos, VoidInAssert);
	}

    //MARKER END INJECT

    private int[] _errPos;

    private int[][] _linkPos;

    private String[] _messages;

    private final StringBuffer humanBuffer;

    private final StringBuffer javacBuffer;

    int _errID;

    private final Pattern linePattern = Pattern.compile("^(.+)[\n\r]*$",
            Pattern.MULTILINE);

    /**
     * Constructs a new SourceCodeException with the given message, at the
     * specified error Location, and with any additional "linked" lines which
     * may be relevent.
     * 
     * @param messages
     *            String[2] array of the messages. messages[JAVAC_MESSAGE] and
     *            messages[HUMAN_MESSAGE] are what u'd expect.
     * @param humanMessage
     *            String of the human Style message.
     * @param errPos
     *            The co-ordinates of the error ( errPos[LINE_NUM] is the line
     *            number, errPos[CHAR_NUM] is the character position ).
     * @param linkPos
     *            An array of co-ordinates of places that may be relevent
     *            linkPos[x][LINE_NUM] and linkPos[x][CHAR_NUM] as above.
     */
    SourceCodeException(String[] messages, int[] errPos, int[][] linkPos,
            int errID) {
        super();

        _messages = messages;
        _errPos = errPos;
        _linkPos = linkPos;
        _errID = errID;

        StringWriter sWriter = new StringWriter();
        PrintWriter pwriter = new PrintWriter(sWriter);
        Matcher m = linePattern.matcher(_messages[HUMAN_MESSAGE]);
        boolean first = true;
        while (m.find()) {
            if (first)
                first = false;
            else
                pwriter.println();
            pwriter.print(m.group());
        }
        humanBuffer = sWriter.getBuffer();

        sWriter = new StringWriter();
        pwriter = new PrintWriter(sWriter);
        m = linePattern.matcher(_messages[JAVAC_MESSAGE]);
        first = true;
        while (m.find()) {
            if (first)
                first = false;
            else
                pwriter.println();
            pwriter.print(m.group(1));
        }
        javacBuffer = sWriter.getBuffer();
    }

    /**
     * Overriden for the custom message. The message given is in the form:
     * <code>
     * [lineNo,charNo]
     * ( [lineNo,charNo] [lineNo,charNo] ... )
     * Javac Style error message
     * Human Style error message
     * { errID }
     * </code>
     * 
     * @see java.lang.Throwable#getMessage()
     */
    public String getMessage() {
        StringBuffer sb = new StringBuffer();

        sb.append(getErrorLocation());
        sb.append(getLinkLocations());
        sb.append(IJavaCode.NEWLINE);
        sb.append(javacBuffer);
        sb.append(IJavaCode.NEWLINE);
        sb.append(humanBuffer);
        sb.append(IJavaCode.NEWLINE);
        sb.append("{ ");
        sb.append(_errID);
        sb.append(" }");
        return sb.toString();
    }

    /**
     * @return String javac style error message for this error.
     */
    public StringReader getJavacStyleMessage() {
        return new StringReader(javacBuffer.toString());
    }

    /**
     * @return String human style error message for this error.
     */
    public StringReader getHumanStyleMessage() {
        return new StringReader(humanBuffer.toString());

    }

    /**
     * @return String representing where this error is in the format
     *         [lineNo,charNo]
     */
    public String getErrorLocation() {
        StringBuffer sb = new StringBuffer();

        sb.append("[");
        sb.append(_errPos[LINE_NUM]);
        sb.append(",");
        sb.append(_errPos[CHAR_NUM]);
        sb.append("]");

        return sb.toString();
    }

    /**
     * @return A string representing the "linked errors" to this error in the
     *         format: "" if there are none. ( [lineNo,charNo] [lineNo,charNo]
     *         ... )
     */
    public String getLinkLocations() {
        if (_linkPos == null || _linkPos.length == 0) {
            return "";
        }

        StringBuffer sb = new StringBuffer();
        sb.append("(");
        for (int i = 0; i < _linkPos.length; i++) {
            sb.append(" ");
            sb.append("[");
            sb.append(_linkPos[i][LINE_NUM]);
            sb.append(",");
            sb.append(_linkPos[i][CHAR_NUM]);
            sb.append("]");
        }
        sb.append(" )");

        return sb.toString();
    }

    /**
     * @see kenya.sourceCodeInformation.interfaces.ISourceCodeInformation#getLocation()
     */
    public ISourceCodeLocation getLocation() {
        return new SourceCodeLocation(_errPos[LINE_NUM], _errPos[CHAR_NUM],
                _errPos[TOKEN_LEN]);
    }

    /**
     * @see kenya.sourceCodeInformation.interfaces.ISourceCodeInformation#getLinkedInformation()
     */
    public ISourceCodeLocation[] getLinkedInformation() {

        if (_linkPos == null) {
            return new ISourceCodeLocation[0];
        }

        ISourceCodeLocation[] out = new ISourceCodeLocation[_linkPos.length];
        for (int i = 0; i < _linkPos.length; i++) {
            out[i] = new SourceCodeLocation(_linkPos[i][LINE_NUM],
                    _linkPos[i][CHAR_NUM], _linkPos[i][TOKEN_LEN]);
        }

        return out;
    }

    /**
     * @see kenya.sourceCodeInformation.interfaces.ISourceCodeInformation#getErrorNumber()
     */
    public int getInformationNumber() {
        return _errID;
    }

}