/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Aug-2004 by toa02
 */
package kenya.passes;

import kenya.errors.KenyaPreconditionError;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.types.tables.VariableLookupTable;
import mediator.stackMachine.IStackMachineInformationManager;
import mediator.util.InformationHolder;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AConstDecDeclaration;
import minijava.node.AVarDecInnerDeclaration;
import minijava.node.Node;

import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.ops.StackOpsFactory;
import org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure;
import org.wellquite.kenya.stackMachine.types.ClassTypeFactory;
import org.wellquite.kenya.stackMachine.types.NullTypeFactory;
import org.wellquite.kenya.stackMachine.types.PrimitiveTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IBuildableClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;

/**
 * @author toa02
 */
class RestrictedStackMachineBuilder extends DepthFirstAdapter {

    final IInterpretedClass _clazz;

    private final IBuildableClosure _initClosure;
    
    private final StackMachineBuilder _smb;
    
    private final InformationHolder _ih;
    private final IStackMachineInformationManager _isim;
    
    private final IClass _cIClass;

    private RestrictedStackMachineBuilder(IInterpretedClass clazz,
            IBuildableClosure initClosure, IStackMachineInformationManager isim, 
            InformationHolder ih, IClass cIClass) {
        _clazz = clazz;
        _initClosure = initClosure;
        _isim = isim;
        _ih = ih;
        _cIClass = cIClass;
        _smb = new StackMachineBuilder(isim,ih,false,_initClosure, cIClass);
    }

    static void doClassVariable(IInterpretedClass theClass,
            IBuildableClosure initClosure, Node node, IStackMachineInformationManager isim, InformationHolder ih, IClass cClass) {
        if (!(node instanceof AVarDecInnerDeclaration
                || node instanceof AConstDecDeclaration || node instanceof AArrayDecInnerDeclaration)) {
            throw new IllegalArgumentException(
                    "Node but be a variable, constant or array declaration: "
                            + node);
        }

        RestrictedStackMachineBuilder rsmb = new RestrictedStackMachineBuilder(
                theClass, initClosure, isim, ih, cClass);
        
        node.apply(rsmb);
        
    }
    
    /**
     * These are variable declarations inside classes.
     * @see minijava.analysis.Analysis#caseAVarDecInnerDeclaration(minijava.node.AVarDecInnerDeclaration)
     */
    public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
    
        String name = node.getIdentifier().getText().trim();
        IVariable decl;
        
        outLoop: {
            for( int i = 0 ; i < _cIClass.getDeclarations().length ; i++ ){
                if( _cIClass.getDeclarations()[i].getName().equals(name)){
                    decl = _cIClass.getDeclarations()[i];
                    break outLoop;
                }
        	}
            
            throw KenyaPreconditionError.get();
        }
        
        final String smName = _isim.lookupVariable(decl);
        
        VariableLookupTable vlt = new VariableLookupTable( _cIClass.getDeclarations());
        vlt.push();
        _smb.setVariableLookupTable(vlt);
        
        IClass type = decl.getType();
        
        if ( node.getInitialiser() != null ) {
            node.getInitialiser().apply(_smb);
        } else {
            
            if (type.getName().equals("boolean")) {
                _initClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(false)));
            } else if (type.getName().equals("char")) {
                _initClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType('\0')));
            } else if (type.getName().equals("int")) {
                _initClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0)));
            } else if (type.getName().equals("double")) {
                _initClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0.0)));
            } else if( type.isClass() || type.isEnum() || type.isArray()  || type.isParam() || type.getName().equals("String") ){
                _initClosure.addClosure( StackOpsFactory.push(
                        NullTypeFactory.createNullType()));
            } else{
                throw KenyaPreconditionError.get();
            }
            
        }
        
        _initClosure.addClosure(StackOpsFactory.storeNewVariable(smName,false));
        _initClosure.addClosure(StackOpsFactory.fetch(smName));
    
        _initClosure.addClosure( new AbstractAtomicClosure(){
            public void execute(StackMachine sm){
                _clazz.addMutableInstanceVariable(smName, sm.pop() );        
            }
        });
        
    }
    
    /**
     * These are array declarations inside classes
     * @see minijava.analysis.Analysis#caseAArrayDecInnerDeclaration(minijava.node.AArrayDecInnerDeclaration)
     */
    public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
        
        String name = node.getIdentifier().getText().trim();
        IVariable decl;
        
        outLoop: {
            for( int i = 0 ; i < _cIClass.getDeclarations().length ; i++ ){
                if( _cIClass.getDeclarations()[i].getName().equals(name)){
                    decl = _cIClass.getDeclarations()[i];
                    break outLoop;
                }
        	}
            
            throw KenyaPreconditionError.get();
        }
        
        final String smName = _isim.lookupVariable(decl);
        
        VariableLookupTable vlt = new VariableLookupTable( _cIClass.getDeclarations());
        vlt.push();
        _smb.setVariableLookupTable(vlt);
        if (node.getArrayInitialiser() != null) {
            node.getArrayInitialiser().apply(_smb);
        } else {
            _initClosure.addClosure( StackOpsFactory.push(NullTypeFactory.createNullType()));
        }
        
        _initClosure.addClosure(StackOpsFactory.storeNewVariable(smName,false));
        _initClosure.addClosure(StackOpsFactory.fetch(smName));
        
        _initClosure.addClosure( new AbstractAtomicClosure(){
            public void execute(StackMachine sm){
                _clazz.addMutableInstanceVariable(smName, sm.pop() );        
            }
        });
    }
    
    public void caseAConstDecDeclaration(AConstDecDeclaration node) {
        inAConstDecDeclaration(node);

        String name = node.getIdentifier().getText().trim();
        
        IVariable cons7 = _ih.getConstantByName(name);
        String smName = _isim.lookupVariable(cons7);
        
        _initClosure.addClosure(StackOpsFactory.push(ClassTypeFactory.createClassStaticType(_clazz)));
        
        VariableLookupTable vlt = new VariableLookupTable(_ih.getConstants());
        vlt.push();

        _smb.setVariableLookupTable(vlt);
        
        node.getInitialiser().apply(_smb);

        _initClosure.addClosure(StackOpsFactory.declareStaticImmutable(smName));
            
        outAConstDecDeclaration(node);
    }
    
        

}