/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

 /* Created on 10-Aug-2004 by toa02
 *
 */
package kenya.passes;

import kenya.builtIns.SMLibraryContainer;
import kenya.errors.KenyaBadLocationError;
import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import kenya.types.KBasicType;
import kenya.types.KFunction;
import kenya.types.tables.VariableLookupTable;
import mediator.stackMachine.IStackMachineInformationManager;
import mediator.util.AVariableInstance;
import mediator.util.InformationHolder;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.*;

import org.wellquite.kenya.stackMachine.InterpretedBuildableClosure;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.ops.ArrayOpsFactory;
import org.wellquite.kenya.stackMachine.ops.ControlFlowOpsFactory;
import org.wellquite.kenya.stackMachine.ops.LogicalOpsFactory;
import org.wellquite.kenya.stackMachine.ops.NumericOpsFactory;
import org.wellquite.kenya.stackMachine.ops.PrimitiveTypeConversionOpsFactory;
import org.wellquite.kenya.stackMachine.ops.StackOpsFactory;
import org.wellquite.kenya.stackMachine.ops.StringOpsFactory;
import org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure;
import org.wellquite.kenya.stackMachine.types.ClassTypeFactory;
import org.wellquite.kenya.stackMachine.types.EnumTypeFactory;
import org.wellquite.kenya.stackMachine.types.NullTypeFactory;
import org.wellquite.kenya.stackMachine.types.PrimitiveTypeFactory;
import org.wellquite.kenya.stackMachine.types.StringTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IBuildableClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedEnumeration;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * @author toa02 Hopefully the last of these tree walkers I need to write. 
 * Woo hoo!
 */
public class StackMachineBuilder extends DepthFirstAdapter {

    // The current closure we are building, null if not building a closure
    private IBuildableClosure _cClosure;

    // The current method closure we are in, null if not inside a method closure
    private IBuildableClosure _methodClosure;

    // The current method we are in, null if not inside a function
    private IFunction _cFunction;

    // The current class we are in, null if not building a class (or are
    // building the non-static class - null )
    private IInterpretedClass _cClass;

    // The current IClass we are in (if we are in one of course )
    private IClass _cIClass;

    // The current breakable closure we are in, null if not in a breakable
    // closure
    private IBuildableClosure _breakClosure;

    // The lookup table mapping kenya names to IVariables.
    private VariableLookupTable _variableTable;

    // The "stack" variable table used inbetween names and things
    private VariableLookupTable _stackVariableTable;
    
    // The global class into which all constants and methods go
    private final IInterpretedClass _gloClass;

    // stack machine information manager
    private final IStackMachineInformationManager _stackInfoManager;

    // holder of the iclass information
    private final InformationHolder _informationHolder;

    // if this is true, the first name should do an assignment down.
    // will be set to false once the assignment has been done.
    private boolean _doAssig;

    //if this is true, markering will be done,
    //if this if false then it will not.
    private boolean _doMarkers;
    
    //Help building switch statements
    private SwitchHelper _switchHelper;
    
    public StackMachineBuilder(IStackMachineInformationManager isim,
            InformationHolder ih, boolean doMarkers) {
        _stackInfoManager = isim;
        _informationHolder = ih;
        _gloClass = isim.getGlobalClassBody();
        _doAssig = false;
        _doMarkers = doMarkers;
        _variableTable = _stackVariableTable = null;
    }
    
    StackMachineBuilder(IStackMachineInformationManager isim,
            InformationHolder ih, boolean doMarkers, IBuildableClosure cClosure,
            IClass cIClass ){
        this(isim,ih,doMarkers);
        _cClosure = cClosure;
        _cIClass = cIClass;
    }
    
    void setVariableLookupTable( VariableLookupTable vlt ){
        _variableTable = vlt;
    }

    
    private IClass getNodeType( Node node ){
     
        IClass nameType;
        
        if( _cFunction != null ){
            /* we're in a function */
            nameType = _informationHolder.getTypeFromNode(node, _cFunction.getTypeParams());
        }else if( _cIClass != null ){
            /* we're in a class */
            nameType = _informationHolder.getTypeFromNode(node, _cIClass.getTemplateParams());
        }else{
            /* we're in a constant */
            nameType = _informationHolder.getTypeFromNode(node, new IClass[]{});
        }
        
        return nameType;
    }
    
    
    /**
     * @param node
     */
    private void checkCast(Node node) {
        IClass type = getNodeType(node); 
        
        if( type.getName().equals("char") ) {
            _cClosure.addClosure(PrimitiveTypeConversionOpsFactory.toChar());
        } else if( type.getName().equals("int")) {
            _cClosure.addClosure(PrimitiveTypeConversionOpsFactory.toInt());
        } else if( type.getName().equals("double")) {
            _cClosure.addClosure(PrimitiveTypeConversionOpsFactory.toDouble());
        }
        
        
    }
    
    
    /**
     * Adds a position reached marker to the current closure.
     * 
     * @param t
     */
    private void doMarker(Token t) {
        if (_cClosure == null || !_doMarkers) {
            return;
        }
        _cClosure.addClosure(StackOpsFactory.positionReached( new
                SourceCodeLocation( t.getLine(), t.getPos(),
                        t.getText().trim().length())));

        _stackInfoManager.addCodePoint(new SourceCodeLocation( t.getLine(), 
                t.getPos(), t.getText().trim().length()));
    }

    /* ********************************************************************** */
    /* Here begins the cases, and for the last time, one by one they will ... */
    /* ********************************************************************** */

    /*
    public void caseAListStatements(AListStatements node) {
        inAListStatements(node); 

        if(node.getStatement() != null) {
            node.getStatement().apply(this); 
        } 

        if(node.getStatements() != null) {
            node.getStatements().apply(this); 
        } 
        outAListStatements(node); 
    }
    */

     /* 
     * public void caseAEmptyStatements(AEmptyStatements node) {
     * inAEmptyStatements(node); outAEmptyStatements(node); }
     */

    //Done (ish)
    public void caseAClassDecDeclaration(AClassDecDeclaration node) {
        inAClassDecDeclaration(node);

        
        String ident = node.getIdentifier().getText().trim();
        _cClosure = null;

        IClass thisClass = _informationHolder.getTypeFromName(ident);
        _cClass = _stackInfoManager.getClassBody( thisClass );

        _variableTable = new VariableLookupTable( thisClass.getDeclarations());
        _variableTable.push();
        
        _cIClass = thisClass;
        
        Object temp[] = node.getClassInnerDeclaration().toArray();
        for (int i = 0; i < temp.length; i++) {
            ((PClassInnerDeclaration) temp[i]).apply(this);
        }

        _cClass = null;
        _cIClass = null;
        
        /*
         * node.getRBrace().apply(this);
         */

        outAClassDecDeclaration(node);
    }

    //DONE
    public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
        // Don't recurse down here
    }

    public void caseAEnumList(AEnumList node) {
        throw KenyaBadLocationError.get();
    }

    public void caseACommaEnumList(ACommaEnumList node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
        inAFuncDecDeclaration(node);


        _cClosure = new InterpretedBuildableClosure();
        _methodClosure = _cClosure;
        
        _cClosure.addClosure(StackOpsFactory.startNewVariableScope());
        /*
         * Parameter passing convention:
         * 
         * void blah(arg1, arg2, arg3){ arg1 = pop(); arg2 = pop(); arg3 =
         * pop(); }
         * 
         * push arg3; push arg2; push arg1; blah()
         * 
         *  
         */

        IFunction theFunc = _stackInfoManager.getFunctionFromNode(node);

        _variableTable = new VariableLookupTable( _informationHolder.getConstants() );

        
        _cFunction = theFunc;
        IInterpretedMethod method = _stackInfoManager.getMethodBody(theFunc);

        _variableTable.push();
        
        for (int i = 0; i < theFunc.getArguments().length; i++) {
            _variableTable.add(theFunc.getArguments()[i]);
            String smName = _stackInfoManager.addNewVariable(theFunc
                    .getArguments()[i]);
            _cClosure.addClosure(StackOpsFactory
                    .storeNewVariable(smName, true));
        }

        node.getBlock().apply(this);

        _variableTable = null;
     
        
        _cClosure.addClosure(StackOpsFactory.endNewVariableScope());
        
        method.setMethodBody(_methodClosure);

        _cClosure = null;
        _cFunction = null;
        _methodClosure = null;
     
        outAFuncDecDeclaration(node);
    }

    public void caseAFormalParamList(AFormalParamList node) {
        throw new KenyaInternalError("Shouldn't be in here");
    }

    public void caseATypeName(ATypeName node) {
        throw new KenyaInternalError("Shouldn't be in here");
    }

    public void caseACommaTypeName(ACommaTypeName node) {
        throw new KenyaInternalError("Shouldn't be in here");
    }

    //DONE
    public void caseAConstDecDeclaration(AConstDecDeclaration node) {
        inAConstDecDeclaration(node);

        _variableTable = new VariableLookupTable(_informationHolder.getConstants());
        
        _cClosure = new InterpretedBuildableClosure();

        RestrictedStackMachineBuilder.doClassVariable(_gloClass, _stackInfoManager.getPreInitClosure(), 
                node, _stackInfoManager, _informationHolder, _cIClass);

        _variableTable = null;
        
        outAConstDecDeclaration(node);
    }

    //DONE
    public void caseAFunctioncallStatement(AFunctioncallStatement node) {
        inAFunctioncallStatement(node);
        
        if (node.getFunctionApplication() != null) {
            node.getFunctionApplication().apply(this);
        }

        /* do we have a return type? */
        KFunction kf = _informationHolder.getFunction((AFunctionApplication)
                node.getFunctionApplication());
        
        if( kf.getReturnType() != KBasicType.getVoid()){
            _cClosure.addClosure(StackOpsFactory.pop());
        }
        
        outAFunctioncallStatement(node);
    }

    public void caseAFunctioncallForRightStat(AFunctioncallForRightStat node){
        
        inAFunctioncallForRightStat(node);
        
        node.getFunctionApplication().apply(this);
        
        /* do we have a return type? */
        KFunction kf = _informationHolder.getFunction((AFunctionApplication)
                node.getFunctionApplication());
        
        if( kf.getReturnType() != KBasicType.getVoid()){
            _cClosure.addClosure(StackOpsFactory.pop());
        }
        
        
        
        outAFunctioncallForRightStat(node);
    }
    
    //DONE - swap field access and expression around
    public void caseAAssignmentStatement(AAssignmentStatement node) {
        inAAssignmentStatement(node);
        
        node.getAssignment().apply(this);
        
        outAAssignmentStatement(node);
    }

    public void caseAAssignment(AAssignment node){
        if (node.getExpression() != null) {
            node.getExpression().apply(this);
        }
        
        checkCast(node);
        
        if (node.getAssign() != null) {
            node.getAssign().apply(this);
        }
        
        boolean tmpDoAssig = _doAssig;
        _doAssig = true;
        if (node.getFieldAccess() != null) {
            node.getFieldAccess().apply(this);
        }
        _doAssig = tmpDoAssig;
    }
    
    // DONE
    public void caseAWhileStatement(AWhileStatement node) {
        inAWhileStatement(node);

        IBuildableClosure tmpBreak = _breakClosure;

        node.getWhile().apply(this);
        IBuildableClosure globalClosure = _cClosure;

        IBuildableClosure whileCond = new InterpretedBuildableClosure();
        _cClosure = whileCond;
        node.getBoolExpression().apply(this);

        IBuildableClosure body = new InterpretedBuildableClosure();
        _cClosure = body;
        _breakClosure = new InterpretedBuildableClosure();
        node.getBlock().apply(this);

        _breakClosure.addClosure(ControlFlowOpsFactory.whileClosure(whileCond,body));
        
        globalClosure.addClosure(_breakClosure);
        _breakClosure = tmpBreak;

        _cClosure = globalClosure;

        outAWhileStatement(node);
    }

    //DONE
    public void caseAReturnStatement(AReturnStatement node) {
        inAReturnStatement(node);

        

        if (node.getExpression() != null) {
            node.getExpression().apply(this);
        }

        node.getReturn().apply(this);
        
        checkCast(node);
        _cClosure.addClosure(_methodClosure.returnFromThisClosure());

        outAReturnStatement(node);
    }

    //DONE
    public void caseASwitchStatement(ASwitchStatement node) {
        inASwitchStatement(node);

        SwitchHelper tmpSwitchHelper = _switchHelper;
        _switchHelper = new SwitchHelper();
        
        IBuildableClosure tmpBreak = _breakClosure;
        
        if (node.getSwitch() != null) {
            node.getSwitch().apply(this);
        }

        IBuildableClosure globalClosure = _cClosure;
        
        _breakClosure = new InterpretedBuildableClosure();
        
        _cClosure = new InterpretedBuildableClosure();
        node.getBoolExpression().apply(this);
        _switchHelper.setCond(_cClosure);

        _cClosure = null;
        if (node.getSwitchBlock() != null) {
            node.getSwitchBlock().apply(this);
        }
        
        _breakClosure.addClosure(_switchHelper.getSwitchClosure());

        
        globalClosure.addClosure( _breakClosure );
        
        _breakClosure = tmpBreak;
        _switchHelper = tmpSwitchHelper;
        
        _cClosure = globalClosure;
        
        outASwitchStatement(node);
    }

    
    //DONE
    public void caseASwitchBlock(ASwitchBlock node) {
        inASwitchBlock(node);
        {
            Object temp[] = node.getPossibleCase().toArray();
            
            _switchHelper.initArrays(temp.length);
            
            for (int i = 0; i < temp.length; i++) {
                ((PPossibleCase) temp[i]).apply(this);
            }
        }
        outASwitchBlock(node);
    }

    // DONE
    public void caseACasePossibleCase(ACasePossibleCase node) {
        inACasePossibleCase(node);
        
        if( _cClosure != null ){
            throw KenyaPreconditionError.get();
        }
        
        boolean tmpMarkers = _doMarkers;
        _doMarkers = false;
        
        IBuildableClosure cond = new InterpretedBuildableClosure();
        IBuildableClosure code = new InterpretedBuildableClosure();
        
        _cClosure = cond;
        node.getBoolExpression().apply(this);

        _doMarkers = tmpMarkers;
        
        _cClosure = code;
        node.getCase().apply(this);
        node.getBlock().apply(this);
        
        _switchHelper.setNextCase(cond, code);
        
        _cClosure = null;
        
        outACasePossibleCase(node);
    }

    //DONE
    public void caseADefaultPossibleCase(ADefaultPossibleCase node) {
        inADefaultPossibleCase(node);
        
        if( _cClosure != null ){
            throw KenyaPreconditionError.get();
        }
        
        IBuildableClosure code = new InterpretedBuildableClosure();
        
        _cClosure = code;
        node.getDefault().apply(this);
        node.getBlock().apply(this);
        
        _switchHelper.setNextCase(null, code);
        _cClosure = null;
        
        outADefaultPossibleCase(node);
    }

    /**
     * DONE For loops: Translation:
     * 
     * for( init ; cond ; eval ){ body }
     * 
     * to:
     * 
     * init while( cond ){ body eval }
     * 
     * @see minijava.analysis.Analysis#caseAForStatement(minijava.node.AForStatement)
     */
    public void caseAForStatement(AForStatement node) {
        inAForStatement(node);

        
        node.getFor().apply(this);
        _cClosure.addClosure(StackOpsFactory.startNewVariableScope());
        _variableTable.push();
        IBuildableClosure stackedClosure = _cClosure;
        _cClosure = new InterpretedBuildableClosure();
        
        
        node.getForControl().apply(this);

        _cClosure.addClosure(StackOpsFactory.endNewVariableScope());
        stackedClosure.addClosure(_cClosure);
        _cClosure = stackedClosure;
        _variableTable.pop();
        
        /*
         * Handled from below: node.getBlock().apply(this);
         */
        outAForStatement(node);
    }

    //DONE
    public void caseAJavaForControl(AJavaForControl node) {
        inAJavaForControl(node);

        // assig part
        node.getForLeftStat().apply(this);

        IBuildableClosure tmpBreak = _breakClosure;
        
        _breakClosure = new InterpretedBuildableClosure();
        
        IBuildableClosure globalClosure = _cClosure;

        IBuildableClosure cond = new InterpretedBuildableClosure();
        _cClosure = cond;
        // cond part
        node.getBoolExpression().apply(this);

        IBuildableClosure blockPart = new InterpretedBuildableClosure();
        _cClosure = blockPart;
        // block part
        ((AForStatement) node.parent()).getBlock().apply(this);

        // final bit part
        if( node.getForRightStat() != null ){
            node.getForRightStat().apply(this);
        }

        //add a while loop:
        _breakClosure.addClosure(ControlFlowOpsFactory.whileClosure(cond,
                blockPart));
        globalClosure.addClosure(_breakClosure);
        _cClosure = globalClosure;
        _breakClosure = tmpBreak;

        outAJavaForControl(node);
    }

    //DONE
    public void caseABreakStatement(ABreakStatement node) {
        inABreakStatement(node);
        node.getBreak().apply(this);

        _cClosure.addClosure(_breakClosure.returnFromThisClosure());

        outABreakStatement(node);
    }

    //DONE (ish)
    public void caseAAssertStatement(AAssertStatement node) {
        inAAssertStatement(node);

        node.getAssert().apply(this);

        IBuildableClosure globalClosure = _cClosure;

        //do the condition
        IBuildableClosure cond = new InterpretedBuildableClosure();
        _cClosure = cond;
        node.getBoolExpression().apply(this);

        IBuildableClosure failure = new InterpretedBuildableClosure();
        _cClosure = failure;
        if (node.getColonString() != null) {
            node.getColonString().apply(this);
        }

        IAtomicClosure assertionFail;

        if (node.getColonString() != null) {
            assertionFail = new AbstractAtomicClosure() {
                /**
                 * @see org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure#execute(org.wellquite.kenya.stackMachine.StackMachine)
                 */
                public void execute(StackMachine sm) {
                    throw new AssertionError(sm.pop()); 
                }
            };
        } else {
            assertionFail = new AbstractAtomicClosure() {
                /**
                 * @see org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure#execute(org.wellquite.kenya.stackMachine.StackMachine)
                 */
                public void execute(StackMachine sm) {
                    throw new AssertionError(); 
                }
            };
        }

        failure.addClosure(assertionFail);

        globalClosure.addClosure(ControlFlowOpsFactory.ifClosure(cond, null,
                failure));
        _cClosure = globalClosure;

        outAAssertStatement(node);
    }

    //DONE
    public void caseAColonString(AColonString node) {
        inAColonString(node);
        if (node.getExpression() != null) {
            node.getExpression().apply(this);
        }
        outAColonString(node);
    }

    // DONE
    public void caseAIfStat(AIfStat node) {
        inAIfStat(node);

        node.getIf().apply(this); // we are in an if

        IBuildableClosure globalClosure = _cClosure;

        // do the if boolean expression
        IBuildableClosure ifPart = new InterpretedBuildableClosure();
        _cClosure = ifPart;
        node.getBoolExpression().apply(this);

        // do the if-true part [ lhs ]
        IBuildableClosure lhs = new InterpretedBuildableClosure();
        _cClosure = lhs;
        node.getBlock1().apply(this);

        // do the if-false part [ rhs ]
        IBuildableClosure rhs = null;
        if (node.getElsePart() != null) {
            rhs = new InterpretedBuildableClosure();
            _cClosure = rhs;
            node.getElsePart().apply(this);
        }

        // add the closure
        globalClosure.addClosure(ControlFlowOpsFactory.ifClosure(ifPart, lhs,
                rhs));

        // reset back global
        _cClosure = globalClosure;

        outAIfStat(node);
    }

    //DONE
    public void caseAElsePart(AElsePart node) {
        inAElsePart(node);
        node.getElseFollow().apply(this);
        outAElsePart(node);
    }

    //DONE ( do I need to stack the closures? bah, why not? )
    public void caseABlock(ABlock node) {
        inABlock(node);
        node.getLBrace().apply(this);

        _variableTable.push();
        
        IBuildableClosure stackedClosure = _cClosure;
        _cClosure = new InterpretedBuildableClosure();
        _cClosure.addClosure(StackOpsFactory.startNewVariableScope());
        
        node.getStatements().apply(this);
        
        node.getRBrace().apply(this);
        _cClosure.addClosure(StackOpsFactory.endNewVariableScope());
        
        stackedClosure.addClosure(_cClosure);
        _cClosure = stackedClosure;

        _variableTable.pop();
        

        outABlock(node);
    }

    //DONE
    public void caseAFunctionApplication(AFunctionApplication node) {
        inAFunctionApplication(node);

        /*
         * Parameter passing convention:
         * 
         * void blah(arg1, arg2, arg3){ arg1 = pop(); arg2 = pop(); arg3 =
         * pop(); }
         * 
         * push arg3; push arg2; push arg1; blah()
         * 
         *  
         */

        // process the arguments
        if (node.getActualParamList() != null) {
            node.getActualParamList().apply(this);
        }


        ((ASimpleName)((ASimpleNameName)node.getName()).getSimpleName()).getIdentifier().apply(this);
        
        final KFunction thisFunc = _informationHolder.getFunction(node);
        
        final int numArgs = thisFunc.getArgs().size();
        
        _cClosure.addClosure(StackOpsFactory.reverseStackTop(numArgs));
        
        if (thisFunc.isBuiltin()) {
            _cClosure.addClosure(SMLibraryContainer.getLibrary().
                    invokeBuiltInMethod(thisFunc.getBuiltinMethod().
                    getInterpretedMethod()));
        } else {
            IFunction invokeF = _stackInfoManager.getFunctionFromNode(thisFunc
                    .getNode());
            String smFN = _stackInfoManager.lookupFunction(invokeF);
            _cClosure.addClosure(StackOpsFactory.invokeLocalMethod(smFN));
        }
        
        final boolean hasRet = thisFunc.getReturnType() != KBasicType.getVoid();
        final int size = thisFunc.getArgs().size();
        
        _cClosure.addClosure( new AbstractAtomicClosure( ){
          
            public void execute(StackMachine sm) {
        
                IType temp = null;
                
                if( hasRet ){
                    temp = sm.pop();
                }

                for( int i = 0 ; i < size ; i++ ){
                    sm.pop();
                }
                
                if( hasRet ){
                    sm.push(temp);
                }
                
            }
            
        });
        
        checkCast(node);

        outAFunctionApplication(node);
    }

    /*
     * DONE
     */
    public void caseAListActualParamList(AListActualParamList node) {
        inAListActualParamList(node);

        if (node.getActualParamList() != null) {
            node.getActualParamList().apply(this);
        }
        
        if (node.getExpression() != null) {
            node.getExpression().apply(this);
        }



        outAListActualParamList(node);
    }

    // DONE 
    public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
        inAVarDecInnerDeclaration(node);

        if( _cIClass != null ){
            RestrictedStackMachineBuilder.doClassVariable(_cClass, _stackInfoManager.getPreInitClosure(), 
                    node, _stackInfoManager, _informationHolder, _cIClass);
            outAVarDecInnerDeclaration(node);
            return;
        }
        
        node.getIdentifier().apply(this); // mark on the ident
        final String origName = node.getIdentifier().getText().trim();

        final IVariable var;
        
        if (_cFunction != null) {
            final int[] pos = Util.getFirstIdent(node.getIdentifier());
            IClass[] tyParams = _cFunction.getTypeParams();
            IClass type = _informationHolder.getTypeFromNode(node,tyParams);
            var = new AVariableInstance(origName, 
                    new SourceCodeLocation(pos[0],pos[1],pos[2]),type);
        } else {
            throw KenyaPreconditionError.get();
        }
        
        if (node.getInitialiser() != null) {
            node.getInitialiser().apply(this);
        } else {
            final IClass type = var.getType();
            /* this really should be a lot better */
            if (type.getName().equals("boolean")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(false)));
            } else if (type.getName().equals("char")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType('\0')));
            } else if (type.getName().equals("int")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0)));
            } else if (type.getName().equals("double")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0.0)));
            } else if (type.getName().equals("Boolean")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(false)));
            } else if (type.getName().equals("Character")){
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType('\0')));
            } else if (type.getName().equals("Integer")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0)));
            } else if (type.getName().equals("Double")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0.0)));
            } else if( type.isClass() ) {
                
                final IStackMachineInformationManager isim = _stackInfoManager;
                
                _cClosure.addClosure( 
                        new AbstractAtomicClosure(){
                            
                            public void execute(StackMachine sm) {
                                
                                sm.push(ClassTypeFactory.createClassInstanceType(
                                                isim.getClassBody(type).
                                                createInstance()));
                            }
                            
                        }
                    );
                
            } else if( type.isEnum() || type.isArray()  || type.isParam() || type.getName().equals("String") ){
                _cClosure.addClosure( StackOpsFactory.push(
                	NullTypeFactory.createNullType()));
            } else{
                throw KenyaPreconditionError.get();
            }
        }
        
        String smName = _stackInfoManager.addNewVariable(var);
        _variableTable.add(var);
        
        checkCast(node);
        
        _cClosure.addClosure(StackOpsFactory.storeNewVariable(smName, true));

        outAVarDecInnerDeclaration(node);
    }

    // DONE
    public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
        inAArrayDecInnerDeclaration(node);

        if( _cIClass != null ){
            RestrictedStackMachineBuilder.doClassVariable(_cClass, _stackInfoManager.getPreInitClosure(),
                    node, _stackInfoManager, _informationHolder, _cIClass);
            outAArrayDecInnerDeclaration(node);
            return;
        }
        
        node.getIdentifier().apply(this);
        final String origName = node.getIdentifier().getText().trim();
        
        final IVariable var;
        
        if (_cFunction != null) {
            final int[] pos = Util.getFirstIdent(node.getIdentifier());
            IClass[] tyParams = _cFunction.getTypeParams();
            IClass type = _informationHolder.getTypeFromNode(node,tyParams);
            tyParams = _cFunction.getTypeParams();
            var = new AVariableInstance(origName, 
                    new SourceCodeLocation(pos[0],pos[1],pos[2]),type);
        } else {
            throw KenyaPreconditionError.get();
        }
        
        
        if (node.getArrayInitialiser() != null) {
            node.getArrayInitialiser().apply(this);
        } else {
            _cClosure.addClosure( StackOpsFactory.push(NullTypeFactory.createNullType()));
        }
        
        String smName = _stackInfoManager.addNewVariable(var);
        _variableTable.add(var);
        _cClosure.addClosure(StackOpsFactory.storeNewVariable(smName, true));
        
        outAArrayDecInnerDeclaration(node);
    }

    //DONE
    public void caseAInitialiser(AInitialiser node) {
        inAInitialiser(node);

        if (node.getBoolExpression() != null) {
            node.getBoolExpression().apply(this);
        }
        
        
        
        if (node.getAssign() != null) {
            node.getAssign().apply(this);
        }
        

        outAInitialiser(node);
    }
    
    //DONE
    public void caseAStaticArrayInitialiser(AStaticArrayInitialiser node) {
        inAStaticArrayInitialiser(node);

        if (node.getArrayInit() != null) {
            node.getArrayInit().apply(this);
        }
        
        if (node.getAssign() != null) {
            node.getAssign().apply(this);
        }
        
        outAStaticArrayInitialiser(node);
    }
    
    //DONE
    public void caseADynamicArrayInitialiser(ADynamicArrayInitialiser node) {
        inADynamicArrayInitialiser(node);
        
        if (node.getArrayAllocate() != null) {
            node.getArrayAllocate().apply(this);
        }
        
        if (node.getAssign() != null) {
            node.getAssign().apply(this);
        }
        
        outADynamicArrayInitialiser(node);
    }
    
    //DONE - dynamic creation
    public void caseAArrayInit(AArrayInit node) {
        inAArrayInit(node);

        if (node.getInitList() != null) {
            node.getInitList().apply(this);
        }
        
        outAArrayInit(node);
    }
    
    //DONE
    public void caseAScalarInitList(AScalarInitList node) {
        inAScalarInitList(node);
        if (node.getExpression() != null) {
            node.getExpression().apply(this);
        }
        {
            Object temp[] = node.getCommaExp().toArray();
            for (int i = 0; i < temp.length; i++) {
                ((PCommaExp) temp[i]).apply(this);
            }
        
            int length = 1 + temp.length;
            
            _cClosure.addClosure( StackOpsFactory.push( PrimitiveTypeFactory.createPrimitiveType( length )));
            _cClosure.addClosure( ArrayOpsFactory.buildArrayWithValues());
            
        }
        
        outAScalarInitList(node);
    }
    
    //DONE
    public void caseAArrayInitList(AArrayInitList node) {
        inAArrayInitList(node);
        if (node.getArrayInit() != null) {
            node.getArrayInit().apply(this);
        }
        {
            Object temp[] = node.getCommaArrayInit().toArray();
            for (int i = 0; i < temp.length; i++) {
                ((PCommaArrayInit) temp[i]).apply(this);
            }
        
        int length = 1 + temp.length;        
        _cClosure.addClosure( StackOpsFactory.push( PrimitiveTypeFactory.createPrimitiveType( length )));
        _cClosure.addClosure( ArrayOpsFactory.buildArrayWithValues());
        
        }
        
        outAArrayInitList(node);
    }
    
    //DONE
    public void caseACommaExp(ACommaExp node) {
        inACommaExp(node);
        if (node.getComma() != null) {
            node.getComma().apply(this);
        }
        if (node.getExpression() != null) {
            node.getExpression().apply(this);
        }
        outACommaExp(node);
    }

    //DONE
    public void caseACommaArrayInit(ACommaArrayInit node) {
        inACommaArrayInit(node);
        if (node.getComma() != null) {
            node.getComma().apply(this);
        }
        if (node.getArrayInit() != null) {
            node.getArrayInit().apply(this);
        }
        outACommaArrayInit(node);
    }

    //DONE
    public void caseATrueBooleanliteral(ATrueBooleanliteral node) {
        inATrueBooleanliteral(node);
        node.getTrue().apply(this);
        
        _cClosure.addClosure(StackOpsFactory.push(
                PrimitiveTypeFactory.createPrimitiveType(true)));
                
        outATrueBooleanliteral(node);
    }

    //DONE
    public void caseAFalseBooleanliteral(AFalseBooleanliteral node) {
        inAFalseBooleanliteral(node);
        node.getFalse().apply(this);
        _cClosure.addClosure(StackOpsFactory.push(
                PrimitiveTypeFactory.createPrimitiveType(false)));
        
        outAFalseBooleanliteral(node);
    }

    
    public void caseAStringliteralFactor(AStringliteralFactor node)
    {
        inAStringliteralFactor(node);
        node.getStringliteral().apply(this);

        String text = node.getStringliteral().getText();
        
        int start = text.indexOf("\"");
        int end   = text.lastIndexOf("\"");
        
        text = text.substring(start+1, end);
        _cClosure.addClosure(StackOpsFactory.push( StringTypeFactory.createStringType(
                Util.escapeString(text))));
        
        outAStringliteralFactor(node);
    }
    
    public void caseACharliteralFactor(ACharliteralFactor node)
    {
        inACharliteralFactor(node);
        node.getCharliteral().apply(this);
        
        char c = Util.parseCharacter(node);

        _cClosure.addClosure(StackOpsFactory.push( PrimitiveTypeFactory.
                createPrimitiveType(c)));
        
        outACharliteralFactor(node);
    }
    
    
    //DONE
    public void caseACharBasicType(ACharBasicType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseAIntBasicType(AIntBasicType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseADoubleBasicType(ADoubleBasicType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseAStringBasicType(AStringBasicType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseABooleanBasicType(ABooleanBasicType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseAVoidBasicType(AVoidBasicType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseAClassTypeReferenceType(AClassTypeReferenceType node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseATypeParam(ATypeParam node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseATypeParamList(ATypeParamList node) {
        throw KenyaBadLocationError.get();
    }

    //DONE
    public void caseACommaType(ACommaType node) {
        throw KenyaBadLocationError.get();
    }


    //DONE
    public void caseAArrayAllocate(AArrayAllocate node) {
        inAArrayAllocate(node);
        if (node.getNew() != null) {
            node.getNew().apply(this);
        }

        /*
         * node.getType();
         * 
         */
        
        {
            Object temp[] = node.getArrayAccess().toArray();
            for (int i = 0; i < temp.length; i++) {
                VariableLookupTable tmp = _stackVariableTable;
                _stackVariableTable = null;
                ((PArrayAccess) temp[i]).apply(this);
                _stackVariableTable = tmp;
            }

            _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(temp.length)));
        }
        
        if ( node.getBracketPair().size() != 0 ) {
            _cClosure.addClosure(StackOpsFactory.push(NullTypeFactory.createNullType()));
        } else {

           IClass type = getNodeType(node);

            String typeName = type.getName().substring(0,
                    type.getName().indexOf('[')
            );
            
            if (typeName.equals("boolean")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(false)));
            } else if (typeName.equals("char")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType('\0')));
            } else if (typeName.equals("int")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0)));
            } else if (typeName.equals("double")) {
                _cClosure.addClosure(StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(0.0)));
            } else {
                _cClosure.addClosure(StackOpsFactory.push(NullTypeFactory.createNullType()));
            }
            
            
        }
        
        _cClosure.addClosure(ArrayOpsFactory.buildMultidimensionalArray());
        
        outAArrayAllocate(node);
    }


    //DONE
    public void caseAPlusMathExpression(APlusMathExpression node) {
        inAPlusMathExpression(node);

        node.getMathExpression().apply(this);
        node.getTerm().apply(this);
        
        _cClosure.addClosure(StackOpsFactory.reverseStackTop(2));
        
        
        node.getPlus().apply(this);
        
        IClass[] typeParams;
        if( _cIClass != null ){
            typeParams = _cIClass.getTemplateParams();
        }else if( _cFunction != null ){
            typeParams = _cFunction.getTypeParams();
        }else{
            /* we are in a constant */
            typeParams = new IClass[] {};
        }
         
        IClass tc = _informationHolder.getTypeFromNode(node, typeParams );

        if( tc == _informationHolder.getTypeFromName("String") ){
            _cClosure.addClosure( StringOpsFactory.concat());            
        }else{
            _cClosure.addClosure( NumericOpsFactory.add());
        }
        
        outAPlusMathExpression(node);
    }
    
    //DONE
    public void caseAMinusMathExpression(AMinusMathExpression node) {
        inAMinusMathExpression(node);
        
        node.getMathExpression().apply(this);
        node.getTerm().apply(this);
        
        _cClosure.addClosure(StackOpsFactory.reverseStackTop(2));
        
        node.getMinus().apply(this);
        
        _cClosure.addClosure( NumericOpsFactory.subtract());
        
        outAMinusMathExpression(node);
    }
    
    //DONE
    public void caseAMultTerm(AMultTerm node) {
        inAMultTerm(node);
        
        node.getTerm().apply(this);
        node.getUnaryExp().apply(this);
        
        node.getTimes().apply(this);
        
        _cClosure.addClosure( NumericOpsFactory.multiply());
        outAMultTerm(node);
    }

    //DONE
    public void caseADivTerm(ADivTerm node) {
        inADivTerm(node);
        
        node.getTerm().apply(this);
        node.getUnaryExp().apply(this);
        
        _cClosure.addClosure(StackOpsFactory.reverseStackTop(2));
        
        node.getDivide().apply(this);
        
        _cClosure.addClosure( NumericOpsFactory.divide());
        outADivTerm(node);
    }

    //DONE
    public void caseAModTerm(AModTerm node) {
        inAModTerm(node);
        
        node.getTerm().apply(this);
        node.getUnaryExp().apply(this);
            
        _cClosure.addClosure(StackOpsFactory.reverseStackTop(2));
        
        node.getMod().apply(this);
        
        _cClosure.addClosure( NumericOpsFactory.modulus());
        outAModTerm(node);
    }

    //DONE
    public void caseAMinusUnaryExp(AMinusUnaryExp node) {
        inAMinusUnaryExp(node);
        node.getUnaryExp().apply(this);
        node.getMinus().apply(this);

        _cClosure.addClosure( StackOpsFactory.push(PrimitiveTypeFactory.createPrimitiveType(-1)));
        _cClosure.addClosure( NumericOpsFactory.multiply());
        outAMinusUnaryExp(node);
    }

    //DONE (don't really need to do anything )
    public void caseAPlusUnaryExp(APlusUnaryExp node) {
        inAPlusUnaryExp(node);
        
        node.getUnaryExp().apply(this);
        
        node.getPlus().apply(this);
        
        outAPlusUnaryExp(node);
    }

    
    //DONE
    public void caseANegateUnaryExp(ANegateUnaryExp node) {
        inANegateUnaryExp(node);

        node.getUnaryExp().apply(this);
        
        node.getNot().apply(this);

        _cClosure.addClosure( LogicalOpsFactory.not() );
        
        outANegateUnaryExp(node);
    }

    //DONE
    public void caseAPostdecr(APostdecr node) {
        inAPostdecr(node);
        node.getFieldAccess().apply(this);
        
        node.getMinusminus().apply(this);
        _cClosure.addClosure( NumericOpsFactory.dec());
        
        boolean tmp = _doAssig;
        _doAssig = true;
        node.getFieldAccess().apply(this);
        _doAssig = tmp;
        
        outAPostdecr(node);
    }

    //DONE
    public void caseAPostincr(APostincr node) {
        inAPostincr(node);
        
        boolean tmp = _doAssig;
        
        _doAssig = false;
        node.getFieldAccess().apply(this);
        node.getPlusplus().apply(this);

        _cClosure.addClosure( NumericOpsFactory.inc());
        
        
        _doAssig = true;
        node.getFieldAccess().apply(this);
        _doAssig = tmp;
        
        outAPostincr(node);
    }

    //DONE
    public void caseAPredecr(APredecr node) {
        inAPredecr(node);

        boolean tmp = _doAssig;
        
        _doAssig = false;
        node.getFieldAccess().apply(this);
        node.getMinusminus().apply(this);
        
        _cClosure.addClosure( NumericOpsFactory.dec());
        
        
        _doAssig = true;
        node.getFieldAccess().apply(this);
        _doAssig = tmp;
        
        outAPredecr(node);
    }

    //DONE
    public void caseAPreincr(APreincr node) {
        inAPreincr(node);
        
        boolean tmp = _doAssig;
        
        _doAssig = false;
        node.getFieldAccess().apply(this);
        node.getPlusplus().apply(this);
        
        
        _cClosure.addClosure( NumericOpsFactory.inc());
        
        
        
        _doAssig = true;
        node.getFieldAccess().apply(this);
        _doAssig = tmp;
        
        outAPreincr(node);
    }

    //DONE
    public void caseANullFactor(ANullFactor node) {
        inANullFactor(node);
        node.getNull().apply(this);
        _cClosure.addClosure( StackOpsFactory.push( NullTypeFactory.createNullType()));
        outANullFactor(node);
    }


    //DONE
    public void caseAExpressionFactor(AExpressionFactor node) {
        inAExpressionFactor(node);
        
        if (node.getBoolExpression() != null) {
            node.getBoolExpression().apply(this);
        }
        
        outAExpressionFactor(node);
    }

    //DONE
    public void caseAINumber(AINumber node) {
        inAINumber(node);

        node.getIntnumber().apply(this);
        
        int i;
        try{
            i = Integer.parseInt(node.getIntnumber().getText().trim());
            
            IClass type = _informationHolder.getTypeFromNode(node, new IClass[]{});
            if ( type.getName().equals("int")) {
                _cClosure.addClosure(StackOpsFactory.push( 
                        PrimitiveTypeFactory.createPrimitiveType(i)));
            } else if ( type.getName().equals("char") ) {
                _cClosure.addClosure(StackOpsFactory.push( 
                        PrimitiveTypeFactory.createPrimitiveType((char)i)));
            } else {
                throw KenyaPreconditionError.get();
            }
            
        }catch(NumberFormatException e){
            throw KenyaPreconditionError.get();
        }
        
        outAINumber(node);
    }

    //DONE
    public void caseADNumber(ADNumber node) {
        inADNumber(node);
        node.getDpnumber().apply(this);

        double d;
        try{
            d = Double.parseDouble(node.getDpnumber().getText().trim());
            _cClosure.addClosure(StackOpsFactory.push(
                    PrimitiveTypeFactory.createPrimitiveType(d)));
        } catch ( NumberFormatException e){
            throw KenyaPreconditionError.get();
        }
        
        outADNumber(node);
    }


    //DONE
    public void caseAOrBoolExpression(AOrBoolExpression node) {
        inAOrBoolExpression(node);

        IBuildableClosure globalClosure = _cClosure;
        
        IBuildableClosure lhs = new InterpretedBuildableClosure();
        IBuildableClosure rhs = new InterpretedBuildableClosure();
        
        _cClosure = lhs;
        node.getBoolExpression().apply(this);
        
        _cClosure = rhs;
        node.getBoolTerm().apply(this);
        
        node.getOr().apply(this);
        
        _cClosure = globalClosure;
        
        _cClosure.addClosure(LogicalOpsFactory.or(lhs,rhs));
        
        outAOrBoolExpression(node);
    }

    //DONE
    public void caseAXorBoolExpression(AXorBoolExpression node) {
        inAXorBoolExpression(node);
        
        node.getBoolExpression().apply(this);
        node.getBoolTerm().apply(this);
        
        node.getXor().apply(this);
        
        _cClosure.addClosure(LogicalOpsFactory.xor());
        
        outAXorBoolExpression(node);
    }

    //DONE
    public void caseAAndBoolTerm(AAndBoolTerm node) {
        inAAndBoolTerm(node);

        IBuildableClosure globalClosure = _cClosure;
        
        IBuildableClosure lhs = new InterpretedBuildableClosure();
        IBuildableClosure rhs = new InterpretedBuildableClosure();
        
        _cClosure = lhs;
        node.getBoolTerm().apply(this);
        
        _cClosure = rhs;
        node.getEquality().apply(this);

        node.getAnd().apply(this);

        _cClosure = globalClosure;
        
        _cClosure.addClosure(LogicalOpsFactory.and(lhs,rhs));
        
        outAAndBoolTerm(node);
    }


    //DONE
    public void caseAEqEquality(AEqEquality node) {
        inAEqEquality(node);

        node.getEquality().apply(this);
        node.getRelational().apply(this);

        node.getEqual().apply(this);
        
        _cClosure.addClosure( LogicalOpsFactory.equal());        

        outAEqEquality(node);
    }

    //DONE
    public void caseANeqEquality(ANeqEquality node) {
        inANeqEquality(node);
        
        node.getEquality().apply(this);
        node.getRelational().apply(this);
        
        node.getNotequal().apply(this);
        
        _cClosure.addClosure( LogicalOpsFactory.notEqual());
        
        outANeqEquality(node);
    }

    
    //DONE
    public void caseALtRelational(ALtRelational node) {
        inALtRelational(node);
        
        node.getRelational().apply(this);
        node.getMathExpression().apply(this);        
        
        _cClosure.addClosure( StackOpsFactory.reverseStackTop(2));
        
        node.getLess().apply(this);
        _cClosure.addClosure( LogicalOpsFactory.lessThan());
        
        outALtRelational(node);
    }

    //DONE
    public void caseAGtRelational(AGtRelational node) {
        inAGtRelational(node);

        node.getRelational().apply(this);
        node.getMathExpression().apply(this);
        
        _cClosure.addClosure( StackOpsFactory.reverseStackTop(2));
        
        node.getGreater().apply(this);
        _cClosure.addClosure( LogicalOpsFactory.greaterThan());
        
        
        outAGtRelational(node);
    }

    //DONE    
    public void caseALteqRelational(ALteqRelational node) {
        inALteqRelational(node);

        node.getRelational().apply(this);
        node.getMathExpression().apply(this);

        _cClosure.addClosure( StackOpsFactory.reverseStackTop(2));

        node.getLessequal().apply(this);

        _cClosure.addClosure( LogicalOpsFactory.lessThanEqual());
        
        outALteqRelational(node);
    }
    //DONE
    public void caseAGteqRelational(AGteqRelational node) {
        inAGteqRelational(node);
        
        node.getRelational().apply(this);
        node.getMathExpression().apply(this);
        
        _cClosure.addClosure( StackOpsFactory.reverseStackTop(2));
        
        node.getGreaterequal().apply(this);
        
        _cClosure.addClosure( LogicalOpsFactory.greaterThanEqual());

        outAGteqRelational(node);
    }

    //DONE
    public void caseAArrayFieldAccess(AArrayFieldAccess node) {
        inAArrayFieldAccess(node);
        
        boolean tmpDoAssig = _doAssig;
        _doAssig = false;
        
        if (node.getName() != null) {
            boolean wasNull = _stackVariableTable == null;
            
            if( wasNull ){
                _stackVariableTable = _variableTable;
            }
            node.getName().apply(this);
            
            if( wasNull ){
                _stackVariableTable = null;
            }
        }

        

        {
            Object temp[] = node.getArrayAccess().toArray();
            
            if( temp.length == 0 ){
                //must be at least 1 array access
                throw KenyaPreconditionError.get();
            }
            
            for (int i = 0; i < temp.length; i++) {
                VariableLookupTable tmp = _stackVariableTable;
                _stackVariableTable = null;
                ((PArrayAccess) temp[i]).apply(this);
                _stackVariableTable = tmp;
                if ( i+1 == temp.length) {
                
                    if( tmpDoAssig ){
                        _cClosure.addClosure(ArrayOpsFactory.set());
                    } else {
                        _cClosure.addClosure(ArrayOpsFactory.get());
                        checkCast(node);
                    }
                } else {
                    _cClosure.addClosure(ArrayOpsFactory.get());
                }
                
            }
        }
        
        
        
        _doAssig = tmpDoAssig;
        outAArrayFieldAccess(node);
    }
    


    //DONE
    public void caseAArrayAccess(AArrayAccess node) {
        inAArrayAccess(node);
        node.getMathExpression().apply(this);
        outAArrayAccess(node);
    }
    

    //DONE
    public void caseAScalarFieldAccess(AScalarFieldAccess node) {
        inAScalarFieldAccess(node);
        node.getName().apply(this);
        outAScalarFieldAccess(node);
    }

    //DONE (ish)
    public void caseASimpleName(ASimpleName node) {
        inASimpleName(node);
        
        node.getIdentifier().apply(this);
        
        boolean wasNull = _stackVariableTable == null;
        if( wasNull ){
            _stackVariableTable = _variableTable;
        }
        
        IClass nameType = getNodeType(node);


        if( _informationHolder.isEnumChildNode( node )){
            /* we are in a switch an this is the child of an enum */

                IInterpretedEnumeration ienum = _stackInfoManager.getEnumBody( nameType );
                String chName = node.getIdentifier().getText().trim();

                IType enumType = EnumTypeFactory.createEnumType( ienum, chName );
                _cClosure.addClosure( StackOpsFactory.push( enumType ));
            
        } else if( _doAssig ) {

            IVariable var = _stackVariableTable.lookupByName(node.getIdentifier().getText().trim());
            String smName = _stackInfoManager.lookupVariable(var);

            _cClosure.addClosure(StackOpsFactory.store(smName));

        } else if( _stackVariableTable.hasVariable(node.getIdentifier().getText().trim())) {
            IVariable var = _stackVariableTable.lookupByName(node.getIdentifier().getText().trim());
            String smName = _stackInfoManager.lookupVariable(var);

            _cClosure.addClosure(StackOpsFactory.fetch(smName));

            // even if var was pointing to an enumeration it wouldn't matter because we will
            // know when to look at its children.
            _stackVariableTable = new VariableLookupTable( nameType.getDeclarations() );

        } else if( _informationHolder.hasEnum(node.getIdentifier().getText().trim())){
            /* this will be dereferenced, so i don't need to do anything: precondition layed down in
             * typechecker and structure checker
             */
        } else {
            throw KenyaPreconditionError.get();
        }

        if( wasNull ){
            _stackVariableTable = null;
        }
        
        outASimpleName(node);
    }

    //DONE (ish)
    public void caseAQualifiedName(AQualifiedName node) {
        inAQualifiedName(node);

        boolean wasNull = _stackVariableTable == null;
        
        if( wasNull ) {
            _stackVariableTable = _variableTable;
        }
        
        boolean tmpDoAssig = _doAssig;
        _doAssig = false;
        
        node.getFieldAccess().apply(this);
        
        _doAssig = tmpDoAssig;
        
        node.getIdentifier().apply(this);
        
        if( _doAssig ) {
        
            IVariable var = _stackVariableTable.lookupByName(node.getIdentifier().getText().trim());
            String smName = _stackInfoManager.lookupVariable(var);
            
            /* a qualified name in an assignment must must must be from a class in kenya */
            /* mustn't it? :s */
            _cClosure.addClosure(
                    new AbstractAtomicClosure(){
                     
                        public void execute(StackMachine sm){
	                        	// The value and type are the wrong way round here - they need swapping.
                            IType a = sm.pop();
	                        IType b = sm.pop();
	                        sm.push(a);
	                        sm.push(b);
                        }
                    });
            
            _cClosure.addClosure(StackOpsFactory.storeInClassInstance(smName));
            
        } else if( _informationHolder.isArrayLengthReference(node)){
            _cClosure.addClosure(ArrayOpsFactory.getLength());

        } else {

            IClass type; // what the resultant type of this node is.
            
            if( _cIClass != null ){
                type = _informationHolder.getTypeFromNode(node, _cIClass.getTemplateParams());
            }else if( _cFunction != null ){
                type = _informationHolder.getTypeFromNode(node, _cFunction.getTypeParams());                
            }else{
                /* not in a class or a function and constants can't have 
                 * dereferences in them.
                 */
                throw KenyaPreconditionError.get();
            }


            if( _informationHolder.isEnumChildNode( node )){

                IInterpretedEnumeration ienum = _stackInfoManager.getEnumBody( type );
                String chName = node.getIdentifier().getText().trim();

                IType enumType = EnumTypeFactory.createEnumType( ienum, chName );
                _cClosure.addClosure( StackOpsFactory.push( enumType ));

            } else {
                /* we are not an enum child node, we could be an enum variable, or anything else for that matter,
                 * but since we are a . derefence we must be working on a live class */
                IVariable thisVar = _stackVariableTable.lookupByName(node.getIdentifier().getText().trim());
                String smName = _stackInfoManager.lookupVariable(thisVar);
                _cClosure.addClosure( StackOpsFactory.fetchFromClassInstance( smName ));
                _stackVariableTable = new VariableLookupTable( type.getDeclarations() );
            }

        }
        
        if( wasNull ){
            _stackVariableTable = null;
        }
        
        outAQualifiedName(node);
    }



    /*
     * TOKENS: do positional with them? maybe not such a good idea - keep it
     * controlled :s
     */

    public void caseTReservedWord(TReservedWord node) {
        doMarker(node);
    }

    public void caseTNewLine(TNewLine node) {
        doMarker(node);
    }

    public void caseTBlank(TBlank node) {
        doMarker(node);
    }

    public void caseTComment(TComment node) {
        doMarker(node);
    }

    public void caseTTraditionalComment(TTraditionalComment node) {
        doMarker(node);
    }

    public void caseTBoolean(TBoolean node) {
        doMarker(node);
    }

    public void caseTChar(TChar node) {
        doMarker(node);
    }

    public void caseTInt(TInt node) {
        doMarker(node);
    }

    public void caseTDouble(TDouble node) {
        doMarker(node);
    }

    public void caseTString(TString node) {
        doMarker(node);
    }

    public void caseTVoid(TVoid node) {
        doMarker(node);
    }

    public void caseTKlass(TKlass node) {
        doMarker(node);
    }

    public void caseTConst(TConst node) {
        doMarker(node);
    }

    public void caseTIf(TIf node) {
        doMarker(node);
    }

    public void caseTElse(TElse node) {
        doMarker(node);
    }

    public void caseTWhile(TWhile node) {
        doMarker(node);
    }

    public void caseTReturn(TReturn node) {
        doMarker(node);
    }

    public void caseTSwitch(TSwitch node) {
        doMarker(node);
    }

    public void caseTCase(TCase node) {
        doMarker(node);
    }

    public void caseTBreak(TBreak node) {
        doMarker(node);
    }

    public void caseTDefault(TDefault node) {
        doMarker(node);
    }

    public void caseTFor(TFor node) {
        doMarker(node);
    }

    public void caseTAssert(TAssert node) {
        doMarker(node);
    }

    public void caseTNew(TNew node) {
        doMarker(node);
    }

    public void caseTEnum(TEnum node) {
        doMarker(node);
    }

    public void caseTTrue(TTrue node) {
        doMarker(node);
    }

    public void caseTFalse(TFalse node) {
        doMarker(node);
    }

    public void caseTNull(TNull node) {
        doMarker(node);
    }

    public void caseTAnd(TAnd node) {
        doMarker(node);
    }

    public void caseTOr(TOr node) {
        doMarker(node);
    }

    public void caseTXor(TXor node) {
        doMarker(node);
    }

    public void caseTNot(TNot node) {
        doMarker(node);
    }

    public void caseTIdentifier(TIdentifier node) {
        doMarker(node);
    }

    public void caseTStringliteral(TStringliteral node) {
        doMarker(node);
    }

    public void caseTCharliteral(TCharliteral node) {
        doMarker(node);
    }

    public void caseTLParenthese(TLParenthese node) {
        doMarker(node);
    }

    public void caseTRParenthese(TRParenthese node) {
        doMarker(node);
    }

    public void caseTLBrace(TLBrace node) {
        doMarker(node);
    }

    public void caseTRBrace(TRBrace node) {
        doMarker(node);
    }

    public void caseTLBracket(TLBracket node) {
        doMarker(node);
    }

    public void caseTRBracket(TRBracket node) {
        doMarker(node);
    }

    public void caseTBracketPair(TBracketPair node) {
        doMarker(node);
    }

    public void caseTSemicolon(TSemicolon node) {
        doMarker(node);
    }

    public void caseTColon(TColon node) {
        doMarker(node);
    }

    public void caseTComma(TComma node) {
        doMarker(node);
    }

    public void caseTDot(TDot node) {
        doMarker(node);
    }

    public void caseTPlus(TPlus node) {
        doMarker(node);
    }

    public void caseTPlusplus(TPlusplus node) {
        doMarker(node);
    }

    public void caseTMinus(TMinus node) {
        doMarker(node);
    }

    public void caseTMinusminus(TMinusminus node) {
        doMarker(node);
    }

    public void caseTTimes(TTimes node) {
        doMarker(node);
    }

    public void caseTDivide(TDivide node) {
        doMarker(node);
    }

    public void caseTMod(TMod node) {
        doMarker(node);
    }

    public void caseTLess(TLess node) {
        doMarker(node);
    }

    public void caseTLessequal(TLessequal node) {
        doMarker(node);
    }

    public void caseTGreater(TGreater node) {
        doMarker(node);
    }

    public void caseTGreaterequal(TGreaterequal node) {
        doMarker(node);
    }

    public void caseTEqual(TEqual node) {
        doMarker(node);
    }

    public void caseTNotequal(TNotequal node) {
        doMarker(node);
    }

    public void caseTAssign(TAssign node) {
        doMarker(node);
    }

    public void caseTIntnumber(TIntnumber node) {
        doMarker(node);
    }

    public void caseTDpnumber(TDpnumber node) {
        doMarker(node);
    }

}
