/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 20-Jul-2004 by toa02
 *
 */
package kenya.passes;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import kenya.errors.KenyaBadLocationError;
import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.errors.SourceCodeException;
import kenya.sourceCodeInformation.util.SourceCodeWarningFactory;
import kenya.types.KBasicType;
import kenya.types.KBoundClassType;
import kenya.types.KEnumType;
import kenya.types.KFunction;
import kenya.types.KType;
import kenya.types.KVariable;
import kenya.types.tables.ClassTable;
import kenya.types.tables.FunctionTable;
import kenya.types.tables.SymbolTable;
import kenya.types.tables.VariableTable;
import kenya.values.IKNumericValue;
import kenya.values.IKValue;
import kenya.values.KBooleanValue;
import kenya.values.KCharValue;
import kenya.values.KDoubleValue;
import kenya.values.KEnumValue;
import kenya.values.KIntValue;
import kenya.values.KNullValue;
import kenya.values.KStringValue;
import kenya.values.KUnassignedValue;
import kenya.values.KUnknownValue;
import kenya.values.KVoidValue;
import kenya.values.util.KValueCalculator;
import kenya.values.util.MarkerKVariable;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AAndBoolTerm;
import minijava.node.AArrayAllocate;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AArrayFieldAccess;
import minijava.node.AArrayInit;
import minijava.node.AArrayInitList;
import minijava.node.AAssertStatement;
import minijava.node.AAssignment;
import minijava.node.AAssignmentForRightStat;
import minijava.node.AAssignmentStatement;
import minijava.node.ABasicTypeType;
import minijava.node.ABlock;
import minijava.node.ABreakStatement;
import minijava.node.ACasePossibleCase;
import minijava.node.ACharliteralFactor;
import minijava.node.AClassDecDeclaration;
import minijava.node.ACommaEnumList;
import minijava.node.ACommaTypeName;
import minijava.node.AConstDecDeclaration;
import minijava.node.ADNumber;
import minijava.node.ADefaultPossibleCase;
import minijava.node.ADivTerm;
import minijava.node.ADynamicArrayInitialiser;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AEnumList;
import minijava.node.AEqEquality;
import minijava.node.AFalseBooleanliteral;
import minijava.node.AForStatement;
import minijava.node.AFormalParamList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.AFunctionApplication;
import minijava.node.AFunctioncallForRightStat;
import minijava.node.AFunctioncallStatement;
import minijava.node.AGtRelational;
import minijava.node.AGteqRelational;
import minijava.node.AINumber;
import minijava.node.AIfStat;
import minijava.node.AJavaForControl;
import minijava.node.AListStatements;
import minijava.node.ALtRelational;
import minijava.node.ALteqRelational;
import minijava.node.AMinusMathExpression;
import minijava.node.AMinusUnaryExp;
import minijava.node.AModTerm;
import minijava.node.AMultTerm;
import minijava.node.ANegateUnaryExp;
import minijava.node.ANeqEquality;
import minijava.node.ANullFactor;
import minijava.node.AOrBoolExpression;
import minijava.node.APlusMathExpression;
import minijava.node.APlusUnaryExp;
import minijava.node.APostdecr;
import minijava.node.APostincr;
import minijava.node.APredecr;
import minijava.node.APreincr;
import minijava.node.AQualifiedName;
import minijava.node.AReferenceTypeType;
import minijava.node.AReturnStatement;
import minijava.node.AScalarInitList;
import minijava.node.ASimpleName;
import minijava.node.AStringliteralFactor;
import minijava.node.ASwitchBlock;
import minijava.node.ASwitchStatement;
import minijava.node.ATrueBooleanliteral;
import minijava.node.ATypeName;
import minijava.node.AVarDecInnerDeclaration;
import minijava.node.AWhileStatement;
import minijava.node.AXorBoolExpression;
import minijava.node.Node;
import minijava.node.PArrayAccess;
import minijava.node.PCommaArrayInit;
import minijava.node.PCommaExp;
import minijava.node.PPossibleCase;

/**
 * This does checks on more structural things in the code:
 * Values / Variables / Constants are not assigned to.
 * 
 * Unreachable code.
 * 
 * As a precondition, this must have a Type-Checked tree.
 * If the tree is not type-checked, then the behaviour here is
 * undefined.
 * @author toa02
 *
 */
public class StructureChecker extends DepthFirstAdapter {



   private ClassTable _classTab;	// Map names -> KTypes
   private SymbolTable _symTab;		// Holds globals and variable types
   //private FunctionTable _funcTab;	// Holds functions
   private VariableTable _varTab; // Holds KVariables as we go up and down the tree
   
   private Set _errStats;			// Statements with errors on them
   private List _errors;			// List of errors
   private List _infos;				// List of information links
   
   
   private Stack _valStack;			// Values propogated as we go up / down the tree
   private KVariable _lastVar; 		// The last KVariable seen.
   
   private Map _typeMap;			// Map of nodes -> KTypes
   private Map _funcMap;			// Map of nodes -> KFunction
   
   // Return statement tracking:
   private boolean _returned;		// if a return has happened
   private Set _returnPos;			// Set of int[3] of return positions
   
   
   private String _cScope;
   
   //Break statments
   private boolean _breakAllowed;	// if its ok for a break statement to happen.
   private boolean _breaked;		// if a break statement has been hit.
   private Set _breakPos;			// Set of int[3] of break'd positions.
   
   //Any form of loop rendering somewhere unreachable
   private boolean _loopUnreachable; // set true if a while loop makes somewhere unreachable.
   private Set _loopPos;			// Set of int[3] of loop (true) positions.
   
   private Set _arrayLengthSet;		// Set of nodes referring to array.length lookups
   private Set _enumChildSet;
   
   //Marker so we know we are doing the lhs of an assignment
   //i.e. we have to have non-constant variables and we return a KVariable aswell as the type.
   private boolean _doingAssig;
   
   public StructureChecker(ClassTable ct, SymbolTable st, FunctionTable ft,
	 VariableTable vt, Set errorStats, Map typeMap, Map funcMap, Set arrayLengths,
	 	Set enumChildSet){

      _classTab = ct;
      _symTab = st;
      //_funcTab = ft;
      _varTab = vt;

      _cScope = Util.GLOBAL;
      
      _arrayLengthSet = arrayLengths;
      _enumChildSet = enumChildSet;
      
      _valStack = new Stack();
      
      _errStats = errorStats;
      _errors = new LinkedList();
      _infos = new LinkedList();
      
      _returned = false;
      _returnPos = null;
      
      _breakAllowed = false;
      _breaked = false;
      _breakPos = null;

      
      _typeMap = typeMap;
      _funcMap = funcMap;
      
      _loopUnreachable = false;
      
      _doingAssig = false;
      _lastVar = null;
   }
   
   /**
    * Returns the list of errors that occured during using this pass.
    * @return List of errors.
    */
   public List getErrors(){
       return _errors;
   }
   
   /**
    * Returns the list of informations that occured during using this pass.
    * @return List of infos.
    */
   public List getInfos(){
       return _infos;
   }
   
   
   public IKValue getTopOfValueStack(){
       return (IKValue) _valStack.pop();
   }

   /*************************************************************************
    *  Here begins the cases. One by one, they will slowly steal my sanity. *
    *************************************************************************/
   
   
   /**
    * List statements: done 
    */
   public void caseAListStatements(AListStatements node){
       inAListStatements(node);
       
       try{
           /* If this statement errored on the type-checker, ignore it now. */
           if(!_errStats.contains(node.getStatement() )){
               
               // We have an unreachable statement due to something.
               if( _returned || _breaked || _loopUnreachable ){
                   int[] pos = Util.getFirstIdent(node.getStatement());
                   
                   int[][] lnkPos = new int[
                       (_returned ? _returnPos.size() : 0) +
                       (_breaked  ? _breakPos.size()  : 0) +
                       ( _loopUnreachable ? _loopPos.size() : 0)
                       ][];

                   int i = 0;
                   
                   
                   if( _returned ){
                       int[][] lnkRet = (int[][]) _returnPos.toArray(new int[0][0]);
                       for(int j = 0 ; j < lnkRet.length ; i++,j++){
                           lnkPos[i] = lnkRet[j];
                       }
                   }
                   
                   if( _breaked ){
                       int[][] lnkBrk = (int[][]) _breakPos.toArray(new int[0][0]);
	                   for(int j = 0 ; j < lnkBrk.length ; i++,j++){
	                       lnkPos[i] = lnkBrk[j];
	                   }
                   }
                   
                   if( _loopUnreachable ){
                       int[][] lnkLp = (int[][]) _loopPos.toArray(new int[0][0]);
                       for(int j = 0 ; j < lnkLp.length ; i++,j++ ){
                           lnkPos[i] = lnkLp[j];
                       }
                   }
                   
                   SourceCodeException.throwUnreachable(pos[0],pos[1],pos[2], lnkPos);
                       
               }
               
               /// Just for safety before each statement
               _valStack.clear();
               _lastVar = null; // lets just be safe !
               _doingAssig = false;
               node.getStatement().apply(this);
           }
       }catch(SourceCodeException sce){
           _errors.add(sce);
           _errStats.add(node.getStatement());
           _valStack.clear();
       }
       
       
 
       
       node.getStatements().apply(this);
       
       outAListStatements(node);
   }


   
     
   /**
    * Assignment statements: Done
    * @see minijava.analysis.Analysis#caseAAssignmentStatement(minijava.node.AAssignmentStatement)
    */
   public void caseAAssignmentStatement(AAssignmentStatement node){
       inAAssignmentStatement(node);
 
       node.getAssignment().apply(this);
       
       outAAssignmentStatement(node);
   }
   

   public void caseAAssignment(AAssignment node){
       inAAssignment(node);
       
       _lastVar = null;
       _doingAssig = true;
       // Lookup whats on the lhs of this assignment
       node.getFieldAccess().apply(this);
       _doingAssig = false;
       IKValue lhs = (IKValue) _valStack.pop();
       
       if(_lastVar == null ){
           // ok, we *didn't* find a variable!
           int[] pos = Util.getFirstIdent(node.getFieldAccess());
           String value = lhs.getString();
           SourceCodeException.throwAssignToValue(pos[0],pos[1], pos[2], value);
       }else if( _lastVar == MarkerKVariable.getArbitary() ){
           /* Safe? */
       }else if( _lastVar == MarkerKVariable.getEnumConst() ){
           throw KenyaPreconditionError.get();
       }else if ( _lastVar.isConstant()){
           int[] pos = Util.getFirstIdent(node.getFieldAccess());
           int[][] lnkPos = { Util.getFirstIdent(_lastVar.getNode()) };
           SourceCodeException.throwAssignToConstant(pos[0], pos[1], pos[2], lnkPos, _lastVar);
       }
       
       KVariable kv = _lastVar;
       
       _lastVar = null;
       node.getExpression().apply(this);
       IKValue rhs = (IKValue) _valStack.pop();
       
       if( kv != MarkerKVariable.getArbitary() &&
           kv != MarkerKVariable.getEnumConst() && 
           kv.getType() == KBasicType.getChar() ){
           if( rhs instanceof IKNumericValue ){
               IKNumericValue iknv = (IKNumericValue) rhs;
               
               if (iknv.inCharRange() ){
                   // yes its fine.
               }else{
                   // no its not!
                   int[] pos = Util.getFirstIdent(node.getExpression());
                   SourceCodeException.throwPossibleLossOfPrecisionCharInt(pos[0],pos[1],pos[2]);
               }
           }else if( Util.isSingleVariableExpression(node.getExpression() )){
               // then ok
           }else{
               int[] pos = Util.getFirstIdent(node.getExpression());
               SourceCodeException.throwPossibleLossOfPrecisionCharInt(pos[0],pos[1],pos[2]);
           }
       }
       
       _varTab.assignTo(kv, rhs);
       
       _lastVar = null;
       
       
       outAAssignment(node);
   }

   /**
    * If statements: Done ( I think )
    * @see minijava.analysis.Analysis#caseAIfStat(minijava.node.AIfStat)
    */
   public void caseAIfStat(AIfStat node){
       inAIfStat(node);
       
       node.getBoolExpression().apply(this);
       
       IKValue bexp = (IKValue) _valStack.pop();
   
       if( bexp instanceof KBooleanValue ){
           int[] pos = Util.getFirstIdent(node.getBoolExpression());
           
           if( bexp == KBooleanValue.getTrue() ){
               _infos.add( SourceCodeWarningFactory.alwaysTrueIf(pos[0],pos[1],pos[2]));
           }else if( bexp == KBooleanValue.getFalse() ){
               _infos.add( SourceCodeWarningFactory.alwaysFalseIf(pos[0],pos[1],pos[2]));
           }
       }

       // If part: stack and save where we are, do the if part, then destack, do the else part, then compare.
       // right: what do i stack?
       // SymbolTable can handle itself
       // variable table needs stacking.
       // as a pre to being here, _breaked and _returned must be false.
       if( _breaked || _returned || _loopUnreachable ){ throw KenyaPreconditionError.get();}
       
       boolean lhsRet;
       boolean lhsBrk;
       boolean lhsLp;
       
       VariableTable tmp = _varTab.createClone();
       
       // Do the lhs
       node.getBlock1().apply(this);
       lhsRet = _returned;
       lhsBrk = _breaked;
       lhsLp = _loopUnreachable;
       
       _returned = false;
       _breaked = false;
       _loopUnreachable = false;
       
       VariableTable lhs = _varTab;
       
       _varTab = tmp;
       
       // Do the rhs
       if( node.getElsePart() != null ){
           node.getElsePart().apply(this);
       }
       
       boolean rhsRet;
       boolean rhsBrk;
       boolean rhsLp;
       
       rhsRet = _returned;
       rhsBrk = _breaked;
       rhsLp = _loopUnreachable;
 
       boolean rhsBoth = rhsRet || rhsBrk || rhsLp;
       boolean lhsBoth = lhsRet || lhsBrk || lhsLp;
       
       if( (lhsRet || lhsBrk || lhsLp ) && (rhsRet || rhsBrk || rhsLp) ){

           
           _returned = (lhsRet && rhsBoth) || (rhsRet && lhsBoth);
           _breaked  = (lhsBrk && rhsBoth) || (rhsBrk && lhsBoth);
           _loopUnreachable = (lhsLp && rhsBoth) || (rhsLp && lhsBoth);
           
       }else{
           _returned = false;
           _breaked = false;
           _loopUnreachable = false;
       }
       
       if( lhsBoth ){
           // if we return from the lhs,
           // _varTab = rhs;
           _varTab = lhs;
       }else if( rhsBoth ){
           // if we don't return from the rhs
           _varTab = lhs;
       }else{
           _varTab = VariableTable.merge(lhs,_varTab);    
       }
       
 
       
       outAIfStat(node);
   }

   
   
	/**
	 * While statements : Done
	 * @see minijava.analysis.Analysis#caseAWhileStatement(minijava.node.AWhileStatement)
	 */
   	public void caseAWhileStatement(AWhileStatement node){
       inAWhileStatement(node);
       
       node.getBoolExpression().apply(this);

       IKValue condVal = (IKValue) _valStack.pop();
       
       if( condVal == KBooleanValue.getFalse() ){
           
           int ln = node.getWhile().getLine();
           int pos = node.getWhile().getPos();
           int len = node.getWhile().getText().trim().length();
           
           int[][] lnkPos = {{ln,pos,len}};
           
           int[] blockPos = Util.getFirstIdent(node.getBlock());
           
           
           _infos.add( SourceCodeWarningFactory.alwaysFalseWhile( ln,pos,len));
           
           try{
               SourceCodeException.throwUnreachable(blockPos[0],blockPos[1],blockPos[2], lnkPos);
           }catch(SourceCodeException sce){
               _errors.add(sce);
           }
           
           return;
           
       }else if( condVal == KBooleanValue.getTrue() ){
           int ln = node.getWhile().getLine();
           int pos = node.getWhile().getPos();
           int len = node.getWhile().getText().trim().length();
           _infos.add( SourceCodeWarningFactory.alwaysTrueWhile( ln,pos,len));
       }
       
       if( _breaked || _returned || _loopUnreachable ){ throw KenyaPreconditionError.get(); }
       
       
       boolean tmpBrk = _breakAllowed;
       _breakAllowed = true;
       
       VariableTable tmp = _varTab.createClone();
       
       node.getBlock().apply(this);
       
       _breakAllowed = tmpBrk;
       
       // If you don't break out of this while loop or return from it, and its condition is true,
       // then anything further is unreachable
       if( (!_breaked ) && (condVal == KBooleanValue.getTrue())){
           _loopUnreachable = true;
           int ln = node.getWhile().getLine();
           int pos = node.getWhile().getPos();
           int len = node.getWhile().getText().trim().length();
           int[] lnkPos = {ln,pos,len};
           _loopPos.add(lnkPos);
       }
       
       if( condVal == KBooleanValue.getTrue() ){
           // then we definatly enter the loop so the varTabs don't need merging
           // ( current one is the one used in the loop )
       }else{
           // we don't know if we enter the loop or not, so merge them.
           _varTab = VariableTable.merge(tmp, _varTab);
       }
       
       /*
        * If you break out of this loop somewhere, then
        * you can't have returned in it. 
        */
       if( _breaked ){
           _returnPos = new HashSet();
           _returned = false;
       }
       
       // At the end of a while loop, you've always "not breaked"..if that makes sense.
       _breaked = false;
       
       outAWhileStatement(node);
   }
   
   
   /**
    * Returns have already been type checked..and lookups will throw if we try and
    * use a variable that has not been assigned to yet. So this should be ok. I hope.
    * @see minijava.analysis.Analysis#caseAReturnStatement(minijava.node.AReturnStatement)
    */
   public void caseAReturnStatement(AReturnStatement node){
       
       inAReturnStatement(node);
       
       if( node.getExpression() != null ){
           node.getExpression().apply(this);
           _valStack.pop();
       }
       
       _returned = true;
       _returnPos.add(new int[] { 	node.getReturn().getLine(),
               						node.getReturn().getPos(),
               						node.getReturn().getText().trim().length()
       });
       
       outAReturnStatement(node);
   }

   
   /**
    * Just check to see if we have a constant or not.
    * @see minijava.analysis.Analysis#caseASwitchStatement(minijava.node.ASwitchStatement)
    */
   public void caseASwitchStatement(ASwitchStatement node){
       inASwitchStatement(node);
       
       node.getBoolExpression().apply(this);
       IKValue _switchValue = (IKValue) _valStack.pop();
       
       if( _switchValue != KUnknownValue.get()){
           int ln = node.getSwitch().getLine();
           int pos = node.getSwitch().getPos();
           int len = node.getSwitch().getText().trim().length();
           _infos.add( SourceCodeWarningFactory.alwaysConstantInSwitch(ln,pos,len));
       }
       
       
       
       if(node.getSwitchBlock() != null)
       {
           boolean tmp = _breakAllowed;
           node.getSwitchBlock().apply(this);
           _breakAllowed = tmp;
       }
       
       
       outASwitchStatement(node);
   }

   /**
    *  Need to reset the break and return stats after each one.
    * There could be some complications relating to returns aswell..
    */
   public void caseASwitchBlock(ASwitchBlock node){
       inASwitchBlock(node);
       
       Iterator it = node.getPossibleCase().iterator();
      
       if( _returned || _loopUnreachable){
           throw KenyaPreconditionError.get();
       }
       
       boolean allLoopRet = true;
       boolean allLoop = false;
       boolean allRet = false;
       boolean hasDefault = false;
       
       VariableTable tmp = _varTab.createClone();
       
       VariableTable allTrack = null;
       
       Map caseVals = new HashMap(); // Maps caseValues (IKValues) to nodes;
       
       while(it.hasNext()){
           _breakAllowed = true;
           
           PPossibleCase ppc = (PPossibleCase) it.next();
       
           hasDefault |= ppc instanceof ADefaultPossibleCase;
       
           ppc.apply(this);
           
           // check for duplicate keys
           if( ppc instanceof ACasePossibleCase ){
	           IKValue lastCaseVal = (IKValue) _valStack.pop();
	           if( caseVals.containsKey(lastCaseVal) ){
	               int[] thisPos = Util.getFirstIdent(ppc);
	               int[][] theirPos = { Util.getFirstIdent((Node)caseVals.get(lastCaseVal)) };
	               SourceCodeException.throwSwitchDuplicateLabel(thisPos[0], thisPos[1], thisPos[2], theirPos);
	           }else{
	               caseVals.put(lastCaseVal,ppc);
	           }
           }
           
           if( !_returned && !_loopUnreachable ){
           
	           if( _breaked || !it.hasNext()){
	               // reset for the next one
	               if( allTrack == null ){
	                   allTrack = _varTab.createClone();
	               }
	               allTrack = VariableTable.merge( _varTab, allTrack);
	               _varTab = tmp.createClone();
	           }else{
	               // fall through on variable table.
	               _varTab = VariableTable.merge( tmp.createClone(), _varTab );
	           }
           }else{
               
               _varTab = tmp.createClone();
           }
           
           if( _breaked || _loopUnreachable || _returned || !it.hasNext()){
               allLoopRet &= (_returned || _loopUnreachable);
               allRet |= _returned;
               allLoop |= _loopUnreachable;
           }
           
           _breaked = false;
           _returned = false;
           _loopUnreachable = false;
       }
       
       if( hasDefault){
           // then we are guaranteed to go through the switch.
           if( allTrack == null ){
               allTrack = _varTab.createClone();
           }
           _varTab = allTrack;
       }else{
           // then we are not.
           if( allTrack == null ){
               _varTab = tmp;
           }else{
               _varTab = VariableTable.merge(tmp,allTrack);
           }
       }
       
       if( hasDefault && allLoopRet ){
           _returned |= allRet;
           _loopUnreachable |= allLoop;
       }
       
       outASwitchBlock(node);
   }
   
   
   /**
    * Switch possible case: ensure that the boolean expressions are constant (i.e. not unknown)
    */
   public void caseACasePossibleCase(ACasePossibleCase node){
       inACasePossibleCase(node);
       
       node.getBoolExpression().apply(this);
       IKValue caseValue = (IKValue) _valStack.pop();
       
       if( caseValue == KUnknownValue.get() ){
           int[] pos = Util.getFirstIdent(node.getBoolExpression());
           SourceCodeException.throwSwitchNeedsConstant(pos[0], pos[1], pos[2]);
       }
       
       node.getBlock().apply(this);
   
       _valStack.push(caseValue);
       
       outACasePossibleCase(node);
   }

   /* Fall through :)
   public void caseADefaultPossibleCase(ADefaultPossibleCase node){
   }
   */

   /**
    * Due to historic reasons, half the logic is split between here and the
    * java for control below.  Oh well.
    */
   public void caseAForStatement(AForStatement node){
       inAForStatement(node);
       
       _symTab.newLocalScope();
       _varTab.pushScope();
       
       try{
	       node.getForControl().apply(this);
	       
	       IKValue condVal = (IKValue)  _valStack.pop();
	       
	       boolean tmpBrk = _breakAllowed;
	       _breakAllowed = true;
	       
	       VariableTable tmp = _varTab.createClone();
	       
	       node.getBlock().apply(this);
	       
	       _breakAllowed = tmpBrk;
	       
	       // If you don't break out of this while loop, and its condition is true,
	       // then anything further is unreachable
	       if( (!_breaked) && (condVal == KBooleanValue.getTrue())){
	           _loopUnreachable = true;
	           int ln = node.getFor().getLine();
	           int pos = node.getFor().getPos();
	           int len = node.getFor().getText().trim().length();
	           int[] lnkPos = {ln,pos,len};
	           _loopPos.add(lnkPos);
	       }
	       
	       if( condVal == KBooleanValue.getTrue() ){
	           // then we definatly enter the loop so the varTabs don't need merging
	           // ( current one is the one used in the loop )
	       }else{
	           // we don't know if we enter the loop or not, so merge them.
	           _varTab = VariableTable.merge(tmp, _varTab);
	       }
	       
	       /*
	        * If you break out of this loop somewhere, then
	        * you can't have returned in it. 
	        */
	       if( _breaked ){
	           _returnPos = new HashSet();
	           _returned = false;
	       }
	       
	       // At the end of a for loop, you've always "not breaked"..if that makes sense.
	       _breaked = false;
	       
       }finally{
           _symTab.pop();
       		_varTab.popScope();
       }
       outAForStatement(node);
   }
   
   
   /**
    * Links with and pushed back up to above.
    */
   public void caseAJavaForControl(AJavaForControl node){
       
       inAJavaForControl(node);
       
        
       node.getForLeftStat().apply(this);
             
       
       // The boolean expression: is it true or is it false?
       node.getBoolExpression().apply(this);
       IKValue condVal = (IKValue) _valStack.pop();
      
       
       if( condVal == KBooleanValue.getFalse() ){
           
           int ln = ((AForStatement)node.parent()).getFor().getLine();
           int pos = ((AForStatement)node.parent()).getFor().getPos();
           int len = ((AForStatement)node.parent()).getFor().getText().trim().length();
           
           int[][] lnkPos = {{ln,pos,len}};
           
           int[] blockPos = Util.getFirstIdent( ((AForStatement)node.parent()).getBlock() );
           
           _infos.add( SourceCodeWarningFactory.alwaysFalseFor( ln,pos,len));
           
           try{
               SourceCodeException.throwUnreachable(blockPos[0],blockPos[1],blockPos[2], lnkPos);
           }catch(SourceCodeException sce){
               _errors.add(sce);
           }
           _valStack.push(condVal);
           return;
           
       }else if( condVal == KBooleanValue.getTrue() ){
           int ln = ((AForStatement)node.parent()).getFor().getLine();
           int pos = ((AForStatement)node.parent()).getFor().getPos();
           int len = ((AForStatement)node.parent()).getFor().getText().trim().length();
           _infos.add( SourceCodeWarningFactory.alwaysTrueFor( ln,pos,len));
       }
       
       if( node.getForRightStat() != null ){
           node.getForRightStat().apply(this);
       }
       
       _valStack.push(condVal);
       

       
       outAJavaForControl(node);
       
   }

   public void caseAAssignmentForRightStat(AAssignmentForRightStat node){
       inAAssignmentForRightStat(node);
       
       node.getAssignment().apply(this);
       
       outAAssignmentForRightStat(node);
   }

   public  void caseAFunctioncallForRightStat(AFunctioncallForRightStat node){
       inAFunctioncallForRightStat(node);
       
       node.getFunctionApplication().apply(this);
       
       IKValue bexp = (IKValue) _valStack.pop();
       
       if( bexp != KVoidValue.get() ){
          	KFunction kf = (KFunction) _funcMap.get(node.getFunctionApplication());
          	int[][] lnkPos ;
          	if( kf.getNode() != null ){
          	    lnkPos = new int[][] { Util.getFirstIdent(kf.getNode().getIdentifier()) };
          	}else{
          	    lnkPos = new int[0][0];
          	}
          	int[] pos = Util.getFirstIdent(node);
           _infos.add( SourceCodeWarningFactory.ignoredReturnValue(pos[0], pos[1], pos[2], lnkPos));
       }
       

       outAFunctioncallForRightStat(node);
   }
   
   /**
    * Break statements:
    * @see minijava.analysis.Analysis#caseABreakStatement(minijava.node.ABreakStatement)
    */
   public void caseABreakStatement(ABreakStatement node){
       if( ! _breakAllowed ){
           int ln = node.getBreak().getLine();
           int pos = node.getBreak().getPos();
           int len = node.getBreak().getText().trim().length();
           SourceCodeException.throwIllegalBreakPlace(ln, pos, len);
       }
       
       _breaked = true;
       _breakPos.add(new int[] { 	node.getBreak().getLine(),
					node.getBreak().getPos(),
					node.getBreak().getText().trim().length()
       	});
       
   }

   /**
    * Assert - don't do much, except give an info if the assert is always true or false
    * @see minijava.analysis.Analysis#caseAAssertStatement(minijava.node.AAssertStatement)
    */
   public void caseAAssertStatement(AAssertStatement node){
       inAAssertStatement(node);
       
       node.getBoolExpression().apply(this);
       IKValue bexp = (IKValue) _valStack.pop();
       
       if( bexp instanceof KBooleanValue ){
           int[] pos = Util.getFirstIdent( node.getBoolExpression() );
           
           if( bexp == KBooleanValue.getTrue() ){
               _infos.add( SourceCodeWarningFactory.alwaysTrueAssert(pos[0],pos[1],pos[2]));
	       }else if( bexp == KBooleanValue.getFalse() ){
	           _infos.add( SourceCodeWarningFactory.alwaysFalseAssert(pos[0],pos[1],pos[2]));
	       }
       }
       
       if( node.getColonString() != null ){
	       node.getColonString().apply(this);
	       _valStack.pop();
       }
       
       outAAssertStatement(node);
   }
   
   /*
   public void caseAColonString(AColonString node){
   }
   */
   
   /**
    * Check to see if we are ignoring the return type of a non-void function.
    * If so give an info on it.
    * @see minijava.analysis.Analysis#caseAFunctioncallStatement(minijava.node.AFunctioncallStatement)
    */
   public void caseAFunctioncallStatement(AFunctioncallStatement node){
       
       inAFunctioncallStatement(node);

       node.getFunctionApplication().apply(this);
       
       
       IKValue bexp = (IKValue) _valStack.pop();
       
       if( bexp != KVoidValue.get() ){
          	KFunction kf = (KFunction) _funcMap.get(node.getFunctionApplication());
          	int[][] lnkPos ;
          	if( kf.getNode() != null ){
          	    lnkPos = new int[][] { Util.getFirstIdent(kf.getNode().getIdentifier()) };
          	}else{
          	    lnkPos = new int[0][0];
          	}
          	int[] pos = Util.getFirstIdent(node);
           _infos.add( SourceCodeWarningFactory.ignoredReturnValue(pos[0], pos[1], pos[2], lnkPos));
       }
       
       outAFunctioncallStatement(node);
       
   }


   /**
    * New block: Stack the scopes of the two tables.
    * @see minijava.analysis.Analysis#caseABlock(minijava.node.ABlock)
    */
   public void caseABlock(ABlock node){
       inABlock(node);
       
       _symTab.newLocalScope();
       _varTab.pushScope();
       
       try{
           node.getStatements().apply(this);
       }finally{
           _varTab.popScope();
           _symTab.pop();
       }
       
       outABlock(node);
   }

   /**
    * Funcion application:
    * Recurse down just to ensure we don't have unitialised variables
    * Lookup the return type: if its void then push void, if its not void, then push unknown.
    * @see minijava.analysis.Analysis#caseAFunctionApplication(minijava.node.AFunctionApplication)
    */
   public void caseAFunctionApplication(AFunctionApplication node){
       inAFunctionApplication(node);
       
       Stack tmp = _valStack;
       _valStack = new Stack();
       
       if( node.getActualParamList() != null ){
           node.getActualParamList().apply(this);       
       }
       
       _valStack = tmp;
       
       KType kt = (KType) _typeMap.get(node);
       
       if( kt == KBasicType.getVoid() ){
           _valStack.push(KVoidValue.get());
       }else{
           _valStack.push(KUnknownValue.get());
       }
       
       outAFunctionApplication(node);
   }

   /* Should recursively handle itself
   public void caseAListActualParamList(AListActualParamList node){
   }
   public void caseAExpActualParamList(AExpActualParamList node){
   }

   public void caseAEmptyActualParamList(AEmptyActualParamList node){
   }
   */



   /**
    * Setup the two tables to track this decalred variable:
    * & set its initial value if its initialised.
    */
   public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node){
       inAVarDecInnerDeclaration(node);

       //Ok first of all lets get this variable's name
       String ident = node.getIdentifier().getText().trim();
       KType type = (KType) _typeMap.get(node);
       
       KVariable kv = new KVariable(ident,type,false,node);

       if( node.getInitialiser() != null ){
           node.getInitialiser().apply(this);
           
           IKValue ikv = (IKValue) _valStack.pop();
           
           if( kv != MarkerKVariable.getArbitary() &&
                   kv != MarkerKVariable.getEnumConst() && 
                   kv.getType() == KBasicType.getChar() ){
               if( ikv instanceof IKNumericValue ){
                   IKNumericValue iknv = (IKNumericValue) ikv;
                   
                   if (iknv.inCharRange() ){
                       // yes its fine.
                   }else{
                       // no its not!
                       int[] pos = Util.getFirstIdent(node.getInitialiser());
                       SourceCodeException.throwPossibleLossOfPrecisionCharInt(pos[0],pos[1],pos[2]);
                   }
               }else if( Util.isSingleVariableExpression(node.getInitialiser() )){
                   // then ok
               }else{
                   int[] pos = Util.getFirstIdent(node.getInitialiser());
                   SourceCodeException.throwPossibleLossOfPrecisionCharInt(pos[0],pos[1],pos[2]);
               }
           }
           
           _symTab.pushLocal(ident,kv);
           if(!_varTab.declareNew(kv)){throw KenyaPreconditionError.get(); }
           
 
           _varTab.assignTo(kv,ikv);
       }else if( type instanceof KBoundClassType ){
           	// class type variables are assigned.
           
           _symTab.pushLocal(ident,kv);
           if(!_varTab.declareNew(kv)){throw KenyaPreconditionError.get(); }
           
           
           _varTab.assignTo(kv,KUnknownValue.get());
       }else {
           _symTab.pushLocal(ident,kv);
           if(!_varTab.declareNew(kv)){throw KenyaPreconditionError.get(); }
       }
       
 
       
       outAVarDecInnerDeclaration(node);
   }

   /**
    * Pretty much as above.
    * @see minijava.analysis.Analysis#caseAArrayDecInnerDeclaration(minijava.node.AArrayDecInnerDeclaration)
    */
   public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node){
       inAArrayDecInnerDeclaration(node);
       
       //Ok first of all lets get this variable's name
       String ident = node.getIdentifier().getText().trim();
       KType type = (KType) _typeMap.get(node);
       
       KVariable kv = new KVariable(ident,type,false,node);
       _symTab.pushLocal(ident,kv);
       _varTab.declareNew(kv);
       
       if( node.getArrayInitialiser() != null ){
           node.getArrayInitialiser().apply(this);
           IKValue ikv = (IKValue) _valStack.pop();
           _varTab.assignTo(kv,ikv);
       }
       
       outAArrayDecInnerDeclaration(node);
   }

   /**
    * Don't recurse down, don't do nothing :)
    * @see minijava.analysis.Analysis#caseAClassDecDeclaration(minijava.node.AClassDecDeclaration)
    */
   public void caseAClassDecDeclaration(AClassDecDeclaration node){
       inAClassDecDeclaration(node);
       outAClassDecDeclaration(node);
   }

   /**
    * Don't do anything.
    * @see minijava.analysis.Analysis#caseAEnumDecDeclaration(minijava.node.AEnumDecDeclaration)
    */
   public void caseAEnumDecDeclaration(AEnumDecDeclaration node){
       inAEnumDecDeclaration(node);
       outAEnumDecDeclaration(node);
   }

   
   public void caseAEnumList(AEnumList node){
       throw KenyaBadLocationError.get();
   }

   public void caseACommaEnumList(ACommaEnumList node){
       throw KenyaBadLocationError.get();
   }
   
   
   /**
    * Function declaration: check the block and if if should have returned when it didn't - give an error
    * @see minijava.analysis.Analysis#caseAFuncDecDeclaration(minijava.node.AFuncDecDeclaration)
    */
   public void caseAFuncDecDeclaration(AFuncDecDeclaration node){
       if( _errStats.contains(node)){ return; }
       inAFuncDecDeclaration(node);
       
       
       
       KFunction kf = (KFunction) _funcMap.get(node);
       SymbolTable st = kf.getSymbolTable();
       
       _cScope = "method " + kf.getName() + FunctionTable.paramsToString(kf.getArgs());
       
       if(!_symTab.push(st)){throw KenyaPreconditionError.get(); }
       _varTab.pushSymbolTable(st);
       
       
       _returned = false;
       _breaked = false;
       _breakAllowed = false;
       _loopUnreachable = false;
       _breakPos = new HashSet();
       _loopPos = new HashSet();
       _returnPos = new HashSet();
       
       try{
           node.getBlock().apply(this);
       }finally{
           _varTab.popScope();
           _symTab.pop();
       }
       
       // If we are not a void function, we havn't returned and loops don't stop us
       if( kf.getReturnType() != KBasicType.getVoid() && !_returned && !_loopUnreachable ){
           try{
               int ln = ((ABlock)node.getBlock()).getRBrace().getLine();
               int pos = ((ABlock)node.getBlock()).getRBrace().getPos();
               int len = ((ABlock)node.getBlock()).getRBrace().getText().trim().length();
               int[][] lnkPos = { Util.getFirstIdent(node.getIdentifier()) };
               SourceCodeException.throwMissingReturnStatement(ln, pos, len, lnkPos, kf.getReturnType());
           }catch(SourceCodeException sce){
               _errors.add(sce);
           }
       }
       
       outAFuncDecDeclaration(node);
   }

   /**
    * Do nothing: constants should already have been covered
    * @see minijava.analysis.Analysis#caseAConstDecDeclaration(minijava.node.AConstDecDeclaration)
    */
   public void caseAConstDecDeclaration(AConstDecDeclaration node){
       inAConstDecDeclaration(node);
       outAConstDecDeclaration(node);
   }

   /**
    * Should never be called.
    * @see minijava.analysis.DepthFirstAdapter#inAFormalParamList(minijava.node.AFormalParamList)
    */
   public void inAFormalParamList(AFormalParamList node) {
       throw KenyaBadLocationError.get();
   }


   /**
    * Should never be called.
    * @see minijava.analysis.DepthFirstAdapter#inATypeName(minijava.node.ATypeName)
    */
   public void inATypeName(ATypeName node) {
       throw KenyaBadLocationError.get();
   }

   /**
    * Should never be called.
    * @see minijava.analysis.DepthFirstAdapter#inACommaTypeName(minijava.node.ACommaTypeName)
    */
   public void inACommaTypeName(ACommaTypeName node) {
       throw new KenyaInternalError("Shouldn't be in here");
   }
   
   /*
   public void caseAInitialiser(AInitialiser node){
   }
   */
   /*
   public void caseAStaticArrayInitialiser(AStaticArrayInitialiser node){
   }
   */

   /**
	 * @see minijava.analysis.DepthFirstAdapter#caseAArrayAllocate(minijava.node.AArrayAllocate)
	*/
	public void caseAArrayAllocate(AArrayAllocate node) {
	    inAArrayAllocate(node);
	    
	    Iterator it = node.getArrayAccess().iterator();
	    
	    while(it.hasNext()){
	        ((PArrayAccess)it.next()).apply(this);
	        _valStack.pop();
	    }
	    
	    outAArrayAllocate(node);
	}
   
   /**
    * Right the type checker should have made sure this is legal:
    */
   public void caseADynamicArrayInitialiser(ADynamicArrayInitialiser node){
       inADynamicArrayInitialiser(node);
       
       node.getArrayAllocate().apply(this);
       _valStack.pop();
       _valStack.push(KUnknownValue.get());
       
       outADynamicArrayInitialiser(node);
   }

   public void caseAArrayInit(AArrayInit node){
       inAArrayInit(node);
       
       node.getInitList().apply(this);
       // the init lists do the pops for you...
       
       _valStack.push(KUnknownValue.get());
       
       outAArrayInit(node);
   }

   public void caseAScalarInitList(AScalarInitList node){
       inAScalarInitList(node);
       
       node.getExpression().apply(this);
       _valStack.pop();
       
       Iterator i = node.getCommaExp().iterator();
       while(i.hasNext()){
           ((PCommaExp) i.next()).apply(this);
           _valStack.pop();
       }
       
       outAScalarInitList(node);
   }

   public void caseAArrayInitList(AArrayInitList node){
       inAArrayInitList(node);
       
       node.getArrayInit().apply(this);
       _valStack.pop();
       
       Iterator i = node.getCommaArrayInit().iterator();
       while(i.hasNext()){
           ((PCommaArrayInit) i.next()).apply(this);
           _valStack.pop();
       }
       
       outAArrayInitList(node);
   }

   public void caseABasicTypeType(ABasicTypeType node){
       throw KenyaBadLocationError.get();
   }

   /**
    * @see minijava.analysis.DepthFirstAdapter#caseAReferenceTypeType(minijava.node.AReferenceTypeType)
    */
   public void caseAReferenceTypeType(AReferenceTypeType node) {
	    throw KenyaBadLocationError.get();
   }
   
   
   /**
    */
   public void outAArrayAllocate(AArrayAllocate node){
       _valStack.push(KUnknownValue.get());
   }
   

   public void caseAPlusMathExpression(APlusMathExpression node){
       inAPlusMathExpression(node);
       
       node.getMathExpression().apply(this);
       IKValue lhs = (IKValue) _valStack.pop();
       
       
       node.getTerm().apply(this);
       IKValue rhs = (IKValue) _valStack.pop();
       
       _valStack.push(KValueCalculator.add(lhs, rhs));
       
       outAPlusMathExpression(node);
   }

   public void caseAMinusMathExpression(AMinusMathExpression node){
       inAMinusMathExpression(node);
       
       node.getMathExpression().apply(this);
       IKValue lhs = (IKValue) _valStack.pop();
       
       node.getTerm().apply(this);
       IKValue rhs = (IKValue) _valStack.pop();
       
       _valStack.push(KValueCalculator.sub(lhs, rhs));
       
       outAMinusMathExpression(node);
   }

   public void caseAMultTerm(AMultTerm node){
       inAMultTerm(node);
       
       node.getTerm().apply(this);
       IKValue lhs = (IKValue) _valStack.pop();
       
       node.getUnaryExp().apply(this);
       IKValue rhs = (IKValue) _valStack.pop();
       
       _valStack.push(KValueCalculator.mult(lhs, rhs));
       
       outAMultTerm(node);
   }

   public void caseADivTerm(ADivTerm node){
   
       inADivTerm(node);
       
       node.getTerm().apply(this);
       IKValue lhs = (IKValue) _valStack.pop();
       
       node.getUnaryExp().apply(this);
       IKValue rhs = (IKValue) _valStack.pop();
       
       if( rhs instanceof IKNumericValue && 
               ((IKNumericValue)rhs).equalsZero() ){
           
           int ln = node.getDivide().getLine();
           int pos = node.getDivide().getPos();
           int len = node.getDivide().getText().trim().length();
           
           _infos.add( SourceCodeWarningFactory.divideByZero(ln, pos, len));
       }
       
       _valStack.push(KValueCalculator.div(lhs, rhs));
       
       outADivTerm(node);
       
   }

   public void caseAModTerm(AModTerm node){
       inAModTerm(node);
       
       node.getTerm().apply(this);
       IKValue lhs = (IKValue) _valStack.pop();
       
       node.getUnaryExp().apply(this);
       IKValue rhs = (IKValue) _valStack.pop();
       
       if( rhs instanceof IKNumericValue && 
               ((IKNumericValue)rhs).equalsZero() ){
           
           int ln = node.getMod().getLine();
           int pos = node.getMod().getPos();
           int len = node.getMod().getText().trim().length();
           
           _infos.add( SourceCodeWarningFactory.moduloByZero(ln, pos, len));
       }
       
       _valStack.push(KValueCalculator.mod(lhs,rhs));
       
       outAModTerm(node);
   }

   public void caseAMinusUnaryExp(AMinusUnaryExp node){
       inAMinusUnaryExp(node);

       node.getUnaryExp().apply(this);
       IKValue lhs = (IKValue) _valStack.pop();
       
       _valStack.push(KValueCalculator.unSub(lhs));
       
       outAMinusUnaryExp(node);
   }

   /**
    * Pass through the variable.
    * @see minijava.analysis.Analysis#caseAPlusUnaryExp(minijava.node.APlusUnaryExp)
    */
   public void caseAPlusUnaryExp(APlusUnaryExp node){

       inAPlusUnaryExp(node);
       
       node.getUnaryExp().apply(this);
       
       outAPlusUnaryExp(node);
   
   }

   public void caseANegateUnaryExp(ANegateUnaryExp node){
       inANegateUnaryExp(node);
       
       node.getUnaryExp().apply(this);
       
       IKValue lhs = (IKValue) _valStack.pop();
       
       _valStack.push(KValueCalculator.unNeg(lhs));
       
       outANegateUnaryExp(node);
   }

   
   public void caseAPostdecr(APostdecr node){
   		inAPostdecr(node);
   		
   		_lastVar = null;
   		
   		node.getFieldAccess().apply(this);
   		
   		if( _lastVar == null ){ 
   		    int[] pos = Util.getFirstIdent(node);
   		    SourceCodeException.throwUnopValueMM(pos[0],pos[1],pos[2]);
   		}

        if( _lastVar.isConstant()){
            int[] pos = Util.getFirstIdent(node.getFieldAccess());
            int[][] lnkPos = { Util.getFirstIdent(_lastVar.getNode()) };
            SourceCodeException.throwAssignToConstant(pos[0], pos[1], pos[2], lnkPos, _lastVar);
        }
   		
   		IKValue lhs = (IKValue) _valStack.pop();
   		IKValue res = KValueCalculator.minusminus(lhs);
   		_valStack.push(res);
   		_varTab.assignTo(_lastVar,res);
   		_lastVar = null;
   		
   		outAPostdecr(node);
   }

   public void caseAPostincr(APostincr node){
 		inAPostincr(node);
   		
   		_lastVar = null;
   		
   		node.getFieldAccess().apply(this);
   		
   		if( _lastVar == null ){ 
   		    int[] pos = Util.getFirstIdent(node.getFieldAccess());
   		    SourceCodeException.throwUnopValuePP(pos[0],pos[1],pos[2]);
   		}
   		
        if( _lastVar.isConstant()){
            int[] pos = Util.getFirstIdent(node.getFieldAccess());
            int[][] lnkPos = { Util.getFirstIdent(_lastVar.getNode()) };
            SourceCodeException.throwAssignToConstant(pos[0], pos[1], pos[2], lnkPos, _lastVar);
        }
   		

   		IKValue lhs = (IKValue) _valStack.pop();
   		IKValue res = KValueCalculator.plusplus(lhs);
   		_valStack.push(res);
   		_varTab.assignTo(_lastVar,res);
   		_lastVar = null;
   		
   		outAPostincr(node);
   }

   public void caseAPredecr(APredecr node){
  		inAPredecr(node);
   		
   		_lastVar = null;
   		
   		node.getFieldAccess().apply(this);
   		
   		if( _lastVar == null ){ 
   		    int[] pos = Util.getFirstIdent(node.getFieldAccess());
   		    SourceCodeException.throwUnopValueMM(pos[0],pos[1],pos[2]);
   		}
   		
        if( _lastVar.isConstant()){
            int[] pos = Util.getFirstIdent(node.getFieldAccess());
            int[][] lnkPos = { Util.getFirstIdent(_lastVar.getNode()) };
            SourceCodeException.throwAssignToConstant(pos[0], pos[1], pos[2], lnkPos, _lastVar);
        }

   		IKValue lhs = (IKValue) _valStack.pop();
   		IKValue res = KValueCalculator.minusminus(lhs);
   		_valStack.push(res);
   		_varTab.assignTo(_lastVar,res);
   		
   		_lastVar = null;
   		
   		outAPredecr(node);
   }

   public void caseAPreincr(APreincr node){
		inAPreincr(node);
   		
   		_lastVar = null;
   		
   		node.getFieldAccess().apply(this);
   		
   		if( _lastVar == null ){ 
   		    int[] pos = Util.getFirstIdent(node.getFieldAccess());
   		    SourceCodeException.throwUnopValuePP(pos[0],pos[1],pos[2]);
   		}
   		
        if( _lastVar.isConstant()){
            int[] pos = Util.getFirstIdent(node.getFieldAccess());
            int[][] lnkPos = { Util.getFirstIdent(_lastVar.getNode()) };
            SourceCodeException.throwAssignToConstant(pos[0], pos[1], pos[2], lnkPos, _lastVar);
        }

   		IKValue lhs = (IKValue) _valStack.pop();
   		IKValue res = KValueCalculator.plusplus(lhs);
   		_valStack.push(res);
   		_varTab.assignTo(_lastVar,res);
   		_lastVar = null;
   		
   		outAPreincr(node);
   }

   public void caseANullFactor(ANullFactor node){
       inANullFactor(node);
       _valStack.push(KNullValue.get());
       outANullFactor(node);
   }

   public void caseAStringliteralFactor(AStringliteralFactor node){
       inAStringliteralFactor(node);
       String text = node.getStringliteral().getText();
       
       int start = text.indexOf("\"");
       int end   = text.lastIndexOf("\"");
       
       text = text.substring(start+1, end);
       _valStack.push( new KStringValue(Util.escapeString(text)));
       
       outAStringliteralFactor(node);
   }

   public void caseACharliteralFactor(ACharliteralFactor node){
       inACharliteralFactor(node);
       
       char c = Util.parseCharacter(node);
       _valStack.push( new KCharValue(c));
       
       outACharliteralFactor(node);
   }

   public void caseATrueBooleanliteral(ATrueBooleanliteral node){
       inATrueBooleanliteral(node);
       _valStack.push(KBooleanValue.getTrue());
       outATrueBooleanliteral(node);
   }

   public void caseAFalseBooleanliteral(AFalseBooleanliteral node){
       inAFalseBooleanliteral(node);
       _valStack.push(KBooleanValue.getFalse());
       outAFalseBooleanliteral(node);
   }

   public void caseAINumber(AINumber node){
       inAINumber(node);
       
       int i;
       try{
           i = Integer.parseInt(node.getIntnumber().getText().trim());
           KIntValue kiv = new KIntValue(i);
           if( kiv.inCharRange() ){
               KCharValue kcv = new KCharValue((char)i);
               _valStack.push(kcv);
           }else{
               _valStack.push(kiv);
           }
       }catch(NumberFormatException e){
           int ln = node.getIntnumber().getLine();
           int pos = node.getIntnumber().getPos();
           int len = node.getIntnumber().getText().trim().length();
           SourceCodeException.throwBadInt(ln, pos, len, node.getIntnumber().getText().trim());
       }
       
       outAINumber(node);
   }

   public void caseADNumber(ADNumber node){
       
       inADNumber(node);
       
       double d;
       
       int ln = node.getDpnumber().getLine();
       int pos = node.getDpnumber().getPos();
       int len = node.getDpnumber().getText().trim().length();
       
       try{
           d = Double.parseDouble(node.getDpnumber().getText().trim());
           if( Double.isInfinite(d) || Double.isNaN(d)){
               SourceCodeException.throwBadDouble(ln,pos,len,node.getDpnumber().getText().trim());
               return; // unreachable but to make it clearer for the moment
           }
           
           KDoubleValue kdv = new KDoubleValue(d);
           _valStack.push(kdv);
       }catch(NumberFormatException e){
           SourceCodeException.throwBadDouble(ln,pos,len,node.getDpnumber().getText().trim());
       }
       
       outADNumber(node);
       
   }

   public void caseAOrBoolExpression(AOrBoolExpression node){
      inAOrBoolExpression(node);

      node.getBoolExpression().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getBoolTerm().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.or(lhs, rhs);
      
      _valStack.push(res);
      
      outAOrBoolExpression(node);
   }

   public void caseAXorBoolExpression(AXorBoolExpression node){
      inAXorBoolExpression(node);
      
      node.getBoolExpression().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getBoolTerm().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.xor(lhs, rhs);
      
      _valStack.push(res);
      
      
      outAXorBoolExpression(node);
   }

   public void caseAAndBoolTerm(AAndBoolTerm node){
      inAAndBoolTerm(node);

      node.getBoolTerm().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getEquality().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.and(lhs, rhs);
      _valStack.push(res);
      
      outAAndBoolTerm(node);
   }

   public void caseAEqEquality(AEqEquality node){
      inAEqEquality(node);

      node.getEquality().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getRelational().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();

      IKValue res = KValueCalculator.equal(lhs, rhs);
      _valStack.push(res);
      
      outAEqEquality(node);
   }

   public void caseANeqEquality(ANeqEquality node){
      inANeqEquality(node);
      
      node.getEquality().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getRelational().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();

      IKValue res = KValueCalculator.unequal(lhs, rhs);
      _valStack.push(res);
      
      outANeqEquality(node);
   }

   public void caseALtRelational(ALtRelational node){
      inALtRelational(node);
      
      node.getRelational().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getMathExpression().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.less(lhs, rhs);
      _valStack.push(res);
      
      outALtRelational(node);
   }

   public void caseAGtRelational(AGtRelational node){
      inAGtRelational(node);
      
      node.getRelational().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getMathExpression().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.greater(lhs, rhs);
      _valStack.push(res);
      outAGtRelational(node);
   }

   public void caseALteqRelational(ALteqRelational node){
      inALteqRelational(node);

      node.getRelational().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getMathExpression().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.lessEq(lhs, rhs);
      _valStack.push(res);
      outALteqRelational(node);
   }

   public void caseAGteqRelational(AGteqRelational node){
      inAGteqRelational(node);
      
      node.getRelational().apply(this);
      IKValue lhs = (IKValue) _valStack.pop();
      
      node.getMathExpression().apply(this);
      IKValue rhs = (IKValue) _valStack.pop();
      
      IKValue res = KValueCalculator.greaterEq(lhs, rhs);
      _valStack.push(res);  
      outAGteqRelational(node);
   }

   
   public void caseAArrayFieldAccess(AArrayFieldAccess node){
      inAArrayFieldAccess(node);
      
      node.getName().apply(this);
      IKValue ikv = (IKValue) _valStack.pop();
      
      Iterator it = node.getArrayAccess().iterator();

      // Save the last KVariable and
      // whether we are doing an assig
      // so variables inside the array brackets don't get too confused.
      boolean tmpDoingAssig = _doingAssig;
      KVariable tmp = _lastVar;
      _doingAssig = false;

      while(it.hasNext()){
          ((PArrayAccess)it.next()).apply(this);
          _valStack.pop();
      }
      _lastVar = tmp;
      _valStack.push(ikv);
      
      if( _lastVar == MarkerKVariable.getEnumConst() ){
          _lastVar = MarkerKVariable.getArbitary();
      }
      
      _doingAssig = tmpDoingAssig;
      
      
      outAArrayFieldAccess(node);
   }

   /**
    * @see minijava.analysis.Analysis#caseASimpleName(minijava.node.ASimpleName)
    */
   public void caseASimpleName(ASimpleName node){
      inASimpleName(node);

      String ident = node.getIdentifier().getText().trim();
      
      KVariable kv;
      IKValue ikv;

      if( _enumChildSet.contains(node)){
          KEnumType ket = (KEnumType) _typeMap.get(node);
          _lastVar = null;
          _valStack.add( new KEnumValue(ket,ident));
          outASimpleName(node);
          return;
      }
      
      if( _symTab.containsVariable(ident)){
          kv = _symTab.lookupVariable(ident);
          ikv = _varTab.lookup(kv);
      }else if( _classTab.isEnum(ident)){
          kv = MarkerKVariable.getEnumConst();
          ikv = KUnknownValue.get();
      }else{
          //throw new KenyaInternalError("Have a variable taht we don't know what it is!");
          int ln = node.getIdentifier().getLine();
          int pos = node.getIdentifier().getPos();
          int len = node.getIdentifier().getText().trim().length();
          SourceCodeException.throwLost_Variable(ln,pos,len,ident,_cScope);
          return; // unreachable
      }
      
      if( ikv == KUnassignedValue.get() && !_doingAssig ){
          int ln = node.getIdentifier().getLine();
          int pos = node.getIdentifier().getPos();
          int len = node.getIdentifier().getText().trim().length();
          int[][] lnkPos = { Util.getFirstIdent(kv.getNode()) };
          SourceCodeException.throwUninitialisedVariable(ln, pos, len, lnkPos, node.getIdentifier().getText().trim());
      }
      
      _lastVar = kv;
      _valStack.push(ikv);
      
      outASimpleName(node);
   }

   public void caseAQualifiedName(AQualifiedName node){
      inAQualifiedName(node);
      
      node.getFieldAccess().apply(this);
      //IKValue ikv = (IKValue)_valStack.pop();
      //KVariable kv = _lastVar;
      
      KType kt = (KType)  _typeMap.get(node);

      if( _arrayLengthSet.contains(node)){
          if( _doingAssig ){
              int ln = node.getIdentifier().getLine();
              int pos = node.getIdentifier().getPos();
              int len = node.getIdentifier().getText().trim().length();
              SourceCodeException.throwInvalidArrayLengthAssig(ln, pos, len);
          }else{
              _lastVar = MarkerKVariable.getArbitary();
              _valStack.push(KUnknownValue.get());
          }
      }else if( _lastVar == MarkerKVariable.getEnumConst() || ( _lastVar != MarkerKVariable.getArbitary() && _lastVar.getType() instanceof KEnumType) ){
          if( _doingAssig ){
              int ln = node.getIdentifier().getLine();
              int pos = node.getIdentifier().getPos();
              int len = node.getIdentifier().getText().trim().length();
              String name = node.getIdentifier().getText().trim();
              SourceCodeException.throwInvalidEnumAssig(ln, pos, len, name);
          }else if( kt instanceof KEnumType ){
              KEnumType ket = (KEnumType) kt;
              String child = node.getIdentifier().getText().trim();
              _valStack.push(new KEnumValue(ket,child));
          }else{
              _valStack.push(KUnknownValue.get());
              _lastVar = MarkerKVariable.getArbitary();
          }
      }else{
          _valStack.push(KUnknownValue.get());
          _lastVar = MarkerKVariable.getArbitary();
      }
      
      
      
      outAQualifiedName(node);
   }

}
