/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Aug-2004 by toa02
 *
 */
package kenya.passes;

import java.util.List;
import java.util.Map;

import kenya.builtIns.IBuiltInMethod;
import kenya.errors.KenyaPreconditionError;
import kenya.types.KBasicType;
import kenya.types.KBoundClassType;
import kenya.types.KClassType;
import kenya.types.KFunction;
import kenya.types.KType;
import mediator.IJavaCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.*;


/**
 * Here we are, the translator.
 * Wow.
 * @author toa02
 *
 */
public class Translator extends DepthFirstAdapter {

    public static final String INDENT =  "    ";
    private static final int MAX_WIDTH = 80;
    
    
    public static final int FUNCTIONS 	= 				1;
    public static final int CLASSES		= FUNCTIONS <<  1;
    public static final int ENUMS			= CLASSES 	<< 	1;
    public static final int CONSTANTS		= ENUMS		<< 	1;
    
    
    private static final char DELIM = '\ubabe';
    
    private StringBuffer _code;
    private final Map _functionMap;	// Maps nodes to KFunctions that are called.
    private final Map _typeMap;		// Maps nodes to KTypes of what the node type is.
    private final Node _globalMain;
    private int _options;
    
    private int _indentLevel;
    private final boolean _doBreaking;
    
    private boolean _inTypeParams;
    
    public Translator( Map functionMap , Node globalMain, Map typeMap, boolean doBreaking ){
        _functionMap = functionMap;
        _code = new StringBuffer();
        _options = 0;
        _indentLevel = 0;
        _oldIndent = 0;
        _globalMain = globalMain;
        _lastPos = 0;
        _lastNlPos = 0;
        _doBreaking = doBreaking;
        _inTypeParams = false;
        _typeMap = typeMap;
    }
    
    
    public void reset(int newOptions){
        _code = new StringBuffer();
        _options = newOptions;
        _indentLevel = 0;
        _oldIndent = 0;
        _lastPos = 0;
        _lastNlPos = 0;
    }
    
    public String getLastCode(){
        if( !_doBreaking ){
            throw KenyaPreconditionError.get();
        }
        
        return _code.toString();
    }

    public String[] getCodeArray(){
        if( _doBreaking ){
            throw KenyaPreconditionError.get();
        }
        
        return _code.toString().split(""+ DELIM);
    }
    
    private void doIndent(){
        for(int i = 0 ; i < _indentLevel ; i++){
            _code.append(INDENT);
        }
    }

    private void doSpace(){
        _code.append(" ");
        tokenOut();
    }
    
    private int _lastNlPos; 
    private int _lastPos;
    private int _oldIndent;
    
    private void doNewLine(){
        _code.append(IJavaCode.NEWLINE);
        _lastNlPos = _code.length();
        _lastPos = _code.length();
    }
    
    
    /**
     * @see minijava.analysis.DepthFirstAdapter#defaultOut(minijava.node.Node)
     */
    public void tokenOut() {
        if( !_doBreaking ){ 
            _code.append(DELIM);
            return; 
        }
        
        int lnLen = _code.length() - _lastNlPos; 
        int lastItemLen = _code.length() - _lastPos;
        
        if( lnLen > MAX_WIDTH && lastItemLen < MAX_WIDTH){
            String lastAdded = _code.substring(_lastPos);
            _code = new StringBuffer( _code.substring(0,_lastPos));
            
            int tmp = _indentLevel;
            
            _indentLevel = _oldIndent + 1;
            doNewLine();
            doIndent();
            _indentLevel = tmp;
            
            _code.append(lastAdded);
        }

        _oldIndent = _indentLevel;
        _lastPos = _code.length();        
    }
    
    /* -------------------------------------------------------------------- */
    /* Here begin the cases, one by one they will steal my sanity slowly... */
    /* -------------------------------------------------------------------- */
    
    public void caseAClassDecDeclaration(AClassDecDeclaration node){
        if( (_options & CLASSES) != CLASSES){ return; }
        
        inAClassDecDeclaration(node);
        
        _indentLevel = 0;
        
        doNewLine();
        doNewLine();
        
        node.getKlass().apply(this);
        doSpace();

        node.getIdentifier().apply(this);

        if(node.getTypeParam() != null)
        {
            node.getTypeParam().apply(this);
        }

        node.getLBrace().apply(this);
        
        _indentLevel = 1;
        
        {
            Object temp[] = node.getClassInnerDeclaration().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                doNewLine();
                doIndent();
                ((PClassInnerDeclaration) temp[i]).apply(this);
            }
        }

        doNewLine();
        node.getRBrace().apply(this);
        
        outAClassDecDeclaration(node);
        
    }


    public void caseAEnumDecDeclaration(AEnumDecDeclaration node){
        if( (_options & ENUMS) != ENUMS){ return; }

            inAEnumDecDeclaration(node);
            
            doNewLine();
            node.getEnum().apply(this);
            doSpace();

            node.getIdentifier().apply(this);

            node.getLBrace().apply(this);
            doNewLine();
            _indentLevel = 1;
            doIndent();
            
            if(node.getEnumList() != null)
            {
                node.getEnumList().apply(this);
            }

            node.getSemicolon().apply(this);
            doNewLine();
            node.getRBrace().apply(this);

            outAEnumDecDeclaration(node);
    }


    public void caseAFuncDecDeclaration(AFuncDecDeclaration node){
        if( (_options & FUNCTIONS) != FUNCTIONS){ return; }
  
        inAFuncDecDeclaration(node);

        doNewLine();
        doNewLine();
        
        _indentLevel = 1;
        doIndent();
        
        if( _globalMain == node ){
            _code.append("public ");
        }
        
        _code.append("static");
        
        if(node.getTypeParam() != null)
        {
            node.getTypeParam().apply(this);
        }
        
        doSpace();
        
        node.getType().apply(this);
        
        {
            Object temp[] = node.getBracketPair().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((TBracketPair) temp[i]).apply(this);
            }
        }
        
        doSpace();
        
        node.getIdentifier().apply(this);
        node.getLParenthese().apply(this);

        if(node.getFormalParamList() != null)
        {
            node.getFormalParamList().apply(this);
        }

        node.getRParenthese().apply(this);
        
        doSpace();
        
        node.getBlock().apply(this);

        outAFuncDecDeclaration(node);
        
        
    }


    public void caseAConstDecDeclaration(AConstDecDeclaration node){
        if( (_options & CONSTANTS) != CONSTANTS){ return; }
        
        inAConstDecDeclaration(node);
        doNewLine();

        _indentLevel = 1;
        doIndent();
        
        node.getConst().apply(this);
        doSpace();
        node.getType().apply(this);
        doSpace();
        node.getIdentifier().apply(this);
        doSpace();
        node.getInitialiser().apply(this);
        node.getSemicolon().apply(this);
        
        outAConstDecDeclaration(node);
    }
    
    /**
     * @see minijava.analysis.DepthFirstAdapter#caseAListStatements(minijava.node.AListStatements)
     */
    public void caseAListStatements(AListStatements node) {
        inAListStatements(node);
        
        doNewLine();
        doIndent();
        node.getStatement().apply(this);

        if(node.getStatements() != null)
        {
            node.getStatements().apply(this);
        }
        
        outAListStatements(node);
    }


    public void caseAFunctioncallStatement(AFunctioncallStatement node){
      
        inAFunctioncallStatement(node);
        node.getFunctionApplication().apply(this);
        node.getSemicolon().apply(this);
        outAFunctioncallStatement(node);
    }

    
    public void caseAInnerDecStatement(AInnerDecStatement node){
        inAInnerDecStatement(node);
        
        node.getInnerDeclaration().apply(this);
        node.getSemicolon().apply(this);
        
        outAInnerDecStatement(node);
    }
    
    public void caseAAssignmentStatement(AAssignmentStatement node){
        inAAssignmentStatement(node);

        node.getAssignment().apply(this);
        
        node.getSemicolon().apply(this);
        
        outAAssignmentStatement(node);
    }
    
    public void caseAAssignment(AAssignment node){
        inAAssignment(node);
        
        node.getFieldAccess().apply(this);
        doSpace();
        node.getAssign().apply(this);
        doSpace();
        node.getExpression().apply(this);
        
        outAAssignment(node);
    }

    
    public void caseAWhileStatement(AWhileStatement node){
        inAWhileStatement(node);
        
        doNewLine();
        doIndent();
        
    	node.getWhile().apply(this);
    	doSpace();
    	node.getLParenthese().apply(this);
        doSpace();
    	node.getBoolExpression().apply(this);
        doSpace();
    	node.getRParenthese().apply(this);
        doSpace();
    	node.getBlock().apply(this);

        outAWhileStatement(node);
    }


    public void caseAReturnStatement(AReturnStatement node){
        inAReturnStatement(node);
        
        node.getReturn().apply(this);
    
        if(node.getExpression() != null)
        {
            doSpace();
            node.getExpression().apply(this);
        }
        
        node.getSemicolon().apply(this);
        outAReturnStatement(node);
    }

    
    public void caseASwitchStatement(ASwitchStatement node){
        inASwitchStatement(node);
        
        doNewLine();
        doIndent();
        
        node.getSwitch().apply(this);
        doSpace();
        node.getLParenthese().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);
        doSpace();
        node.getRParenthese().apply(this);
        doSpace();
        node.getSwitchBlock().apply(this);
        outASwitchStatement(node);
    }

    public void caseASwitchBlock(ASwitchBlock node){
        inASwitchBlock(node);
        _indentLevel++;
        node.getLBrace().apply(this);
        {
            Object temp[] = node.getPossibleCase().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                doNewLine();
                doIndent();
                ((PPossibleCase) temp[i]).apply(this);
            }
        }
        _indentLevel--;
        doNewLine();
        doIndent();
        node.getRBrace().apply(this);
        outASwitchBlock(node);
    }


    public void caseACasePossibleCase(ACasePossibleCase node){
        inACasePossibleCase(node);
        node.getCase().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);
        node.getColon().apply(this);
        node.getBlock().apply(this);
        outACasePossibleCase(node);
    }


    public void caseADefaultPossibleCase(ADefaultPossibleCase node){
        inADefaultPossibleCase(node);
        node.getDefault().apply(this);
        node.getColon().apply(this);
        node.getBlock().apply(this);
        outADefaultPossibleCase(node);
    }
    
    
    public void caseAForStatement(AForStatement node){
        inAForStatement(node);
        doNewLine();
        doIndent();
        node.getFor().apply(this);
        doSpace();
        node.getForControl().apply(this);
        doSpace();
        node.getBlock().apply(this);
        outAForStatement(node);
    }


    public void caseAAssertStatement(AAssertStatement node){
        inAAssertStatement(node);

        node.getAssert().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);

        if(node.getColonString() != null)
        {
            node.getColonString().apply(this);
        }
        
    	if(node.getSemicolon() != null)
        {
            node.getSemicolon().apply(this);
        }
        outAAssertStatement(node);
    }

    
    public void caseAColonString(AColonString node){
        inAColonString(node);
        doSpace();
        node.getColon().apply(this);
        doSpace();
        node.getExpression().apply(this);
        outAColonString(node);
    }
    

    /**
     * @see minijava.analysis.DepthFirstAdapter#caseAIfStatement(minijava.node.AIfStatement)
     */
    public void caseAIfStatement(AIfStatement node) {
        
        inAIfStatement(node);
        
        doNewLine();
        doIndent();
        
        node.getIfStat().apply(this);
        outAIfStatement(node);
        
    }
    
    public void caseAIfStat(AIfStat node){
        inAIfStat(node);
        
        node.getIf().apply(this);
        doSpace();
        node.getLParenthese().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);
        doSpace();
        node.getRParenthese().apply(this);
        doSpace();
        node.getBlock1().apply(this);
        
        if(node.getElsePart() != null)
        {
            node.getElsePart().apply(this);
        }
        outAIfStat(node);
    }

    
    public void caseAElsePart(AElsePart node){
        inAElsePart(node);
        
        doSpace();
        node.getElse().apply(this);
        doSpace();
        node.getElseFollow().apply(this);

        outAElsePart(node);
    }


    public void caseABlockElseFollow(ABlockElseFollow node){
        inABlockElseFollow(node);
        
        if(node.getBlock() != null)
        {
            node.getBlock().apply(this);
        }
        
        outABlockElseFollow(node);
    }


    public void caseAIfElseFollow(AIfElseFollow node){
        inAIfElseFollow(node);
        node.getIfStat().apply(this);
        outAIfElseFollow(node);
    }




    public void caseAJavaForControl(AJavaForControl node){
        inAJavaForControl(node);
        node.getLParenthese().apply(this);
        
        node.getForLeftStat().apply(this);
        doSpace();
        node.getS1().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);
        doSpace();
        node.getS2().apply(this);
        doSpace();
        if( node.getForRightStat() != null){
            node.getForRightStat().apply(this);
        }
        doSpace();
        node.getRParenthese().apply(this);
        
        outAJavaForControl(node);
    }


    public void caseABlock(ABlock node){
        inABlock(node);
        
        node.getLBrace().apply(this);

        _indentLevel ++;
        
        if(node.getStatements() != null)
        {
            node.getStatements().apply(this);
        }

        doNewLine();
        _indentLevel--;
        doIndent();
        node.getRBrace().apply(this);
        outABlock(node);
    }


    public void caseAFunctionApplication(AFunctionApplication node){
        inAFunctionApplication(node);
        
        KFunction kf = (KFunction) _functionMap.get(node);
        
        if( kf.isBuiltin() ){
            List args = RestrictedTranslator.getParams(node.getActualParamList(), _functionMap, _typeMap, node);
            
            IBuiltInMethod ibm = kf.getBuiltinMethod();
            String[] codeToAppend = ibm.getCode(args);
            
            for(int i = 0 ; i < codeToAppend.length ; i++){
                _code.append( codeToAppend[i]);
                tokenOut();
            }
            
        }else{
            node.getName().apply(this);
            node.getLParenthese().apply(this);
            
            if( node.getActualParamList() != null ){
                doSpace();
                node.getActualParamList().apply(this);
                doSpace();
            }
            
            node.getRParenthese().apply(this);    
        }
        
        outAFunctionApplication(node);
    }

    
    public void caseAListActualParamList(AListActualParamList node){
        inAListActualParamList(node);
        
        node.getActualParamList().apply(this);

        node.getComma().apply(this);
        doSpace();
        node.getExpression().apply(this);
        
        outAListActualParamList(node);
    }


    public void caseAExpActualParamList(AExpActualParamList node){

        inAExpActualParamList(node);
        
        node.getExpression().apply(this);
        
        outAExpActualParamList(node);
    }


    
    public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node){
        inAVarDecInnerDeclaration(node);

        node.getType().apply(this);
        doSpace();
        node.getIdentifier().apply(this);

	    if(node.getInitialiser() != null)
	    {
	        doSpace();
	        node.getInitialiser().apply(this);
	    }else{
	        KType ct = (KType) _typeMap.get(node);
	        
	        if( ct instanceof KBoundClassType ){
	            doSpace();
	            caseTAssign(null);
	            doSpace();
	            caseTNew(null);
	            doSpace();
	            _code.append(ct.getName());
	            
	            caseTLParenthese(null);
	            
	            KClassType underlying = ((KBoundClassType)ct).getBase();
	            if( underlying == KClassType.getCharacter() ){
	                doSpace();
	                _code.append("\'\\0\'");
	                doSpace();
	            }else if( underlying == KClassType.getInteger() ){
	                doSpace();
	                _code.append("0");
	                doSpace();
	            }else if( underlying == KClassType.getDouble() ){
	                doSpace();
	                _code.append("0.0");
	                doSpace();
	            }else if( underlying == KClassType.getBoolean()){
	                doSpace();
	                _code.append("false");
	                doSpace();
	            }
	            
	            caseTRParenthese(null);
	        }
	        
	    }
	    


        outAVarDecInnerDeclaration(node);
    }


    public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node){
        inAArrayDecInnerDeclaration(node);
        
        node.getType().apply(this);
        
        {
            Object temp[] = node.getBracketPair().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((TBracketPair) temp[i]).apply(this);
            }
        }
        
        doSpace();
        
        node.getIdentifier().apply(this);

        if(node.getArrayInitialiser() != null)
        {
            doSpace();
            node.getArrayInitialiser().apply(this);
        }
        
        outAArrayDecInnerDeclaration(node);
    }

    
    public void caseAFormalParamList(AFormalParamList node){
        inAFormalParamList(node);
        
        node.getTypeName().apply(this);

        {
            Object temp[] = node.getCommaTypeName().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((PCommaTypeName) temp[i]).apply(this);
            }
        }
        
        outAFormalParamList(node);
    }

    public void caseATypeName(ATypeName node){
        inATypeName(node);
        
        node.getType().apply(this);

        {
            Object temp[] = node.getBracketPair().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((TBracketPair) temp[i]).apply(this);
            }
        }
        doSpace();
        node.getIdentifier().apply(this);

        outATypeName(node);
    }

    public void caseATypeParam(ATypeParam node) {
        inATypeParam(node);
        boolean tmp = _inTypeParams;
        _inTypeParams = true;
        node.getLess().apply(this);
        node.getTypeParamList().apply(this);
        node.getGreater().apply(this);
        _inTypeParams = tmp;
        outATypeParam(node);
    }
    

    public void caseACommaTypeName(ACommaTypeName node){
        
        inACommaTypeName(node);
        node.getComma().apply(this);
        doSpace();
        node.getTypeName().apply(this);
        outACommaTypeName(node);
        
    }
    
    public void caseAEnumList(AEnumList node){
        inAEnumList(node);
        
        node.getIdentifier().apply(this);

        {
            Object temp[] = node.getCommaEnumList().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((PCommaEnumList) temp[i]).apply(this);
            }
        }
        
        outAEnumList(node);
    }


    public void caseACommaEnumList(ACommaEnumList node){
        inACommaEnumList(node);
        node.getComma().apply(this);
        doSpace();
        node.getIdentifier().apply(this);
        outACommaEnumList(node);
    }


    public void caseAInitialiser(AInitialiser node){
        inAInitialiser(node);
        node.getAssign().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);
        outAInitialiser(node);
    }

    
    public void caseAStaticArrayInitialiser(AStaticArrayInitialiser node){
        
        inAStaticArrayInitialiser(node);
        node.getAssign().apply(this);
        doSpace();
        node.getArrayInit().apply(this);
        outAStaticArrayInitialiser(node);
        
    }


    public void caseADynamicArrayInitialiser(ADynamicArrayInitialiser node){
        inADynamicArrayInitialiser(node);

        node.getAssign().apply(this);
        doSpace();
        node.getArrayAllocate().apply(this);
        
        outADynamicArrayInitialiser(node);
    }

    
    public void caseAArrayInit(AArrayInit node){
        inAArrayInit(node);
        
        node.getLBrace().apply(this);
        doSpace();
        node.getInitList().apply(this);
        doSpace();
        node.getRBrace().apply(this);
        
        outAArrayInit(node);
    }

    public void caseAScalarInitList(AScalarInitList node){
        inAScalarInitList(node);
        
        node.getExpression().apply(this);
        {
            Object temp[] = node.getCommaExp().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((PCommaExp) temp[i]).apply(this);
            }
        }
        
        outAScalarInitList(node);
    }

    
    public void caseACommaExp(ACommaExp node){
        inACommaExp(node);
        
        node.getComma().apply(this);
        doSpace();
        node.getExpression().apply(this);
        
        outACommaExp(node);
    }
    
    
    public void caseAArrayInitList(AArrayInitList node){
        
        inAArrayInitList(node);

        node.getArrayInit().apply(this);
        
        {
            Object temp[] = node.getCommaArrayInit().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((PCommaArrayInit) temp[i]).apply(this);
            }
        }
        
        outAArrayInitList(node);
        
    }

    public void caseACommaArrayInit(ACommaArrayInit node){
        inACommaArrayInit(node);
        node.getComma().apply(this);
        doSpace();
        node.getArrayInit().apply(this);
        outACommaArrayInit(node);
    }



    public void caseACommaType(ACommaType node){
        inACommaType(node);
        
        node.getComma().apply(this);
        doSpace();
        node.getType().apply(this);

        {
            Object temp[] = node.getBracketPair().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((TBracketPair) temp[i]).apply(this);
            }
        }
        
        outACommaType(node);
    }


    public void caseAArrayAllocate(AArrayAllocate node){
        
        inAArrayAllocate(node);
        node.getNew().apply(this);
        doSpace();
        node.getType().apply(this);
        {
            Object temp[] = node.getArrayAccess().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((PArrayAccess) temp[i]).apply(this);
            }
        }
        {
            Object temp[] = node.getBracketPair().toArray();
            for(int i = 0; i < temp.length; i++)
            {
                ((TBracketPair) temp[i]).apply(this);
            }
        }
        outAArrayAllocate(node);
        
    }


    public void caseAPlusMathExpression(APlusMathExpression node){
        inAPlusMathExpression(node);
        node.getMathExpression().apply(this);
        doSpace();
        node.getPlus().apply(this);
        doSpace();
        node.getTerm().apply(this);
        outAPlusMathExpression(node);
    }


    public void caseAMinusMathExpression(AMinusMathExpression node){
        inAMinusMathExpression(node);
        node.getMathExpression().apply(this);
        doSpace();
        node.getMinus().apply(this);
        doSpace();
        node.getTerm().apply(this);
        outAMinusMathExpression(node);
    }


    public void caseAMultTerm(AMultTerm node){
        inAMultTerm(node);
        node.getTerm().apply(this);
        doSpace();
        node.getTimes().apply(this);
        doSpace();
        node.getUnaryExp().apply(this);
        outAMultTerm(node);
    }


    public void caseADivTerm(ADivTerm node){
        inADivTerm(node);
        node.getTerm().apply(this);
        doSpace();
        node.getDivide().apply(this);
        doSpace();
        node.getUnaryExp().apply(this);
        outADivTerm(node);
    }


    public void caseAModTerm(AModTerm node){
        inAModTerm(node);
        node.getTerm().apply(this);
        doSpace();
        node.getMod().apply(this);
        doSpace();
        node.getUnaryExp().apply(this);
        outAModTerm(node);
    }


    public void caseAExpressionFactor(AExpressionFactor node){
        inAExpressionFactor(node);
        node.getLParenthese().apply(this);
        doSpace();
        node.getBoolExpression().apply(this);
        doSpace();
        node.getRParenthese().apply(this);
        outAExpressionFactor(node);
    }


    public void caseAOrBoolExpression(AOrBoolExpression node){
        inAOrBoolExpression(node);
        node.getBoolExpression().apply(this);
        doSpace();
        node.getOr().apply(this);
        doSpace();
        node.getBoolTerm().apply(this);
        outAOrBoolExpression(node);
    }


    public void caseAXorBoolExpression(AXorBoolExpression node){
        inAXorBoolExpression(node);
        node.getBoolExpression().apply(this);
        doSpace();
        node.getXor().apply(this);
        doSpace();
        node.getBoolTerm().apply(this);
        outAXorBoolExpression(node);
    }


    public void caseAAndBoolTerm(AAndBoolTerm node){
        inAAndBoolTerm(node);
        node.getBoolTerm().apply(this);
        doSpace();
        node.getAnd().apply(this);
        doSpace();
        node.getEquality().apply(this);
        outAAndBoolTerm(node);
    }


    public void caseAEqEquality(AEqEquality node){
        inAEqEquality(node);
        
        if(_typeMap.get(node) == KBasicType.getString() ){
            caseTLParenthese(null);
            doSpace();
            node.getEquality().apply(this);
            doSpace();
            caseTRParenthese(null);
	        caseTDot(null);
	        
	        _code.append("equals");
	        tokenOut();
            caseTLParenthese(null);
	        doSpace();
	        node.getRelational().apply(this);
	        doSpace();
	        caseTRParenthese(null);
        }else{
            node.getEquality().apply(this);
	        doSpace();
	        node.getEqual().apply(this);
	        doSpace();
	        node.getRelational().apply(this);
        }
        outAEqEquality(node);
    }


    public void caseANeqEquality(ANeqEquality node){
        inANeqEquality(node);

        if(_typeMap.get(node) == KBasicType.getString() ){
            caseTNot(null);
            caseTLParenthese(null);
            doSpace();
            node.getEquality().apply(this);
            doSpace();
            caseTRParenthese(null);
	        caseTDot(null);
	        
	        _code.append("equals");
	        tokenOut();
            caseTLParenthese(null);
	        doSpace();
	        node.getRelational().apply(this);
	        doSpace();
	        caseTRParenthese(null);
        }else{
	        node.getEquality().apply(this);
	        doSpace();
	        node.getNotequal().apply(this);
	        doSpace();
	        node.getRelational().apply(this);
	        outANeqEquality(node);
        }
    }


    public void caseALtRelational(ALtRelational node){
        inALtRelational(node);
        node.getRelational().apply(this);
        doSpace();
        node.getLess().apply(this);
        doSpace();
        node.getMathExpression().apply(this);
        outALtRelational(node);
    }


    public void caseAGtRelational(AGtRelational node){
        inAGtRelational(node);
        node.getRelational().apply(this);
        doSpace();
        node.getGreater().apply(this);
        doSpace();
        node.getMathExpression().apply(this);
        outAGtRelational(node);
    }


    public void caseALteqRelational(ALteqRelational node){
        inALteqRelational(node);
        node.getRelational().apply(this);
        doSpace();
        node.getLessequal().apply(this);
        doSpace();
        node.getMathExpression().apply(this);
        outALteqRelational(node);
    }


    public void caseAGteqRelational(AGteqRelational node){
        inAGteqRelational(node);
        node.getRelational().apply(this);
        doSpace();
        node.getGreaterequal().apply(this);
        doSpace();
        node.getMathExpression().apply(this);
        outAGteqRelational(node);
    
    }

	public void caseATypeParamList(ATypeParamList node) {
	    inATypeParamList(node);
	    boolean iTP = _inTypeParams;
	    
	    if( node.getBracketPair().size() != 0 ){
	        _inTypeParams = false;
	    }

	    node.getType().apply(this);
        _inTypeParams = iTP;
	        
	    {
	        Object temp[] = node.getBracketPair().toArray();
	        for(int i = 0; i < temp.length; i++)
	        {
	            ((TBracketPair) temp[i]).apply(this);
	        }
	    }
	    
	    {
	        Object temp[] = node.getCommaType().toArray();
	        for(int i = 0; i < temp.length; i++)
	        {
	            ((PCommaType) temp[i]).apply(this);
	        }
	    }
	    outATypeParamList(node);
	}

    public void caseTBoolean(TBoolean node){
        defaultIn(node);
        if( _inTypeParams ){
            _code.append("Boolean");
        }else{
            _code.append("boolean");
        }
        tokenOut();
    }

    public void caseTChar(TChar node){
        defaultIn(node);
        if( _inTypeParams){
            _code.append("Character");
        }else{
            _code.append("char");
        }
        tokenOut();
    }

    public void caseTInt(TInt node){
        defaultIn(node);
        if( _inTypeParams ){
            _code.append("Integer");
        }else{
            _code.append("int");
        }
        tokenOut();
    }

    public void caseTDouble(TDouble node){
        defaultIn(node);
        if( _inTypeParams ){
            _code.append("Double");
        }else{
            _code.append("double");
        }
        tokenOut();
    }

    public void caseTString(TString node){
        defaultIn(node);
        _code.append("String");
        tokenOut();
    }

    public void caseTVoid(TVoid node){
        defaultIn(node);
        _code.append("void");
        tokenOut();
    }

    public void caseTKlass(TKlass node){
        defaultIn(node);
        _code.append("class");
        tokenOut();
    }

    public void caseTConst(TConst node){
        defaultIn(node);
        _code.append("static final");
        tokenOut();
    }

    public void caseTIf(TIf node){
        defaultIn(node);
        _code.append("if");
        tokenOut();
    }

    public void caseTElse(TElse node){
        defaultIn(node);
        _code.append("else");
        tokenOut();
    }

    public void caseTWhile(TWhile node){
        defaultIn(node);
        _code.append("while");
        tokenOut();
    }

    public void caseTReturn(TReturn node){
        defaultIn(node);
        _code.append("return");
        tokenOut();
    }

    public void caseTSwitch(TSwitch node){
        defaultIn(node);
        _code.append("switch");
        tokenOut();
    }

    public void caseTCase(TCase node){
        defaultIn(node);
        _code.append("case");
        tokenOut();
    }

    public void caseTBreak(TBreak node){
        defaultIn(node);
        _code.append("break");
        tokenOut();
    }

    public void caseTDefault(TDefault node){
        defaultIn(node);
        _code.append("default");
        tokenOut();
    }

    public void caseTFor(TFor node){
        defaultIn(node);
        _code.append("for");
        tokenOut();
    }

    public void caseTAssert(TAssert node){
        defaultIn(node);
        _code.append("assert");
        tokenOut();
    }

    public void caseTNew(TNew node){
        defaultIn(node);
        _code.append("new");
        tokenOut();
    }

    public void caseTEnum(TEnum node){
        defaultIn(node);
        _code.append("enum");
        tokenOut();
    }

    public void caseTTrue(TTrue node){
        defaultIn(node);
        _code.append("true");
        tokenOut();
    }

    public void caseTFalse(TFalse node){
        defaultIn(node);
        _code.append("false");
        tokenOut();
    }

    public void caseTNull(TNull node){
        defaultIn(node);
        _code.append("null");
        tokenOut();
    }

    public void caseTAnd(TAnd node){
        defaultIn(node);
        _code.append("&&");
        tokenOut();
    }

    public void caseTOr(TOr node){
        defaultIn(node);
        _code.append("||");
        tokenOut();
    }

    public void caseTXor(TXor node){
        defaultIn(node);
        _code.append("^");
        tokenOut();
    }

    public void caseTNot(TNot node){
        defaultIn(node);
        _code.append("!");
        tokenOut();
    }

    public void caseTIdentifier(TIdentifier node){
        defaultIn(node);
        _code.append(node.getText().trim());
        tokenOut();
    }

    public void caseTStringliteral(TStringliteral node){
        defaultIn(node);
        _code.append(node.getText().trim());
        tokenOut();
    }

    public void caseTCharliteral(TCharliteral node){
        defaultIn(node);
        _code.append(node.getText().trim());
        tokenOut();
    }

    public void caseTLParenthese(TLParenthese node){
        defaultIn(node);
        _code.append("(");
        tokenOut();
    }

    public void caseTRParenthese(TRParenthese node){
        defaultIn(node);
        _code.append(")");
        tokenOut();
    }

    public void caseTLBrace(TLBrace node){
        defaultIn(node);
        _code.append("{");
        tokenOut();
    }

    public void caseTRBrace(TRBrace node){
        defaultIn(node);
        _code.append("}");
        tokenOut();
    }

    public void caseTLBracket(TLBracket node){
        defaultIn(node);
        _code.append("[");
        tokenOut();
    }

    public void caseTRBracket(TRBracket node){
        defaultIn(node);
        _code.append("]");
        tokenOut();
    }

    public void caseTBracketPair(TBracketPair node){
        defaultIn(node);
        _code.append("[]");
        tokenOut();
    }

    public void caseTSemicolon(TSemicolon node){
        defaultIn(node);
        _code.append(";");
        tokenOut();
    }

    public void caseTColon(TColon node){
        defaultIn(node);
        _code.append(":");
        tokenOut();
    }

    public void caseTComma(TComma node){
        defaultIn(node);
        _code.append(",");
        tokenOut();
    }

    public void caseTDot(TDot node){
        defaultIn(node);
        _code.append(".");
        tokenOut();
    }

    public void caseTPlus(TPlus node){
        defaultIn(node);
        _code.append("+");
        tokenOut();
    }

    public void caseTPlusplus(TPlusplus node){
        defaultIn(node);
        _code.append("++");
        tokenOut();
    }

    public void caseTMinus(TMinus node){
        defaultIn(node);
        _code.append("-");
        tokenOut();
    }

    public void caseTMinusminus(TMinusminus node){
        defaultIn(node);
        _code.append("--");
        tokenOut();
    }

    public void caseTTimes(TTimes node){
        defaultIn(node);
        _code.append("*");
        tokenOut();
    }

    public void caseTDivide(TDivide node){
        _code.append("/");
        tokenOut();
    }

    public void caseTMod(TMod node){
        _code.append("%");
        tokenOut();
    }

    public void caseTLess(TLess node){
        _code.append("<");
        tokenOut();
    }

    public void caseTLessequal(TLessequal node){
        _code.append("<=");
        tokenOut();
    }

    public void caseTGreater(TGreater node){
        _code.append(">");
        tokenOut();
    }

    public void caseTGreaterequal(TGreaterequal node){
        _code.append(">=");
        tokenOut();
    }

    public void caseTEqual(TEqual node){
        _code.append("==");
        tokenOut();
    }

    public void caseTNotequal(TNotequal node){
        _code.append("!=");
        tokenOut();
    }

    public void caseTAssign(TAssign node){
        _code.append("=");
        tokenOut();
    }

    public void caseTIntnumber(TIntnumber node){
        _code.append(node.getText().trim());
        tokenOut();
    }

    public void caseTDpnumber(TDpnumber node){
        _code.append(node.getText().trim());
        tokenOut();
    }

    
}
