/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004 by toa02
 *
 */
package kenya.passes;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.wellquite.kenya.stackMachine.types.interfaces.IType;

import kenya.errors.KenyaBadLocationError;
import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.errors.SourceCodeException;
import kenya.types.KArrayType;
import kenya.types.KBasicType;
import kenya.types.KBoundClassType;
import kenya.types.KClassType;
import kenya.types.KEnumType;
import kenya.types.KFunction;
import kenya.types.KNullType;
import kenya.types.KParamType;
import kenya.types.KType;
import kenya.types.KVariable;
import kenya.types.tables.ClassTable;
import kenya.types.tables.FunctionTable;
import kenya.types.tables.SymbolTable;
import kenya.types.tables.util.PairKFunctionReturnType;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AAndBoolTerm;
import minijava.node.AArrayAccess;
import minijava.node.AArrayAllocate;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AArrayFieldAccess;
import minijava.node.AArrayInit;
import minijava.node.AArrayInitList;
import minijava.node.AAssertStatement;
import minijava.node.AAssignment;
import minijava.node.AAssignmentForLeftStat;
import minijava.node.AAssignmentForRightStat;
import minijava.node.AAssignmentStatement;
import minijava.node.ABlock;
import minijava.node.ABooleanBasicType;
import minijava.node.ABoolliteralFactor;
import minijava.node.ACasePossibleCase;
import minijava.node.ACharBasicType;
import minijava.node.ACharliteralFactor;
import minijava.node.AClassDecDeclaration;
import minijava.node.AClassTypeReferenceType;
import minijava.node.AColonString;
import minijava.node.ACommaArrayInit;
import minijava.node.ACommaEnumList;
import minijava.node.ACommaExp;
import minijava.node.ACommaType;
import minijava.node.ACommaTypeName;
import minijava.node.AConstDecDeclaration;
import minijava.node.ADNumber;
import minijava.node.ADefaultPossibleCase;
import minijava.node.ADivTerm;
import minijava.node.ADoubleBasicType;
import minijava.node.ADynamicArrayInitialiser;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AEnumList;
import minijava.node.AEqEquality;
import minijava.node.AExpActualParamList;
import minijava.node.AFalseBooleanliteral;
import minijava.node.AForStatement;
import minijava.node.AFormalParamList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.AFunctionApplication;
import minijava.node.AFunctioncallForRightStat;
import minijava.node.AFunctioncallStatement;
import minijava.node.AGtRelational;
import minijava.node.AGteqRelational;
import minijava.node.AINumber;
import minijava.node.AIfStat;
import minijava.node.AInitialiser;
import minijava.node.AIntBasicType;
import minijava.node.AJavaForControl;
import minijava.node.AListActualParamList;
import minijava.node.AListStatements;
import minijava.node.ALtRelational;
import minijava.node.ALteqRelational;
import minijava.node.AMinusMathExpression;
import minijava.node.AMinusUnaryExp;
import minijava.node.AModTerm;
import minijava.node.AMultTerm;
import minijava.node.ANegateUnaryExp;
import minijava.node.ANeqEquality;
import minijava.node.ANullFactor;
import minijava.node.AOrBoolExpression;
import minijava.node.APlusMathExpression;
import minijava.node.APlusUnaryExp;
import minijava.node.APostdecr;
import minijava.node.APostincr;
import minijava.node.APredecr;
import minijava.node.APreincr;
import minijava.node.AQualifiedName;
import minijava.node.AReturnStatement;
import minijava.node.AScalarInitList;
import minijava.node.ASimpleName;
import minijava.node.ASimpleNameName;
import minijava.node.AStaticArrayInitialiser;
import minijava.node.AStringBasicType;
import minijava.node.AStringliteralFactor;
import minijava.node.ASwitchBlock;
import minijava.node.ASwitchStatement;
import minijava.node.ATrueBooleanliteral;
import minijava.node.ATypeName;
import minijava.node.ATypeParam;
import minijava.node.ATypeParamList;
import minijava.node.AVarDecInnerDeclaration;
import minijava.node.AVoidBasicType;
import minijava.node.AWhileStatement;
import minijava.node.AXorBoolExpression;
import minijava.node.PArrayAccess;
import minijava.node.PCommaArrayInit;
import minijava.node.PCommaExp;
import minijava.node.PName;
import minijava.node.PPossibleCase;

/**
 * The Type Checker.
 * 
 * This should be used either partially or completely, so it can be called turn
 * by turn on functions, assignments etc. or negotiate the whole tree by itself.
 * This only checks **types**. There are other passes to check the other things.
 * 
 * Lets rock!
 */
public class TypeChecker extends DepthFirstAdapter {

    private ClassTable _classTable;

    private FunctionTable _funcTable;

    private SymbolTable _symTab;

    //private TypeTable _typeTab;
    private Map _paramMap; // Constains a mapping of String Name to

    // KParamType of the parameter types that
    // are "in play".

    // General stack used by expressions.
    private Stack _typeStack;

    // Used by functions to keep track of return values.
    private KType _retVal;

    // Used by Switches to keep track of their properties.
    private KType _switchType;

    // Used by Array's to check the scalar init lists are correct.
    private KType _arrType;

    private List _errors; //List of SourceCodeExceptions that are the errors
                          // encoounterd

    private Set _errNodes; // HashSet of PStatements + Class/Func declarations
                           // that are statements with errors in them.

    private List _warnings;

    // String name of the current function/class name we are in (or null if
    // none).
    private String _cScope;

    // Set true iff we want any basic integers to come back as chars.
    private KType _typeContext;

    // Maps nodes to KTypes
    private Map _typeMap;

    // Maps nodes to KFunctions that are being called.
    private Map _funcMap;

    // Set of nodes that make reference to an array.length
    private Set _arrayLengthRef;

    // Set of nodes that use an enum child as a direct name (in switches)
    private Set _enumChildSet;

    /**
     * Sets up the TypeChecker with a ClassTable and a FunctionTable.
     * 
     * @param ct
     *            The ClassTable this TypeChecker should use. All Classes should
     *            be fully defined.
     * @param ft
     *            The FunctionTable this TypeChecker should use. All Functions
     *            should be fully defined.
     */
    public TypeChecker(ClassTable ct, SymbolTable symTab, FunctionTable ft,
            Set errNodes, Map typeMap ) {
        _classTable = ct;
        _funcTable = ft;
        _symTab = symTab;
        _paramMap = new HashMap();
        _typeStack = new Stack();
        _errors = new LinkedList();
        _warnings = new LinkedList();
        _errNodes = errNodes;
        _typeContext = null;
        _cScope = Util.GLOBAL;
        _typeMap = typeMap;
        _funcMap = new HashMap();
        _arrayLengthRef = new HashSet();
        _enumChildSet = new HashSet();
    }

    TypeChecker(ClassTable ct, SymbolTable st, Map paramMap, Set errNodes, Map typeMap, KType retType) {
        _classTable = ct;
        _funcTable = new FunctionTable();
        _symTab = st;
        _paramMap = paramMap;
        _typeStack = new Stack();
        _errors = new LinkedList();
        _warnings = new LinkedList();
        _errNodes = errNodes;
        _typeContext = retType;
        _cScope = Util.GLOBAL;
        _typeMap = typeMap;
        _funcMap = new HashMap();
        _arrayLengthRef = new HashSet();
        _enumChildSet = new HashSet();
        _arrType = retType;
    }

    /**
     * Gives back the error List, which will have been propogated by the last
     * failed run of this TypeChecker. It will be empty if the last run
     * succeeded or no run has taken place yet.
     * 
     * @return List of SourceCodeException.
     */
    public List getErrors() {
        return _errors;
    }

    public List getWarnings() {
        return _warnings;
    }

    /**
     * @return Set of nodes (AQualifiedName) that refer to the name part of an
     *         array.length lookup.
     */
    public Set getArrayLengthRefs() {
        return _arrayLengthRef;
    }

    /**
     * @return Set of nodes where an enum child has been referenced directly
     */
    public Set getEnumChildSet() {
        return _enumChildSet;
    }

    /**
     * @return Map mapping nodes -> KTypes.
     */
    public Map getTypeMap() {
        return _typeMap;
    }

    /**
     * @return Set of nodes with errors on them.
     */
    public Set getErrorNodes() {
        return _errNodes;
    }

    /**
     * @return Map mapping nodes -> KFunctions
     */
    public Map getFuncMap() {
        return _funcMap;
    }

    /**
     * Gives back the error statements set, which will have been propogated by
     * the last failed run of this TypeChecker. Will be empty if last run
     * succeeded or no run has taken place yet.
     * 
     * @return Set of PStatements.
     */
    public Set getErrorStatements() {
        return _errNodes;
    }

    /**
     * Returns the top element on the TypeStack.
     * 
     * @return
     */
    public KType getTopOfTypeStack() {
        return (KType) _typeStack.pop();
    }

    /**
     * Little method for RestrictedTypeChecker to use.
     * 
     * @param value
     */
    void setWantChar() {
        _typeContext = KBasicType.getChar();
    }

    /***************************************************************************
     * Here begins the cases. One by one, they will slowly steal my sanity. *
     **************************************************************************/

    public void caseAListStatements(AListStatements node) {
        inAListStatements(node);

        try {
            if (!_errNodes.contains(node.getStatement())) {
                node.getStatement().apply(this);
            }
        } catch (SourceCodeException sce) {
            _errors.add(sce);
            _errNodes.add(node.getStatement());
            _typeStack.clear();
        }

        node.getStatements().apply(this);

        outAListStatements(node);
    }

    /**
     * Assignment Statements. Get the left and right hand types. Check they are
     * compatible.
     * 
     * @see minijava.analysis.Analysis#caseAAssignmentStatement(minijava.node.AAssignmentStatement)
     */
    public void caseAAssignmentStatement(AAssignmentStatement node) {
        inAAssignmentStatement(node);

        node.getAssignment().apply(this);
        _typeMap.put(node,_typeStack.pop());
        outAAssignmentStatement(node);
    }

    
    /**
     * @see minijava.analysis.DepthFirstAdapter#caseAAssignment(minijava.node.AAssignment)
     */
    public void caseAAssignment(AAssignment node) {
        
        inAAssignment(node);
        node.getFieldAccess().apply(this);
        KType fla = (KType) _typeStack.pop();

        if (fla instanceof KArrayType) {
            _typeContext = ((KArrayType) fla).getBaseType();
        } else {
            _typeContext = fla;
        }

        node.getExpression().apply(this);
        KType exp = (KType) _typeStack.pop();

        
        if (!Util.isAssignmentCompatible(fla, exp)) {
            int[] pos = Util.getFirstIdent(node.getExpression());
            SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
                    pos[1], pos[2], fla, exp);
        }
        
        //put into the type map the type of the assigned expression
        _typeMap.put(node,fla);
        
        _typeStack.push(fla);
        
        inAAssignment(node);
        
    }
    /**
     * If statements: check the boolean expression is actaully boolean. The else
     * part should take care of itself. I hope.
     * 
     * @see minijava.analysis.Analysis#caseAIfStatement(minijava.node.AIfStatement)
     */
    public void caseAIfStat(AIfStat node) {
        inAIfStat(node);

        if (node.getBoolExpression() != null) {
            node.getBoolExpression().apply(this);
        }

        KType boolRet = (KType) _typeStack.pop();
        if (!Util.isAssignmentCompatible(boolRet, KBasicType.getBoolean())) {
            int[] pos = Util.getFirstIdent(node.getBoolExpression());
            SourceCodeException.throwIncompatibleTypes_IfBoolean(pos[0],
                    pos[1], pos[2], boolRet);
        }

        if (node.getBlock1() != null) {
            node.getBlock1().apply(this);
        }

        if (node.getElsePart() != null) {
            node.getElsePart().apply(this);
        }

        outAIfStat(node);
    }

    /**
     * While statments: Check the boolean is actually boolean.
     * 
     * @see minijava.analysis.Analysis#caseAWhileStatement(minijava.node.AWhileStatement)
     */
    public void caseAWhileStatement(AWhileStatement node) {
        inAWhileStatement(node);

        node.getBoolExpression().apply(this);

        KType boolRet = (KType) _typeStack.pop();
        if (! Util.isAssignmentCompatible(boolRet,KBasicType.getBoolean())) {
            int[] pos = Util.getFirstIdent(node.getBoolExpression());
            SourceCodeException.throwIncompatibleTypes_WhileBoolean(pos[0],
                    pos[1], pos[2], boolRet);
        }

        if (node.getBlock() != null) {
            node.getBlock().apply(this);
        }

        outAWhileStatement(node);
    }

    /**
     * Return statements: Evaluate the expression and check its return type
     * matches the currently expected return type (_retVal).
     */
    public void caseAReturnStatement(AReturnStatement node) {
        inAReturnStatement(node);

        KType retted;
        
        _typeContext = _retVal;
        
        if (node.getExpression() != null) {
            node.getExpression().apply(this);
            retted = (KType) _typeStack.pop();
        } else {
            retted = null;
        }

        /* Now check the return value is the same as the expected return value. */
        int ln = node.getReturn().getLine();
        int pos = node.getReturn().getPos();
        int len = node.getReturn().getText().trim().length();
        if (retted == null) {
            if (_retVal == KBasicType.getVoid()) {
                // This is ok.
            } else {
                SourceCodeException.throwReturn_MissingReturnValue(ln, pos,
                        len, _retVal);
                // missing return value.
            }
        } else if (!Util.isAssignmentCompatible(_retVal, retted)) {
            if (_retVal == KBasicType.getVoid()) {
                // trying to return from a void
                SourceCodeException.throwReturn_ValueInVoid(ln, pos, len,
                        retted);
            } else {
                // just generally wrong
                SourceCodeException.throwIncompatibleTypes_Return(ln, pos, len,
                        retted, _retVal);
            }
        }
        
        _typeMap.put(node,_retVal);
        _typeContext = null;

        outAReturnStatement(node);
    }

    /**
     * Switch (top level). Get the type of the bool expression and ensure its
     * either basic ( not void or string ) or Enum type.
     */
    public void caseASwitchStatement(ASwitchStatement node) {
        inASwitchStatement(node);

        KType tmp = _switchType;
        _switchType = null;

        node.getBoolExpression().apply(this);

        _switchType = (KType) _typeStack.pop();

        if (_switchType instanceof KEnumType
                || (Util.isAssignmentCompatible(KBasicType.getInt(),
                        _switchType))) {

            if (node.getSwitchBlock() != null) {
                node.getSwitchBlock().apply(this);
            }

        } else {
            int[] pos = Util.getFirstIdent(node.getBoolExpression());
            SourceCodeException.throwIncompatibleTypes_SwitchTopLevel(pos[0],
                    pos[1], pos[2], _switchType);
        }

        _switchType = tmp;

        outASwitchStatement(node);
    }

    /**
     * Switch( block level ). Ensure there is only one default case. this
     * bypasses one level of indirection for you :)
     */
    public void caseASwitchBlock(ASwitchBlock node) {
        inASwitchBlock(node);

        Iterator it = node.getPossibleCase().iterator();
        boolean foundDefault = false;

        KType tmpSwitchType = _switchType;

        while (it.hasNext()) {
            PPossibleCase ppc = (PPossibleCase) it.next();

            if (ppc instanceof ACasePossibleCase) {
                ACasePossibleCase acpc = (ACasePossibleCase) ppc;

                // toa02 28th -> int / char / problems. I think this makes us
                // more restrictive than java, but thats ok.
                KType tmpContext = _typeContext;
                _typeContext = _switchType;

                acpc.getBoolExpression().apply(this);

                // toa02 28th -> int / char / problems.
                _typeContext = tmpContext;

                KType retType = (KType) _typeStack.pop();
                if (!Util.isAssignmentCompatible(_switchType, retType)) {
                    int[] pos = Util.getFirstIdent(acpc.getBoolExpression());

                    if (_switchType instanceof KEnumType) {
                        SourceCodeException.throwSwitch_EnumCaseExpected(
                                pos[0], pos[1], pos[2], _switchType);
                    } else {
                        SourceCodeException
                                .throwIncompatibleTypes_SwitchCaseLevel(pos[0],
                                        pos[1], pos[2], _switchType, retType);
                    }
                }

                _switchType = null;
                acpc.getBlock().apply(this);
                _switchType = tmpSwitchType;
            } else if (ppc instanceof ADefaultPossibleCase) {
                ADefaultPossibleCase adpc = (ADefaultPossibleCase) ppc;

                if (foundDefault) {
                    int ln = adpc.getDefault().getLine();
                    int pos = adpc.getDefault().getPos();
                    int len = adpc.getDefault().getText().trim().length();
                    SourceCodeException.throwSwitch_DuplicateDefault(ln, pos,
                            len);
                }
                foundDefault = true;

                _switchType = null;
                adpc.getBlock().apply(this);
                _switchType = tmpSwitchType;
            } else {
                throw new KenyaInternalError(
                        "Got a switch possible case that isn't default or case.");
            }
        }

        outASwitchBlock(node);
    }

    /**
     * @see minijava.analysis.DepthFirstAdapter#caseAForStatement(minijava.node.AForStatement)
     */
    public void caseAForStatement(AForStatement node) {
        inAForStatement(node);
        
        _symTab.newLocalScope();
        try{
            node.getForControl().apply(this);
            node.getBlock().apply(this);
        }finally{
            _symTab.pop();    
        }
        
        outAForStatement(node);
    }
    
    /**
     * Java style for control - check the x = blah bit, check the bool_exp is,
     * infact, boolean and check the last bit.
     * 
     * @see minijava.analysis.Analysis#caseAJavaForControl(minijava.node.AJavaForControl)
     */
    public void caseAJavaForControl(AJavaForControl node) {
        inAJavaForControl(node);

        
        
        node.getForLeftStat().apply(this);
        
        node.getBoolExpression().apply(this);
        // On the top of the stack should be a KBasicType.getBoolean();
        KType boolTop = (KType) _typeStack.pop();
        if (!(Util.isAssignmentCompatible(boolTop,KBasicType.getBoolean()))) {
            int pos[] = Util.getFirstIdent(node.getBoolExpression());
            SourceCodeException.throwIncompatibleTypes_ForBoolean(pos[0],
                    pos[1], pos[2], boolTop);
        }

        if( node.getForRightStat() != null ){
            node.getForRightStat().apply(this);
        }
        // On the top of the stack will be the return type of the unary exp or assignment
        // May aswell free up the stack.
        
        
        outAJavaForControl(node);
    }

    public void caseAFunctioncallForRightStat(AFunctioncallForRightStat node){
        inAFunctioncallForRightStat(node);
        
        node.getFunctionApplication().apply(this);
        _typeStack.pop();
        
        outAFunctioncallForRightStat(node);
    }
    
    public void caseAAssignmentForRightStat(AAssignmentForRightStat node){
        inAAssignmentForRightStat(node);
        
        node.getAssignment().apply(this);
        _typeStack.pop();
        
        outAAssignmentForRightStat(node);
        
    }
    
    /**
     * @see minijava.analysis.DepthFirstAdapter#caseAAssignmentForLeftStat(minijava.node.AAssignmentForLeftStat)
     */
    public void caseAAssignmentForLeftStat(AAssignmentForLeftStat node) {
        inAAssignmentForLeftStat(node);
        
        node.getAssignment().apply(this);
        _typeStack.pop();
        
        outAAssignmentForLeftStat(node);
    }


    /**
     * Assertions: Check the boolean expression is, infact, boolean!
     */
    public void caseAAssertStatement(AAssertStatement node) {
        inAAssertStatement(node);

        node.getBoolExpression().apply(this);
        //Top of stack should be boolean.
        KType tos = (KType) _typeStack.pop();
        if (!Util.isAssignmentCompatible(tos,KBasicType.getBoolean())) {
            int[] pos = Util.getFirstIdent(node.getBoolExpression());
            SourceCodeException.throwIncompatibleTypes_AssertBoolean(pos[0],
                    pos[1], pos[2], tos);
        }

        if (node.getColonString() != null) {
            node.getColonString().apply(this);
        }

        outAAssertStatement(node);
    }

    /**
     * The name is a bit of a misleader. The "String" part can be any
     * expression. Its resulting type is just printed - but here we just pop the
     * stop of the stack since its not necessary.
     * 
     * @see minijava.analysis.Analysis#caseAColonString(minijava.node.AColonString)
     */
    public void caseAColonString(AColonString node) {
        inAColonString(node);
        node.getExpression().apply(this);
        // There will be a return type on the stack. May aswell drop it.
        
        KType type = (KType) _typeStack.pop();
        
        if( type == KBasicType.getVoid() ) {
            
            int[] pos = Util.getFirstIdent(node);
            SourceCodeException.throwVoidInAssert(pos[0],pos[1],pos[2]); 
        }

        outAColonString(node);
    }

    /**
     * FunctionCall - just clear the return type.
     * 
     * @see minijava.analysis.DepthFirstAdapter#caseAFunctioncallStatement(minijava.node.AFunctioncallStatement)
     */
    public void caseAFunctioncallStatement(AFunctioncallStatement node) {
        inAFunctioncallStatement(node);

        node.getFunctionApplication().apply(this);
        _typeStack.pop();

        outAFunctioncallStatement(node);

    }

    /**
     * Entering a new block: Create a new local scope in the symbol table, and
     * then pop it at the end.
     * 
     * @see minijava.analysis.Analysis#caseABlock(minijava.node.ABlock)
     */
    public void caseABlock(ABlock node) {
        inABlock(node);

        _symTab.newLocalScope();
        try{
            node.getStatements().apply(this);
        }finally{
            _symTab.pop();
        }
        outABlock(node);
    }

    /**
     * A Function Application. There's a fair bit that goes on here, at some
     * point I need to check this all works!
     * 
     * @see minijava.analysis.Analysis#caseAFunctionApplication(minijava.node.AFunctionApplication)
     */
    public void caseAFunctionApplication(AFunctionApplication node) {
        inAFunctionApplication(node);

        /*
         * Right, the names to functions can only be simple names. If they are
         * more complex than that then something has gone wrong. (Classes can't
         * have methods in this version of kenya).
         */

        PName unName = node.getName();
        String name;
        if (unName instanceof ASimpleNameName) {
            name = ((ASimpleName) ((ASimpleNameName) unName).getSimpleName())
                    .getIdentifier().getText().trim();
        } else {
            int[] errPos = Util.getFirstIdent(unName);
            SourceCodeException.throwFunctionApplication_NoDots(errPos[0],
                    errPos[1], errPos[2]);
            return; // technically unreachable.
        }

        // Save the type - stack for the moment.
        Stack tmpStack = _typeStack;
        _typeStack = new Stack();

        if (node.getActualParamList() != null) {
            node.getActualParamList().apply(this);
        }

        int[] pos = Util.getFirstIdent(node.getName());
        PairKFunctionReturnType pkfrt = _funcTable.getFuncAndRetType(name,
                _typeStack, pos[0], pos[1], pos[2], _warnings, _cScope);
        KType retType = pkfrt.getRetType();
        KFunction kf = pkfrt.getKFunction();

        if (retType instanceof KParamType && !_paramMap.containsKey(retType.toString())) {
            retType = _typeContext;
        }

        _typeStack = tmpStack;

        //>>> 26th
        _typeMap.put(node, retType);
        _funcMap.put(node, kf);
        _typeStack.push(retType);

        outAFunctionApplication(node);
    }

    /**
     * A list of parameters for function applications. Need to make sure the
     * expression isn't returning void. That could be messy.
     * 
     * @see minijava.analysis.Analysis#caseAListActualParamList(minijava.node.AListActualParamList)
     */
    public void caseAListActualParamList(AListActualParamList node) {
        inAListActualParamList(node);

        node.getActualParamList().apply(this);
        node.getComma().apply(this);

        node.getExpression().apply(this);
        KType retType = (KType) _typeStack.peek();
        if (retType == KBasicType.getVoid()) {
            int[] pos = Util.getFirstIdent(node.getExpression());
            SourceCodeException.throwFunctionApplication_Void(pos[0], pos[1],
                    pos[2]);
        }

        outAListActualParamList(node);
    }

    /**
     * An expression in function parameter lists. As above, need to make sure
     * the expression isn't returning void. That could be messy.
     * 
     * @see minijava.analysis.Analysis#caseAExpActualParamList(minijava.node.AExpActualParamList)
     */
    public void caseAExpActualParamList(AExpActualParamList node) {
        inAExpActualParamList(node);

        node.getExpression().apply(this);

        KType retType = (KType) _typeStack.peek();
        if (retType == KBasicType.getVoid()) {
            int[] pos = Util.getFirstIdent(node.getExpression());
            SourceCodeException.throwFunctionApplication_Void(pos[0], pos[1],
                    pos[2]);
        }

        outAExpActualParamList(node);
    }

    /**
     * A Variable decalaration inside a method / class. Fairly simple procedure:
     * 1) get my name 2) get my type 3) convert to variable 4) see if variable
     * already exists 5) see if we have an initialiser 6) check initialiser is
     * of the right type.
     * 
     * @see minijava.analysis.Analysis#caseAVarDecInnerDeclaration(minijava.node.AVarDecInnerDeclaration)
     */
    public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
        inAVarDecInnerDeclaration(node);

        // Ok first of all lets get this variable's name
        String ident = node.getIdentifier().getText().trim();

        // Next up, lets get its type.
        KType kt;
        kt = Util.getType(node.getType(), _classTable, _paramMap, 0, _cScope);

        if (kt == KBasicType.getVoid()) {
            int[] pos = Util.getFirstIdent(node.getType());
            SourceCodeException.throwVariableDeclaration_Void(pos[0], pos[1],
                    pos[2]);
        }

        // Right we have a type, and we have an identifier. I wonder if its
        // valid.
        KVariable kv = new KVariable(ident, kt, false, node);

        //toa02 : [ I wrote this originally but still... ]
        // Associate this node with the type of its owning variable

        _typeContext = kt;
        
        if (node.getInitialiser() != null) {
        	node.getInitialiser().apply(this);
        	// Top of stack will have the type of this initialiser on the top.
        	KType assigType = (KType) _typeStack.pop();
        	
        	if (!Util.isAssignmentCompatible(kt, assigType)) {
        		int[] pos = Util.getFirstIdent(((AInitialiser) node
        				.getInitialiser()).getBoolExpression());
        		SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
        				pos[1], pos[2], kt, assigType);
        	}
        }
        if (!_symTab.pushLocal(ident, kv)) {
        	int ln = node.getIdentifier().getLine();
        	int pos = node.getIdentifier().getPos();
        	int len = node.getIdentifier().getText().trim().length();
        	KVariable conf = _symTab.lookupVariable(ident);
        	int[][] lnkPos = { Util.getFirstIdent(conf.getNode()) };
        	SourceCodeException.throwDuplicateVariable(ln, pos, len, lnkPos,
        			ident, _cScope);
        }
        _typeMap.put(node, kt);

        outAVarDecInnerDeclaration(node);
    }

    /**
     * In theory very simple to a variable declaration (above). Just with a few
     * brackets. & Some annoying casting depending on if this is a static or a
     * dynamic array thats just been declared.
     * 
     * @see minijava.analysis.Analysis#caseAArrayDecInnerDeclaration(minijava.node.AArrayDecInnerDeclaration)
     */
    public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
        inAArrayDecInnerDeclaration(node);

        // Ok first of all lets get this variable's name
        String ident = node.getIdentifier().getText().trim();

        // Now get the number of brackets.
        int numBrackets = node.getBracketPair().size();

        // Next up, lets get its type.
        KType kt;
        kt = Util.getType(node.getType(), _classTable, _paramMap, numBrackets,
                _cScope);
        _typeMap.put(node, kt);

        if (kt == KBasicType.getVoid()) {
            int[] pos = Util.getFirstIdent(node.getType());
            SourceCodeException.throwVariableDeclaration_Void(pos[0], pos[1],
                    pos[2]);
        }

        // Right we have a type, and we have an identifier. I wonder if its
        // valid.
        KVariable kv = new KVariable(ident, kt, false, node);

        if (kt == KBasicType.getVoid()) {
            int ln = node.getIdentifier().getLine();
            int pos = node.getIdentifier().getPos();
            int len = node.getIdentifier().getText().trim().length();
            SourceCodeException.throwVoidArrayType(ln, pos, len);
        }

        
        
        if (node.getArrayInitialiser() != null) {
            /* Do we have generic array creation? */
        	boolean genericClassType = false;
        	
        	if( ((KArrayType) kt).getBaseType() instanceof KBoundClassType ){
        		KBoundClassType kbct = (KBoundClassType) ((KArrayType) kt).getBaseType();
        		if( kbct.getBase().getTypeParamList().size() != 0 ){
        			/*we are attempting creation of an array of a class with template params*/
        			genericClassType = true;
        		}
        	}
        	
        	if (((KArrayType) kt).getBaseType() instanceof KParamType || genericClassType) {
                int[] pos;
                if (node.getArrayInitialiser() instanceof AStaticArrayInitialiser) {
                    pos = Util.getFirstIdent(((AStaticArrayInitialiser) node
                            .getArrayInitialiser()).getArrayInit());
                    SourceCodeException.throwGeneric_Array_Creation(pos[0], pos[1],
                            pos[2]);
                } else if (node.getArrayInitialiser() instanceof ADynamicArrayInitialiser) {
                    pos = Util.getFirstIdent(((ADynamicArrayInitialiser) node
                            .getArrayInitialiser()).getArrayAllocate());
                    SourceCodeException.throwGeneric_Array_Creation(pos[0], pos[1],
                            pos[2]);
                } else {
                	/* we are ok - it comes from a method */
                }
                
            }

            KType tmp = _arrType;

            _arrType = kt;
            node.getArrayInitialiser().apply(this);

            _arrType = tmp;
            // Top of stack will have the type of this initialiser on the top.
            KType assigType = (KType) _typeStack.pop();

            if (!Util.isAssignmentCompatible(kt, assigType)) {
                int[] pos = Util.getFirstIdent(node.getArrayInitialiser());
				SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
                        pos[1], pos[2], kt, assigType);
            }
        }

        if (!_symTab.pushLocal(ident, kv)) {
            int ln = node.getIdentifier().getLine();
            int pos = node.getIdentifier().getPos();
            int len = node.getIdentifier().getText().trim().length();
            KVariable conf = _symTab.lookupVariable(ident);
            int[][] lnkPos = { Util.getFirstIdent(conf.getNode()) };
            SourceCodeException.throwDuplicateVariable(ln, pos, len, lnkPos,
                    ident, _cScope);
        }

        outAArrayDecInnerDeclaration(node);
    }

    /**
     * Do nothing.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAClassDecDeclaration(minijava.node.AClassDecDeclaration)
     */
    public void caseAClassDecDeclaration(AClassDecDeclaration node) {

        inAClassDecDeclaration(node);
        outAClassDecDeclaration(node);

    }

    /**
     * An enumeration has been declared. Thats nice. Don't do anything, anything
     * at all. :)
     * 
     * @see minijava.analysis.Analysis#caseAEnumDecDeclaration(minijava.node.AEnumDecDeclaration)
     */
    public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
        inAEnumDecDeclaration(node);

        outAEnumDecDeclaration(node);
    }

    /**
     * Should never be called by the TypeChecker.
     * 
     * @see minijava.analysis.Analysis#caseAEnumList(minijava.node.AEnumList)
     */
    public void caseAEnumList(AEnumList node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called by TypeChecker.
     * 
     * @see minijava.analysis.Analysis#caseACommaEnumList(minijava.node.ACommaEnumList)
     */
    public void caseACommaEnumList(ACommaEnumList node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * In a function declaration. Ok, now life gets interesting.
     * 
     * @see minijava.analysis.Analysis#caseAFuncDecDeclaration(minijava.node.AFuncDecDeclaration)
     */
    public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
        if (_errNodes.contains(node)) {
            return;
        } // Leave if this node is marked as errored
        try {
            inAFuncDecDeclaration(node);

            KFunction kf;

            kf = Util.makeFunction(node, _classTable);
            _funcMap.put(node, kf);

            _paramMap = kf.getTypeParamMap();

            if (node.getBlock() != null) {
                _cScope = "method " + kf.getName()
                        + FunctionTable.paramsToString(kf.getArgs());
                _symTab.push(kf.getSymbolTable());
                _retVal = kf.getReturnType();
                try{
                    node.getBlock().apply(this);
                }finally{
                _symTab.pop();
                _cScope = null;
                }
            }
        } catch (SourceCodeException sce) {
            _errors.add(sce);
            _errNodes.add(node);
        }

        outAFuncDecDeclaration(node);
    }

    /**
     * Constant declaration: Should already have been covered by the
     * DefinitionFinder / TypeFinder passes. So don't do anything here.
     * 
     * @see minijava.analysis.Analysis#caseAConstDecDeclaration(minijava.node.AConstDecDeclaration)
     */
    public void caseAConstDecDeclaration(AConstDecDeclaration node) {
        inAConstDecDeclaration(node);
        outAConstDecDeclaration(node);
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAFormalParamList(minijava.node.AFormalParamList)
     */
    public void inAFormalParamList(AFormalParamList node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inATypeName(minijava.node.ATypeName)
     */
    public void inATypeName(ATypeName node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inACommaTypeName(minijava.node.ACommaTypeName)
     */
    public void inACommaTypeName(ACommaTypeName node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * The nodes under here will return the underlying type of the init, just
     * need to wrap it up to a KArrayType before passing it on.
     */
    public void caseAArrayInit(AArrayInit node) {
        inAArrayInit(node);

        KType tmp = _arrType;

        if (!(_arrType instanceof KArrayType)) {
            int ln = node.getLBrace().getLine();
            int pos = node.getLBrace().getPos();
            int len = node.getLBrace().getText().trim().length();
            SourceCodeException.throwArrayDeclaration_TooManyBraces(ln, pos,
                    len, _arrType);
        }

        _arrType = ((KArrayType) _arrType).getChildType();

        node.getInitList().apply(this);
        // Top of the stack should be a type.
        // Wrap it in an array and return it.
        KType kt = (KType) _typeStack.pop();

        if (kt == KBasicType.getVoid()) {
            int[] pos = Util.getFirstIdent(node.getInitList());
            SourceCodeException.throwVoidArrayType(pos[0], pos[1], pos[2]);
        }

        /*
         * need to do a check incase the initList is of type int, and we are of
         * type double.
         */
        if (Util.isAssignmentCompatible(_arrType, kt)) {

            //>>> 26th
            _typeMap.put(node, new KArrayType(_arrType));
            _typeStack.push(new KArrayType(_arrType)); // push _arrType as its
                                                       // more general.
        } else {

            //>>> 26th
            _typeMap.put(node, new KArrayType(kt));
            _typeStack.push(new KArrayType(kt));
        }

        _arrType = tmp;

        outAArrayInit(node);
    }

    /**
     * Need to go through each element of the list, and make sure they are all
     * of the same type and return that type, or the most general type (ints and
     * doubles).
     * 
     * @see minijava.analysis.Analysis#caseAScalarInitList(minijava.node.AScalarInitList)
     */
    public void caseAScalarInitList(AScalarInitList node) {
        inAScalarInitList(node);

        node.getExpression().apply(this);
        KType cType = (KType) _typeStack.pop();

        if (!Util.isAssignmentCompatible(_arrType, cType)) {
            int[] pos = Util.getFirstIdent(node.getExpression());
            SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
                    pos[1], pos[2], _arrType, cType);
        }

        Object temp[] = node.getCommaExp().toArray();
        for (int i = 0; i < temp.length; i++) {
            ((PCommaExp) temp[i]).apply(this);
            KType nType = (KType) _typeStack.pop();

            if (!Util.isAssignmentCompatible(_arrType, nType)) {
                int[] pos = Util.getFirstIdent(((ACommaExp) temp[i])
                        .getExpression());
                SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
                        pos[1], pos[2], _arrType, nType);
            }

        }

        //>>> 26th
        _typeMap.put(node, cType);
        _typeStack.push(cType);

        outAScalarInitList(node);
    }

    /**
     * Again, go through the list and make sure they are all of the same type.
     * No need to generalise worry tho as int[] and double[] cannot be assigned
     * to in either order.
     * 
     * @see minijava.analysis.Analysis#caseAArrayInitList(minijava.node.AArrayInitList)
     */
    public void caseAArrayInitList(AArrayInitList node) {
        inAArrayInitList(node);
        node.getArrayInit().apply(this);
        KType cType = (KType) _typeStack.pop();

        if (!Util.isAssignmentCompatible(_arrType, cType)) {
            int[] pos = Util.getFirstIdent(node.getArrayInit());
            SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
                    pos[1], pos[2], _arrType, cType);
        }

        Object temp[] = node.getCommaArrayInit().toArray();
        for (int i = 0; i < temp.length; i++) {
            ((PCommaArrayInit) temp[i]).apply(this);
            KType nType = (KType) _typeStack.pop();

            if (!Util.isAssignmentCompatible(_arrType, nType)) {
                int[] pos = Util.getFirstIdent(((ACommaArrayInit) temp[i])
                        .getArrayInit());
                SourceCodeException.throwIncompatibleTypes_Assignment(pos[0],
                        pos[1], pos[2], _arrType, nType);
            }
        }

        //>>> 26th
        _typeMap.put(node, cType);
        _typeStack.push(cType);
        outAArrayInitList(node);
    }

    /**
     * An array allocate. Return the type of the array we are trying to allocate
     * on the top of the stack.
     * 
     * @see minijava.analysis.Analysis#caseAArrayAllocate(minijava.node.AArrayAllocate)
     */
    public void caseAArrayAllocate(AArrayAllocate node) {
        inAArrayAllocate(node);

        int numBrackets;

        numBrackets = node.getArrayAccess().size();

        Object temp[] = node.getArrayAccess().toArray();
        for (int i = 0; i < temp.length; i++) {
            ((PArrayAccess) temp[i]).apply(this);
        }

        numBrackets += node.getBracketPair().size();

        KType kt = Util.getType(node.getType(), _classTable, _paramMap,
                numBrackets, _cScope);

        /* Do we have generic array creation? */
        boolean genericBase = false;
        if( ((KArrayType) kt).getBaseType() instanceof KBoundClassType ){
    		KBoundClassType kbct = (KBoundClassType) ((KArrayType) kt).getBaseType();
    		if( kbct.getBase().getTypeParamList().size() != 0 ){
    			/*we are attempting creation of an array of a class with template params*/
    			genericBase = true;
    		}
    	}
        
        if (((KArrayType) kt).getBaseType() instanceof KParamType || genericBase) {
            int[] pos;
            pos = Util.getFirstIdent(node);
            SourceCodeException.throwGeneric_Array_Creation(pos[0], pos[1],
                    pos[2]);
        }

        //>>> 26th
        _typeMap.put(node, kt);
        _typeStack.push(kt);

        outAArrayAllocate(node);
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inATrueBooleanliteral(minijava.node.ATrueBooleanliteral)
     */
    public void inATrueBooleanliteral(ATrueBooleanliteral node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAFalseBooleanliteral(minijava.node.AFalseBooleanliteral)
     */
    public void inAFalseBooleanliteral(AFalseBooleanliteral node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inACharBasicType(minijava.node.ACharBasicType)
     */
    public void inACharBasicType(ACharBasicType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAIntBasicType(minijava.node.AIntBasicType)
     */
    public void inAIntBasicType(AIntBasicType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inADoubleBasicType(minijava.node.ADoubleBasicType)
     */
    public void inADoubleBasicType(ADoubleBasicType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAStringBasicType(minijava.node.AStringBasicType)
     */
    public void inAStringBasicType(AStringBasicType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inABooleanBasicType(minijava.node.ABooleanBasicType)
     */
    public void inABooleanBasicType(ABooleanBasicType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAVoidBasicType(minijava.node.AVoidBasicType)
     */
    public void inAVoidBasicType(AVoidBasicType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inAClassTypeReferenceType(minijava.node.AClassTypeReferenceType)
     */
    public void inAClassTypeReferenceType(AClassTypeReferenceType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inATypeParam(minijava.node.ATypeParam)
     */
    public void inATypeParam(ATypeParam node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inATypeParamList(minijava.node.ATypeParamList)
     */
    public void inATypeParamList(ATypeParamList node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Should never be called.
     * 
     * @see minijava.analysis.DepthFirstAdapter#inACommaType(minijava.node.ACommaType)
     */
    public void inACommaType(ACommaType node) {
        throw KenyaBadLocationError.get();
    }

    /**
     * Plus expressions: Check things are char, integer, string or double.
     * Special cases for plus since if the lhs/rhs is a string, assume the whole
     * thing is to be a string (i.e. string concatenation). Could all get a bit
     * broken, but ah, what the heck.
     * 
     * @see minijava.analysis.Analysis#caseAPlusMathExpression(minijava.node.APlusMathExpression)
     */
    public void caseAPlusMathExpression(APlusMathExpression node) {
        inAPlusMathExpression(node);

        node.getMathExpression().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getTerm().apply(this);
        KType rhs = (KType) _typeStack.pop();

        // one of lhs and rhs must be something 'plussable');
        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), lhs) || 
             Util.isAssignmentCompatible(KBasicType.getDouble(), rhs) ||
            lhs == KBasicType.getString() ||  rhs == KBasicType.getString())) {
            int ln = node.getPlus().getLine();
            int pos = node.getPlus().getPos();
            int len = node.getPlus().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "+", rhs);
            return;
        }

        if (lhs == KBasicType.getString() && rhs != KBasicType.getVoid() ) {
            // do nothing ( "" + .. will be of string type - rhs can be anything
            // pretty much?).
        } else if (rhs == KBasicType.getString() && lhs != KBasicType.getVoid() ) {
            lhs = rhs; // string concatenation.
        } else if (Util.isAssignmentCompatible(lhs, rhs)) {
            // do nothing
        } else if (Util.isAssignmentCompatible(rhs, lhs)) {
            lhs = rhs; // generalise the type of this expression.
        } else {
            int ln = node.getPlus().getLine();
            int pos = node.getPlus().getPos();
            int len = node.getPlus().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "+", rhs);
        }

        //>> 26th
        _typeMap.put(node, lhs);
        _typeStack.push(lhs);

        outAPlusMathExpression(node);
    }

    /**
     * Minus, ints, chars and doubles - check the casting aswell, make most
     * general back on the stack.
     * 
     * @see minijava.analysis.Analysis#caseAMinusMathExpression(minijava.node.AMinusMathExpression)
     */
    public void caseAMinusMathExpression(AMinusMathExpression node) {

        node.getMathExpression().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getTerm().apply(this);
        KType rhs = (KType) _typeStack.pop();

        // lhs must equal int or char or double;
        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), lhs))) {
            int ln = node.getMinus().getLine();
            int pos = node.getMinus().getPos();
            int len = node.getMinus().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "-", rhs);
        }

        if (Util.isAssignmentCompatible(lhs, rhs)) {
            // do nothing
        } else if (Util.isAssignmentCompatible(rhs, lhs)) {
            lhs = rhs; // generalise the type of this expression.
        } else {
            int ln = node.getMinus().getLine();
            int pos = node.getMinus().getPos();
            int len = node.getMinus().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "-", rhs);
        }

        //>> 26th
        _typeMap.put(node, lhs);
        _typeStack.push(lhs);
    }

    /**
     * Ints, chars and doubles only, most general back on the stack.
     * 
     * @see minijava.analysis.Analysis#caseAMultTerm(minijava.node.AMultTerm)
     */
    public void caseAMultTerm(AMultTerm node) {
        inAMultTerm(node);

        node.getTerm().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getUnaryExp().apply(this);
        KType rhs = (KType) _typeStack.pop();

        // lhs must equal int or double;
        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), lhs))) {
            int ln = node.getTimes().getLine();
            int pos = node.getTimes().getPos();
            int len = node.getTimes().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "*", rhs);
        }

        if (Util.isAssignmentCompatible(lhs, rhs)) {
            // do nothing
        } else if (Util.isAssignmentCompatible(rhs, lhs)) {
            lhs = rhs; // generalise the type of this expression.
        } else {
            int ln = node.getTimes().getLine();
            int pos = node.getTimes().getPos();
            int len = node.getTimes().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "*", rhs);
        }

        //>> 26th
        _typeMap.put(node, lhs);
        _typeStack.push(lhs);

        outAMultTerm(node);
    }

    /**
     * Ints, char and doubles only, most general back on the stack.
     * 
     * @see minijava.analysis.Analysis#caseADivTerm(minijava.node.ADivTerm)
     */
    public void caseADivTerm(ADivTerm node) {
        inADivTerm(node);

        node.getTerm().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getUnaryExp().apply(this);
        KType rhs = (KType) _typeStack.pop();

        // lhs must equal int or double;
        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), lhs))) {
            int ln = node.getDivide().getLine();
            int pos = node.getDivide().getPos();
            int len = node.getDivide().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "/", rhs);
        }

        if (Util.isAssignmentCompatible(lhs, rhs)) {
            // do nothing
        } else if (Util.isAssignmentCompatible(rhs, lhs)) {
            lhs = rhs; // generalise the type of this expression.
        } else {
            int ln = node.getDivide().getLine();
            int pos = node.getDivide().getPos();
            int len = node.getDivide().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "/", rhs);
        }

        //>> 26th
        _typeMap.put(node, lhs);
        _typeStack.push(lhs);

        outADivTerm(node);
    }

    /**
     * Ints and doubles only, most general back on the stack.
     * 
     * @see minijava.analysis.Analysis#caseADivTerm(minijava.node.ADivTerm)
     */
    public void caseAModTerm(AModTerm node) {
        inAModTerm(node);

        node.getTerm().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getUnaryExp().apply(this);
        KType rhs = (KType) _typeStack.pop();

        // lhs must equal int, char or double;
        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), lhs))) {
            int ln = node.getMod().getLine();
            int pos = node.getMod().getPos();
            int len = node.getMod().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "%", rhs);
        }

        if (Util.isAssignmentCompatible(lhs, rhs)) {
            // do nothing
        } else if (Util.isAssignmentCompatible(rhs, lhs)) {
            lhs = rhs; // generalise the type of this expression.
        } else {
            int ln = node.getMod().getLine();
            int pos = node.getMod().getPos();
            int len = node.getMod().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "%", rhs);
            return;
        }

        //>> 26th
        _typeMap.put(node, lhs);
        _typeStack.push(lhs);

        outAModTerm(node);
    }

    /**
     * Unary Minus, only char, int or double types allowed.
     * 
     * @see minijava.analysis.Analysis#caseAMinusUnaryExp(minijava.node.AMinusUnaryExp)
     */
    public void caseAMinusUnaryExp(AMinusUnaryExp node) {
        inAMinusUnaryExp(node);

        node.getUnaryExp().apply(this);

        KType kt = (KType) _typeStack.peek();

        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), kt))) {
            int ln = node.getMinus().getLine();
            int pos = node.getMinus().getPos();
            int len = node.getMinus().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len, "-",
                    kt);
        }

        outAMinusUnaryExp(node);
    }

    /**
     * Unary plus, like minus, only char ints or double's allowed.
     * 
     * @see minijava.analysis.Analysis#caseAPlusUnaryExp(minijava.node.APlusUnaryExp)
     */
    public void caseAPlusUnaryExp(APlusUnaryExp node) {

        inAPlusUnaryExp(node);

        node.getUnaryExp().apply(this);

        KType kt = (KType) _typeStack.peek();

        if (!(Util.isAssignmentCompatible(KBasicType.getDouble(), kt))) {
            int ln = node.getPlus().getLine();
            int pos = node.getPlus().getPos();
            int len = node.getPlus().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len, "+",
                    kt);
        }

        outAPlusUnaryExp(node);
    }

    /**
     * Unary negation (!). Only boolean's allowed.
     * 
     * @see minijava.analysis.Analysis#caseANegateUnaryExp(minijava.node.ANegateUnaryExp)
     */
    public void caseANegateUnaryExp(ANegateUnaryExp node) {
        inANegateUnaryExp(node);
        node.getUnaryExp().apply(this);

        KType kt = (KType) _typeStack.peek();

        if ( ! Util.isAssignmentCompatible(kt,KBasicType.getBoolean())) {
            int ln = node.getNot().getLine();
            int pos = node.getNot().getPos();
            int len = node.getNot().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len, "!",
                    kt);
        }
        outANegateUnaryExp(node);
    }

    /**
     * Post decrement - only int /char s allowed.
     * 
     * @see minijava.analysis.Analysis#caseAPostdecr(minijava.node.APostdecr)
     */
    public void caseAPostdecr(APostdecr node) {
        inAPostdecr(node);
        node.getFieldAccess().apply(this);

        KType kt = (KType) _typeStack.pop();
        if (!(Util.isAssignmentCompatible(KBasicType.getInt(), kt))) {
            int ln = node.getMinusminus().getLine();
            int pos = node.getMinusminus().getPos();
            int len = node.getMinusminus().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len,
                    "--", kt);
        }

        outAPostdecr(node);
    }

    /**
     * Post increment, only int /char s allowed.
     * 
     * @see minijava.analysis.Analysis#caseAPostincr(minijava.node.APostincr)
     */
    public void caseAPostincr(APostincr node) {
        inAPostincr(node);

        node.getFieldAccess().apply(this);

        KType kt = (KType) _typeStack.pop();
        if (!(Util.isAssignmentCompatible(KBasicType.getInt(), kt))) {
            int ln = node.getPlusplus().getLine();
            int pos = node.getPlusplus().getPos();
            int len = node.getPlusplus().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len,
                    "++", kt);
        }

        outAPostincr(node);
    }

    /**
     * Pre decrement, only int / char s allowed.
     * 
     * @see minijava.analysis.Analysis#caseAPredecr(minijava.node.APredecr)
     */
    public void caseAPredecr(APredecr node) {
        inAPredecr(node);

        node.getFieldAccess().apply(this);

        KType kt = (KType) _typeStack.pop();
        if (!(Util.isAssignmentCompatible(KBasicType.getInt(), kt))) {
            int ln = node.getMinusminus().getLine();
            int pos = node.getMinusminus().getPos();
            int len = node.getMinusminus().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len,
                    "--", kt);
        }

        outAPredecr(node);
    }

    /**
     * Pre increment, only int / char s allowed.
     * 
     * @see minijava.analysis.Analysis#caseAPreincr(minijava.node.APreincr)
     */
    public void caseAPreincr(APreincr node) {
        inAPreincr(node);

        node.getFieldAccess().apply(this);

        KType kt = (KType) _typeStack.pop();
        if (!(Util.isAssignmentCompatible(KBasicType.getInt(), kt))) {
            int ln = node.getPlusplus().getLine();
            int pos = node.getPlusplus().getPos();
            int len = node.getPlusplus().getText().trim().length();
            SourceCodeException.throwUnop_InvalidApplication(ln, pos, len,
                    "++", kt);
        }

        outAPreincr(node);
    }

    /***************************************************************************
     * Literals
     **************************************************************************/

    /**
     * Puts KNullType onto the _typeStack. KNullType isnt really a valid type
     * but a magic placeholder.
     * 
     * @see minijava.analysis.Analysis#caseANullFactor(minijava.node.ANullFactor)
     */
    public void caseANullFactor(ANullFactor node) {
        inANullFactor(node);

        //>> 26th
        _typeMap.put(node, KNullType.get());
        _typeStack.push(KNullType.get());
        outANullFactor(node);
    }

    /**
     * @see minijava.analysis.Analysis#caseAINumber(minijava.node.AINumber)
     */
    public void caseAINumber(AINumber node) {
        inAINumber(node);
        if (_typeContext == KBasicType.getChar()) {

            //>> 26th
            _typeMap.put(node, KBasicType.getChar());
            _typeStack.push(KBasicType.getChar());
        } else {

            //>> 26th
            _typeMap.put(node, KBasicType.getInt());
            _typeStack.push(KBasicType.getInt());
        }
        outAINumber(node);
    }

    /**
     * Puts a double onto the stack
     * 
     * @see minijava.analysis.Analysis#caseADNumber(minijava.node.ADNumber)
     */
    public void caseADNumber(ADNumber node) {
        inADNumber(node);

        //>> 26th
        _typeMap.put(node, KBasicType.getDouble());
        _typeStack.push(KBasicType.getDouble());
        outADNumber(node);
    }

    /**
     * Puts a char onto the stack
     * 
     * @see minijava.analysis.Analysis#caseACharliteralFactor(minijava.node.ACharliteralFactor)
     */
    public void caseACharliteralFactor(ACharliteralFactor node) {
        inACharliteralFactor(node);

        //>> 26th
        _typeMap.put(node, KBasicType.getChar());
        _typeStack.push(KBasicType.getChar());
        outACharliteralFactor(node);
    }

    /**
     * Puts a boolean onto the stack
     * 
     * @see minijava.analysis.Analysis#caseABoolliteralFactor(minijava.node.ABoolliteralFactor)
     */
    public void caseABoolliteralFactor(ABoolliteralFactor node) {
        inABoolliteralFactor(node);

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());
        outABoolliteralFactor(node);
    }

    public void caseAStringliteralFactor(AStringliteralFactor node) {
        inAStringliteralFactor(node);

        //>> 26th
        _typeMap.put(node, KBasicType.getString());
        _typeStack.push(KBasicType.getString());
        outAStringliteralFactor(node);
    }

    /***************************************************************************
     * Bool expressions, terms, equalities and other fun stuff
     **************************************************************************/

    /**
     * OR, lfs and rhs must be boolean type.
     * 
     * @see minijava.analysis.Analysis#caseAOrBoolExpression(minijava.node.AOrBoolExpression)
     */
    public void caseAOrBoolExpression(AOrBoolExpression node) {
        inAOrBoolExpression(node);

        node.getBoolExpression().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getBoolTerm().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if (!Util.isAssignmentCompatible(lhs,KBasicType.getBoolean())) {
            int ln = node.getOr().getLine();
            int pos = node.getOr().getPos();
            int len = node.getOr().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, node.getOr().getText().trim(), rhs);
        }

        if (!Util.isAssignmentCompatible(rhs,KBasicType.getBoolean())) {
            int ln = node.getOr().getLine();
            int pos = node.getOr().getPos();
            int len = node.getOr().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, node.getOr().getText().trim(), rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outAOrBoolExpression(node);
    }

    /**
     * As above, lhs, rhs should be boolean type.
     * 
     * @see minijava.analysis.Analysis#caseAXorBoolExpression(minijava.node.AXorBoolExpression)
     */
    public void caseAXorBoolExpression(AXorBoolExpression node) {
        inAXorBoolExpression(node);

        node.getBoolExpression().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getBoolTerm().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if (! Util.isAssignmentCompatible(lhs,KBasicType.getBoolean())) {
            int ln = node.getXor().getLine();
            int pos = node.getXor().getPos();
            int len = node.getXor().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, node.getXor().getText().trim(), rhs);
        }

        if (!Util.isAssignmentCompatible(rhs, KBasicType.getBoolean())) {
            int ln = node.getXor().getLine();
            int pos = node.getXor().getPos();
            int len = node.getXor().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, node.getXor().getText().trim(), rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outAXorBoolExpression(node);
    }

    /**
     * Again, lhs, rhs must be boolean type.
     * 
     * @see minijava.analysis.Analysis#caseAAndBoolTerm(minijava.node.AAndBoolTerm)
     */
    public void caseAAndBoolTerm(AAndBoolTerm node) {
        inAAndBoolTerm(node);

        node.getBoolTerm().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getEquality().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if ( !Util.isAssignmentCompatible(lhs, KBasicType.getBoolean())) {
            int ln = node.getAnd().getLine();
            int pos = node.getAnd().getPos();
            int len = node.getAnd().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, node.getAnd().getText().trim(), rhs);
        }

        if (!Util.isAssignmentCompatible(rhs, KBasicType.getBoolean())) {
            int ln = node.getAnd().getLine();
            int pos = node.getAnd().getPos();
            int len = node.getAnd().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, node.getAnd().getText().trim(), rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outAAndBoolTerm(node);
    }

    /**
     * Check the lhs and rhs are both of the same type. :|
     * 
     * @see minijava.analysis.Analysis#caseAEqEquality(minijava.node.AEqEquality)
     */
    public void caseAEqEquality(AEqEquality node) {
        inAEqEquality(node);

        node.getEquality().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getRelational().apply(this);
        KType rhs = (KType) _typeStack.pop();

       /* if (!(lhs instanceof KBasicType || rhs instanceof KBasicType)) {
            // its all good.
        } else */ if (Util.isAssignmentCompatible(rhs, lhs)) {
            // its also good
        } else if (Util.isAssignmentCompatible(lhs, rhs)) {
            // its also good.
        } else {
            // trying to equate two things that can't be equated.
            int ln = node.getEqual().getLine();
            int pos = node.getEqual().getPos();
            int len = node.getEqual().getText().trim().length();
            SourceCodeException.throwIncomparable(ln, pos, len, lhs, "==", rhs);
        }

        //>> 26th
        //>> 10th
        if ( (lhs == KBasicType.getString() && rhs != KNullType.get()) ||
        	  rhs == KBasicType.getString() && lhs != KNullType.get()) {
            _typeMap.put(node, KBasicType.getString());
        } else {
            _typeMap.put(node, KBasicType.getBoolean());
        }

        _typeStack.push(KBasicType.getBoolean());

        outAEqEquality(node);
    }

    /**
     * Both "the same type", see above.
     * 
     * @see minijava.analysis.Analysis#caseANeqEquality(minijava.node.ANeqEquality)
     */
    public void caseANeqEquality(ANeqEquality node) {
        inANeqEquality(node);

        node.getEquality().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getRelational().apply(this);
        KType rhs = (KType) _typeStack.pop();

       /* if (!(lhs instanceof KBasicType || rhs instanceof KBasicType)) {
            // its all good.
        } else */ if (Util.isAssignmentCompatible(rhs, lhs)) {
            // its also good
        } else if (Util.isAssignmentCompatible(lhs, rhs)) {
            // its also good.
        } else {
            // trying to equate two things that can't be equated.
            int ln = node.getNotequal().getLine();
            int pos = node.getNotequal().getPos();
            int len = node.getNotequal().getText().trim().length();
            SourceCodeException.throwIncomparable(ln, pos, len, lhs, "!=", rhs);
        }

        //>> 26th
        //>> 10th
        if ( (lhs == KBasicType.getString() && rhs != KNullType.get()) ||
          	  rhs == KBasicType.getString() && lhs != KNullType.get()) {
            _typeMap.put(node, KBasicType.getString());
        } else {
            _typeMap.put(node, KBasicType.getBoolean());
        }

        _typeStack.push(KBasicType.getBoolean());

        outANeqEquality(node);
    }

    /**
     * Relational: int or double or char lhs, int or double or char rhs.
     * 
     * @see minijava.analysis.Analysis#caseALtRelational(minijava.node.ALtRelational)
     */
    public void caseALtRelational(ALtRelational node) {
        inALtRelational(node);

        node.getRelational().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getMathExpression().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if (Util.isAssignmentCompatible(KBasicType.getDouble(), lhs)
                && Util.isAssignmentCompatible(KBasicType.getDouble(), rhs)) {
            // its ok
        } else {
            int ln = node.getLess().getLine();
            int pos = node.getLess().getPos();
            int len = node.getLess().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "<", rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outALtRelational(node);
    }

    /**
     * Relational, int or double lhs, int or double rhs.
     * 
     * @see minijava.analysis.Analysis#caseAGtRelational(minijava.node.AGtRelational)
     */
    public void caseAGtRelational(AGtRelational node) {
        inAGtRelational(node);

        node.getRelational().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getMathExpression().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if (Util.isAssignmentCompatible(KBasicType.getDouble(), lhs)
                && Util.isAssignmentCompatible(KBasicType.getDouble(), rhs)) {
            // its ok
        } else {
            int ln = node.getGreater().getLine();
            int pos = node.getGreater().getPos();
            int len = node.getGreater().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, ">", rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outAGtRelational(node);
    }

    /**
     * int or double <= int or double.
     * 
     * @see minijava.analysis.Analysis#caseALteqRelational(minijava.node.ALteqRelational)
     */
    public void caseALteqRelational(ALteqRelational node) {
        inALteqRelational(node);

        node.getRelational().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getMathExpression().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if (Util.isAssignmentCompatible(KBasicType.getDouble(), lhs)
                && Util.isAssignmentCompatible(KBasicType.getDouble(), rhs)) {
            // its ok
        } else {
            int ln = node.getLessequal().getLine();
            int pos = node.getLessequal().getPos();
            int len = node.getLessequal().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, "<=", rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outALteqRelational(node);
    }

    /**
     * int or double >= int or double.
     * 
     * @see minijava.analysis.Analysis#caseAGteqRelational(minijava.node.AGteqRelational)
     */
    public void caseAGteqRelational(AGteqRelational node) {
        inAGteqRelational(node);

        node.getRelational().apply(this);
        KType lhs = (KType) _typeStack.pop();

        node.getMathExpression().apply(this);
        KType rhs = (KType) _typeStack.pop();

        if (Util.isAssignmentCompatible(KBasicType.getDouble(), lhs)
                && Util.isAssignmentCompatible(KBasicType.getDouble(), rhs)) {
            // its ok
        } else {
            int ln = node.getGreaterequal().getLine();
            int pos = node.getGreaterequal().getPos();
            int len = node.getGreaterequal().getText().trim().length();
            SourceCodeException.throwBinop_InvalidApplication(ln, pos, len,
                    lhs, ">=", rhs);
        }

        //>> 26th
        _typeMap.put(node, KBasicType.getBoolean());
        _typeStack.push(KBasicType.getBoolean());

        outAGteqRelational(node);
    }

    /**
     * Array field accesses, the return type is the child's type - the number of
     * accesses there are.
     * 
     * @see minijava.analysis.Analysis#caseAArrayFieldAccess(minijava.node.AArrayFieldAccess)
     */
    public void caseAArrayFieldAccess(AArrayFieldAccess node) {
        inAArrayFieldAccess(node);

        node.getName().apply(this);
        // on the stack should be the type of this things "name".

        KType kt = (KType) _typeStack.pop();

        {
            Object temp[] = node.getArrayAccess().toArray();
            for (int i = 0; i < temp.length; i++) {
                ((PArrayAccess) temp[i]).apply(this);
            }
        }

        /* Lets do some checking: */

        for (int i = 0; i < node.getArrayAccess().size(); i++) {
            if (!(kt instanceof KArrayType)) {
                int ln = ((AArrayAccess) node.getArrayAccess().get(i))
                        .getLBracket().getLine();
                int pos = ((AArrayAccess) node.getArrayAccess().get(i))
                        .getLBracket().getPos();
                int len = ((AArrayAccess) node.getArrayAccess().get(i))
                        .getLBracket().getText().trim().length();

                SourceCodeException
                        .throwArray_TooManyAccesses(ln, pos, len, kt);
            }
            kt = ((KArrayType) kt).getChildType();
        }

        //>> 26th
        _typeMap.put(node, kt);
        _typeStack.push(kt);

        outAArrayFieldAccess(node);
    }

    /**
     * Simple names: look them up in the symbol table and return them.
     * 
     * @see minijava.analysis.Analysis#caseASimpleName(minijava.node.ASimpleName)
     */
    public void caseASimpleName(ASimpleName node) {
        inASimpleName(node);

        String ident = node.getIdentifier().getText().trim();

        if (_switchType != null && _switchType instanceof KEnumType) {
            /*
             * WE are in a switch block header, and somebody is using an enum to
             * switch on:
             */

            KEnumType enumSwi = (KEnumType) _switchType;

            if (enumSwi.hasChild(ident)) {
                _typeStack.push(enumSwi);
                _typeMap.put(node, enumSwi);
                _enumChildSet.add(node);
                outASimpleName(node);
                return;
            } else {
                int[] pos = Util.getFirstIdent(node);
                SourceCodeException.throwSwitch_EnumCaseExpected(pos[0],
                        pos[1], pos[2], _switchType);
            }
        }

        if (_symTab.containsVariable(ident)) {
            KType kt = _symTab.lookupVariable(ident).getType();

            //>> 26th
            _typeMap.put(node, kt);
            _typeStack.push(kt);
        } else if (_classTable.isEnum(ident)) {

            KType kt = _classTable.getType(ident);

            if (node.parent() != null
                    && node.parent().parent() != null
                    && node.parent().parent().parent() instanceof AQualifiedName) {
                /* we are going to be dereferenced so thats fine */
            } else {
                int ln = node.getIdentifier().getLine();
                int pos = node.getIdentifier().getPos();
                int len = node.getIdentifier().getText().trim().length();
                SourceCodeException.throwLost_Variable(ln, pos, len, ident,
                        _cScope);
            }

            //>> 26th
            _typeMap.put(node, kt);
            _typeStack.push(kt);

        } else {
            int ln = node.getIdentifier().getLine();
            int pos = node.getIdentifier().getPos();
            int len = node.getIdentifier().getText().trim().length();
            SourceCodeException
                    .throwLost_Variable(ln, pos, len, ident, _cScope);
        }

        outASimpleName(node);
    }

    /**
     * @see minijava.analysis.Analysis#caseAQualifiedName(minijava.node.AQualifiedName)
     */
    public void caseAQualifiedName(AQualifiedName node) {
        inAQualifiedName(node);

        if (_switchType != null && _switchType instanceof KEnumType) {
            /*
             * WE are in a switch block header, and somebody is using an enum to
             * switch on: failed.
             */
            int[] pos = Util.getFirstIdent(node);
            SourceCodeException.throwSwitch_EnumCaseExpected(pos[0], pos[1],
                    pos[2], _switchType);
        }

        node.getFieldAccess().apply(this);

        KType type = (KType) _typeStack.pop();

        if (type instanceof KBoundClassType) {
            KBoundClassType kbct = (KBoundClassType) type;

            SymbolTable st = kbct.getBoundChildren();

            String partName = node.getIdentifier().getText().trim();
            if (st.containsVariable(partName)) {
                KVariable kv = st.lookupVariable(partName);

                //>> 26th
                _typeMap.put(node, kv.getType());
                _typeStack.push(kv.getType());
            } else {
                int ln = node.getIdentifier().getLine();
                int pos = node.getIdentifier().getPos();
                int len = node.getIdentifier().getText().trim().length();
                SourceCodeException.throwLost_Variable(ln, pos, len, partName,
                        kbct.toString());
            }
        } else if (type instanceof KClassType) {
            throw KenyaPreconditionError.get();
        } else if (type instanceof KEnumType) {
            String chName = node.getIdentifier().getText().trim();
            KEnumType et = (KEnumType) type;

            if (et.hasChild(chName)) {
                //>> 26th
                _enumChildSet.add(node);
                _typeMap.put(node, type);
                _typeStack.push(type);
            } else {
                int ln = node.getIdentifier().getLine();
                int pos = node.getIdentifier().getPos();
                int len = node.getIdentifier().getText().trim().length();
                SourceCodeException.throwLost_Variable(ln, pos, len, chName, et
                        .getName());
            }
        } else if (type instanceof KArrayType) {
            String partName = node.getIdentifier().getText().trim();
            if (partName.equals("length")) {
                //>> 26th
                _arrayLengthRef.add(node);
                _typeMap.put(node, KBasicType.getInt());
                _typeStack.push(KBasicType.getInt());
            } else {
                int ln = node.getIdentifier().getLine();
                int pos = node.getIdentifier().getPos();
                int len = node.getIdentifier().getText().trim().length();
                SourceCodeException.throwArray_Dereference(ln, pos, len,
                        partName, "class " + type);
            }

        } else if (type instanceof KParamType) {
            int ln = node.getIdentifier().getLine();
            int pos = node.getIdentifier().getPos();
            int len = node.getIdentifier().getText().trim().length();
            String name = node.getIdentifier().getText().trim();
            SourceCodeException.throwTypeParam_Dereference(ln, pos, len, name);
            return;
        } else if (type instanceof KBasicType) {
            int ln = node.getIdentifier().getLine();
            int pos = node.getIdentifier().getPos();
            int len = node.getIdentifier().getText().trim().length();
            SourceCodeException.throwBasicType_Dereference(ln, pos, len, type);
            return;
        } else {
            throw new KenyaInternalError("Unknown type" + type);
        }

        outAQualifiedName(node);
    }

    /**
     * Array access have to be integer type.
     * 
     * @see minijava.analysis.Analysis#caseAArrayAccess(minijava.node.AArrayAccess)
     */
    public void caseAArrayAccess(AArrayAccess node) {

        inAArrayAccess(node);

        KType tmpType = _typeContext;
        _typeContext = KBasicType.getInt();
        
        node.getMathExpression().apply(this);

        KType kt = (KType) _typeStack.pop();
        if (!Util.isAssignmentCompatible(KBasicType.getInt(), kt)) {
            int[] pos = Util.getFirstIdent(node.getMathExpression());
            SourceCodeException.throwIncompatibleTypes_ArrayIndex(pos[0],
                    pos[1], pos[2], kt);
            return;
        }
        
        _typeContext = tmpType;
        
        outAArrayAccess(node);

    }

}