/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

//Rewritten completely by Tristan Allwood
package kenya.passes;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import kenya.errors.SourceCodeException;
import kenya.types.KClassType;
import kenya.types.KEnumType;
import kenya.types.tables.ClassTable;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AClassDecDeclaration;
import minijava.node.ACommaEnumList;
import minijava.node.AConstDecDeclaration;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AEnumList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.ATypeParam;
import minijava.node.ATypeParamList;

/**
 * TypeFinder: Finds Enums and Class's and sets up native types in the Set
 * aswell It only type checks that enum's and class's don't duplicate share
 * names.
 */
public class TypeFinder extends DepthFirstAdapter {

	private final ClassTable _ct;
	private List _errors = new LinkedList();

	private Set _errNodes;
	
	public TypeFinder( ClassTable ct) {
	    _ct = ct;
	    _errNodes = new HashSet();
	}

	public List getErrors() {
		return _errors;
	}

	public Set getErrorNodes(){
	    return _errNodes;
	}
	
	public ClassTable getClassTable() {
		return _ct;
	}

	/**
	 * A class declaration node.
	 * Gets the class name, and any template parameters and creates a KClassType. Then puts that, the name and
	 * its node into the ClassTable.
	 * @see minijava.analysis.Analysis#caseAClassDecDeclaration(minijava.node.AClassDecDeclaration)
	 */
	public void caseAClassDecDeclaration(AClassDecDeclaration node) {
		try{
		    if (node.getIdentifier() != null) {
				String className = node.getIdentifier().getText().trim();
				List params = null;
				if (node.getTypeParam() != null) {
					params = Util.getParameterTemplates((ATypeParamList) ((ATypeParam) node
							.getTypeParam()).getTypeParamList(),className);
				}
				_ct.put(className,new KClassType(className,params),node);
			}
		}catch(SourceCodeException sce){
			_errors.add(sce);
			_errNodes.add(node);
			return;
		}
	}

	/**
	 * Enum declaration.
	 * Generate a new KEnumType and add it to the ClassTable along with its node. 
	 * @see minijava.analysis.Analysis#caseAEnumDecDeclaration(minijava.node.AEnumDecDeclaration)
	 */
	public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
	    try {
			String enumName = node.getIdentifier().toString().trim();
			Set enumChildren = new HashSet();

			AEnumList ael = (AEnumList) node.getEnumList();
			String s = ael.getIdentifier().getText().trim();
			enumChildren.add(s);

			if (ael.getCommaEnumList() != null) {
				Iterator it = ael.getCommaEnumList().iterator();

				while (it.hasNext()) {
					ACommaEnumList acel = (ACommaEnumList) it.next();

					s = acel.getIdentifier().getText().trim();
					if (!enumChildren.add(s)) {
						int ln = acel.getIdentifier().getLine();
						int pos = acel.getIdentifier().getPos();
						int len = acel.getIdentifier().getText().trim().length();
						SourceCodeException.throwDuplicateEnumChild(ln, pos, len, s, enumName);
					}
				}
			}

			_ct.put(enumName, new KEnumType(enumName,enumChildren),node);
			
		} catch (SourceCodeException sce) {
			_errNodes.add(node);
			_errors.add(sce);
			return;
		}
	}

	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
		//Don't allow this to go down into code.
		//There's no point.
	}

	public void caseAConstDecDeclaration(AConstDecDeclaration node) {
		//Don't allow this to go down into the code.
	}

}