/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 01-Jul-2004 by toa02
 *
 */
package kenya.passes;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.errors.SourceCodeException;
import kenya.types.KArrayType;
import kenya.types.KBasicType;
import kenya.types.KBoundClassType;
import kenya.types.KClassType;
import kenya.types.KEnumType;
import kenya.types.KFunction;
import kenya.types.KNullType;
import kenya.types.KParamType;
import kenya.types.KType;
import kenya.types.KVariable;
import kenya.types.tables.ClassTable;
import kenya.types.tables.TypeTable;
import minijava.node.ABasicTypeType;
import minijava.node.ABooleanBasicType;
import minijava.node.ACharBasicType;
import minijava.node.ACharliteralFactor;
import minijava.node.AClassTypeReferenceType;
import minijava.node.ACommaType;
import minijava.node.ACommaTypeName;
import minijava.node.ADoubleBasicType;
import minijava.node.AFormalParamList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.AIntBasicType;
import minijava.node.AReferenceTypeType;
import minijava.node.AStringBasicType;
import minijava.node.ATypeName;
import minijava.node.ATypeParam;
import minijava.node.ATypeParamList;
import minijava.node.AVoidBasicType;
import minijava.node.Node;
import minijava.node.PBasicType;
import minijava.node.PType;
import minijava.node.TBracketPair;

/**
 * Utility methods for the Passes and Type Checker
 * @author toa02
 *
 */
public class Util {

    
    public static final String GLOBAL = "<global>";


    /**
	 * Returns the KType of the given PType node, from the ClassTable / paramTypes variables.
	 * It wraps ArrayTypes round this object as necessary.
	 * If this type is a bound class type, it will return a KBoundType as appropriate.
	 * @param type The PType node that defines this type.
	 * @param ct The current ClassTable.
	 * @param paramTypes A Map of String -> ParamType for the ParamTypes in this scope
	 * @param numBrackets The depth of the array for this type.
	 * @return
	 */
	static KType getType(PType type, ClassTable ct, Map paramTypes, int numBrackets, String cScope){
		KType child;
		
		/* Is it a Basic type? */
		if (type instanceof ABasicTypeType) {
			child = getBasicType((ABasicTypeType)type);
			
		/* Oh no, the more complex. Is it a Reference type? If so...what the heck is it? */
		} else if (type instanceof AReferenceTypeType) {
			AClassTypeReferenceType actrt = (AClassTypeReferenceType) ((AReferenceTypeType) type).getReferenceType();
			
			String ident = actrt.getIdentifier().getText().trim();
			
			/* Right, firstly do we know you as a paramType? */
			if(paramTypes.containsKey(ident)) {
				
				
				/* Ok, now you better not have typeParameters yourself. */
				if( actrt.getTypeParam() == null ){
					/*Excellent, well you look well formed. */
					child = (KType) paramTypes.get(ident);
					
				}else {
					/* Well time to write home about u being bad. */
					int ln = ((ATypeParam)actrt.getTypeParam()).getLess().getLine();
					int pos = ((ATypeParam)actrt.getTypeParam()).getLess().getPos();
					int len = ((ATypeParam)actrt.getTypeParam()).getLess().getText().trim().length();
					SourceCodeException.throwTypeParam_WithTypeParam(ln, pos, len);
					return null; //not actually possible to get here.
				}
			
			/* Ok, so you're not a paramType, are you a class Type or enumType? */
			}else if( ct.containsMapping(ident) ) {
				
				/* So which class or enum? */
				if(ct.isClass(ident) ) {
					/* Right, I'm a class type.
					 * 
					 * This is not simple.
					 * Do I have Type parameters?
					 * 		If I do, then I need to build a list of types of my parameters
					 * 		And then check that they match who I think I am :).
					 * */
					KClassType kc = (KClassType) ct.getType(ident);
					List params = getTypeParamList( actrt, ct, paramTypes, cScope );
					
					if( kc.doTypeParametersMatch(params)) {
					    TypeTable tt = kc.genTypeTable(params);
					    child = new KBoundClassType(kc,tt);
					}else {
						int ln = actrt.getIdentifier().getLine();
						int pos = actrt.getIdentifier().getPos();
						int len = actrt.getIdentifier().getText().trim().length();
						SourceCodeException.throwTypeParam_MisMatch(ln, pos, len, kc, kc.getTypeParamList().size());
						return null; //not actually possible to get here
					}
					
				}else if( ct.containsMapping(ident) ) {
					/* Ah, I'm an enum type. Better not have type parameters. */
					if(actrt.getTypeParam() == null){
						/*Excellent, well you look well formed. */
						child = ct.getType(ident);
					}else {
						/* Well time to write home about u being bad. */
						int ln = ((ATypeParam)actrt.getTypeParam()).getLess().getLine();
						int pos = ((ATypeParam)actrt.getTypeParam()).getLess().getPos();
						int len = ((ATypeParam)actrt.getTypeParam()).getLess().getText().trim().length();
						SourceCodeException.throwTypeParam_OnEnum(ln, pos,len);
						return null; // not actually possible to get here
					}
					
				
				}else {
					/* This really should be impossible */
					throw new KenyaInternalError("Got a type that is in the known types, but isn't a class, or an enum type!");
				}
			
			/* So you ain't in the known types. Well, sorry. Clear off. */
			}else {
				int ln = actrt.getIdentifier().getLine();
				int pos = actrt.getIdentifier().getPos();
				int len = actrt.getIdentifier().getText().trim().length();
				SourceCodeException.throwLost_Class(ln, pos, len, ident, cScope);
				return null; //not actually possible to get here.
			}
			
			
		/* Well, then, what on earth are you? */
		}else {
			throw new KenyaInternalError("Got a type that isn't Reference or Basic!" + type);
		}
		
		/* Now we have the class, add its brackets. */
		for(int i = 0 ; i < numBrackets ; i++ ) {
		    if( child == KBasicType.getVoid() ){
		        int[] pos = getFirstIdent(type);
		        SourceCodeException.throwVoidArrayType(pos[0],pos[1],pos[2]);
		    }
		    child = new KArrayType(child);
		}
		
		return child;
	}

    /**
     * @param node
     */
    static char parseCharacter(ACharliteralFactor node) {
        String text = node.getCharliteral().getText();
        
        int start = text.indexOf("'");
        int end   = text.lastIndexOf("'");
        
        text = text.substring(start+1, end);
        
        char c;
        
        if( text.length() != 1 ){
            if( text.charAt(0) != '\\' ){
                throw new KenyaInternalError("Impossible");
            }
            
            char tmp = text.charAt(1);
            
            c = escapeOneChar(tmp);
          
        }else{
            c = text.charAt(0);
        }
        
        return c;
    }
	
    
    private static char escapeOneChar(char in){
        char c;
        switch(in){
		case '0':
       		 c = '\0';
       		 break;
	    case 'b':
       	     c = '\b';
       	     break;
	    case 't':
	        c = '\t';
	        break;
	    case 'n':
	        c = '\n';
	        break;
	    case 'f':
	        c = '\f';
	        break;
	    case 'r':
	        c = '\r';
	        break;
	    case '"':
	        c = '\"';
	        break;
	    case '\'':
	        c = '\'';
	     break;
	    case '\\':
	        c = '\\';
	        break;
	    default:
	        throw new KenyaInternalError("Impossible");
        }
        return c;
    }
    
    
    /**
     * 
     * @param unescaped
     * @return
     */
    static String escapeString(String unescaped){
        StringBuffer sb = new StringBuffer(unescaped);
        StringBuffer out = new StringBuffer();
        
        boolean escapeMode = false;
        
        for(int i = 0 ; i < sb.length() ; i++ ){
            char c = sb.charAt(i);
            
            if( escapeMode ){
                c = escapeOneChar(c);
                out.append(c);
                escapeMode = false;
            }else{
                if ( c == '\\') {
                    escapeMode = true;
                } else {
                    out.append(c);
                }
            }
        }
        
        return out.toString();
    }
    
	/**
	 * Returns a list of the KTypes in the Type Parameter list of the provided type.
	 * @param type The PType node to get the Type Parameters for
	 * @param ct The ClassTable of all defined types.
	 * @param paramTypes A Map of String->KParamType of the current-in-play paramTypes
	 * @return
	 */
	static List getTypeParamList(PType type, ClassTable ct, Map paramTypes, String cScope){
	    if (!(type instanceof AReferenceTypeType)) { return new LinkedList(); }

	    AClassTypeReferenceType actrt = (AClassTypeReferenceType) ((AReferenceTypeType) type).getReferenceType();
	    String ident = actrt.getIdentifier().getText().trim();
	    
	    if(!ct.isClass(ident)){ return new LinkedList(); }
	    
	    return getTypeParamList(actrt, ct, paramTypes, cScope);
	}
	
	/**
	 * Returns a List of the Template Parameter Types to a given AClassTypeReferenceType
	 * @param actrt
	 * @param types
	 * @param paramTypes
	 * @return
	 */
	static List getTypeParamList(AClassTypeReferenceType actrt, ClassTable ct, Map paramTypes, String cScope){
				
		if(	actrt.getTypeParam() == null ) {
			return new LinkedList();
		}
		
		ATypeParamList atpl = ((ATypeParamList)((ATypeParam)actrt.getTypeParam()).getTypeParamList());
		
		return getTypeParamList( atpl, ct, paramTypes, cScope);
	}
	
	
	/**
	 * Utility wrapper round TokenPositionFinder to get the first (most left-most) token's position.
	 * Originally this was going to be done by a more complicated method, hence the indirection.
	 * @param node
	 * @return
	 */
	public static int[] getFirstIdent( Node node ){
	    return TokenPositionFinder.find(node);
	}
	
	
	/**
	 * Some javadoc.
	 * @param atpl			I dunno
	 * @param types			I forgot
	 * @param paramTypes	Eh?
	 * @return
	 * @throws SourceCodeException
	 * @throws AnonymousSourceCodeException
	 */
	static List getTypeParamList(ATypeParamList atpl, ClassTable ct, Map paramTypes, String cScope){
	    List l = new LinkedList();
	    
	    int numBrackets = ( atpl.getBracketPair() != null) ? atpl.getBracketPair().size() : 0;
		
	    
	    KType otype = getType(atpl.getType(), ct, paramTypes,numBrackets,cScope); 
		
	    if( otype instanceof KBasicType && otype != KBasicType.getVoid() && otype != KBasicType.getString() ){
	        l.add(new KBoundClassType(((KBasicType)otype).getClassType(),new TypeTable()));
	    }else if( otype == KBasicType.getVoid() ){
	        int[] pos = getFirstIdent(atpl.getType());
	        SourceCodeException.throwVoidParamType(pos[0],pos[1],pos[2]);
	    }else{
			l.add( otype );			        
	    }
	    
	    
	    List commaList = atpl.getCommaType();
		if( commaList != null ) {
			Iterator it = commaList.iterator();
			while( it.hasNext() ) {
				ACommaType act = (ACommaType) it.next();
				numBrackets = ( act.getBracketPair() != null) ? act.getBracketPair().size() : 0;
				KType type = getType( act.getType(), ct, paramTypes, numBrackets, cScope); 
				
			    if( type instanceof KBasicType && type != KBasicType.getVoid() && type != KBasicType.getString() ){
			        l.add(type.bind(new TypeTable()));
			    }else if( type == KBasicType.getVoid() ){
			        int[] pos = getFirstIdent(act.getType());
			        SourceCodeException.throwVoidParamType(pos[0],pos[1],pos[2]);
			    }else{
					l.add( type );			        
			    }
				

			}
		}
		return l;
	    
	}

	/**
	 * Returns true if the KType coming in is a Basic Type, or a Basic Array Type.
	 * @param check
	 * @return
	 */
	static boolean isBasicArray(KType check) {
		if( check instanceof KBasicType ) { return true; }
		
		if( check instanceof KArrayType ) {
			KArrayType kat = (KArrayType) check;
			KType child = kat.getBaseType();
			
			if( child instanceof KBasicType) {
				return true;
			}else {
				return false; 
			}
		}
		return false;
	}

	/**
	 * Convert a ABasicTypType into a KType.
	 * @param type ABasicTypeType to be converted.
	 * @return KBasicType representing the BasicType this KBasicType represents.
	 */
	static KBasicType getBasicType( ABasicTypeType type ) {
		PBasicType bt = type.getBasicType();
		if( bt instanceof ACharBasicType ) {
			return KBasicType.getChar();
		}else if( bt instanceof AIntBasicType) {
			return KBasicType.getInt();
		}else if( bt instanceof ADoubleBasicType) {
			return KBasicType.getDouble();
		}else if( bt instanceof AStringBasicType) {
			return KBasicType.getString();
		}else if( bt instanceof ABooleanBasicType) {
			return KBasicType.getBoolean();
		}else if( bt instanceof AVoidBasicType) {
			return KBasicType.getVoid();
		}else {
			throw new KenyaInternalError("Got a Basic type that isn't known;" + type.toString() );
		}
	}


	/**
	 * Builds a KFunction from the AFuncDecDeclaration and the current ClassTable.
	 * @param fdd The function declaration node in the sablecc ast.
	 * @param ct The current class table.
	 * @return A KFunction
	 * @throws SourceCodeException
	 * @throws AnonymousSourceCodeException
	 */
	static KFunction makeFunction(AFuncDecDeclaration fdd, ClassTable ct){
		
		/* 1) Create the return type.
		 * 1a) Pull off the type parametrs if there are some -> use that in
		 * 		constructing everything else.
		 * 2) Pull off the identifier.
		 * 3) Pull off the param list -> make that list (of KIdentifier's).
		 */
		Map paramMap = new HashMap();
		
		if(fdd.getTypeParam() != null &&
		        ((ATypeParam)fdd.getTypeParam()).getTypeParamList() != null){
		    List paramList = getParameterTemplates((ATypeParamList)((ATypeParam)fdd.getTypeParam()).getTypeParamList(), fdd.getIdentifier().getText().trim());
		    
		    Iterator i = paramList.iterator();
		    while(i.hasNext()){
		        KType type = (KType) i.next();
		        String name = type.toString();
		        paramMap.put( name, type);
		    }
		
		}
		
		
		int numBrackets = (fdd.getBracketPair() == null) ? 0 : fdd.getBracketPair().size();
		KType retType = getType(fdd.getType() , ct, paramMap, numBrackets,Util.GLOBAL);
		
		String fnName = fdd.getIdentifier().getText().trim();
		
		
		
		if( fdd.getFormalParamList() == null ) {
		    if( retType.isBound(new HashMap())){
		        return new KFunction(fnName, paramMap, retType, null, null, null, fdd);
		    }else{
		        int[] pos = Util.getFirstIdent(fdd.getIdentifier());
		        SourceCodeException.throwTypeParamNoBindingFunction(pos[0],pos[1],pos[2]);
		    }
		}
		    
		
		/* Now to create the parameters to this function. */
		Map varIdents = new HashMap();
		String ident;
		List vars = new LinkedList(); /*Holds Variables */
		List args = new LinkedList();	/* Only holds types */
		
		Map refdTypeMap = new HashMap();	// KParamTypes that are referenced in the function
		
		
		AFormalParamList afpl = (AFormalParamList) fdd.getFormalParamList();
		
		numBrackets = (((ATypeName)afpl.getTypeName()).getBracketPair() == null) ?
				0 : ((ATypeName)afpl.getTypeName()).getBracketPair().size();
		KType kt = getType( ((ATypeName)afpl.getTypeName()).getType(), 
				ct, paramMap, numBrackets, fnName);
		
		kt.populateParamMap(refdTypeMap);
		
		
		ident = ((ATypeName)afpl.getTypeName()).getIdentifier().getText().trim();
		
		vars.add( new KVariable(ident, kt,false,afpl.getTypeName()) );
		args.add( kt);
		varIdents.put(ident, afpl.getTypeName());
		
		if( afpl.getCommaTypeName() == null) {
		    if( retType.isBound(refdTypeMap)){
		        return new KFunction(fnName, paramMap, retType, args, vars, null, fdd);
		    }else{
		        int[] pos = Util.getFirstIdent(fdd.getIdentifier());
		        SourceCodeException.throwTypeParamNoBindingFunction(pos[0],pos[1],pos[2]);    
		    }
		}
		
		Iterator it = afpl.getCommaTypeName().iterator();
		while(it.hasNext()){
		    ACommaTypeName actn = (ACommaTypeName) it.next();
		    
		    numBrackets = (((ATypeName)actn.getTypeName()).getBracketPair() == null) ?
					0 : ((ATypeName)actn.getTypeName()).getBracketPair().size();
			kt = getType( ((ATypeName)actn.getTypeName()).getType(), 
					ct, paramMap, numBrackets, fnName);
			kt.populateParamMap(refdTypeMap);
			ident = ((ATypeName)actn.getTypeName()).getIdentifier().getText().trim();
			
			if(varIdents.containsKey(ident)){
			    int ln = ((ATypeName)actn.getTypeName()).getIdentifier().getLine();
			    int pos = ((ATypeName)actn.getTypeName()).getIdentifier().getPos();
			    int len = ((ATypeName)actn.getTypeName()).getIdentifier().getText().trim().length();
			    
			    int oln = ((ATypeName) varIdents.get(ident)).getIdentifier().getLine(); 
			    int opos = ((ATypeName) varIdents.get(ident)).getIdentifier().getPos(); 
			    int olen = ((ATypeName) varIdents.get(ident)).getIdentifier().getText().trim().length();
			    int[][] lnkPos = { { oln, opos, olen } };
			    SourceCodeException.throwDuplicateVariable(ln, pos, len, lnkPos, ident, fnName);
			}
			
			varIdents.put( ident, actn.getTypeName());
			
			vars.add( new KVariable(ident, kt, false,actn.getTypeName()) );
			args.add( kt);
		}		
		
		if( retType.isBound(refdTypeMap)){	    
		    return new KFunction(fnName, paramMap, retType,args,vars,null,fdd);
		}else{
		    int[] pos = Util.getFirstIdent(fdd.getIdentifier());
	        SourceCodeException.throwTypeParamNoBindingFunction(pos[0],pos[1],pos[2]);
	        //unreachable
	        return null;
		}
		
	}
	
	/** Makes a KParamType, from the PType node
	 * @param pt
	 * @return
	 * @throws AnonymousSourceCodeException
	 * @throws SourceCodeException
	 */
	static KParamType mkParamType(PType pt){
		if (pt instanceof ABasicTypeType) {

		    int[] pos = getFirstIdent(pt);
		    /* Syntactically I don't think this can happen. */
		    SourceCodeException.throwTypeParam_BasicType(pos[0], pos[1], pos[2]);
		    
		    return null; //unreachable.

		} else if (pt instanceof AReferenceTypeType) {
			AClassTypeReferenceType actrt = (AClassTypeReferenceType) ((AReferenceTypeType) pt)
					.getReferenceType();
			if (actrt.getTypeParam() != null) {
				int ln = (((ATypeParam) actrt.getTypeParam()).getLess())
						.getLine();
				int pos = (((ATypeParam) actrt.getTypeParam())
						.getLess()).getPos();
				int len = (((ATypeParam) actrt.getTypeParam()).getLess().getText().trim().length());
				SourceCodeException.throwTypeParam_WithTypeParam(ln, pos, len);
			}
			return new KParamType(actrt.getIdentifier().getText().trim(),actrt.getIdentifier());
		} else {
			throw KenyaPreconditionError.get();
		}
	}
	
	/**
	 * This is to get the Paramter templates which are essentially the
	 * Class<...> bit or for methods
	 * public< ... > bit.
	 * @param atpl
	 * @return
	 * @throws AnonymousSourceCodeException
	 */
	static List getParameterTemplates(ATypeParamList atpl, String cScope){
            List l = new LinkedList();
            Set s = new HashSet();
            PType pt;
            KParamType res;

            if (atpl.getBracketPair() != null
                    && atpl.getBracketPair().size() != 0) {
                int ln = ((TBracketPair) atpl.getBracketPair().get(0))
                        .getLine();
                int pos = ((TBracketPair) atpl.getBracketPair().get(0))
                        .getPos();
                int len = ((TBracketPair) atpl.getBracketPair().get(0)).getText().trim().length();
                
                SourceCodeException.throwTypeParam_WithArrayBrackets(ln, pos,len);
            }

            pt = atpl.getType();
            res = mkParamType(pt);
            if (!s.add(res.toString())) {
                int ln = ((AClassTypeReferenceType) ((AReferenceTypeType) atpl
                        .getType()).getReferenceType()).getIdentifier()
                        .getLine();
                int pos = ((AClassTypeReferenceType) ((AReferenceTypeType) atpl
                        .getType()).getReferenceType()).getIdentifier()
                        .getPos();
                int len = ((AClassTypeReferenceType) ((AReferenceTypeType) atpl
                        .getType()).getReferenceType()).getIdentifier().getText().trim().length();
                SourceCodeException.throwTypeParam_Duplicate(ln, pos, len, res.toString(), cScope);
            }
            l.add(res);

            Iterator it = atpl.getCommaType().iterator();

            while (it.hasNext()) {
                ACommaType act = (ACommaType) it.next();
                if (act.getBracketPair() != null
                        && act.getBracketPair().size() != 0) {
                    int ln = ((TBracketPair) act.getBracketPair().get(0))
                            .getLine();
                    int pos = ((TBracketPair) act.getBracketPair().get(0))
                            .getPos();
                    int len = ((TBracketPair) act.getBracketPair().get(0))
                    		.getText().trim().length();
                   SourceCodeException.throwTypeParam_WithArrayBrackets(ln, pos,len);
                }

                pt = act.getType();
                res = mkParamType(pt);
                if (!s.add(res.toString())) {
                    int ln = ((AClassTypeReferenceType) ((AReferenceTypeType) pt)
                            .getReferenceType()).getIdentifier().getLine();
                    int pos = ((AClassTypeReferenceType) ((AReferenceTypeType) pt)
                            .getReferenceType()).getIdentifier().getPos();
                    int len = ((AClassTypeReferenceType) ((AReferenceTypeType) pt)
                            .getReferenceType()).getIdentifier().getText().trim().length();
                    SourceCodeException.throwTypeParam_Duplicate(ln, pos, len, res.toString(), cScope);
                }
                l.add(res);
            }

            return l;
    }
	
	
	/**
	 * Returns true if this PExpression represents only a single variable.
	 * (i.e. something with no operations applied to it)
	 * @param pexp
	 * @return
	 */
	static boolean isSingleVariableExpression( Node pexp ){
	    return SingleVariableExpressionChecker.isSingleVariableExpression(pexp);
	}

	
	 /**
     * Checks if two types are assignment compatible.
     * This assumes that a is being assigned to the type of b. ie. a = b ;
     * So basically they must be identical, unless a is double and b is an int/char.
     * @param a The Left hand side of this assignment.
     * @param b The Right hand side of this assignment.
     */
    public static boolean isAssignmentCompatible(KType a, KType b){
        // Low hanging fruit.
        // cannot assign a void to anything
        if( a == KBasicType.getVoid() ) {
            return false;
        }
        
        if( a == b) { return true; }
        
        if( a instanceof KArrayType ){
            if( b instanceof KArrayType ){
                KArrayType kat = (KArrayType) a;
                KArrayType kbt = (KArrayType) b;

                KType childA = kat.getChildType();
                KType childB = kbt.getChildType();
                
                if( childA instanceof KBasicType && childB instanceof KBasicType ){
                    return childA == childB;
                }else if( childA instanceof KBasicType && childB instanceof KBoundClassType ){
                    return false;
                }else if( childA instanceof KBoundClassType && childB instanceof KBasicType){
                    return false;
                }else if( childA instanceof KBoundClassType && childB instanceof KBoundClassType){
                    KBoundClassType kbA = (KBoundClassType) childA;
                    KBoundClassType kbB = (KBoundClassType) childB;
                    
                    if( kbA.getBase() != kbB.getBase()){
                        return false;
                    }
                }
                
                return isAssignmentCompatible(childA,childB);
                
            }else if( b instanceof KNullType){
                return true;
            }else{
                return false;
            }
        
        }else if( a instanceof KBasicType ){
        	if( a == KBasicType.getString() ) {
        		return b == KBasicType.getString() || b == KNullType.get();
        	} else if( b instanceof KBasicType ){
	            return (
	                	( a == KBasicType.getDouble() && 
	                	        ( b == KBasicType.getInt() || b == KBasicType.getChar() )
	                	) ||
	                	( a == KBasicType.getInt() && b == KBasicType.getChar() )
	            );
            }else if( b instanceof KBoundClassType ){
                KBoundClassType kbB = (KBoundClassType) b;
                return isAssignmentCompatible(a, kbB.getBase());
                
            }else if( b instanceof KClassType ){
                return (
	                	( a == KBasicType.getDouble() && 
	                	        ( b == KClassType.getInteger() || b == KClassType.getCharacter() )
	                	) ||
	                	( a == KBasicType.getInt() && b == KClassType.getCharacter() ) ||
	                	a == KBasicType.getBoolean() && b == KClassType.getBoolean() ||
	                	a == KBasicType.getChar() && b == KClassType.getCharacter() ||
	                	a == KBasicType.getInt() && b == KClassType.getInteger() ||
	                	a == KBasicType.getDouble() && b == KClassType.getDouble()
	                );
            }else{
                return false;
            }
            
            
        }else if( a instanceof KBoundClassType ){
            KBoundClassType kbA = (KBoundClassType) a;
            if( b instanceof KBoundClassType){
                
                KBoundClassType kbB = (KBoundClassType) b;
                
                if( ! isAssignmentCompatible(kbA.getBase(), kbB.getBase())){
                    return false;
                }
                
                boolean isOk = true;
                
                List params = kbA.getBase().getTypeParamList();
                TypeTable ttA = kbA.getBindings();
                TypeTable ttB = kbB.getBindings();
                
                Iterator it = params.iterator();
                while(it.hasNext()){
                    KParamType kpt = (KParamType) it.next();
                    KType cA = ttA.lookup(kpt);
                    KType cB = ttB.lookup(kpt);
                    isOk &= isAssignmentCompatible(cA, cB);
                }
                return isOk;
            }else if( b instanceof KBasicType ){
                return isAssignmentCompatible(kbA.getBase(), b);
            }else if( b instanceof KNullType){
                return true;
            }else{
                return false;
            }
        }else if( a instanceof KClassType ){
            if( b instanceof KClassType ){
             /*	Evidently this isn't allowed
                if(
             	( a == KClassType.getDouble() && 
            	        ( b == KClassType.getInteger() || b == KClassType.getCharacter() )
            	) ||
            	( a == KClassType.getInteger() && b == KClassType.getCharacter() ) 
            	){
             	    return true;
             	}else 
             	*/
             	if( a.getName().equals(b.getName())){
             	    return true;
             	}else{
             	    return false;
             	}
            }else if( b instanceof KBasicType ){
                if(
                /*	Evidently this isn't allowed
                     	( a == KClassType.getDouble() && 
                    	        ( b == KBasicType.getInt() || b == KBasicType.getChar() )
                    	) ||
                    	( a == KClassType.getInteger() && b == KBasicType.getChar() ) ||
                 */
                    	a == KClassType.getBoolean() && b == KBasicType.getBoolean() ||
	                	a == KClassType.getCharacter() && b == KBasicType.getChar() ||
	                	a == KClassType.getInteger() && b == KBasicType.getInt() ||
	                	a == KClassType.getDouble() && b == KBasicType.getDouble()){
                     	    return true;
                }else{
                    return false;
                }
            }else{
                return false;
            }
            
        }else if( a instanceof KEnumType ){
            if( b instanceof KEnumType ){
                return b.getName().equals(a.getName());
            } else if( b instanceof KNullType ){
                return true;
            } else {
                return false;
            }
            
        }else if( a instanceof KParamType ){
            if( b instanceof KNullType ){
                return true;
            }else if( b instanceof KParamType){
               return b.getName().equals(a.getName());
            }else{
                return false;
            }
            
        }else if( a instanceof KNullType ){
        	return false;
        }else{
            throw new KenyaInternalError("Got an unknown type " + a);
        }
        
/*        
        if( a instanceof KArrayType && b instanceof KArrayType){
            KArrayType kat = (KArrayType) a;
            KArrayType kbt = (KArrayType) b;
            
            KType childA = kat.getChildType();
            KType childB = kbt.getChildType();
            if( childA instanceof KBasicType && childB instanceof KBasicType ){
                //A double[] cannot be assigned to an int[] etc. Hence this extra condition.
                return childA == childB;
            }else{
                return isAssignmentCompatible(childA, childB);
            }
        }else if( a instanceof KArrayType && b instanceof KNullType ){
            return true;
        }else if( a instanceof KArrayType ){ 
            return false; 			
        }else if( a instanceof KBasicType && b instanceof KBasicType){
            // If they were both the same basic type, they would have passed the LHF above.:)
            return (
                    	( a == KBasicType.getDouble() && 
                    	        ( b == KBasicType.getInt() || b == KBasicType.getChar() )
                    	) ||
                    	( a == KBasicType.getInt() && b == KBasicType.getChar() )
                    );
        }else if( a == KBasicType.getString() && b instanceof KEnumType){
            return true;
        }else if( a instanceof KBasicType ){
            return false;	// They arn't both of basic type, since failed one above.
        }else if( a instanceof KBoundClassType ){
            if( b instanceof KBoundClassType){
                KBoundClassType kbA = (KBoundClassType) a;
                KBoundClassType kbB = (KBoundClassType) b;
                
                if(! kbA.getBase().getName().equals(kbB.getBase().getName()) ){ return false; }
                
                boolean isOk = true;
                
                List params = kbA.getBase().getTypeParamList();
                TypeTable ttA = kbA.getBindings();
                TypeTable iottB = kbB.getBindings();
                
                Iterator it = params.iterator();
                while(it.hasNext()){
                    KParamType kpt = (KParamType) it.next();
                    KType cA = ttA.lookup(kpt);
                    KType cB = ttB.lookup(kpt);
                    isOk &= isAssignmentCompatible(cA, cB);
                }
                
                return isOk;
            }else if( b instanceof KNullType){
                return true;
            }else if( b instanceof KBasicType && b != KBasicType.getString()){
                return isAssignmentCompatible(a, ((KBasicType)b).bind(new TypeTable()));
            }else{
                return false;
            }
        }else if( a instanceof KClassType && b instanceof KClassType){
            return (a.getName().equals(b.getName()) );

        }else if( a instanceof KEnumType ){
            return ( b instanceof KEnumType && b.getName().equals(a.getName()));
        }else if( a instanceof KParamType ){
            // Since a parameterised type could represent a basic type,
            // you can't assign them to null.
            return ((b instanceof KParamType) && b.getName().equals(a.getName())));
        }else if( a instanceof KNullType){
            return false; // Erm. null = ? ...shouldn't really happen. null is a state.
        }else{
            throw new KenyaInternalError(a + " is not a legal KType!");
        }
*/
    }

}
