/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004 by toa02
 *
 */
package kenya.types;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import kenya.errors.KenyaPreconditionError;
import kenya.types.tables.SymbolTable;
import kenya.types.tables.TypeTable;

/**
 *Holds a representation of an instantiated generic type.
 *Keeps a pointer to the KClassType it represents, and a TypeTable
 *for the type bindings that are going on in here.
 *  @author toa02
 *
 */
public class KBoundClassType extends KType {

    private KClassType _base;
    private TypeTable _bindings;
    
    
    private SymbolTable _boundChildren;
    
    /**
     * Creates a KBoundClassType with the given KClassType as its "base" class, and the TypeTable
     * representing the bindings that are at work here.
     * @param base The base KClassType
     * @param bindings The Parameter Bindings.
     */
    public KBoundClassType(KClassType base, TypeTable bindings){
        _base = base;
        _bindings = bindings;
     
        // Ok this can't be done now because it is entirely possible to
        // be creating a bound class on a class who's children don't exist.
        // So it has to be delayed until later.
        //doBindChildren();
    }
    /**
     * Returns the base name of the class with the bindings in \< \>'s 
     * @see kenya.types.KType#getName()
     */
    public String getName() {
        StringBuffer sb = new StringBuffer();
        sb.append(_base.getName());
        if( _base.getTypeParamList().size() != 0 ){
            sb.append("<");
        
	        Iterator it = _base.getTypeParamList().iterator();
	        
	        while(it.hasNext()){
	            KParamType kpt = (KParamType) it.next();
	            KType kt = _bindings.lookup(kpt);
	            if( kt instanceof KBasicType && kt != KBasicType.getString() ){
	                sb.append( ((KBasicType)kt).getClassType());
	            }else{
	                sb.append(kt);
	            }
	            if( it.hasNext()){
	                sb.append(",");
	            }
	        }
	        
	        sb.append(">");
        }
        return sb.toString();
    }
    
    
    /**
     * Gives back the underlying Base-Class of this BoundClass.
     * @return KClassType representing the base class of this BoundClass.
     */
    public KClassType getBase(){
        return _base;
    }
    
    /**
     * Gives back the Type-Parameter bindings that are mapped onto the base class.
     * @return A TypeTable of bindings in place here.
     */
    public TypeTable getBindings(){
        return _bindings;
    }
    
    /**
     * @see getName()
     * @see java.lang.Object#toString()
     */
    public String toString(){
        return getName();
    }
    
	/**
	 * @see kenya.types.KType#bind(kenya.types.tables.TypeTable)
	 */
    public KType bind(TypeTable tt){
        /* The underlying KClassType stays the same.
         * But the rhs of the bindings in this class
         * need to be updated with TypeTables bindings.
         */
        Map newBindings = new HashMap();
        
        Iterator it = _bindings.keyIterator();
        while(it.hasNext()){
            KParamType lhs = (KParamType) it.next();
            KType rhs = _bindings.lookup(lhs);
            
            newBindings.put( lhs, rhs.bind(tt) );
        }
        
        TypeTable newTT = new TypeTable(newBindings);
        return new KBoundClassType(_base,newTT);
    }

    /**
     * Gives back a SymbolTable of the "children" of this bound class,
     * where the children have been updated with this class's bindings.
     * @return A SymbolTable of the "children" of this class, where the
     * children's types have been updated with the bindings in place here.
     */
    public SymbolTable getBoundChildren(){
        if( _boundChildren == null ){
            doBindChildren();
        }
        
        return _boundChildren;
    }
    
    /**
     * Actually sets up the boundChildren SymbolTable in this class.
     */
    private final void doBindChildren(){
        Map baseChildren = _base.getChildren().collapse();
        
        Iterator it = baseChildren.keySet().iterator();
        
        Map boundChildren = new HashMap();
        while(it.hasNext()){
            String name = (String) it.next();
            KVariable unboundChild = (KVariable) baseChildren.get(name);
            
            KType unboundChildType = unboundChild.getType();
            
            KType boundChildType = unboundChildType.bind(_bindings);
            boundChildren.put(name,new KVariable(name,boundChildType,false,unboundChild.getNode()));
        }

        	_boundChildren = new SymbolTable(boundChildren);
        
    }
    
    /**
     * @see kenya.types.KType#compareAndBind(kenya.types.KType, kenya.types.tables.TypeTable)
     */
    public int compareAndBind(KType target, TypeTable tt) {
        
    	if( target instanceof KBoundClassType ){
            KClassType kt = ((KBoundClassType)target).getBase();
            
        	if( _base == KClassType.getDouble() && kt == KClassType.getDouble() ||
        	    _base == KClassType.getInteger() && kt == KClassType.getInteger()     ||
        	    _base == KClassType.getCharacter() && kt == KClassType.getCharacter()  ||
        	    _base == KClassType.getBoolean() && kt == KClassType.getBoolean() ){
        	    return 0;
        	}            
            
            KBoundClassType kbct = (KBoundClassType) target;
            if(!kbct.getBase().equals(_base)){ return -1; }

            int count = 0;
            Iterator meIt = _bindings.keyIterator();
            while(meIt.hasNext()){
                KParamType myCurrentParam = (KParamType) meIt.next();
                
                KType myBind = _bindings.lookup(myCurrentParam);
                KType hisBind = kbct._bindings.lookup(myCurrentParam);
                
                KType mb = myBind.bind(tt);
                KType hb = hisBind;// = hisBind.bind(tt);

                // check that ints and doubles and chars are not being casted
                if( hb instanceof KBoundClassType ){
                    if( mb instanceof KBoundClassType ){
                        KBoundClassType hbc = (KBoundClassType) hb;
                        KBoundClassType mbc = (KBoundClassType) mb;
                        
                        if( hbc.getBase() != mbc.getBase()){
                            return -1;
                        }
                        
                    }else if( mb instanceof KBasicType ){
                        KBoundClassType hbc = (KBoundClassType) hb;
                        KBasicType mbc = (KBasicType) mb;
                        if( hbc.getBase() != mbc.getClassType() ){
                            return -1;
                        }
                    }
                }else if( hb instanceof KBasicType ){
                    if( mb instanceof KBoundClassType ){
                        KBoundClassType mbc = (KBoundClassType) mb;
                        KBasicType hbc = (KBasicType) hb;
                        if( mbc.getBase() != hbc.getClassType() ){
                            return -1;
                        }
                    }else if( mb instanceof KBasicType ){
                        if( mb != hb){
                            return -1;
                        }
                    }
                }
                
                if( mb instanceof KBasicType &&
                    hb instanceof KBasicType &&
                    mb != hb){
                    return -1;
                }

                myBind.compareAndBind(hisBind, tt);
                int ret = myBind.compareAndBind(hisBind, tt);
                
                if( ret == -1 ){ return -1; }
                count += ret;
                
                
            }

            return count;
           
            
        }else if( target instanceof KClassType){
            throw KenyaPreconditionError.get();
        }else if( target instanceof KBasicType ){

            KBasicType kt = (KBasicType) target;
            
            /* I think these must be allowed allowed */
            if( _base == KClassType.getDouble() && kt == KBasicType.getInt() ){
                return 1;
            }
            
            if( _base == KClassType.getDouble() && kt == KBasicType.getDouble() ){
                return 2;
        	}
        
        	if( _base == KClassType.getInteger() && kt == KBasicType.getChar() ){
        	    return 1;
        	}
        	/**/
            
        	if( _base == KClassType.getDouble() && kt == KBasicType.getDouble() ||
        	    _base == KClassType.getInteger() && kt == KBasicType.getInt()     ||
        	    _base == KClassType.getCharacter() && kt == KBasicType.getChar()  ||
        	    _base == KClassType.getBoolean() && kt == KBasicType.getBoolean() ){
        	    return 0;
        	}     
            
            
        } else if (target == KNullType.get() ) {
        	return 0;
        }
        
        return -1;
    }
    
    
    /**
     * @see kenya.types.KType#isBound(java.util.Map)
     */
    public boolean isBound(Map paramMap) {
        Iterator it = _bindings.keyIterator();
        
        while(it.hasNext()){
            KParamType kpt = (KParamType) it.next();
            KType rhs = _bindings.lookup(kpt);
            if( rhs instanceof KParamType && !paramMap.containsKey(kpt.toString())){
                return false;
            }
        }
        return true;
    }
    
    /**
     * @see kenya.types.KType#populateParamMap(java.util.Map)
     */
    public void populateParamMap(Map paramMap) {
        Iterator it = _bindings.keyIterator();
        while(it.hasNext()){
            KParamType kpt = (KParamType) it.next();
            KType rhs = _bindings.lookup(kpt);
            if( rhs instanceof KParamType ){
                paramMap.put(rhs.toString(),rhs);
            }

        }
    }
}
