/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 30-Jun-2004 by toa02
 *
 */
package kenya.types;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.types.tables.SymbolTable;
import kenya.types.tables.TypeTable;

/**
 * @author toa02
 *
 */
public class KClassType extends KType {
	
	private String _className;
	private List _typeParams;		// List of KParamType
	private SymbolTable _children;
	
	private static HashMap _BasicTypes;
	
	static{
	    _BasicTypes = new HashMap();
	    
	    KClassType kboolean = new KClassType("Boolean",new LinkedList());
	    kboolean.setChildren(new SymbolTable());
	    
	    KClassType kchar = new KClassType("Character",new LinkedList());
	    kchar.setChildren(new SymbolTable());
	    
	    KClassType kint = new KClassType("Integer",new LinkedList());
	    kint.setChildren(new SymbolTable());
	    
	    KClassType kdouble = new KClassType("Double",new LinkedList());
	    kdouble.setChildren(new SymbolTable());
	    
	    //KClassType kstring = new KClassType("String",new LinkedList());
	    //kdouble.setChildren(new SymbolTable());
	    
	    _BasicTypes.put("Boolean",kboolean);
	    _BasicTypes.put("Character",kchar);
	    _BasicTypes.put("Integer",kint);
	    _BasicTypes.put("Double",kdouble);
	    //_BasicTypes.put("String",kdouble);
	}
	
	
	public synchronized static KClassType getBoolean(){
	    return (KClassType) _BasicTypes.get("Boolean");
	}
	
	public synchronized static KClassType getCharacter(){
	    return (KClassType) _BasicTypes.get("Character");
	}
	
	public synchronized static KClassType getInteger(){
	    return (KClassType) _BasicTypes.get("Integer");
	}
	
	public synchronized static KClassType getDouble(){
	    return (KClassType) _BasicTypes.get("Double");
	}
	
//	public static KClassType getString(){
//	    return (KClassType) _BasicTypes.get("String");
//	}
	
	public KClassType( String className, List typeParams ){
		_className = className;
		if( typeParams == null){
			_typeParams = Collections.unmodifiableList(new LinkedList());
		}else{
			_typeParams = Collections.unmodifiableList(typeParams);
		}
		_children = null;
	}
	

	/**
	 * Set the "children" of this class as a SymbolTable.
	 * @param children
	 */
	public void setChildren( SymbolTable children ){
		_children = children;
	}
	
	/**
	 * Gives back the "Children" of this class as a Symbol Table.
	 * @return SymbolTable representing the "children" of this class.
	 */
	public SymbolTable getChildren(){
	    if( _children == null ){
	        throw new KenyaInternalError("Attempting to get children when they havn't" +
	        		"been setup yet.");
	    }
	    
	    return _children;
	}
	
	/**
	 * Returns a Map of Strings to KParamTypes of the param types of this class.
	 * @return
	 */
	public Map getTypeParams(){
	    Iterator it = _typeParams.iterator();
	    Map m = new HashMap();
	    while(it.hasNext()){
	        KParamType kt = (KParamType) it.next();
	        m.put(kt.toString(),kt);
	    }
	        return m;
	}
	

	/**
	 * Gets a List of Type Paramters this class has.
	 * This is needed so that there is a standard order between classes.
	 * @return The type parameters in a list.
	 */
	public List getTypeParamList(){
	    return _typeParams;
	}
	
	
	/**
	 * @see kenya.types.KType#bind(kenya.types.tables.TypeTable)
	 */
	public KType bind(TypeTable tt){
	    Map m = new HashMap();
	    
	    Iterator it = _typeParams.iterator();
	    
	    while(it.hasNext()){
	        KParamType kt = (KParamType) it.next();
	        if( tt.containsKey(kt)){
	            m.put(kt, tt.lookup(kt));
	        }else{
	            m.put(kt,kt);
	        }
	    }
	    
	    TypeTable nTT = new TypeTable(m);
	    
	    return new KBoundClassType(this,nTT);
	}
	
	
	
	/**
	 * Returns true iff the incoming list of classes could match up
	 * with this class's type params.
	 * (e.g. a Class<X> can match up with a Class<SomeOtherClass>. )
	 * I.e. check they are the same length :)  
	 * @param params the list of params to check.
	 * @return
	 */
	public boolean doTypeParametersMatch(List params){
	    return( params.size() == _typeParams.size());
	}
	
	/**
	 * Given a list of bound Paramters, create a TypeTable mapping this class's
	 * type params to the lists' type-params.
	 * @param boundParams
	 * @return A TypeTable.
	 */
	public TypeTable genTypeTable( List boundParams){
	   if(!doTypeParametersMatch(boundParams)) {
	       throw new IllegalArgumentException("Type parameters do not match!");
	   }
	   
	   Map mappings = new HashMap();
	   //Map mappings = new LazyMap();
	   
	   Iterator itB = boundParams.iterator();
	   Iterator itT = _typeParams.iterator();

		while( itB.hasNext() ){
		    mappings.put( itT.next(), itB.next() );
		}

	   return new TypeTable(mappings);
	   
	}
	
	public String getName(){
		return _className;
	}
	
	public String toString(){
	    StringBuffer sb = new StringBuffer();
	    sb.append( getName() );
	    if( _typeParams.size() != 0 ){
		    sb.append("<");
		    
		    Iterator it = _typeParams.iterator();
		    
		    while( it.hasNext()){
		        sb.append(it.next());
		        if( it.hasNext()){
		            sb.append(",");
		        }
		    }
		    sb.append(">");
	    }
	    return sb.toString();
	}
	
	
	/**
     * @see kenya.types.KType#compareAndBind(kenya.types.KType, kenya.types.tables.TypeTable)
     */
    public int compareAndBind(KType target, TypeTable tt) {
        throw KenyaPreconditionError.get();
    }

    /**
     * @see kenya.types.KType#isBound(java.util.Map)
     */
    public boolean isBound(Map paramMap) {
        throw KenyaPreconditionError.get();
    }
    
    /**
     * @see kenya.types.KType#populateParamMap(java.util.Map)
     */
    public void populateParamMap(Map paramMap) {
        throw KenyaPreconditionError.get();
    }
    
}
