/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004 by toa02
 *
 */
package kenya.types.tables;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.NoSuchElementException;

import mediator.IJavaCode;

import kenya.errors.KenyaInternalError;
import kenya.types.KVariable;

/**
 * Symbol Table.
 * Holds scopable mappings between String names for variables
 * and KVariable objects.
 * @author toa02
 *
 */
public class SymbolTable {

    private Map _gloConsts;
    private LinkedList _tableQueue = new LinkedList();
    
    /**
     * Creates a new Symbol table with the given Map representing global constants.
     * @param globalConstants A map of Strings to KVariables for the global constants
     * that should be considered available.
     */
    public SymbolTable(Map globalConstants){
        _gloConsts = Collections.unmodifiableMap(globalConstants);
    }
    
    /**
     * Creates a new Symbol Table with no global constants defined.
     */
    public SymbolTable(){
        _gloConsts = Collections.unmodifiableMap(new HashMap());
    }
    
    /**
     * Takes two symbol tables, and uses the first ones global variables
     * as the global variables now.
     * @param base
     * @param local
     */
    public SymbolTable( SymbolTable base, SymbolTable local ){
        _gloConsts = Collections.unmodifiableMap(base.collapse());
        push(local);
    }
    
  
    /**
     * Pushes a single variable into the symbol table as its own scope.
     * @param name The name of the variable
     * @param var The KVariable it represents.
     * @return True iff the variable has been added into its own scope,
     * False if a variable with that name alread exists in this SymbolTable outside
     * of the _gloConsts area.
     */
    public boolean push(String name, KVariable var){

        newLocalScope();
        return pushLocal(name,var);
    }

    /**
     * Creates a new empty local scope for the symbol table.
     */
    public void newLocalScope(){
        push(new HashMap());
    }
    
    /**
     * Conditionally pushes a name / variable binding into the local scope.
     * @param name The String name of the variable.
     * @param var The KVariable to push.
     * @return true iff the name/variable pair has been added to the local scope,
     * 			false if there was already a mapping for "name" at this or any scope
     * 			above it (the item will not have been added if this returns false).
     */
    public boolean pushLocal(String name, KVariable var){
        Map m = (Map) _tableQueue.getFirst();
        
        if( containsLocalVariable(name)){
            return false;
        }
        
        m.put(name,var);
        return true;
    }
    
    /**
     * Pushes a SymbolTable in to make it the most local scope.
     * This is done by collapsing the parameter into a map and adding that. 
     * @param st The symbol table to add.
     * @return true iff adding the symbol table succeeded.
     * @throws IllegalStateException if the collapse call throws IllegalStateException
     */
    public boolean push(SymbolTable st) throws IllegalStateException{
        Map m = st.collapse();
        return push(m);
    }
    
    
    /**
     * Pushes a map into the most-local scope.
     * @param m
     * @return true iff the Map is successfully added into the local scope of this
     * Symbol table. Returns false if a variable name clash occurs with the local variables
     * in this symbol table. 
     */
    public boolean push(Map m){
        Iterator it = m.keySet().iterator();
        while(it.hasNext()){
            String name = (String) it.next();
            if( containsLocalVariable(name)){
                return false;
            }
        }
        
        _tableQueue.addFirst(m);
        return true;
    }
    
    /**
     * Removes the most local scope from this SymbolTable.
     */
    public void pop(){
        _tableQueue.removeFirst();
    }

    
    /**
     * Collapses all the scopes of this symbol table into one SymbolTable.
     * This includes the Global's map aswell.
     * @return Map of all the mappings that where in this symbol table
     * (including the globals table and all currently defined scopes).
     * @throws IllegalStateException if a duplicate mapping is found anywhere.
     */
    public Map  collapse() throws IllegalStateException{
        Map allMappings = new HashMap();
        
        Iterator it = _tableQueue.iterator();
        
        while(it.hasNext()){
            Map cScope = (Map) it.next();
            
            Iterator nit = cScope.keySet().iterator();
            while(nit.hasNext()){
                Object o = nit.next();
                
                if(allMappings.containsKey(o)){
                    throw new KenyaInternalError("Duplicate Mapping found in SymbolTable during collapse.");
                }
                
                allMappings.put( o, cScope.get(o));
            }
        }
        
        Iterator nit = _gloConsts.keySet().iterator();
        while( nit.hasNext()){
            Object o = nit.next();
            if( allMappings.containsKey(o)){
                throw new KenyaInternalError("Duplicate Mapping found in SymbolTable during collapse.");
            }
            
            allMappings.put( o , _gloConsts.get(o));
        }
        
        return allMappings;
    }
    
    /**
     * Checks to see if the given variable name is in the Symbol Table.
     * @param name The name of the variable to check.
     * @return true iff the given variable name has a mapping in either the
     * local scopes or the global's tables.
     */
    public boolean containsVariable(String name){
       return ( containsLocalVariable(name) || constansGlobalVariable(name));
    }
    
    /**
     * A way to check if a Local Variable with given name exists.
     * @param name The name of the variable to check for.
     * @return true iff a mapping for name to a KVariable is 
     * found in the stack of scopes.
     */
    private boolean containsLocalVariable(String name){
        Iterator it = _tableQueue.iterator();
        while(it.hasNext()){
            Map cScope = (Map) it.next();
            
            if(cScope.containsKey(name)){
                return true;
            }
        }
        
        return false;
    }
    
    
    /**
     * A way to check if a Global Variable with given name exists.
     * @param name The name of the variable to check for.
     * @return true iff a mapping for name to KVariable is
     * found in the globalConstants area.
     */
    private boolean constansGlobalVariable(String name){
        return _gloConsts.containsKey(name);
    }
    
    /**
     * Gives back the KVariable that has this name. In preference a local scope is used
     * followd by a global constsant.
     * @param name The string identifier to use.
     * @return KVariable in this symbol table.
     * @throws java.util.NoSuchElementException if the variable isn't found.
     */
    public KVariable lookupVariable(String name){
        Iterator it = _tableQueue.iterator();
        while(it.hasNext()){
            Map cScope = (Map) it.next();
            
            if(cScope.containsKey(name)){
                return (KVariable) cScope.get(name);
            }
        }
        
        if( _gloConsts.containsKey(name)){
            return (KVariable) _gloConsts.get(name);
        }
        
        throw new NoSuchElementException("Couldn't find variable");
    }
    
    /**
     * returns a string representation of this SymbolTable.
     * @see java.lang.Object#toString()
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        Iterator it;
        
        sb.append("Globals:" + IJavaCode.NEWLINE + "{");
        it = _gloConsts.keySet().iterator();
        while(it.hasNext()){
            KVariable kt = (KVariable) _gloConsts.get(it.next());
            sb.append(kt);
            if( it.hasNext()){
                sb.append(",");
            }
        }
        sb.append("}");
        
        sb.append(IJavaCode.NEWLINE + "Scopes:" + IJavaCode.NEWLINE);
        
        
        it = _tableQueue.iterator();
        while(it.hasNext()){
    
            Map m = (Map) it.next();
            sb.append("{");    
            Iterator it2 = m.keySet().iterator();
            
            while(it2.hasNext()){
                String name = (String) it2.next();
                KVariable kt = (KVariable) m.get(name);
                
                sb.append(kt);
                
                if(it2.hasNext()){
                    sb.append(",");
                }
            }
            
            sb.append("}");
            
            if(it.hasNext()){
                sb.append(IJavaCode.NEWLINE);
            }
        }
        sb.append(IJavaCode.NEWLINE);
        return sb.toString();
    }
   
}
