/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 20-Jul-2004 by toa02
 *
 */
package kenya.types.tables;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.NoSuchElementException;

import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.types.KVariable;
import kenya.values.IKValue;
import kenya.values.KBooleanValue;
import kenya.values.KUnassignedValue;
import kenya.values.KUnknownValue;
import kenya.values.util.KValueCalculator;
import kenya.values.util.MarkerKVariable;
import mediator.IJavaCode;

/**
 * @author toa02
 *
 */

/* **************
 * HUGE NOTE:
 * THERE ARE REFERENCES IN HERE TO 'TRACKED' AND 'UNTRACKED' VARIABLES
 * THE 'TRACKED' VARIABLES ARE NOT RELIABLE AND SHOULD NOT BE USED
 * AS I DO NOT CREATE GEN/KILL/R.D. ANALYSIS FOR IF/WHILE LOOPS. ONLY USE THE
 * UNTRACKED ONES.
 */

public class VariableTable {
    
    private Map _gloConsts;		// Map of KVariable to IKValue (constants only have a value)
    private LinkedList _tableQueue; // LinkedList of Map of KVariable to ValuePair's
    
    /**
     * Constructor: Creates a new, empty, variableTracker.
     *
     */
    public VariableTable(){
        _gloConsts = new HashMap();
        _tableQueue = new LinkedList();
    }
    
    
    private VariableTable(Map gloConsts, LinkedList tableQueue){
        _gloConsts = gloConsts;
        _tableQueue = tableQueue;
    }
    
    /**
     * Adds a KVariable to the "global constants" area of this variable tracker
     * @param name
     * @param val
     * @return true iff the constant was added, false if the constant was not added (i.e. it was there already?)
     */
    public boolean addConstant(KVariable var, IKValue val){
        if( _gloConsts.containsKey(var)){
            return false;
        }
        
        _gloConsts.put(var, val);
        return true;
    }
    
    
    public Map getConstants(){
        return _gloConsts;
    }
    
    /**
     * Takes a SymbolTable and pushes all the variables in it as new
     * KUnknownValues into a new scope. (The SymbolTable is collapsed as
     * part of this process so it needs to be a nice SymbolTable.
     * @param st
     */
    public void pushSymbolTable(SymbolTable st){
        pushScope();
       	Map m = st.collapse();
       	
       	Iterator i = m.keySet().iterator();
       	
       	while(i.hasNext()){
       	    String ident = (String) i.next();
       	    KVariable kv = (KVariable) m.get(ident);
       	    declareNew(kv);
       	    assignTo(kv,KUnknownValue.get());
       	}
    }
    
    /**
     * Makes a new "LocalScope" in the Variable Table that
     * all new declareNew's will be put into.
     */
    public void pushScope(){
        _tableQueue.addFirst(new HashMap());
    }

    /**
     * Removes the most local scope from this variable tracker.
     */
    public void popScope(){
        _tableQueue.removeFirst();
    }
    

    /**
     * Declases a new KVariable in the current scope,
     * initially created with the IKValue of KUnassignedValue
     * @param n3w
     * @return true iff the new variable was added to the local scope,
     * false if the variable is already declared in the local scope.
     */
    public boolean declareNew(KVariable n3w){
        
        Map tsb = (Map) _tableQueue.getFirst();
        if( tsb.containsKey(n3w)){
            return false;
        }
        
        tsb.put( n3w, new ValuePair() );
        return true;
    }
    
    
    /**
     * Assignes an KVariable a given IKValue.
     * This will map it to assigned in the lookup area
     * (unless its assigned to "Unassigned"!)
     * And the given IKValue in the tracked ar
     * @param lhs KVariable to assign a new value to
     * @param rhs IKValue value to assign to the variable.
     * @throws NoSuchElementException if the KVariable key cannot be found.
     */
    public void assignTo(KVariable lhs, IKValue rhs){
        if( lhs == MarkerKVariable.getArbitary() ){
            /* do nothing */ return;
        }else if( lhs == MarkerKVariable.getEnumConst()){
            throw new KenyaInternalError("Impossible");
        }
        
        Iterator it = _tableQueue.iterator();
        while(it.hasNext()){
            Map cScope = (Map) it.next();
            if( cScope.containsKey(lhs) ){
                ValuePair vp = (ValuePair) cScope.get(lhs);
                vp.assign(rhs);
                return;
            }
        }
        
        throw new NoSuchElementException("Cannot find " + lhs + " to assign to.");
    }
    
    
    /**
     * Utility method to check the scope'd queue for a KVariable
     * and get back its ValuePair
     * @param val KVariable to lookup
     * @return ValuePair  of the binding for that KVariable, or null if it was not found.
     */
    private ValuePair lookupScope(KVariable val){
        Iterator it = _tableQueue.iterator();
        
        while(it.hasNext()){
            Map cScope = (Map) it.next();
            if( cScope.containsKey(val)){
                return (ValuePair) cScope.get(val);
            }
        }
        
        return null;
    }
    
    /**
     * Utility method to lookup an IKValue in the global constants map.
     * @param val KVariable to lookup
     * @return IKValue of the binding for that KVariable, or null if none was found.
     */
    private IKValue lookupConsts(KVariable val){
        if(_gloConsts.containsKey(val)){
            return (IKValue) _gloConsts.get(val);
        }
        
        return null;
    }
    
    /**
     * Returns the IKValue for the given KVariable (usu Unassigned or Unknown unless we hit a constant).
     * @param val The KVariable to lookup
     * @return IKValue of the value it has in an untracked setting.
     * @throws NoSuchElementException if we cannot find that value in the lookup.
     */
    public IKValue lookup(KVariable val){
        if( val == MarkerKVariable.getArbitary() ){
            return KUnknownValue.get();
        }else if( val == MarkerKVariable.getEnumConst() ){
            throw KenyaPreconditionError.get();
        }
        
        ValuePair vp = lookupScope(val);

        if( vp != null ){
           return vp.getUnTracked();
       }
       
       IKValue cv = lookupConsts(val);
       if( cv != null ){
           return cv;
       }
       
       throw new NoSuchElementException("Could not lookup KVariable " + val);
    }
    
    /**
     * Returns the *TRACKED* IKValue for the given KVariable,
     * this could have a value.
     * @param val
     * @return
     */
    public IKValue lookupTracked(KVariable val){
        
        if( val == MarkerKVariable.getArbitary() ){
            return KUnknownValue.get();
        }else if( val == MarkerKVariable.getEnumConst() ){
            throw KenyaPreconditionError.get();
        }
        
        ValuePair vp = lookupScope(val);
        
        if( vp != null ){
            return vp.getTracked();
        }
        
        IKValue cv = lookupConsts(val);
        if( cv != null ){
            return cv;
        }
        
        throw new NoSuchElementException("Could not lookup KVariable " + val);
    }

    
    
    /**
     * Creates a clone of this Variable tracker, for use
     * in splits down if/then/else trees etc.
     * Some thing will still point across, but most things in this
     * are immutable anyways..
     * @return A VariableTable where the underlying globals set and tablequeue / maps are all cloned.
     */
    public VariableTable createClone(){
        Map cloneGlobals = new HashMap();
        Iterator cgIt = _gloConsts.keySet().iterator();
        
        while( cgIt.hasNext() ){
            KVariable kv = (KVariable) cgIt.next();
            IKValue ikv  = (IKValue) _gloConsts.get(kv);
            cloneGlobals.put( kv, ikv );
        }
        
        
        LinkedList cloneScopes = new LinkedList();
        Iterator scIt = _tableQueue.iterator();
        while( scIt.hasNext() ){
            Map cScopeMap = (Map) scIt.next();
            Map cScopeCloneMap = new HashMap();
            Iterator cScopeKeyIt = cScopeMap.keySet().iterator();
            
            while( cScopeKeyIt.hasNext() ){
                KVariable kv = (KVariable) cScopeKeyIt.next();
                ValuePair vp = (ValuePair) cScopeMap.get(kv);
                cScopeCloneMap.put(kv, vp.clone());
            }
            
            cloneScopes.addLast(cScopeCloneMap);
        }
        
        return new VariableTable(cloneGlobals,cloneScopes);
    }
    
    
    /**
     * Merges two VariableTrackers together, taking advantage of knowledge about
     * assignments on the lhs and rhs to create correct updates. 
     * 
     * The resulting table will be an essential clone, so references shouldn't be live
     * across them. :s
     * 
     * The rules are in precedence order:
     * If either side is unassigned, the resulting table will have that variable as unassigned.
     * If either side is unknown, the resulting table will have that variable as unknown.
     * If both sides have a value that is different, the resulting table will be unknown
     * If both sides have a value that is the same, the resulting table will have that value.
     * 
     * This applies to both tracked and untracked values.
     * 
     * @param lhs VariableTable to merge.
     * @param rhs VariableTable to merge.
     * @return VariableTable of the variables merged.
     * 
     * @throws IllegalArgumentException if either side have mis-matching variables (e.g. variables in one side that
     * are not in the other, or a different number of scopes). Or if the constants between the two have different values.
     * (Since constants should stay, ahem, constant :) )
     */
    public static VariableTable merge(VariableTable lhs,
            VariableTable rhs){
        
        //1: Check the constants are the same.
        Map lhsConsts = lhs._gloConsts;
        Map rhsConsts = rhs._gloConsts;
        
        if(!lhsConsts.equals(rhsConsts)){
            throw new IllegalArgumentException("Constants are not the same." + lhsConsts + IJavaCode.NEWLINE + rhsConsts);
        }
        
        Map cloneGlobals = new HashMap();	// this is the cloned globals map
        {
	        Iterator cgIt = lhsConsts.keySet().iterator();
	        
	        while( cgIt.hasNext() ){
	            KVariable kv = (KVariable) cgIt.next();
	            IKValue ikv  = (IKValue) lhsConsts.get(kv);
	            cloneGlobals.put( kv, ikv );
	        }
        }
        
        
        //2: Check The TableQueues:
        LinkedList cloneScopes = new LinkedList();
        
        LinkedList lhsScopes = lhs._tableQueue;
        LinkedList rhsScopes = rhs._tableQueue;
        
        // LHF - make sure they have the same number of scopes:
        if( lhsScopes.size() != rhsScopes.size() ){
            throw new IllegalArgumentException("Do not have the same number of scopes." + lhsScopes + IJavaCode.NEWLINE + rhsScopes);
        }

        Iterator lhsScopeIt = lhsScopes.iterator();
        Iterator rhsScopeIt = rhsScopes.iterator();
        
        while( lhsScopeIt.hasNext() && rhsScopeIt.hasNext()){
            Map lhsCScope = (Map) lhsScopeIt.next();
            Map rhsCScope = (Map) rhsScopeIt.next();

            Map thisCloneScope = new HashMap();
            
            // Check both have got the same number of elements as keys:
            if( lhsCScope.keySet().size() != rhsCScope.keySet().size() ){
                throw new IllegalArgumentException("Do not have the same number of elements in this scope." + lhsCScope + IJavaCode.NEWLINE + rhsCScope);
            }
            
            Iterator varIt = lhsCScope.keySet().iterator();
            while(varIt.hasNext()){
                KVariable kv = (KVariable) varIt.next();
                
                // Check the rhsCScope has this variable
                if( !rhsCScope.containsKey(kv) ){
                    throw new IllegalArgumentException("Got a variable that is unmatched in other scope." + kv);
                }
                
                ValuePair lvp = (ValuePair) lhsCScope.get(kv);
                ValuePair rvp = (ValuePair) rhsCScope.get(kv);
                // Right, we now have 2 value pairs...one for the lhs, one for the rhs.

                IKValue tracked = mergeValues( lvp.getTracked(), rvp.getTracked() );
                IKValue untracked = mergeValues(lvp.getUnTracked(), rvp.getUnTracked());
        
                ValuePair ovp = new ValuePair(untracked,tracked);
                thisCloneScope.put( kv, ovp );
            }
            
            cloneScopes.addLast(thisCloneScope);
            
        }
        
        return new VariableTable(cloneGlobals, cloneScopes);
    }


    /** 
     * Helper routine for merge above.
     * Takes two IKValues and returns the resultant "merge" IKValue from them,
     * as given by the table in merge.
     * @param lhs IKValue one
     * @param rhs IKValue two
     * @return IKValue of them merged as in "merge".
     * @see VariableTable#merge(VariableTable, VariableTable)
     */
    private static IKValue mergeValues( IKValue lhs, IKValue rhs ){
        if( lhs == KUnassignedValue.get() || rhs == KUnassignedValue.get() ){
            return KUnassignedValue.get();
        }
        
        if( lhs == KUnknownValue.get() || rhs == KUnknownValue.get()){
            return KUnknownValue.get();
        }
        
        if( KValueCalculator.equal(lhs, rhs) == KBooleanValue.getTrue() ){
            return lhs;
        }else{
            return KUnknownValue.get();
        }
    }

    /**
     * returns a string representation of this VariableTable.
     * @see java.lang.Object#toString()
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        Iterator it;
        
        sb.append("Globals:" + IJavaCode.NEWLINE + "{");
        it = _gloConsts.keySet().iterator();
        while(it.hasNext()){
            KVariable kt = (KVariable) it.next();
            IKValue ikv = (IKValue) _gloConsts.get(kt);
            sb.append(kt);
            sb.append("=");
            sb.append(ikv);
            if( it.hasNext()){
                sb.append(",");
            }
        }
        sb.append("}");
        
        sb.append(IJavaCode.NEWLINE);
        sb.append("Scopes:");
        sb.append(IJavaCode.NEWLINE);
        
        
        it = _tableQueue.iterator();
        while(it.hasNext()){
    
            Map m = (Map) it.next();
            sb.append("{");    
            Iterator it2 = m.keySet().iterator();
            
            while(it2.hasNext()){
                KVariable var = (KVariable) it2.next();
                ValuePair vp = (ValuePair) m.get(var);
                
                sb.append(var);
                sb.append("=");
                sb.append(vp.getUnTracked());
                
                if(it2.hasNext()){
                    sb.append(",");
                }
            }
            
            sb.append("}");
            
            if(it.hasNext()){
                sb.append(IJavaCode.NEWLINE);
            }
        }
        sb.append(IJavaCode.NEWLINE);
        return sb.toString();
    }
    
    
}




/**
 * Utility Class to hold a pair of "tracked" and "untracked" values.
 * @author toa02
 */
class ValuePair implements Cloneable{
 
    private IKValue _unTracked;
    private IKValue _tracked;

    /**
     * Default constructor - both are unassigned.
     */
    ValuePair(){
        _unTracked = _tracked = KUnassignedValue.get(); // all get the value of KUnassignedValue.get()
    }

    /**
     * Custom constructor.
     * @param ut IKValue what the untracked value should be.
     * @param t IKValue what the tracked value should be.
     */
    ValuePair(IKValue ut, IKValue t){
        _unTracked = ut;
        _tracked = t;
    }
    
    /**
     * Assign to this value pair. The untracked value will get KUnknownValue (unless the incoming IKValue is KUnassignedValue
     * ) and the tracked value will get the incoming IKValue.
     * @param n3w IKValue of the value that has been assigned to this ValuePair.
     */
    void assign( IKValue n3w){
        if( n3w == KUnassignedValue.get() ){
            _unTracked = _tracked = n3w; // all get the value of n3w
            return;
        }
        
        _unTracked = KUnknownValue.get();
        _tracked = n3w;
    }
    
    /**
     * @return IKValue of the tracked value.
     */
    IKValue getTracked(){
        return _tracked;
    }
    
    /**
     * @return IKValue of the untracked value.
     */
    IKValue getUnTracked(){
        return _unTracked;
    }
    
    /**
     * @see java.lang.Object#clone()
     */
    public Object clone(){
        return new ValuePair(_unTracked,_tracked);
    }
    
    public String toString(){
        return _unTracked.toString();
    }
}
