/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on Aug 2, 2004
 *
 */
package kenya.ui.indentation;

import java.util.regex.Matcher;

import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.StyledTextContent;

/**
 * @author Matthew Sackman
 * @version 1
 */
public class IndentationStrategy {

    private ILineIndentation root = new LineIndentation();

    public IndentationStrategy() {
    }

    public void insertCarriageReturn(StyledText text, int offset) {
        text.setRedraw(false);

        int lineNumber = text.getLineAtOffset(offset);
        StyledTextContent content = text.getContent();

        content.replaceTextRange(offset, 0, content.getLineDelimiter());

        String lineText = content.getLine(lineNumber);
        Matcher trailing = ILineIndentation.TRAILINGSPACE.matcher(lineText);
        if (trailing.matches()) {
            String trailingSpaces = trailing.group(1);
            content.replaceTextRange(offset - trailingSpaces.length(),
                    trailingSpaces.length(), "");
            offset -= trailingSpaces.length();
        }

        offset += content.getLineDelimiter().length();

        lineText = content.getLine(lineNumber);

        int caretDelta = formatLine(text, offset);
        text.setCaretOffset(offset + caretDelta);
        text.setSelection(offset + caretDelta);
        text.showSelection();

        text.setRedraw(true);
    }

    public void insertCloseBrace(StyledText text, int offset) {
        text.setRedraw(false);

        StyledTextContent content = text.getContent();

        content.replaceTextRange(offset, 0, "}");
        offset++;
        int lineLengthDelta = formatLine(text, offset);
        text.setCaretOffset(offset + lineLengthDelta);
        text.setRedraw(true);
    }

    public void insertOpenBrace(StyledText text, int offset) {
        text.setRedraw(false);

        StyledTextContent content = text.getContent();

        content.replaceTextRange(offset, 0, "{");
        offset++;
        int lineLengthDelta = formatLine(text, offset);
        text.setCaretOffset(offset + lineLengthDelta);
        text.setRedraw(true);
    }

    public void tab(StyledText text, int offset) {
        text.setRedraw(false);

        int lineNumber = text.getLineAtOffset(offset);
        int lineOffset = text.getOffsetAtLine(lineNumber);
        StyledTextContent content = text.getContent();

        String textLine = content.getLine(lineNumber);

        Matcher indent = ILineIndentation.INDENT.matcher(textLine);
        if (indent.matches()) {
            String indentText = indent.group(1);
            int caretOffsetInLine = offset - lineOffset;
            if (caretOffsetInLine < indentText.length()
                    || (caretOffsetInLine == indentText.length() && caretOffsetInLine == 0)) {
                int lineLengthDelta = formatLine(text, offset);
                text.setCaretOffset(lineOffset + indentText.length()
                        + lineLengthDelta);
            } else {
                content.replaceTextRange(offset, 0, ILineIndentation.UNIT);
                text.setCaretOffset(offset + ILineIndentation.UNIT.length());
            }
        } else {
            content.replaceTextRange(offset, 0, ILineIndentation.UNIT);
            text.setCaretOffset(offset + ILineIndentation.UNIT.length());
        }

        text.setRedraw(true);
    }

    public void formatText(StyledText text) {
        text.setRedraw(false);
        StyledTextContent content = text.getContent();
        int caretOffset = text.getCaretOffset();
        for (int idx = 0; idx < content.getLineCount(); idx++) {
            int lineOffset = content.getOffsetAtLine(idx);
            formatLine(text, lineOffset);
        }
        if (caretOffset > content.getCharCount()) {
            caretOffset = content.getCharCount();
        }
        boolean inc = true;
        while (!(caretOffset == content.getCharCount())
                && content.getLineDelimiter().indexOf(
                        content.getTextRange(caretOffset, 1).charAt(0)) != -1) {
            if (inc && caretOffset + 1 < content.getCharCount()) {
                caretOffset++;
            } else if (inc && caretOffset + 1 >= content.getCharCount()) {
                inc = false;
            }
            if (!inc && caretOffset > 0) {
                caretOffset--;
            } else if (!inc && caretOffset <= 0) {
                text.setRedraw(true);
                return; // avoid endless loop
            }
        }
        text.setCaretOffset(caretOffset);
        text.setRedraw(true);
    }

    public void parseText(StyledText text) {
        StyledTextContent content = text.getContent();
        for (int idx = 0; idx < content.getLineCount(); idx++) {
            int lineOffset = content.getOffsetAtLine(idx);
            parseLine(text, lineOffset);
        }
    }

    private int formatLine(StyledText text, int offset) {
        int lineNumber = text.getLineAtOffset(offset);
        int lineOffset = text.getOffsetAtLine(lineNumber);
        StyledTextContent content = text.getContent();

        String prevLineText = "";
        if (lineNumber > 0)
            prevLineText = content.getLine(lineNumber - 1);

        String lineText = content.getLine(lineNumber);

        String existingIndent = "";
        Matcher existingIndentM = ILineIndentation.INDENT.matcher(lineText);
        if (existingIndentM.matches()) {
            existingIndent = existingIndentM.group(1);
            content.replaceTextRange(lineOffset, existingIndent.length(), "");
            offset -= existingIndent.length();
        }

        String indent = getAppropriateIndent(lineNumber, prevLineText, lineText);
        content.replaceTextRange(lineOffset, 0, indent);
        return indent.length() - existingIndent.length();
    }

    private void parseLine(StyledText text, int offset) {
        int lineNumber = text.getLineAtOffset(offset);
        StyledTextContent content = text.getContent();

        String prevLineText = "";
        if (lineNumber > 0)
            prevLineText = content.getLine(lineNumber - 1);

        String lineText = content.getLine(lineNumber);

        getAppropriateIndent(lineNumber, prevLineText, lineText);
    }

    private String getAppropriateIndent(int lineNumber, String lineText,
            String nextLineText) {
        lineNumber--;
        if (lineNumber < 0)
            return "";
        ILineIndentation lineIndenter = root.getLine(lineNumber);
        if (lineIndenter == null) {
            lineIndenter = createNewLineIndenter(lineNumber);
        }
        lineIndenter.parseLineText(lineText, nextLineText);
        return lineIndenter.getNextLineIndentation();
    }

    private ILineIndentation createNewLineIndenter(int lineNumber) {
        if (lineNumber < 1)
            return root;
        ILineIndentation previous = root.getLine(lineNumber - 1);
        if (previous == null) {
            if (lineNumber - 1 > 0)
                previous = createNewLineIndenter(lineNumber - 1);
        }
        ILineIndentation middle = new LineIndentation();
        if (previous == null) {
            root = middle;
            System.err.println("Hmm, possible issue - "
                    + "root has been reset when creating line " + lineNumber
                    + " - report to ms02");
        } else {
            ILineIndentation next = previous.getNextLine();
            if (next != null) {
                middle.setNextLine(next);
                next.setPreviousLine(middle);
            }

            previous.setNextLine(middle);
            middle.setPreviousLine(previous);
        }
        return middle;
    }
}