/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation, Matthew
 * Sackman - customisation to kenya
 ******************************************************************************/
package kenya.ui.syntaxhighlighting;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.regex.Matcher;

import kenya.io.VariablePushbackReader;
import kenya.ui.indentation.ILineIndentation;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.LineStyleEvent;
import org.eclipse.swt.custom.LineStyleListener;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledTextContent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;

public class KenyaLineStyler implements LineStyleListener {

    private KenyaScanner scanner;

    private int[] tokenColors;

    private Color[] colors;

    private List blockComments = new ArrayList();

    public static final int EOF = -1;

    public static final int EOL = 10;

    public static final int WORD = 0;

    public static final int WHITE = 1;

    public static final int KEY = 2;

    public static final int COMMENT = 3;

    public static final int STRING = 5;

    public static final int OTHER = 6;

    public static final int NUMBER = 7;

    public static final int OPERATOR = 8;

    public static final int GENERIC = 9;

    public static final int MAXIMUM_TOKEN = 10;

    public KenyaLineStyler() {
        initializeColors();
        scanner = new KenyaScanner();
    }

    Color getColor(int type) {
        if (type < 0 || type >= tokenColors.length) { return null; }
        return colors[tokenColors[type]];
    }

    boolean inBlockComment(int start, int end) {
        for (int idx = 0; idx < blockComments.size(); idx++) {
            Point p = (Point) blockComments.get(idx);
            if (start >= p.x && start <= p.y && end >= p.x && end <= p.y)
                    return true;
        }
        return false;
    }

    void initializeColors() {
        Display display = Display.getDefault();
        colors = new Color[] { new Color(display, new RGB(0, 0, 0)), // black
                new Color(display, new RGB(255, 0, 0)), // red
                new Color(display, new RGB(0, 165, 40)), // green
                new Color(display, new RGB(0, 0, 255)), // blue
                new Color(display, new RGB(150, 0, 255)), // purple
                new Color(display, new RGB(0, 200, 200)) // cyan
        };
        tokenColors = new int[MAXIMUM_TOKEN];
        tokenColors[WORD] = 0;
        tokenColors[WHITE] = 0;
        tokenColors[KEY] = 3;
        tokenColors[COMMENT] = 1;
        tokenColors[STRING] = 2;
        tokenColors[OTHER] = 0;
        tokenColors[NUMBER] = 0;
        tokenColors[OPERATOR] = 4;
        tokenColors[GENERIC] = 5;
    }

    void disposeColors() {
        for (int i = 0; i < colors.length; i++) {
            colors[i].dispose();
        }
    }

    /**
     * Event.detail line start offset (input) Event.text line text (input)
     * LineStyleEvent.styles Enumeration of StyleRanges, need to be in order.
     * (output) LineStyleEvent.background line background color (output)
     */
    public void lineGetStyle(LineStyleEvent event) {
        Vector styles = new Vector();
        int token;
        StyleRange lastStyle;
        // If the line is part of a block comment, create one style for the
        // entire line.
        if (inBlockComment(event.lineOffset, event.lineOffset
                + event.lineText.length())) {
            styles.addElement(new StyleRange(event.lineOffset, event.lineText
                    .length(), getColor(COMMENT), null));
            event.styles = new StyleRange[styles.size()];
            styles.copyInto(event.styles);
            return;
        }
        Color defaultFgColor = ((Control) event.widget).getForeground();
        scanner.setRange(event.lineText);
        token = scanner.nextToken(event.lineOffset);
        while (token != EOF) {
            if (token == OTHER) {
                // do nothing for non-colored tokens
            } else if (token != WHITE) {
                Color color = getColor(token);
                // Only create a style if the token color is different than the
                // widget's default foreground color and the token's style is
                // not
                // bold. Keywords are bolded.
                if ((!color.equals(defaultFgColor)) || (token == KEY)) {
                    StyleRange style = new StyleRange(scanner.getStartOffset()
                            + event.lineOffset, scanner.getLength(), color,
                            null);
                    if (token == KEY) {
                        style.fontStyle = SWT.BOLD;
                    }
                    if (styles.isEmpty()) {
                        styles.addElement(style);
                    } else {
                        // Merge similar styles. Doing so will improve
                        // performance.
                        lastStyle = (StyleRange) styles.lastElement();
                        if (lastStyle.similarTo(style)
                                && (lastStyle.start + lastStyle.length == style.start)) {
                            lastStyle.length += style.length;
                        } else {
                            styles.addElement(style);
                        }
                    }
                }
            } else if ((!styles.isEmpty())
                    && ((lastStyle = (StyleRange) styles.lastElement()).fontStyle == SWT.BOLD)) {
                int start = scanner.getStartOffset() + event.lineOffset;
                lastStyle = (StyleRange) styles.lastElement();
                // A font style of SWT.BOLD implies that the last style
                // represents a java keyword.
                if (lastStyle.start + lastStyle.length == start) {
                    // Have the white space take on the style before it to
                    // minimize the number of style ranges created and the
                    // number of font style changes during rendering.
                    lastStyle.length += scanner.getLength();
                }
            }
            token = scanner.nextToken(event.lineOffset);
        }
        event.styles = new StyleRange[styles.size()];
        styles.copyInto(event.styles);
    }

    public void parseBlockComments(StyledTextContent content) {
        blockComments.clear();
        Point currentPoint = null;

        for (int lineNumber = 0; lineNumber < content.getLineCount(); lineNumber++) {
            String line = content.getLine(lineNumber);
            int offset = content.getOffsetAtLine(lineNumber);
            currentPoint = extractCommentsFromLine(line, currentPoint, offset,
                    content);
        }

        if (currentPoint != null) {
            currentPoint.y = content.getCharCount() - 1;
            blockComments.add(currentPoint);
        }
    }

    private Point extractCommentsFromLine(String line, Point currentPoint,
            int offset, StyledTextContent content) {
        Matcher blockEnd = ILineIndentation.BLOCKEND.matcher(line);
        Matcher blockStart = ILineIndentation.BLOCKSTART.matcher(line);
        if (blockEnd.matches()) {
            String prefix = blockEnd.group(1);
            if (currentPoint == null) {
                blockStart = ILineIndentation.BLOCKSTART.matcher(prefix);
                if (blockStart.matches()) {
                    String prePrefix = blockStart.group(1);
                    currentPoint = new Point(offset + prePrefix.length(),
                            offset + prefix.length() + 2);
                    blockComments.add(currentPoint);
                    currentPoint = null;
                }
            } else {
                currentPoint.y = offset + prefix.length() + 2;
                blockComments.add(currentPoint);
                currentPoint = null;
            }
            String suffix = blockEnd.group(2);
            if (suffix.length() > 0) {
                currentPoint = extractCommentsFromLine(suffix, currentPoint,
                        offset + prefix.length() + 2, content);
            }
        } else if (blockStart.matches()) {
            if (currentPoint == null) {
                int prefixLength = blockStart.group(1).length();
                currentPoint = new Point(offset + prefixLength, 0);
            }
        }
        return currentPoint;
    }

    /**
     * A simple fuzzy scanner for Kenya
     */
    public class KenyaScanner {

        protected Map fgKeys = null;

        protected final StringBuffer fBuffer = new StringBuffer();

        protected VariablePushbackReader reader;

        protected int charsRead;

        protected int length;

        protected int currentTokenStartPosition;

        protected boolean fEofSeen = false;

        private final String[] fgKeywords = { "boolean", "char", "int",
                "double", "String", "void", "class", "const", "null" };

        private final String[] fgOperators = { "if", "else", "while", "return",
                "switch", "case", "break", "default", "for", "assert", "new",
                "enum", "true", "false", };

        public KenyaScanner() {
            initialize();
        }

        /**
         * Returns the ending location of the current token in the document.
         */
        public final int getLength() {
            return charsRead - currentTokenStartPosition;
        }

        /**
         * Initialize the lookup table.
         */
        void initialize() {
            fgKeys = new Hashtable();
            Integer k = new Integer(KEY);
            for (int i = 0; i < fgKeywords.length; i++)
                fgKeys.put(fgKeywords[i], k);

            Integer o = new Integer(OPERATOR);
            for (int i = 0; i < fgOperators.length; i++)
                fgKeys.put(fgOperators[i], o);
        }

        /**
         * Returns the starting location of the current token in the document.
         */
        public final int getStartOffset() {
            return currentTokenStartPosition;
        }

        /**
         * Returns the next lexical token in the document.
         */
        public int nextToken(int lineOffset) {
            int c;
            int preReadCount = 0;
            currentTokenStartPosition = charsRead;

            if (currentTokenStartPosition == 0
                    && inBlockComment(lineOffset, lineOffset)) {
                int d = read();
                if (d == EOF) return EOF;
                while (d != '/' && d != '*' && d != EOL && d != EOF) {
                    d = read();
                    preReadCount++;
                }
                unread(d);
                if (d == EOL || d == EOF) return COMMENT;
            }

            while (true) {

                switch (c = read()) {
                case EOF:
                    return EOF;
                case '<': // generic
                    int depth = 1;
                    List reads = new ArrayList();
                    //reads.append(c);
                    for (;;) {
                        c = read();
                        reads.add(new Integer(c));
                        if (c == '>') {
                            depth--;
                            if (depth == 0) return GENERIC;
                        } else if (c == EOL || c == EOF) {
                            unreadBuffer(reads);
                            return OTHER;
                        } else {
                            if (!(c == ' ' || c == ',' || c == '_' || c == '<'
                                    || c == '>' || c == '[' || c == ']'
                                    || (c >= '0' && c <= '9')
                                    || (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z'))) {
                                unreadBuffer(reads);
                                return OTHER;
                            }
                            if (c == '<') depth++;
                        }
                    }
                case '/': // comment
                    c = read();
                    if (c == '/') {
                        while (true) {
                            c = read();
                            if ((c == EOF) || (c == EOL)) {
                                unread(c);
                                return COMMENT;
                            }
                        }
                    } else if (c == '*') { // block comment
                        for (int idx = 0; idx < blockComments.size(); idx++) {
                            Point p = (Point) blockComments.get(idx);
                            if (p.x == currentTokenStartPosition + lineOffset) {
                                int target = p.y;
                                int count = 2; // already read '/' and '*'
                                while (currentTokenStartPosition + lineOffset
                                        + count <= target
                                        && c != EOL && c != EOF) {
                                    c = read();
                                    count++;
                                }
                                return COMMENT;
                            }
                        }
                        unread(c);
                    } else {
                        unread(c);
                    }
                    return OTHER;
                case '*': // end block comment
                    c = read();
                    if (c == '/') {
                        for (int idx = 0; idx < blockComments.size(); idx++) {
                            Point p = (Point) blockComments.get(idx);
                            if (p.y == currentTokenStartPosition + lineOffset
                                    + preReadCount + 2) return COMMENT;
                        }
                        unread(c);
                    } else {
                        unread(c);
                    }
                    return OTHER;
                case '\'': // char const
                    character: for (;;) {
                        c = read();
                        switch (c) {
                        case '\'':
                            return STRING;
                        case EOF:
                            unread(c);
                            return STRING;
                        case '\\':
                            c = read();
                            break;
                        }
                    }

                case '"': // string
                    string: for (;;) {
                        c = read();
                        switch (c) {
                        case '"':
                            return STRING;
                        case EOF:
                            unread(c);
                            return STRING;
                        case '\\':
                            c = read();
                            break;
                        }
                    }

                case '0':
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9':
                    do {
                        c = read();
                    } while (Character.isDigit((char) c));
                    unread(c);
                    return NUMBER;
                default:
                    if (Character.isWhitespace((char) c)) {
                        do {
                            c = read();
                        } while (Character.isWhitespace((char) c));
                        unread(c);
                        return WHITE;
                    }
                    if (isKenyaIdentifierStart((char) c)) {
                        fBuffer.setLength(0);
                        do {
                            fBuffer.append((char) c);
                            c = read();
                        } while (isKenyaIdentifierPart((char) c));
                        unread(c);
                        if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z')
                                || (c >= 'A' && c <= 'Z') || c == '_')
                                return OTHER;
                        Integer i = (Integer) fgKeys.get(fBuffer.toString());
                        if (i != null) return i.intValue();
                        return WORD;
                    }
                    return OTHER;
                }
            }
        }

        private boolean isKenyaIdentifierStart(char c) {
            synchronized (fgKeys) {
                Iterator it = fgKeys.keySet().iterator();
                while (it.hasNext()) {
                    String key = (String) it.next();
                    if (key.startsWith("" + c)) return true;
                }
            }
            return false;
        }

        private boolean isKenyaIdentifierPart(char c) {
            synchronized (fgKeys) {
                Iterator it = fgKeys.keySet().iterator();
                while (it.hasNext()) {
                    String key = (String) it.next();
                    if (key.indexOf(c) != -1) return true;
                }
            }
            return false;
        }

        /**
         * Returns next character.
         */
        protected int read() {
            try {
                if (reader.ready()) {
                    charsRead++;
                    return reader.read();
                }
            } catch (IOException e) {
                return EOF;
            }
            return EOF;
        }

        public void setRange(String text) {
            reader = new VariablePushbackReader(new StringReader(text));
            charsRead = 0;
            currentTokenStartPosition = 0;
            length = text.length();
        }

        protected void unread(int c) {
            if (c != EOF)
                reader.unread(c);
            
            charsRead--;
            if (charsRead < 0) {
                System.err.println("CharsRead goes -ve ("
                        + charsRead + ") when unreading " + c);
            }
        }

        protected void unreadBuffer(List buff) {
            for (int idx = buff.size() - 1; idx >= 0; idx--) {
                unread(((Integer) buff.get(idx)).intValue());
            }
        }
    }

}