/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 21-Jul-2004 by toa02
 *
 */
package kenya.values.util;

import kenya.errors.KenyaInternalError;
import kenya.errors.KenyaPreconditionError;
import kenya.values.IKNumericValue;
import kenya.values.IKValue;
import kenya.values.KBooleanValue;
import kenya.values.KCharValue;
import kenya.values.KDoubleValue;
import kenya.values.KEnumValue;
import kenya.values.KIntValue;
import kenya.values.KNullValue;
import kenya.values.KStringValue;
import kenya.values.KUnknownValue;


/**
 * @author toa02
 *  
 */
public class KValueCalculator {

    /**
     * Cannot be instantiated - holds utility methods.
     */
    private KValueCalculator(){ /* Do not allow construction */ }
    
    /**
     * Adds two IKValues together.
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue add(IKValue lhs, IKValue rhs) {

        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        
        if (lhs instanceof KStringValue) {
            // do something that returns
            return new KStringValue(lhs.getString() + rhs.getString());
        }

        if (rhs instanceof KStringValue) {
            // do something that returns
            return new KStringValue(lhs.getString() + rhs.getString());
        }

        
        if (lhs instanceof KCharValue) {
            KCharValue ldc = (KCharValue) lhs;
            
            //char + char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                
                int res = 	ldc.getValue() +
                			rdc.getValue();
                
                /* Statically this is still a character */
                if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE){
                    return new KCharValue((char)res); // there *is* something wrong
                    								   // by the fact that i need that cast
                }else{
                /* Oh crud. Well the type checker thought this was a character,
                 * be we have now statically determined that it is, infact, out of range 
                 * an should really be an int. */
                    return new KIntValue(res);
                }
                
            //char + int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() + rdc.getValue();
                return new KIntValue(res);
            //char + double
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() + rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KIntValue) {
            KIntValue ldc = (KIntValue) lhs;
            //int + char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                int res = ldc.getValue() + rdc.getValue();
                return new KIntValue(res);
            //int + int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() + rdc.getValue();
                return new KIntValue(res);
            //int + double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() + rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KDoubleValue) {
            KDoubleValue ldc = (KDoubleValue) lhs;
            //double + char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                double res = ldc.getValue() + rdc.getValue();
                return new KDoubleValue(res);
            //double + int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                double res = ldc.getValue() + rdc.getValue();
                return new KDoubleValue(res);
            //double + double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() + rdc.getValue();
                return new KDoubleValue(res);
            }
        }
        
        throw KenyaPreconditionError.get();
    }

    
    
    /**
     * Statically subtract two IKValues and return the result.
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue sub(IKValue lhs, IKValue rhs) {
    
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        
        if (lhs instanceof KCharValue) {
            KCharValue ldc = (KCharValue) lhs;
            
            //char - char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                
                int res = 	ldc.getValue() -
                			rdc.getValue();
                
                /* Statically this is still a character */
                if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE){
                    return new KCharValue((char)res); // there *is* something wrong
                    								   // by the fact that i need that cast
                }else{
                /* Oh crud. Well the type checker thought this was a character,
                 * be we have now statically determined that it is, infact, out of range 
                 * an should really be an int. */
                    return new KIntValue(res);
                }
                
            //char - int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() - rdc.getValue();
                return new KIntValue(res);
            //char - double
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() - rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KIntValue) {
            KIntValue ldc = (KIntValue) lhs;
            //int - char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                int res = ldc.getValue() - rdc.getValue();
                return new KIntValue(res);
            //int - int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() - rdc.getValue();
                return new KIntValue(res);
            //int - double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() - rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KDoubleValue) {
            KDoubleValue ldc = (KDoubleValue) lhs;
            //double - char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                double res = ldc.getValue() - rdc.getValue();
                return new KDoubleValue(res);
            //double - int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                double res = ldc.getValue() - rdc.getValue();
                return new KDoubleValue(res);
            //double - double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() - rdc.getValue();
                return new KDoubleValue(res);
            }
        }
        
        throw KenyaPreconditionError.get();
    }

    
    /**
     * Multiply two IKValues together
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue mult(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        
        if (lhs instanceof KCharValue) {
            KCharValue ldc = (KCharValue) lhs;
            
            //char * char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                
                int res = 	ldc.getValue() *
                			rdc.getValue();
                
                /* Statically this is still a character */
                if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE){
                    return new KCharValue((char)res); // there *is* something wrong
                    								   // by the fact that i need that cast
                }else{
                /* Oh crud. Well the type checker thought this was a character,
                 * be we have now statically determined that it is, infact, out of range 
                 * an should really be an int. */
                    return new KIntValue(res);
                }
                
            //char * int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() * rdc.getValue();
                return new KIntValue(res);
            //char * double
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() * rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KIntValue) {
            KIntValue ldc = (KIntValue) lhs;
            //int * char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                int res = ldc.getValue() * rdc.getValue();
                return new KIntValue(res);
            //int * int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() * rdc.getValue();
                return new KIntValue(res);
            //int * double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() * rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KDoubleValue) {
            KDoubleValue ldc = (KDoubleValue) lhs;
            //double * char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                double res = ldc.getValue() * rdc.getValue();
                return new KDoubleValue(res);
            //double * int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                double res = ldc.getValue() * rdc.getValue();
                return new KDoubleValue(res);
            //double * double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() * rdc.getValue();
                return new KDoubleValue(res);
            }
        }
        
        throw KenyaPreconditionError.get();
    }

    
    /**
     * Divide two values.
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue div(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        // So we at least do not get a /0 in the middle of this!
        if( rhs instanceof IKNumericValue && 
                ((IKNumericValue)rhs).equalsZero() ){
            return KUnknownValue.get();
        }
        
        if (lhs instanceof KCharValue) {
            KCharValue ldc = (KCharValue) lhs;
            
            //char / char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                
                int res = 	ldc.getValue() /
                			rdc.getValue();
                
                /* Statically this is still a character */
                if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE){
                    return new KCharValue((char)res); // there *is* something wrong
                    								   // by the fact that i need that cast
                }else{
                /* Oh crud. Well the type checker thought this was a character,
                 * be we have now statically determined that it is, infact, out of range 
                 * an should really be an int. */
                    return new KIntValue(res);
                }
                
            //char / int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() / rdc.getValue();
                return new KIntValue(res);
            //char / double
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() / rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KIntValue) {
            KIntValue ldc = (KIntValue) lhs;
            //int / char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                int res = ldc.getValue() / rdc.getValue();
                return new KIntValue(res);
            //int / int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() / rdc.getValue();
                return new KIntValue(res);
            //int / double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() / rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KDoubleValue) {
            KDoubleValue ldc = (KDoubleValue) lhs;
            //double / char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                double res = ldc.getValue() / rdc.getValue();
                return new KDoubleValue(res);
            //double / int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                double res = ldc.getValue() / rdc.getValue();
                return new KDoubleValue(res);
            //double / double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() / rdc.getValue();
                return new KDoubleValue(res);
            }
        }
        
        throw KenyaPreconditionError.get();
    }
    
    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue mod(IKValue lhs, IKValue rhs) {

        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        // So we at least do not get a % 0 in the middle of this!
        if( rhs instanceof IKNumericValue && 
                ((IKNumericValue)rhs).equalsZero() ){
            return KUnknownValue.get();
        }
        
        
        if (lhs instanceof KCharValue) {
            KCharValue ldc = (KCharValue) lhs;
            
            //char % char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                
                int res = 	ldc.getValue() %
                			rdc.getValue();
                
                /* Statically this is still a character */
                if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE){
                    return new KCharValue((char)res); // there *is* something wrong
                    								   // by the fact that i need that cast
                }else{
                /* Oh crud. Well the type checker thought this was a character,
                 * be we have now statically determined that it is, infact, out of range 
                 * an should really be an int. */
                    return new KIntValue(res);
                }
                
            //char % int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() % rdc.getValue();
                return new KIntValue(res);
            //char % double
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() % rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KIntValue) {
            KIntValue ldc = (KIntValue) lhs;
            //int % char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                int res = ldc.getValue() % rdc.getValue();
                return new KIntValue(res);
            //int % int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                int res = ldc.getValue() % rdc.getValue();
                return new KIntValue(res);
            //int % double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() % rdc.getValue();
                return new KDoubleValue(res);
            }
            
        } else if (lhs instanceof KDoubleValue) {
            KDoubleValue ldc = (KDoubleValue) lhs;
            //double % char
            if (rhs instanceof KCharValue) {
                KCharValue rdc = (KCharValue) rhs;
                double res = ldc.getValue() % rdc.getValue();
                return new KDoubleValue(res);
            //double % int    
            } else if (rhs instanceof KIntValue) {
                KIntValue rdc = (KIntValue) rhs;
                double res = ldc.getValue() % rdc.getValue();
                return new KDoubleValue(res);
            //double % double    
            } else if (rhs instanceof KDoubleValue) {
                KDoubleValue rdc = (KDoubleValue) rhs;
                double res = ldc.getValue() % rdc.getValue();
                return new KDoubleValue(res);
            }
        }
        
        throw KenyaPreconditionError.get();
    
    }

    /**
     * @param val
     * @return
     */
    public static IKValue unSub(IKValue val) {
   
        if( val instanceof KUnknownValue ){
            return val;
        }
        
        if( val instanceof KCharValue ){
            KCharValue kcv = (KCharValue) val; 
            return new KIntValue( -kcv.getValue() );
        }else if(val instanceof KIntValue ){
            KIntValue kiv = (KIntValue) val;
            return new KIntValue( -kiv.getValue() );
        }else if(val instanceof KDoubleValue ){
            KDoubleValue kdv = (KDoubleValue) val;
            return new KDoubleValue( -kdv.getValue() );
        }

        throw KenyaPreconditionError.get();
        
    }

    /**
     * In theory this won't actually get called
     * (well as the grammar stands at the moment anyways)
     * @param val
     * @return
     */
    public static IKValue plusplus(IKValue val) {
        if( val instanceof KUnknownValue ){
            return val;
        }
        
        if( val instanceof KCharValue ){
            KCharValue kcv = (KCharValue) val; 
            int res = kcv.getValue() + 1;
            if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE ){
                return new KCharValue((char)res);
            }else{
                return new KIntValue(res);
            }
        }else if(val instanceof KIntValue ){
            KIntValue kiv = (KIntValue) val;
            return new KIntValue( kiv.getValue() +1 );
        }else if(val instanceof KDoubleValue ){
            KDoubleValue kdv = (KDoubleValue) val;
            return new KDoubleValue( kdv.getValue() +1 );
        }

        throw KenyaPreconditionError.get();
    }

    /**
     * In theory this won't actually get called
     * (well as the grammar stands at the moment anyways)
     * @param val
     * @return
     */
    public static IKValue minusminus(IKValue val) {

        if( val instanceof KUnknownValue ){
            return val;
        }
        
        if( val instanceof KCharValue ){
            KCharValue kcv = (KCharValue) val; 
            int res = kcv.getValue() - 1;
            if( res <= Character.MAX_VALUE && res >= Character.MIN_VALUE ){
                return new KCharValue((char)res);
            }else{
                return new KIntValue(res);
            }
        }else if(val instanceof KIntValue ){
            KIntValue kiv = (KIntValue) val;
            return new KIntValue( kiv.getValue() -1 );
        }else if(val instanceof KDoubleValue ){
            KDoubleValue kdv = (KDoubleValue) val;
            return new KDoubleValue( kdv.getValue() -1 );
        }

        throw KenyaPreconditionError.get();
    
    }

    /**
     * @param val
     * @return
     */
    public static IKValue unNeg(IKValue val) {
        if( val instanceof KUnknownValue ){
            return val;
        }
        
        if( val instanceof KBooleanValue ){
            KBooleanValue kbv = (KBooleanValue) val;
            return kbv.getNegation();
        }
        
        throw new KenyaInternalError("Precondition failed for ! (unary negation) in KValueCalculator: " +
                val);
    }

    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue or(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof KBooleanValue && rhs instanceof KBooleanValue ){
            KBooleanValue kbl = (KBooleanValue) lhs;
            KBooleanValue kbr = (KBooleanValue) rhs;
            
            return KBooleanValue.get(
                    kbl.getValue() || kbr.getValue()
                    );
        }
        
        throw KenyaPreconditionError.get();
    }


    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue xor(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof KBooleanValue && rhs instanceof KBooleanValue ){
            KBooleanValue kbl = (KBooleanValue) lhs;
            KBooleanValue kbr = (KBooleanValue) rhs;
            
            return KBooleanValue.get(
                    kbl.getValue() ^ kbr.getValue()
                    );
        }
        
        throw KenyaPreconditionError.get();
    }
    
    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue and(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof KBooleanValue && rhs instanceof KBooleanValue ){
            KBooleanValue kbl = (KBooleanValue) lhs;
            KBooleanValue kbr = (KBooleanValue) rhs;
            
            return KBooleanValue.get(
                    kbl.getValue() && kbr.getValue()
                    );
        }
        
        throw new KenyaInternalError("Precondition failed for binary and in KValueCalculator: " +
                lhs + " && " + rhs);
    }

    
    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue equal(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof IKNumericValue && rhs instanceof IKNumericValue ){
            IKNumericValue lv = (IKNumericValue) lhs;
            IKNumericValue rv = (IKNumericValue) rhs;
            
            return KBooleanValue.get(
                    lv.getDoubleValue() == rv.getDoubleValue()
                    );
        }

        if( lhs instanceof KEnumValue && rhs instanceof KEnumValue){
            return KBooleanValue.get( lhs.equals(rhs) );
        }
        
        if( lhs instanceof KBooleanValue && rhs instanceof KBooleanValue ){
            KBooleanValue lv = (KBooleanValue) lhs;
            KBooleanValue rv = (KBooleanValue) rhs;
            
            return KBooleanValue.get( lv.getValue() == rv.getValue());
            
        }
        
        if( lhs instanceof KNullValue && rhs instanceof KNullValue ) {
            //null == null;
            return KBooleanValue.get(true);
        }
        
        return KUnknownValue.get();
    }

    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue unequal(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof IKNumericValue && rhs instanceof IKNumericValue ){
            IKNumericValue lv = (IKNumericValue) lhs;
            IKNumericValue rv = (IKNumericValue) rhs;
            
            return KBooleanValue.get(
                    lv.getDoubleValue() != rv.getDoubleValue()
                    );
        }

        if( lhs instanceof KBooleanValue && rhs instanceof KBooleanValue ){
            KBooleanValue lv = (KBooleanValue) lhs;
            KBooleanValue rv = (KBooleanValue) rhs;
            
            return KBooleanValue.get( lv.getValue() != rv.getValue());
            
        }
        
        
        if( lhs instanceof KEnumValue && rhs instanceof KEnumValue){
            return KBooleanValue.get( !lhs.equals(rhs) );
        }
        
        return KUnknownValue.get();
        
    }

    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue less(IKValue lhs, IKValue rhs) {
    
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof IKNumericValue && rhs instanceof IKNumericValue ){
            IKNumericValue lv = (IKNumericValue) lhs;
            IKNumericValue rv = (IKNumericValue) rhs;
            
            return KBooleanValue.get(
                    lv.getDoubleValue() < rv.getDoubleValue()
                    );
        }

        throw KenyaPreconditionError.get();
        
    }

    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue greater(IKValue lhs, IKValue rhs) {
    
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof IKNumericValue && rhs instanceof IKNumericValue ){
            IKNumericValue lv = (IKNumericValue) lhs;
            IKNumericValue rv = (IKNumericValue) rhs;
            
            return KBooleanValue.get(
                    lv.getDoubleValue() > rv.getDoubleValue()
                    );
        }

        throw KenyaPreconditionError.get();
        
    }

    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue lessEq(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof IKNumericValue && rhs instanceof IKNumericValue ){
            IKNumericValue lv = (IKNumericValue) lhs;
            IKNumericValue rv = (IKNumericValue) rhs;
            
            return KBooleanValue.get(
                    lv.getDoubleValue() <= rv.getDoubleValue()
                    );
        }

        throw KenyaPreconditionError.get();
    }

    /**
     * @param lhs
     * @param rhs
     * @return
     */
    public static IKValue greaterEq(IKValue lhs, IKValue rhs) {
        if( lhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( rhs instanceof KUnknownValue ){
            return KUnknownValue.get();
        }
        
        if( lhs instanceof IKNumericValue && rhs instanceof IKNumericValue ){
            IKNumericValue lv = (IKNumericValue) lhs;
            IKNumericValue rv = (IKNumericValue) rhs;
            
            return KBooleanValue.get(
                    lv.getDoubleValue() >= rv.getDoubleValue()
                    );
        }

        throw KenyaPreconditionError.get();
    }

}
