/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 23-Jul-2004 by toa02
 *
 */
package mediator;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Reader;


/**
 * @author toa02
 */
public interface IJavaCode {
    
    
    public static final String NEWLINE = System.getProperty("line.separator");
    
    /**
     * Gives back the actual source code representation of this
     * "IJavaCode", with the enclosing class name as specified
     * @param className The name to use for the enclosing class of
     * the static methods.
     * @return A Reader to get the java source code from.
     * @throws IllegalArgumentException if the provided className is
     * either a class name that is already defined in the source code,
     * or is not a valid class name.
     */
    public abstract Reader getCode(String className);
    
    /**
     * Gives back the actual source code representation of this
     * "IJavaCode", with the enclosing class name as a placehold
     *  - note: this code will not compile!
     * This is useful for displaying the java code without actually 
     * having to save it.
     * @param className The name to use for the enclosing class of
     * the static methods.
     * @return A Reader to get the java source code from.
     * @throws IllegalArgumentException if the provided className is
     * either a class name that is already defined in the source code,
     * or is not a valid class name.
     */
    public abstract Reader getPlaceHeldCode();
    
    
    /**
     * @param name String name to check as a ClassName
     * @return true iff this name is a valid class name in this kenya file,
     * i.e. the name does not clash with an existing class name, or a reserved word
     */
    public boolean isValidClassName (String name);
    
    /**
     * @return false iff it is safe to compile this program
     * without java 1.5 extensions turned on.
     */
    public abstract boolean require15();
    
    
    /**
     * Compiles and executes this IJavaCode object.
     * The destination file is made up of <code>filePath + className + ".java"</code>
     * Any compile errors / IO exceptions are caught and the results put onto the 
     * nStdErr outputstream.  
     * 
     * Note This method will block the current thread while it runs javac and java.
     * When this method returns, any processes that this created *should* have terminted
     * (unless this thread was Interrupted in which case its behaviour is undefined and it
     * will return false).
     *
     * The programs called for java and javac are whatever would be if you typed in
     * javac and java normally. If they don't exist, then things are likely to go wrong.
     *  
     * @param String filePath The directory of where to store the file.
     * @param String className The className to use for the enclosing main class.
     * @param nStdIn BufferedReader to use as Standard In for the running program.
     * @param nStdOut BufferedWriter to use as StandardOut for the running program.
     * @param nStdErr BufferedWriter to use as StandardErr for the running program.
     * @param args String[] arguments to give to the program.
     * 
     * @return true iff the program compiled without errors and java was invoked on it
     * without a problem. (Whether the program errored is another matter.
     * False iff an exception/error stopped the destination file being written, or 
     * javac complained during compilation.
     */
    public boolean execute(String filePath, String className, 
            BufferedReader nStdIn, BufferedWriter stdInWriter, BufferedWriter nStdOut, 
            BufferedWriter nStdErr, String[] args);
    
    
    /**
     * Utility method to provide a way to kill the last process
     * that was created.
     */
    public void killLastProcess();
}