/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package mediator;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.IVariable;
import mediator.stackMachine.IStackMachineInformationProvider;
import mediator.util.InformationHolder;

/**
 * Represents a Kenya Program with Invalid Code.
 * @author toa02
 * @see mediator.ICheckedCode
 */
class InvalidCode implements ICheckedCode {

    private final List _errors;
    private final List _infos;
    private final InformationHolder _ih;
    
    public InvalidCode(List errors, InformationHolder ih){
        if( errors == null ){throw new NullPointerException("errors can't be null"); }
        
        _errors = Collections.unmodifiableList(errors);
        _infos = Collections.unmodifiableList(new LinkedList());
        
        if( ih == null ){ throw new NullPointerException("Information holder can't be null");}
        _ih = ih;
    }
    
    public InvalidCode(List errors, List infos, InformationHolder ih){
        if( errors == null ){throw new NullPointerException("errors can't be null"); }
        
        _errors = Collections.unmodifiableList(errors);

        if( infos == null ){throw new NullPointerException("infos can't be null"); }
        _infos = Collections.unmodifiableList(infos);
        
        if( ih == null ){ throw new NullPointerException("Information holder can't be null");}
        _ih = ih;
    }
    
    /**
     * @see mediator.ICheckedCode#isErroredCode()
     */
    public boolean isErroredCode() {
        return true;
    }

    /**
     * @see mediator.ICheckedCode#getInfos()
     */
    public List getInfos() {
        return _infos;
    }
    
    /**
     * @see mediator.ICheckedCode#getErrors()
     */
    public List getErrors() {
        return _errors;
    }

    /**
     * This method is guaranteed to throw an IllegalStateException.
     * @see mediator.ICheckedCode#translate()
     */
    public IJavaCode translate() {
        throw new IllegalStateException("Cannot translate an invalid Kenya Program");
    }

    /**
     * This method is guaranteed to throw an IllegalStateException.
     * @see mediator.ICheckedCode#getBaseStackMachine()
     */
    public IStackMachineInformationProvider getBaseStackMachine() {
        throw new IllegalStateException("There is no Stack Machine to interpret an invalid Kenya Program");
    }
    
    /**
     * @see mediator.ICheckedCode#getClasses()
     */
    public IClass[] getClasses() {
        return _ih.getClasses();
    }
    
    /**
     * @see mediator.ICheckedCode#getConstants()
     */
    public IVariable[] getConstants() {
        return _ih.getConstants();
    }
    
    /**
     * @see mediator.ICheckedCode#getEnums()
     */
    public IClass[] getEnums() {
        return _ih.getEnums();
    }
    
    
    /**
     * @see mediator.ICheckedCode#getFunctions()
     */
    public IFunction[] getFunctions() {
        return _ih.getFunctions();
    }

}
