/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package mediator;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import kenya.errors.KenyaInternalError;
import kenya.passes.Translator;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.types.KFunction;
import mediator.stackMachine.IStackMachineInformationManager;
import mediator.stackMachine.IStackMachineInformationProvider;
import mediator.stackMachine.StackMachineInformationManager;
import mediator.util.InformationHolder;
import mediator.util.JavaCode;
import mediator.util.ValidMainChecker;
import minijava.node.Node;

/**
 * Represents a Kenya Program with Valid Code.
 * @author toa02
 *
 */
class ValidCode implements ICheckedCode {

    private final List _infos;
    private final InformationHolder _ih;
    private final ValidMainChecker _vmc;
    private final Map _functionMap; // Maps nodes to functions used: needed by 
    private final Map _typeMap;		// Maps nodes to the type of the node
    
    public ValidCode( List infos, InformationHolder ih, ValidMainChecker vmc, Map functionMap, Map typeMap ){
        if( infos == null ){throw new NullPointerException("Infos can't be null");}
        if( ih == null ){ throw new NullPointerException("Information Holder can't be null");}
        _infos = Collections.unmodifiableList(infos);
        _ih = ih;
        _vmc = vmc;
        _functionMap = functionMap;
        _typeMap = typeMap;
    }
    /**
     * @see mediator.ICheckedCode#isErroredCode()
     */
    public boolean isErroredCode() {
        return false;
    }

    /**
     * @see mediator.ICheckedCode#getInfos()
     */
    public List getInfos() {
        return _infos;
    }
    
    /**
     * @see mediator.ICheckedCode#getErrors()
     */
    public List getErrors() {
        throw new IllegalStateException("Valid Kenya has no errors.");
    }

    /**
     * @see mediator.ICheckedCode#translate()
     */
    public IJavaCode translate() {
        /** OK to translate what do I need to do:
         * 1: establish used builtin functions & get their imports
         * 2: translate functions
         * 3: translate classes
         * 4: translate enums
         */
        
        String imports;
        String constants;
        String functions;
        String classesAndEnums;
        Set typeNames;
        Set allBuiltinTypes;

        boolean require15 = false;
        
        {
            SortedSet allImports = new TreeSet();		// Sorted set of Strings
	        Iterator it = _functionMap.keySet().iterator();
	        while ( it.hasNext() ){
	            KFunction kf = (KFunction) _functionMap.get(it.next());
	            if( kf.isBuiltin() ){
	                allImports.addAll(kf.getBuiltinMethod().getImports());
	            } else if( kf.getTypeParamMap().entrySet().size() != 0 ){
	                require15 = true;
	            }
	        }
	        
	        StringBuffer sb = new StringBuffer();
	        it = allImports.iterator();
	        
	        while( it.hasNext()){
	            sb.append("import ");
	            sb.append(it.next());
	            sb.append(";");
	            sb.append(IJavaCode.NEWLINE);
	        }
	        
	        imports = sb.toString();
        }
        
        {
            allBuiltinTypes = new HashSet();		// Sorted set of Strings
	        Iterator it = _functionMap.keySet().iterator();
	        while ( it.hasNext() ){
	            KFunction kf = (KFunction) _functionMap.get(it.next());
	            if( kf.isBuiltin() ){
	                allBuiltinTypes.addAll(kf.getBuiltinMethod().getReservedClasses());
	            }
	        }
        }
        

        Node node = (_vmc.getMainWithStringArgs() != null) ? _vmc.getMainWithStringArgs().getDeclarationNode()
                											: null;
        Translator translator = new Translator( _functionMap, node, _typeMap, true );

        /* Constants */
        translator.reset(Translator.CONSTANTS);
        _ih.getRootNode().apply(translator);
        constants = translator.getLastCode();
        
        /* Functions */
        translator.reset(Translator.FUNCTIONS);
        _ih.getRootNode().apply(translator);
        functions = translator.getLastCode();
        
        if( _vmc.haveMainWithStringArgs() ){
            // Then everything is fine
        }else if( _vmc.haveMainNoArgs() ){
            functions =
            IJavaCode.NEWLINE + 
            Translator.INDENT + "public static void main(String[] args) {" + IJavaCode.NEWLINE +
            Translator.INDENT + Translator.INDENT + "main();" + IJavaCode.NEWLINE +
            Translator.INDENT + "}" + IJavaCode.NEWLINE + functions;
        }else{
            throw new KenyaInternalError("Failed Precondition");
        }
        
        
        translator.reset(Translator.CLASSES | Translator.ENUMS );
        _ih.getRootNode().apply(translator);
        classesAndEnums = translator.getLastCode();
        
        
        typeNames = new HashSet();
        typeNames.addAll(allBuiltinTypes);
        
        for(int i = 0 ; i < _ih.getClasses().length ; i++ ){
            typeNames.add(_ih.getClasses()[i].getName());
            require15 |= _ih.getClasses()[i].getTemplateParams().length != 0;
        }
        
        for(int i = 0 ; i < _ih.getEnums().length ; i++ ){
            typeNames.add(_ih.getEnums()[i].getName());
            require15 = true;
        }
        
        /*
         * only do tihs check if we absolutly have to 
         */
        if(!require15){
            require15 = _ih.areClassBasicTypesUsed();
        }
        
        return new JavaCode( imports, constants, functions, classesAndEnums, typeNames, require15 );
    }

    /**
     * @see mediator.ICheckedCode#getBaseStackMachine()
     */
    public IStackMachineInformationProvider getBaseStackMachine() {
        IStackMachineInformationManager smi = new StackMachineInformationManager(_ih);
        return smi.build(_vmc);
    }

    /**
     * @see mediator.ICheckedCode#getClasses()
     */
    public IClass[] getClasses() {
        return _ih.getClasses();
    }
    
    /**
     * @see mediator.ICheckedCode#getConstants()
     */
    public IVariable[] getConstants() {
        return _ih.getConstants();
    }
    
    /**
     * @see mediator.ICheckedCode#getEnums()
     */
    public IClass[] getEnums() {
        return _ih.getEnums();
    }
    
    /**
     * @see mediator.ICheckedCode#getFunctions()
     */
    public IFunction[] getFunctions() {
        return _ih.getFunctions();
    }
}
