/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 10-Aug-2004 by toa02
 *
 */
package mediator.stackMachine;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import kenya.passes.StackMachineBuilder;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.interfaces.IVariable;
import mediator.util.InformationHolder;
import mediator.util.ValidMainChecker;
import minijava.node.AFuncDecDeclaration;

import org.wellquite.kenya.stackMachine.InterpretedBuildableClosure;
import org.wellquite.kenya.stackMachine.InterpretedClass;
import org.wellquite.kenya.stackMachine.InterpretedEnumeration;
import org.wellquite.kenya.stackMachine.InterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IBuildableClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedEnumeration;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;

/**
 * @author toa02
 *
 */
public class StackMachineInformationManager implements IStackMachineInformationManager{

    private final NameGenerator _classGen;
    private final NameGenerator _enumGen;
    private final NameGenerator _funcGen;
    private final NameGenerator _varGen;
    
    final BidirectionalMap _classMap;
    final BidirectionalMap _enumMap;	//map of IClass/IFunction <--> stack mach name
    final BidirectionalMap _funcMap;
    final BidirectionalMap _varMap;
    
    final BidirectionalMap _normClassMap;
    final BidirectionalMap _normEnumMap;
    final BidirectionalMap _normFuncMap;
    final BidirectionalMap _normVarMap;
    
    final Map _classBodies;
    final Map _funcBodies;
    final Map _enumBodies;
    
    final Map _funcNodeMap;
    
    final InformationHolder _ih;

    final IInterpretedClass _globalClass;
    
    final IBuildableClosure _preInitClosure;
    
    final List _codePoints;
    
    public StackMachineInformationManager(InformationHolder ih){
        _classGen = new NameGenerator("_cls");
        _enumGen = new NameGenerator("_enu");
        _funcGen= new NameGenerator("_fun");
        _varGen = new NameGenerator("_var");
        
        _classMap = new BidirectionalMap();
        _enumMap = new BidirectionalMap();
        _funcMap = new BidirectionalMap();
        _varMap = new BidirectionalMap();
        
        _normClassMap = new BidirectionalMap();
        _normEnumMap = new BidirectionalMap();
        _normFuncMap = new BidirectionalMap();
        _normVarMap = new BidirectionalMap();
        
        _classBodies = new HashMap();
        _funcBodies = new HashMap();
        _enumBodies = new HashMap();
        
        _funcNodeMap = new HashMap();
        
        _globalClass = new InterpretedClass(GLOBALCLASS);
        
        _preInitClosure = new InterpretedBuildableClosure();
        
        _ih = ih;
        
        _codePoints = new ArrayList();
        
        addBasicDetails(ih);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#addBasicDetails(mediator.util.InformationHolder)
     */
    private synchronized void addBasicDetails(InformationHolder ih) {
        
        /* Do classes: */
        {
            IClass[] classes = ih.getClasses();
            for(int i = 0 ; i < classes.length ; i++ ){
                String name = _classGen.getNewName();
                _classMap.associate(classes[i], name);
                _normClassMap.associate(classes[i],classes[i].normalise());
                _classBodies.put(name, new InterpretedClass(name));
                
                IVariable[] classDecs = classes[i].getDeclarations();
                for(int j = 0 ; j < classDecs.length ; j++){
                    String smName = _varGen.getNewName();
                    _varMap.associate(classDecs[j], smName);
                    _normVarMap.associate(classDecs[j], classDecs[j].normalise());
                }
            }
        }
        
        /* Do enumerations: */
        {
            IClass[] enums = ih.getEnums();
            for(int i = 0 ; i < enums.length ; i++){
                String name = _enumGen.getNewName();
                _enumMap.associate(enums[i],name);
                _normEnumMap.associate(enums[i],enums[i].normalise());
                String[] kids = enums[i].getEnumConstants();
                _enumBodies.put(name, new InterpretedEnumeration(name, kids ));
            }
        }
        
        /* Do functions */
        {
            IFunction[] functions = ih.getFunctions();
            for(int i = 0 ; i < functions.length ; i++ ){
                String name = _funcGen.getNewName();
                _funcMap.associate( functions[i], name );
                _normFuncMap.associate(functions[i],functions[i].normalise());
                _funcNodeMap.put( functions[i].getDeclarationNode(), functions[i] );
                boolean hasRet = functions[i].getReturnType() != ih.getVoid();
                IInterpretedMethod method = new InterpretedMethod(name,_globalClass,true,hasRet); 
                _funcBodies.put(name, method);
                _globalClass.addStaticMethod(method);
                // function param variables are created in caseAFuncDecDeclaration in StackMachineBuilder
            }
        }
        
        /* Placeholder the constants */
        {
            IVariable[] vars = ih.getConstants();
            for(int i = 0 ; i < vars.length ; i++ ){
                String name = _varGen.getNewName();
                _varMap.associate( vars[i], name );
                _normVarMap.associate(vars[i],vars[i].normalise());
            }
        }
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#addNewVariable(kenya.sourceCodeInformation.interfaces.IVariable)
     */
    public synchronized String addNewVariable(IVariable var) {
        String name = _varGen.getNewName();
        _varMap.associate(var,name);
        _normVarMap.associate(var,var.normalise());
        return name;
    }    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#build()
     */
    public synchronized IStackMachineInformationProvider build(ValidMainChecker vmc) {
        StackMachineBuilder smb = new StackMachineBuilder(this,_ih, true);
        _ih.getRootNode().apply(smb);
        return new StackMachineInformationProvider(this,vmc);
    }
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#getClassBody(kenya.sourceCodeInformation.interfaces.IClass)
     */
    public synchronized IInterpretedClass getClassBody(IClass clazz) {
        // From the class get the name, from the name get the body
        String name = (String) _classMap.lookupRHS(clazz);
        return (IInterpretedClass) _classBodies.get(name);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#getEnumBody(kenya.sourceCodeInformation.interfaces.IClass)
     */
    public IInterpretedEnumeration getEnumBody(IClass enumm) {
        String name = (String) _enumMap.lookupRHS( enumm );
        return( IInterpretedEnumeration) _enumBodies.get(name);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#getMethodBody(kenya.sourceCodeInformation.interfaces.IFunction)
     */
    public synchronized IInterpretedMethod getMethodBody(IFunction func) {
        String name = (String) _funcMap.lookupRHS(func);
        return (IInterpretedMethod) _funcBodies.get(name);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#lookupClass(kenya.sourceCodeInformation.interfaces.IClass)
     */
    public synchronized String lookupClass(IClass clazz) {
        return (String) _classMap.lookupRHS(clazz);
    }
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#lookupEnum(kenya.sourceCodeInformation.interfaces.IClass)
     */
    public synchronized String lookupEnum(IClass zenum) {
        return (String) _enumMap.lookupRHS(zenum);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#lookupFunction(kenya.types.IFunction)
     */
    public synchronized String lookupFunction(IFunction func) {
        return (String) _funcMap.lookupRHS(func);
    }
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#lookupVariable(kenya.sourceCodeInformation.interfaces.IVariable)
     */
    public synchronized String lookupVariable(IVariable var) {
        return (String) _varMap.lookupRHS(var);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#getFunctionFromNode(minijava.node.AFuncDecDeclaration)
     */
    public IFunction getFunctionFromNode(AFuncDecDeclaration node) {
        return (IFunction) _funcNodeMap.get(node);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#getGlobalClassBody()
     */
    public IInterpretedClass getGlobalClassBody() {
        return _globalClass;
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationManager#getPreInitClosure()
     */
    public IBuildableClosure getPreInitClosure() {
        return _preInitClosure;
    }
    
	/**
	 * @see mediator.stackMachine.IStackMachineInformationManager#addCodePoint(kenya.sourceCodeInformation.interfaces.ISourceCodeLocation)
	 */
	public void addCodePoint(ISourceCodeLocation iscl) {
	    _codePoints.add(iscl);
	}
    
    
}
