/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 16-Aug-2004 by toa02
 *
 */
package mediator.stackMachine;

import java.util.List;

import kenya.errors.KenyaPreconditionError;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.IVariable;
import mediator.util.ValidMainChecker;

import org.wellquite.kenya.stackMachine.InterpretedBuildableClosure;
import org.wellquite.kenya.stackMachine.InterpretedMethod;
import org.wellquite.kenya.stackMachine.ops.StackOpsFactory;
import org.wellquite.kenya.stackMachine.types.ClassTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IBuildableClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;

/**
 * @author toa02
 *
 */
class StackMachineInformationProvider implements
        IStackMachineInformationProvider {
   
    private static final String INIT_METHOD_NAME = "<INIT_METHOD>";
    
    private final StackMachineInformationManager _ismim;
    private final ValidMainChecker _vmc;
    private final List _codePoints;
    private final IAtomicClosure _preInitClosure;
    
    
    public StackMachineInformationProvider(StackMachineInformationManager ismim, ValidMainChecker vmc){
        _ismim = ismim;
        _vmc = vmc;
        _codePoints = ismim._codePoints;

        IInterpretedMethod iim = new InterpretedMethod(INIT_METHOD_NAME,_ismim.getGlobalClassBody(),true,false);
        _ismim.getGlobalClassBody().addStaticMethod(iim);
        
        iim.setMethodBody(_ismim._preInitClosure);
        IBuildableClosure ibc = new InterpretedBuildableClosure();
        ibc.addClosure( StackOpsFactory.push( ClassTypeFactory.createClassStaticType(getEntryPointClass())));
        ibc.addClosure( StackOpsFactory.invokeForeignStaticMethod(INIT_METHOD_NAME));
    
        _preInitClosure = ibc;
        
    
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#entryPointTakesArguments()
     */
    public boolean entryPointTakesArguments() {
        return _vmc.haveMainWithStringArgs();
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#getEntryPoint()
     */
    public String getEntryPoint() {
        IFunction mainFunc;
        if( _vmc.haveMainWithStringArgs() ){
            mainFunc = _vmc.getMainWithStringArgs();
        }else if( _vmc.haveMainNoArgs() ){
            mainFunc = _vmc.getMainWithNoArgs();
        }else{
            throw KenyaPreconditionError.get();
        }
        return _ismim.lookupFunction(mainFunc);
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#getEntryPointClass()
     */
    public IInterpretedClass getEntryPointClass() {
        return _ismim.getGlobalClassBody();
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#lookupClass(java.lang.String)
     */
    public IClass lookupClass(String name) {
        IClass cls = (IClass) _ismim._classMap.lookupLHS(name);
        
        if( cls == null ){
            return null;
        } else {
            return (IClass) _ismim._normClassMap.lookupRHS(cls);
        }
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#lookupEnum(java.lang.String)
     */
    public IClass lookupEnum(String name) {
        IClass enu = (IClass) _ismim._enumMap.lookupLHS(name);
        
        if( enu == null ){
            return null;
        } else {
            return (IClass) _ismim._normEnumMap.lookupRHS(enu);
        }
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#lookupFunction(java.lang.String)
     */
    public IFunction lookupFunction(String name) {
        IFunction iff = (IFunction) _ismim._funcMap.lookupLHS(name);
        if( iff == null ){
            return null;
        } else {
            return (IFunction) _ismim._normFuncMap.lookupRHS(iff);
        }
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#lookupVariable(java.lang.String)
     */
    public IVariable lookupVariable(String name) {
        IVariable ivar = (IVariable) _ismim._varMap.lookupLHS(name);
        if( ivar == null ){
            return null;
        } else {
            return (IVariable) _ismim._normVarMap.lookupRHS(ivar);
        }
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#getPreInitClosure()
     */
    public IAtomicClosure getPreInitClosure() {
        return _preInitClosure;
    }
    
    /**
     * @see mediator.stackMachine.IStackMachineInformationProvider#getDiscreteExecutionPointMarkersFromTheAST()
     */
    public List getDiscreteExecutionPointMarkersFromTheAST() {
        return _codePoints;
    }

}
