/*
 * Created on 30-Oct-2004
 */
package mediator.subscription;

/**
 * defines capabilities of a subscriptionservice
 * when defining a service it must take care of
 * providing each subscriber with the data relevant only
 * to their individual subscription
 * 
 * @author Thomas Timbul
 */
public interface ISubscriptionService extends Runnable {
	
	/**
	 * starts this service in a separate thread
	 */
	public void start();
	
	/**
	 * closes this service.
	 * A good server might notify subscribers of its unavailability
	 */
	public void shutdown();
	
	/**
	 * adds a new subscription to the service
	 * @param sub the subscription to add
	 * @return a subscription ID to be used in future references
	 */
	public String addSubscription(ISubscription sub);
	
	/**
	 * retrieves the ISubscription for the given ID
	 * @param id the id which belongs to the ISubscription to be retrieved
	 * @return an ISubscription corresponding to id,
	 *         or null if none exists with this service
	 */
	public ISubscription getSubscription(String id);
	
	/**
	 * cancels a subscription. This subscription will not be
	 * updated subsequently
	 * @param sub the ID of the subscription to cancel
	 */
	public void cancelSubscription(String id);
	
	/**
	 * provides all subscribers with an update by calling the respective
	 * subscriptionUpdate method on each subscriber
	 * This method is meant to ALWAYS execute in asynchronous fashion.
	 * The order in which subscriptions are treated is not necessarily
	 * guaranteed.
	 */
	public void provideSubscriptionUpdate();
	
	/**
	 * provides an update for a single subscription in asynchronous fashion
	 * This means that the calling Thread is not blocked for the execution of
	 * the call. If you require synchronous execution, use the synchronous
	 * method provided.
	 * There is no effect if the ID given is not valid.
	 * @param sub the ID of the subscription to send update for
	 */
	public void provideASyncSubscriptionUpdate(final String id);
	
	/**
	 * provides an update for a single subscription in synchronous fashion
	 * This means that the calling Thread is blocked for the execution of
	 * the call. If you require asynchronous execution, use the asynchronous
	 * method provided.
	 * There is no effect if the ID given is not valid.
	 * @param sub the ID of the subscription to send update for
	 */
	public void provideSyncSubscriptionUpdate(final String id);
}
