/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 04-Aug-2004 by toa02
 *
 */
package mediator.util;

import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeSet;

import kenya.errors.KenyaInternalError;
import kenya.passes.Util;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import kenya.types.KArrayType;
import kenya.types.KBasicType;
import kenya.types.KBoundClassType;
import kenya.types.KClassType;
import kenya.types.KEnumType;
import kenya.types.KFunction;
import kenya.types.KParamType;
import kenya.types.KParamTypeMapper;
import kenya.types.KType;
import kenya.types.KVariable;
import kenya.types.tables.ClassTable;
import kenya.types.tables.FunctionTable;
import kenya.types.tables.SymbolTable;
import kenya.types.tables.TypeTable;
import kenya.types.tables.VariableTable;
import minijava.node.AClassDecDeclaration;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AFunctionApplication;
import minijava.node.AQualifiedName;
import minijava.node.Node;
import minijava.node.Start;

/**
 * @author toa02
 *
 */
public class InformationHolder {
    
    private IClass[] _classes;
    private IClass[] _enums;
    private IVariable[] _consts;
    private IFunction[] _funcs;
 
    private final Start _rootNode;
    private final Map _typeMap;	// String name to IClass instance
    private final Map _funcMap;	// Node to KFunction of nodes.
    
    private final Map _ktMap;		// Node to KType map ( this is gonna start breaking! )
    
    private final Set _arrayLengths; //Set of references to array.length
    private final Set _enumChildSet; //Set of references to enumeration children
    
    public InformationHolder(ClassTable types, FunctionTable functions, VariableTable consts, 
            Set enumChildSet, Set arrayLengths, Map funcMap, Map ktMap, Start rootNode){
        _typeMap = new HashMap();
        _rootNode = rootNode;
        _funcMap = funcMap;
        _ktMap = ktMap;
        
        _arrayLengths = arrayLengths;
        _enumChildSet = enumChildSet;
        doBasics();
        doEnums(types);
        doClasses(types);
        doFunctions( functions );
        doConstants( consts );

    }
    
    public IClass[] getClasses(){
        return _classes;
    }
    
    public IClass getTypeFromName(String name){
        if( _typeMap.containsKey(name)){
            return (IClass) _typeMap.get(name);    
        }else{
            throw new NoSuchElementException(name);
        }
    }
    
    public IVariable getConstantByName(String name){
        for(int i = 0 ; i < _consts.length ; i++ ){
            if( _consts[i].getName().equals(name)){
                return _consts[i];
            }
        }
        
        throw new NoSuchElementException("Cannot find that constant");
    }
    
    public IClass[] getEnums(){
        return _enums;
    }
    
    public IVariable[] getConstants(){
        return _consts;
    }
    
    public IFunction[] getFunctions(){
        return _funcs;
    }

    public Start getRootNode(){
        return _rootNode;
    }
    
    public IClass getVoid(){
        return (IClass) _typeMap.get("void");
    }
    
    /* lets completely break the encapsulation line... */
    public KFunction getFunction(AFunctionApplication func){
        return (KFunction) _funcMap.get(func);
    }
    
    public IClass getTypeFromNode(Node n, IClass[] typeParams){
        KType kt = (KType) _ktMap.get(n);
        return getType(kt, paramArrayToMap(typeParams));
    }
    
    public boolean isArrayLengthReference(AQualifiedName node){
        return _arrayLengths.contains(node);
    }
    
    public boolean isEnumChildNode(Node node){
        return _enumChildSet.contains(node);
    }
    
    public boolean hasEnum(String name){
        for(int i = 0 ; i < _enums.length ; i++ ){
            if( _enums[i].getName().equals(name)){
                return true;
            }
        }
        return false;
    }
    
    private void doBasics(){
		_typeMap.put("void",new AClassInstance("void",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("boolean",new AClassInstance("boolean",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("char",new AClassInstance("char",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("int",new AClassInstance("int",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("double",new AClassInstance("double",new IClass[0],(ISourceCodeLocation) null,AClassInstance.A_BASIC ));
		
		_typeMap.put("Boolean",new AClassInstance("Boolean",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("Character",new AClassInstance("Character",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("Integer",new AClassInstance("Integer",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
		_typeMap.put("Double",new AClassInstance("Double",new IClass[0],(ISourceCodeLocation) null,AClassInstance.A_BASIC ));
		
		
		_typeMap.put("String",new AClassInstance("String",new IClass[0],(ISourceCodeLocation) null, AClassInstance.A_BASIC ));
    }
    
    
    private void doConstants(VariableTable consts){
        Map m  = consts.getConstants();
        List l = new  LinkedList();
        
        Iterator it = m.keySet().iterator();
        
        while(it.hasNext()){
           
            KVariable kv = (KVariable) it.next();
            
            IVariable ivar = getVariable(kv, new HashMap());

            l.add(ivar);
            
        }
        
        _consts = (IVariable[]) l.toArray(new IVariable[0]);
    }
    
    private void doFunctions( FunctionTable functions ){
        Map m = functions.getBackingMap();
        
        TreeSet s = new TreeSet( m.keySet());
        
        LinkedList l = new LinkedList();
        
        Iterator it = s.iterator();
        
        while( it.hasNext()){
            String name = (String) it.next();
            
            Set fns = (Set) m.get(name);

            Iterator fnsIt = fns.iterator();
            
            while( fnsIt.hasNext() ){
                KFunction kf = (KFunction) fnsIt.next();
            
                if( kf.isBuiltin() ){ continue; }
                
                Set typePSet = new HashSet();
                
                Iterator ity = kf.getTypeParamMap().keySet().iterator();
                while( ity.hasNext() ){
                    KParamType kpt = (KParamType) kf.getTypeParamMap().get(ity.next());
                    int[] pos = Util.getFirstIdent(kpt.getNode());
                    ISourceCodeLocation scl = new SourceCodeLocation(pos[0], pos[1], pos[2]);
                    IClass pt = new AClassInstance(kpt.getName(),new IClass[0],scl, AClassInstance.A_PARAM);
                    typePSet.add(pt);
                }
                
                IClass[] typeParams = (IClass[]) typePSet.toArray(new IClass[0]);
                // String name = name
                Map paramMap = paramArrayToMap(typeParams);
                IClass returnType = getType(kf.getReturnType(), paramMap);
                
                List vars = kf.getVars();
                Iterator varIt = vars.iterator();
                
                IVariable[] arguments = new IVariable[vars.size()];
                int i = 0;
                while(varIt.hasNext()){
                    arguments[i] = getVariable((KVariable)varIt.next(), paramMap);
                    i++;
                }
                
                int[] pos = Util.getFirstIdent(kf.getNode().getIdentifier());
                ISourceCodeLocation position = new SourceCodeLocation( pos[0], pos[1], pos[2]);
                
                AFunctionInstance afi = new AFunctionInstance( typeParams,
                        name, returnType,arguments,position,kf.getNode());
                l.add(afi);
                
            }
            
        }
        
        _funcs = (IFunction[]) l.toArray(new IFunction[0]);
    }
    
    private void doEnums(ClassTable types){
        Iterator it = types.getEnumIterator();
        
        Set enumSet = new TreeSet(new IClassNameSorter());
        
        while(it.hasNext()){
            String enumName = (String) it.next();
            
            KEnumType ket = (KEnumType) types.getType(enumName);
            
            IClass[] paramList = new IClass[0];
            
            int[] pos = Util.getFirstIdent( ((AEnumDecDeclaration)types.getNode(enumName)).getIdentifier());
            
            String[] kids = ket.getChildren();
            SourceCodeLocation scl = new SourceCodeLocation( pos[0], pos[1], pos[2]);
            AClassInstance aci = new AClassInstance(enumName,paramList,scl, AClassInstance.AN_ENUM );
            aci.setEnumConsts(kids);
            _typeMap.put( enumName, aci );
            enumSet.add(aci);
        }
     
        _enums = (IClass[]) enumSet.toArray(new IClass[0]);
    }    
    
    private void doClasses(ClassTable types){
        // 2 stage mehod:
        // 1: load up all the classes.
        // 2: do all the variables in each class.
        Iterator it;
        
        it = types.getClassIterator();
        
        Set classSet = new TreeSet(new IClassNameSorter());
        
        while(it.hasNext()){
            String className = (String) it.next();
            
            if( types.getNode(className) == null ){	// don't process the basic class types
                continue;
            }
            
            KClassType kct = (KClassType) types.getType(className);
            
            List params = kct.getTypeParamList();
            Iterator pit = params.iterator();
            IClass[] paramList = new IClass[params.size()];
            int i = 0;
            while( pit.hasNext() ){
                KParamType kpt = (KParamType) pit.next(); 
                
                int[] pos = Util.getFirstIdent( kpt.getNode() );
                SourceCodeLocation scl = new SourceCodeLocation(pos[0],pos[1],pos[2]);
                paramList[i] = new AClassInstance(kpt.getName(),new IClass[0],scl,AClassInstance.A_PARAM );
                i++;
            }
            
            int[] pos = Util.getFirstIdent( ((AClassDecDeclaration)types.getNode(className)).getIdentifier());
            SourceCodeLocation scl = new SourceCodeLocation( pos[0], pos[1], pos[2]);
            AClassInstance aci = new AClassInstance(className,paramList,scl, AClassInstance.A_CLASS );
            _typeMap.put( className, aci );
            classSet.add(aci);
        }
        
        _classes = (IClass[]) classSet.toArray( new IClass[0] );
        
        // now go back and do the declarations:
        
        it = types.getClassIterator();
        
        while(it.hasNext()){
            String thisClassName = (String) it.next();

            if( types.getNode(thisClassName) == null ){	// don't process the basic class types
                continue;
            }

            
            
            KClassType kct = (KClassType) types.getType(thisClassName);
            SymbolTable st = kct.getChildren();
        
            AClassInstance ic = (AClassInstance) _typeMap.get(thisClassName);
            ic.setDeclarations( getVariables(st, paramArrayToMap( ic.getTemplateParams() )));
        }
        
    }
    
    
    private Map paramArrayToMap( IClass[] params ){
        
        Map m = new HashMap();
        for(int i = 0 ; i < params.length ; i ++ ){
            m.put( KParamTypeMapper.get().lookupName(params[i].getName()),params[i]);
        }
        
        return m;
    }
    
    
    private IVariable[] getVariables(SymbolTable st, Map params){
        Map m = st.collapse();
        
        IVariable[] iv = new IVariable[ m.keySet().size() ];
        
        Iterator it = m.keySet().iterator();
        int i = 0;
        while( it.hasNext() ){
            final String name = (String) it.next();
            KVariable kv = (KVariable) m.get(name);
            iv[i] = getVariable(kv, params);
            i++;
        }
        return iv;
    }

    private IVariable getVariable( KVariable kv, Map params){

        final String name = kv.getName();
        
        int[] pos = Util.getFirstIdent(kv.getNode());
        final ISourceCodeLocation scl = new SourceCodeLocation( pos[0], pos[1], pos[2]);

        final IClass type = getType( kv.getType(), params );
        
        return new AVariableInstance(name, scl, type );

    }
    
    public IClass getType( KType type, Map params ){

        if( type instanceof KArrayType ){
            KArrayType kat = (KArrayType) type;
            IClass underlying = getType( kat.getChildType(), params);
            
            return new AClassInstance(underlying.getName() + "[]",
                    underlying.getTemplateParams(),
                    underlying, AClassInstance.AN_ARRAY);
            
        }else if( type instanceof KBasicType ){
            return (IClass) _typeMap.get(type.getName());
        
        }else if( type instanceof KBoundClassType ){
            KBoundClassType kat = (KBoundClassType) type;
            
            
            IClass underlying = getType(kat.getBase(), params);
            TypeTable tt = kat.getBindings();
            
            /* By some twisted logic this is Integer, Boolean, etc. */
            if( underlying == null ){
                return (IClass) _typeMap.get(type.getName().toLowerCase());
            }
            
            IClass[] newParams = new IClass[underlying.getTemplateParams().length];
            for(int i = 0 ; i < underlying.getTemplateParams().length ; i++ ){
                newParams[i] = getType(tt.lookup( new KParamType(underlying.getTemplateParams()[i].getName(), false)),params);
            }
            return new AClassInstance(underlying.getName(),newParams,underlying, AClassInstance.A_CLASS);
            
        }else if( type instanceof KClassType ){
            return (IClass) _typeMap.get(type.getName());

        }else if( type instanceof KParamType ){
            return (IClass) params.get(KParamTypeMapper.get().lookupName(type.getName()));

        }else if( type instanceof KEnumType ){
            return (IClass) _typeMap.get(type.getName());
            
        }else{
            throw new KenyaInternalError("Failed precondition with type : " + type);
        }
    }
    
    public boolean areClassBasicTypesUsed(){
        
        KType kt;
        for(Iterator it = _ktMap.entrySet().iterator() ; it.hasNext() ; ){
            
            kt = (KType) ((Map.Entry) it.next()).getValue();
            if( kt instanceof KBoundClassType){
                KBoundClassType kbct = (KBoundClassType) kt;
                
                if( 	kbct.getBase() == KClassType.getCharacter() ||
                        kbct.getBase() == KClassType.getDouble() ||
                        kbct.getBase() == KClassType.getInteger() ||
                        kbct.getBase() == KClassType.getBoolean()){
                    return true;
                }
                        
            }
            
        }
        
        return false;
        
    }
}

class IClassNameSorter implements Comparator{
    
    /**
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    public int compare(Object arg0, Object arg1) {
        IClass lhs = (IClass) arg0;
        IClass rhs = (IClass) arg1;
        return lhs.getName().compareTo(rhs.getName());
    }
    
}