/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package org.wellquite.kenya.stackMachine;

import java.util.Map;
import java.util.TreeMap;

import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * Instances of this class represent Classes in the stack machine. This class
 * should be used to detail classes which you then invoke methods on,
 * instantiate etc. The correct way to get a class onto the stack is to use the
 * closure:
 * StackOpsFactory.push(ClassTypeFactory.createClassStaticType([instance of this
 * class]);
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class InterpretedClass implements IInterpretedClass {

    private final Map staticMutableVariables = new TreeMap();

    private final Map staticImmutableVariables = new TreeMap();

    private final Map instanceMutableVariables = new TreeMap();

    private final Map instanceImmutableVariables = new TreeMap();

    private final Map staticMethods = new TreeMap();

    private final Map instanceMethods = new TreeMap();

    private String name = null;

    /**
     * Creates an InterpretedClass instance with the supplied class name.
     * 
     * @param name
     *            The name of the class.
     */
    public InterpretedClass(String name) {
        this.name = name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#addStaticMutableVariable(org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void addMutableStaticVariable(String name,
            IType variable) {
        staticMutableVariables.put(name, variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#addStaticImmutableVariable(org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void addImmutableStaticVariable(String name,
            IType variable) {
        staticImmutableVariables.put(name, variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#addMutableInstanceVariable(org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void addMutableInstanceVariable(String name,
            IType variable) {
        instanceMutableVariables.put(name, variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#addImmutableInstanceVariable(org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void addImmutableInstanceVariable(String name,
            IType variable) {
        instanceImmutableVariables.put(name, variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#addStaticMethod(java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IMethod)
     */
    public synchronized void addStaticMethod(IInterpretedMethod method) {
        if (!method.isStatic())
            throw new RuntimeException(
                    "Attempt to add non-static method to class as static method in class "
                            + name + ": " + method);
        staticMethods.put(method.getName(), method);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#addInstanceMethod(java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IMethod)
     */
    public synchronized void addInstanceMethod(IInterpretedMethod method) {
        instanceMethods.put(method.getName(), method);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#getStaticMethod(java.lang.String)
     */
    public synchronized IInterpretedMethod getMethod(String methodName) {
        if (staticMethods.containsKey(methodName)) {
            return (IInterpretedMethod) staticMethods.get(methodName);
        }

        if (instanceMethods.containsKey(methodName)) {
            throw new RuntimeException(
                    "Attempt made to access non-static method from static context. "
                            + methodName);
        }

        throw new RuntimeException("Method '" + methodName
                + "' not found in class " + this.name);

    }

    public synchronized IInterpretedClassInstance createInstance() {
        return new InterpretedClassInstance(this, instanceMethods,
                instanceImmutableVariables, instanceMutableVariables);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#setName(java.lang.String)
     */
    public synchronized void setName(String name) {
        if (this.name != null)
            throw new RuntimeException(
                    "Attempt made to redefine InterpretedClass's name: "
                            + this.name);
        this.name = name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#getName()
     */
    public synchronized String getName() {
        return name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    public synchronized String toString() {
        return "InterpretedClass: " + name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#getStaticMutableVariables()
     */
    public synchronized Map getMutableStaticVariables() {
        return staticMutableVariables;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass#getStaticImmutableVariables()
     */
    public synchronized Map getImmutableStaticVariables() {
        return staticImmutableVariables;
    }

    public boolean equals(Object obj) {
        return obj == this;
    }

    public int hashCode() {
        return 0;
    }
}