/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package org.wellquite.kenya.stackMachine;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;

/**
 * Instances of this class represent class instances. These are the values that
 * the IClassInstanceType type takes. The correct way to get a class instance
 * onto the stack is to use the closure:
 * StackOpsFactory.push(ClassTypeFactory.createClassInstanceType([instance of
 * this class]);
 * <p>
 * You should never create instances of this class directly, instead take your
 * instance of IInterpretedClass and invoke createInstance() on it.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class InterpretedClassInstance implements IInterpretedClassInstance {

    private final Map methods;

    private final Map mutableVariables;

    private final Map immutableVariables;

    private final IInterpretedClass myClass;

    /**
     * Creates a new InterpretedClassInstance instance. The constructor takes
     * copies of both the immutable variables and the mutable variables, thus
     * changes to the registered mutable and immutable instance variables in the
     * IInterpretedClass instance representing the class of this instance are
     * not reflected in the class instance. And if you understood that sentence
     * then you're doing well.
     * 
     * @param myClass
     *            The class of which this instance represents an instance of.
     * @param instanceMethods
     *            The instance methods this class contains.
     * @param instanceImmutableVariables
     *            The immutable variables this class instance contains.
     * @param instanceMutableVariables
     *            The mutable variables this class instance contains.
     */
    public InterpretedClassInstance(IInterpretedClass myClass,
            Map instanceMethods, Map instanceImmutableVariables,
            Map instanceMutableVariables) {
        this.myClass = myClass;
        methods = Collections.unmodifiableMap(instanceMethods);
        immutableVariables = Collections.unmodifiableMap(new HashMap(
                instanceImmutableVariables));
        mutableVariables = new HashMap(instanceMutableVariables);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance#getInterpretedClass()
     */
    public synchronized IInterpretedClass getInterpretedClass() {
        return myClass;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance#getMethod(java.lang.String)
     */
    public synchronized IInterpretedMethod getMethod(String methodName) {
        if (methods.containsKey(methodName)) {
            return (IInterpretedMethod) methods.get(methodName);
        }
        return myClass.getMethod(methodName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    public synchronized String toString() {
        return "InterpretedClassInstance of: "
                + getInterpretedClass().getName();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance#getMutableInstanceVariables()
     */
    public synchronized Map getMutableInstanceVariables() {
        return mutableVariables;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance#getImmutableInstanceVariables()
     */
    public synchronized Map getImmutableInstanceVariables() {
        return immutableVariables;
    }

    public boolean equals(Object obj) {
        return obj == this;
    }

    public int hashCode() {
        return 0;
    }
}