/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004
 */
package org.wellquite.kenya.stackMachine;

import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import kenya.builtIns.BuiltInMethodsLoader;
import kenya.builtIns.IBuiltInMethod;
import kenya.builtIns.SMLibraryContainer;

import org.wellquite.kenya.stackMachine.ops.ArrayOpsFactory;
import org.wellquite.kenya.stackMachine.ops.ControlFlowOpsFactory;
import org.wellquite.kenya.stackMachine.ops.LogicalOpsFactory;
import org.wellquite.kenya.stackMachine.ops.NumericOpsFactory;
import org.wellquite.kenya.stackMachine.ops.StackOpsFactory;
import org.wellquite.kenya.stackMachine.types.ArrayTypeFactory;
import org.wellquite.kenya.stackMachine.types.ClassTypeFactory;
import org.wellquite.kenya.stackMachine.types.EnumTypeFactory;
import org.wellquite.kenya.stackMachine.types.IntType;
import org.wellquite.kenya.stackMachine.types.PrimitiveTypeFactory;
import org.wellquite.kenya.stackMachine.types.StringTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IBuildableClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IClassInstanceType;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedEnumeration;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * This class contains a main and various tests which demonstrate the stack
 * machine in glorified battle.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class StackMachineTest {

    private static Set builtInMethods = BuiltInMethodsLoader
            .getBuiltInMethods();

    private static Map methods = new TreeMap();

    /**
     * Call which ever test methods you want in here.
     * 
     * @param args
     */
    public static void main(String[] args) {
        Iterator it = builtInMethods.iterator();
        while (it.hasNext()) {
            IBuiltInMethod method = (IBuiltInMethod) it.next();
            methods.put(method.getName(), method);
        }

        equalityTest();

        mathTest();

        test1();
        test2();

        long totalTime = 0;
        for (int idx = 0; idx < 3; idx++) {
            Date start = new Date();
            primeTest();
            Date end = new Date();
            totalTime += end.getTime() - start.getTime();
        }
        System.out.println(totalTime / 3);

    }

    /**
     * This test finds primes.
     *  
     */
    private static void primeTest() {
        final IInterpretedClass primeClass = new InterpretedClass("prime");

        // isPrime method. Takes 1 arg (the number to test)
        /*
         * The plan: public void isPrime(int testNumber) { double target =
         * Math.ceil(Math.sqrt(testNumber)); int counter = 2; while (target >
         * counter && prime) { if (testNumber % counter == 0) { return false; }
         * else { counter++; } } return true; }
         */
        final IInterpretedMethod isPrime = new InterpretedMethod("isPrime",
                primeClass, true, true); // isStatic, with return type

        IBuildableClosure methodBody = new InterpretedBuildableClosure();
        methodBody.addClosure(StackOpsFactory.storeNewVariable("testNumber",
                false));
        methodBody.addClosure(StackOpsFactory.fetch("testNumber"));
        methodBody.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("sqrt"))
                                .getInterpretedMethod()));
        methodBody.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("ceil"))
                                .getInterpretedMethod()));
        methodBody
                .addClosure(StackOpsFactory.storeNewVariable("target", false));

        methodBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(2)));
        methodBody
                .addClosure(StackOpsFactory.storeNewVariable("counter", true));

        methodBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(true)));
        methodBody.addClosure(StackOpsFactory.storeNewVariable("prime", true));

        IBuildableClosure whileCond = new InterpretedBuildableClosure();
        IBuildableClosure leftAnd = new InterpretedBuildableClosure();
        leftAnd.addClosure(StackOpsFactory.fetch("counter"));
        leftAnd.addClosure(StackOpsFactory.fetch("target"));
        leftAnd.addClosure(LogicalOpsFactory.greaterThanEqual());
        IBuildableClosure rightAnd = new InterpretedBuildableClosure();
        rightAnd.addClosure(StackOpsFactory.fetch("prime"));
        whileCond.addClosure(LogicalOpsFactory.and(leftAnd, rightAnd));

        IBuildableClosure whileBody = new InterpretedBuildableClosure();

        IBuildableClosure ifCond = new InterpretedBuildableClosure();
        ifCond.addClosure(StackOpsFactory.fetch("counter"));
        ifCond.addClosure(StackOpsFactory.fetch("testNumber"));
        ifCond.addClosure(NumericOpsFactory.modulus());
        ifCond.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(0)));
        ifCond.addClosure(LogicalOpsFactory.equal());

        IBuildableClosure ifTrue = new InterpretedBuildableClosure();
        ifTrue.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(false)));
        ifTrue.addClosure(methodBody.returnFromThisClosure());

        IBuildableClosure ifFalse = new InterpretedBuildableClosure();
        ifFalse.addClosure(StackOpsFactory.fetch("counter"));
        ifFalse.addClosure(NumericOpsFactory.inc());
        ifFalse.addClosure(StackOpsFactory.store("counter"));

        whileBody.addClosure(ControlFlowOpsFactory.ifClosure(ifCond, ifTrue,
                ifFalse));

        methodBody.addClosure(ControlFlowOpsFactory.whileClosure(whileCond,
                whileBody));

        methodBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(true)));
        methodBody.addClosure(methodBody.returnFromThisClosure());

        isPrime.setMethodBody(methodBody);

        final IInterpretedMethod main = new InterpretedMethod("main",
                primeClass, true, false); // isStatic, no returnType
        /*
         * General idea: int current = 0; while (current < 10000) { if
         * (primeClass.isPrime(current)) { println(current); } }
         */
        methodBody = new InterpretedBuildableClosure();
        methodBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(0)));
        methodBody
                .addClosure(StackOpsFactory.storeNewVariable("current", true));

        methodBody.addClosure(StackOpsFactory.push(ClassTypeFactory
                .createClassStaticType(primeClass)));
        methodBody.addClosure(StackOpsFactory.storeNewVariable("primeClass",
                false));

        whileCond = new InterpretedBuildableClosure();
        whileCond.addClosure(StackOpsFactory.fetch("current"));
        whileCond.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(20000)));
        whileCond.addClosure(LogicalOpsFactory.notEqual());

        whileBody = new InterpretedBuildableClosure();
        whileBody.addClosure(StackOpsFactory.fetch("current"));

        ifCond = new InterpretedBuildableClosure();
        ifCond.addClosure(StackOpsFactory.fetch("primeClass"));
        ifCond.addClosure(StackOpsFactory.invokeForeignStaticMethod("isPrime"));
        ifTrue = new InterpretedBuildableClosure();
        ifTrue.addClosure(StackOpsFactory.fetch("current"));
        ifTrue.addClosure(SMLibraryContainer.getLibrary().invokeBuiltInMethod(
                ((IBuiltInMethod) methods.get("println"))
                        .getInterpretedMethod()));

        whileBody.addClosure(ControlFlowOpsFactory.ifClosure(ifCond, ifTrue,
                null));

        whileBody.addClosure(NumericOpsFactory.inc());
        whileBody.addClosure(StackOpsFactory.store("current"));

        methodBody.addClosure(ControlFlowOpsFactory.whileClosure(whileCond,
                whileBody));

        main.setMethodBody(methodBody);

        primeClass.addStaticMethod(main);
        primeClass.addStaticMethod(isPrime);

        new StackMachine().invokeMethod(primeClass, "main");

    }

    /**
     * This test demonstrates class variables class Pair { int x = 5; int y = 5;
     * static int z = 0; }
     * 
     * class myClass { void add(Pair pair) { boolean result = pair.y > pair.x;
     * pair.z = result; // no error! z is now boolean (still static) }
     * 
     * static void main() { println("Main running"); add(new Pair());
     * println("Main finishing"); } }
     */
    private static void test1() {
        final IInterpretedClass pairClass = new InterpretedClass("pair");
        pairClass.addMutableInstanceVariable("x", PrimitiveTypeFactory
                .createPrimitiveType(5));
        pairClass.addMutableInstanceVariable("y", PrimitiveTypeFactory
                .createPrimitiveType(6));
        pairClass.addMutableStaticVariable("z", PrimitiveTypeFactory
                .createPrimitiveType(0));

        final IInterpretedClass myClass = new InterpretedClass("myClass");

        IInterpretedMethod add = new InterpretedMethod("add", myClass, false,
                false);
        IBuildableClosure irmb = new InterpretedBuildableClosure();
        irmb.addClosure(StackOpsFactory.storeNewVariable("pair", false));
        irmb.addClosure(StackOpsFactory.fetch("pair"));
        irmb.addClosure(StackOpsFactory.fetchFromClassInstance("x"));
        irmb.addClosure(StackOpsFactory.fetch("pair"));
        irmb.addClosure(StackOpsFactory.fetchFromClassInstance("y"));
        irmb.addClosure(LogicalOpsFactory.greaterThan());
        irmb.addClosure(StackOpsFactory.storeNewVariable("result", false));
        irmb.addClosure(StackOpsFactory.fetch("pair"));
        irmb.addClosure(StackOpsFactory.fetch("result"));
        irmb.addClosure(StackOpsFactory.storeInClassInstance("z"));
        irmb.addClosure(StackOpsFactory.printlnStackPeek());
        add.setMethodBody(irmb);

        myClass.addInstanceMethod(add);

        IInterpretedMethod main = new InterpretedMethod("main", myClass, true,
                false);
        irmb = new InterpretedBuildableClosure();
        irmb.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("Main running.")));
        irmb.addClosure(StackOpsFactory.printlnStackPeek());
        irmb.addClosure(StackOpsFactory.push(ClassTypeFactory
                .createClassInstanceType(pairClass.createInstance())));
        irmb.addClosure(StackOpsFactory.push(ClassTypeFactory
                .createClassInstanceType(myClass.createInstance())));
        irmb.addClosure(StackOpsFactory.invokeForeignInstanceMethod("add"));
        irmb.addClosure(StackOpsFactory.printlnStackPeek());
        irmb.addClosure(StackOpsFactory.fetchFromClassInstance("z"));
        irmb.addClosure(StackOpsFactory.printlnStackPeek());
        irmb.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("Main finishing.")));
        irmb.addClosure(StackOpsFactory.printlnStackPeek());

        main.setMethodBody(irmb);

        myClass.addStaticMethod(main);

        new StackMachine().invokeMethod(myClass, "main");
    }

    /**
     * This test demonstrates arrays. class fooClass { } class myClass ( static
     * void main() { myClass [] ary = {new myClass(), new myClass(), new
     * myClass()}; println(ary); ary[0] = new myClass(); println(ary); } }
     */
    private static void test2() {
        final IInterpretedClass myClass = new InterpretedClass("myClass");
        final IInterpretedMethod main = new InterpretedMethod("main", myClass,
                true, false);
        final IBuildableClosure mainBody = new InterpretedBuildableClosure();
        mainBody.addClosure(StackOpsFactory.push(ArrayTypeFactory
                .createArrayType(new IClassInstanceType[] {
                        ClassTypeFactory.createClassInstanceType(myClass
                                .createInstance()),
                        ClassTypeFactory.createClassInstanceType(myClass
                                .createInstance()),
                        ClassTypeFactory.createClassInstanceType(myClass
                                .createInstance()) }, myClass.getName())));
        mainBody.addClosure(StackOpsFactory.storeNewVariable("ary", false));
        mainBody.addClosure(StackOpsFactory.fetch("ary"));
        mainBody.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(StackOpsFactory.push(ClassTypeFactory
                .createClassInstanceType(myClass.createInstance())));
        mainBody.addClosure(StackOpsFactory.fetch("ary"));
        mainBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(0)));

        mainBody.addClosure(ArrayOpsFactory.set());
        mainBody.addClosure(StackOpsFactory.fetch("ary"));
        mainBody.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(3)));
        mainBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(2)));
        mainBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(4)));
        mainBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(3)));
        mainBody.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('M')));
        mainBody.addClosure(ArrayOpsFactory.buildMultidimensionalArray());
        mainBody.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        main.setMethodBody(mainBody);

        myClass.addStaticMethod(main);
        new StackMachine().invokeMethod(myClass, "main");
    }

    private static void equalityTest() {
        final IInterpretedClass classA = new InterpretedClass("classA");
        final IInterpretedClass mainClass = new InterpretedClass("mainClass");

        final IInterpretedMethod main = new InterpretedMethod("main",
                mainClass, true, false);
        final IBuildableClosure mainBody = new InterpretedBuildableClosure();

        IBuildableClosure condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(65)));
        condition.addClosure(LogicalOpsFactory.equal());
        IBuildableClosure trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("A == 65")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(65)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("65 == A")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("5 == 5")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5.0)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5.0)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("5.0 == 5.0")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(false)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(false)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("false == false")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(65.0)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("65.0 == A")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(65.0)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("A == 65.0")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("A")));
        condition.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("A")));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("\"A\" == \"A\"")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        IType wibble = ClassTypeFactory.createClassInstanceType(classA
                .createInstance());
        condition.addClosure(StackOpsFactory.push(wibble));
        condition.addClosure(StackOpsFactory.push(wibble));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("" + wibble + " == " + wibble)));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        IntType one = PrimitiveTypeFactory.createPrimitiveType(1);
        IntType two = PrimitiveTypeFactory.createPrimitiveType(2);
        IntType three = PrimitiveTypeFactory.createPrimitiveType(3);
        IType ary1 = ArrayTypeFactory.createArrayType(new IntType[] { one, two,
                three }, one.getTypeName());
        condition.addClosure(StackOpsFactory.push(ary1));
        condition.addClosure(StackOpsFactory.push(ary1));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("" + ary1 + " == " + ary1)));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        IType ary2 = ArrayTypeFactory.createArrayType(new IntType[] { one, two,
                three }, one.getTypeName());
        condition.addClosure(StackOpsFactory.push(ary1));
        condition.addClosure(StackOpsFactory.push(ary2));
        condition.addClosure(LogicalOpsFactory.notEqual());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("" + ary1 + " != " + ary2)));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        IInterpretedEnumeration enumValue = new InterpretedEnumeration(
                "numbers", new Object[] { one, two, three });
        IType enumType1 = EnumTypeFactory.createEnumType(enumValue, two);
        condition.addClosure(StackOpsFactory.push(enumType1));
        condition.addClosure(StackOpsFactory.push(enumType1));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("" + enumType1 + " == " + enumType1)));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        IType enumType2 = EnumTypeFactory.createEnumType(enumValue, two);
        condition.addClosure(StackOpsFactory.push(enumType1));
        condition.addClosure(StackOpsFactory.push(enumType2));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("" + enumType1 + " == " + enumType2)));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        main.setMethodBody(mainBody);
        mainClass.addStaticMethod(main);

        new StackMachine().invokeMethod(mainClass, "main");
    }

    private static void mathTest() {
        final IInterpretedClass mainClass = new InterpretedClass("mainClass");

        final IInterpretedMethod main = new InterpretedMethod("main",
                mainClass, true, false);
        final IBuildableClosure mainBody = new InterpretedBuildableClosure();

        IBuildableClosure condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(10)));
        condition.addClosure(LogicalOpsFactory.equal());
        IBuildableClosure trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("5 + 5 == 10")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(10)));
        condition.addClosure(NumericOpsFactory.subtract());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("10 - 5 == 5")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(2)));
        condition.addClosure(NumericOpsFactory.multiply());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(10)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("5 * 2 == 10")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(2)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(10)));
        condition.addClosure(NumericOpsFactory.divide());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("10 / 2 == 5")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(2)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(5)));
        condition.addClosure(NumericOpsFactory.modulus());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("5 % 2 == 1")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(66)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("'A' + 1 == 66")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(66.0)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("'A' + 1 == 66.0")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('B')));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("'A' + 1 == 'B'")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(66)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("1 + 'A' == 66")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(66.0)));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("1 + 'A' == 66.0")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        condition = new InterpretedBuildableClosure();
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType(1)));
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('A')));
        condition.addClosure(NumericOpsFactory.add());
        condition.addClosure(StackOpsFactory.push(PrimitiveTypeFactory
                .createPrimitiveType('B')));
        condition.addClosure(LogicalOpsFactory.equal());
        trueBlock = new InterpretedBuildableClosure();
        trueBlock.addClosure(StackOpsFactory.push(StringTypeFactory
                .createStringType("1 + 'A' == 'B'")));
        trueBlock.addClosure(SMLibraryContainer.getLibrary()
                .invokeBuiltInMethod(
                        ((IBuiltInMethod) methods.get("println"))
                                .getInterpretedMethod()));

        mainBody.addClosure(ControlFlowOpsFactory.ifClosure(condition,
                trueBlock, null));

        main.setMethodBody(mainBody);
        mainClass.addStaticMethod(main);

        new StackMachine().invokeMethod(mainClass, "main");
    }
}