/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004
 */
package org.wellquite.kenya.stackMachine.misc;

import java.util.ArrayList;
import java.util.List;

/**
 * This class deals with dispatching threads to deal with jobs as they come in.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class JobDispatch {

    private static final int MAX_IDLE_THREAD_COUNT = 10;

    private static final int PRESPAWN_THREAD_COUNT = 5;

    private static boolean shutDown = false;

    private static final List runningThreads = new ArrayList();

    private static final Object lock = new Object();

    private static final List outStandingJobs = new ArrayList();

    static {
        for (int idx = 0; idx < PRESPAWN_THREAD_COUNT; idx++)
            spawnThread();
    }

    /**
     * Enqueue a job. The job will not be run in the current thread. The job
     * will be run with no locks held when the job execution begins. There is no
     * guarentee on the ordering of job execution ie, a job "b" enqueued after
     * job "a" may execute concurrently with job "a".
     * 
     * @param job
     *            The Job to run.
     */
    public static void enqueueJob(AbstractJob job) {
        if (kenya.KenyaCore.DEBUG_OUTPUT) {
            System.out.println("Enqueuing");
        }
        synchronized (lock) {
            outStandingJobs.add(job);
            while (getIdleThreadCount() < outStandingJobs.size())
                spawnThread();
            lock.notifyAll();
        }
    }

    /**
     * Enqueue a job. The job will not be run in the current thread. The job
     * will be run with no locks held when the job execution begins. There is no
     * guarentee on the ordering of job execution ie, a job "b" enqueued after
     * job "a" may execute concurrently with job "a".
     * <p>
     * The calling thread is blocked until the Job is complete.
     * 
     * @param job
     *            The Job to run.
     */
    public static void enqueueJobAndWaitForFinish(AbstractJob job) {
        if (kenya.KenyaCore.DEBUG_OUTPUT) {
            System.out.println("Enqueuing");
        }
        synchronized (job) {
            synchronized (lock) {
                outStandingJobs.add(job);
                while (getIdleThreadCount() < outStandingJobs.size())
                    spawnThread();
                lock.notifyAll();
            }
            while (!job.isFinished()) {
                try {
                    job.wait();
                } catch (InterruptedException e) {
                }
            }
        }
    }

    public static void shutDownAllJobs() {
        synchronized (lock) {
            shutDown = true;
            runningThreads.clear();
            lock.notifyAll();
        }
    }

    private static void spawnThread() {
        synchronized (lock) {
            JobRunner jr = new JobRunner(lock, outStandingJobs);
            runningThreads.add(jr);
            new Thread(jr).start();
            while (!jr.isRunning()) {
                try {
                    lock.wait();
                } catch (InterruptedException e) {
                }
            }
        }
    }

    private static int getIdleThreadCount() {
        synchronized (lock) {
            int count = 0;
            for (int idx = 0; idx < runningThreads.size(); idx++) {
                JobRunner jr = (JobRunner) runningThreads.get(idx);
                if (jr.isRunning() && jr.isIdle()) count++;
            }
            return count;
        }
    }

    static boolean isThreadRunning(JobRunner jr) {
        synchronized (lock) {
            if (shutDown)
                return false;
            else if (runningThreads.contains(jr))
                return getIdleThreadCount() <= MAX_IDLE_THREAD_COUNT;
            else
                return false;
        }
    }

    static void threadExiting(JobRunner jr) {
        synchronized (lock) {
            runningThreads.remove(jr);
            if (kenya.KenyaCore.DEBUG_OUTPUT) {
                System.out.println("Thread exited.");
            }
        }
    }
}