/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 07-Jul-2004
 */
package org.wellquite.kenya.stackMachine.ops;

import org.wellquite.kenya.stackMachine.ClosureMiscHelper;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure;
import org.wellquite.kenya.stackMachine.types.IntType;
import org.wellquite.kenya.stackMachine.types.PrimitiveTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IBooleanType;
import org.wellquite.kenya.stackMachine.types.interfaces.ICharType;
import org.wellquite.kenya.stackMachine.types.interfaces.INumericType;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * This class contains static methods which return closures which manipulate the
 * stack providing boolean logical functions. Unless otherwise stated, all
 * methods remove from the stack any arguements they require.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class LogicalOpsFactory {

    private LogicalOpsFactory() {
    }

    private static final IAtomicClosure lessThan = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            double left = 0;
            double right = 0;

            IType arg0 = sm.pop();
            IType arg1 = sm.pop();

            if (arg0 instanceof INumericType)
                left = ((INumericType) arg0).getValue().doubleValue();
            else if (arg0 instanceof ICharType)
                left = ((ICharType) arg0).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform < op on arg: " + arg0);

            if (arg1 instanceof INumericType)
                right = ((INumericType) arg1).getValue().doubleValue();
            else if (arg1 instanceof ICharType)
                right = ((ICharType) arg1).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform < op on arg: " + arg1);

            if (left < right) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            }
        }

        public String toString() {
            return "<";
        }
    };

    private static final IAtomicClosure lessThanEqual = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            double left = 0;
            double right = 0;

            IType arg0 = sm.pop();
            IType arg1 = sm.pop();

            if (arg0 instanceof INumericType)
                left = ((INumericType) arg0).getValue().doubleValue();
            else if (arg0 instanceof ICharType)
                left = ((ICharType) arg0).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform <= op on arg: " + arg0);

            if (arg1 instanceof INumericType)
                right = ((INumericType) arg1).getValue().doubleValue();
            else if (arg1 instanceof ICharType)
                right = ((ICharType) arg1).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform <= op on arg: " + arg1);

            if (left <= right) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            }
        }

        public String toString() {
            return "<=";
        }
    };

    private static final IAtomicClosure greaterThan = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            double left = 0;
            double right = 0;

            IType arg0 = sm.pop();
            IType arg1 = sm.pop();

            if (arg0 instanceof INumericType)
                left = ((INumericType) arg0).getValue().doubleValue();
            else if (arg0 instanceof ICharType)
                left = ((ICharType) arg0).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform > op on arg: " + arg0);

            if (arg1 instanceof INumericType)
                right = ((INumericType) arg1).getValue().doubleValue();
            else if (arg1 instanceof ICharType)
                right = ((ICharType) arg1).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform > op on arg: " + arg1);

            if (left > right) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            }
        }

        public String toString() {
            return ">";
        }
    };

    private static final IAtomicClosure greaterThanEqual = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            double left = 0;
            double right = 0;

            IType arg0 = sm.pop();
            IType arg1 = sm.pop();

            if (arg0 instanceof INumericType)
                left = ((INumericType) arg0).getValue().doubleValue();
            else if (arg0 instanceof ICharType)
                left = ((ICharType) arg0).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform >= op on arg: " + arg0);

            if (arg1 instanceof INumericType)
                right = ((INumericType) arg1).getValue().doubleValue();
            else if (arg1 instanceof ICharType)
                right = ((ICharType) arg1).getValue().charValue();
            else
                throw new IllegalArgumentException(
                        "Can not perform >= op on arg: " + arg1);

            if (left >= right) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            }
        }

        public String toString() {
            return ">=";
        }
    };

    private static final IAtomicClosure equal = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            if (arg0.equals(arg1)) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            }
        }

        public String toString() {
            return "equal";
        }
    };

    private static final IAtomicClosure notEqual = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            if (arg0.equals(arg1)) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            }
        }

        public String toString() {
            return "notEqual";
        }
    };

    private static final IAtomicClosure xor = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {

            IType arg0 = sm.pop();
            IType arg1 = sm.pop();

            if (arg0 instanceof IntType && arg1 instanceof IntType) {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType(((IntType) arg0).getValue()
                                .intValue()
                                ^ ((IntType) arg1).getValue().intValue()));
            } else if (arg0 instanceof IBooleanType
                    && arg1 instanceof IBooleanType) {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType(((IBooleanType) arg0).getValue()
                                .booleanValue()
                                ^ ((IBooleanType) arg1).getValue()
                                        .booleanValue()));
            } else {
                throw new IllegalArgumentException(
                        "Can not perform xor on args: " + arg0 + ", " + arg1);
            }
        }

        public String toString() {
            return "xor";
        }
    };

    private static final IAtomicClosure not = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            boolean arg = ((IBooleanType) sm.pop()).getValue().booleanValue();
            if (arg) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(false));
            } else {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(true));
            }
        }

        public String toString() {
            return "not";
        }
    };

    /**
     * The returned closure takes the top two items on the stack which must both
     * be INumericType instances and computes whether one is smaller than the
     * other, placing an IBooleanType instance representing the result of the
     * comparison on the stack.
     * <p>
     * If arg0 is the item on the top of the stack and arg1 is the next item
     * then the result is arg0 < arg1.
     * 
     * @return
     */
    public static IAtomicClosure lessThan() {
        return lessThan;
    }

    /**
     * The returned closure takes the top two items on the stack which must both
     * be INumericType instances and computes whether one is smaller or equal
     * than the other, placing an IBooleanType instance representing the result
     * of the comparison on the stack.
     * <p>
     * If arg0 is the item on the top of the stack and arg1 is the next item
     * then the result is arg0 <= arg1.
     * 
     * @return
     */
    public static IAtomicClosure lessThanEqual() {
        return lessThanEqual;
    }

    /**
     * The returned closure takes the top two items on the stack which must both
     * be INumericType instances and computes whether one is greater than the
     * other, placing an IBooleanType instance representing the result of the
     * comparison on the stack.
     * <p>
     * If arg0 is the item on the top of the stack and arg1 is the next item
     * then the result is arg0 > arg1.
     * 
     * @return
     */
    public static IAtomicClosure greaterThan() {
        return greaterThan;
    }

    /**
     * The returned closure takes the top two items on the stack which must both
     * be INumericType instances and computes whether one is greater or equal
     * than the other, placing an IBooleanType instance representing the result
     * of the comparison on the stack.
     * <p>
     * If arg0 is the item on the top of the stack and arg1 is the next item
     * then the result is arg0 >= arg1.
     * 
     * @return
     */
    public static IAtomicClosure greaterThanEqual() {
        return greaterThanEqual;
    }

    /**
     * The returned closure takes the top two items on the stack which must both
     * be INumericType instances and computes whether they are equal, placing an
     * IBooleanType instance representing the result of the comparison on the
     * stack.
     * <p>
     * If arg0 is the item on the top of the stack and arg1 is the next item
     * then the result is arg0 == arg1.
     * 
     * @return
     */
    public static IAtomicClosure equal() {
        return equal;
    }

    /**
     * The returned closure takes the top two items on the stack which must both
     * be INumericType instances and computes whether they are equal, placing an
     * IBooleanType instance representing the result of the comparison on the
     * stack.
     * <p>
     * If arg0 is the item on the top of the stack and arg1 is the next item
     * then the result is arg0 != arg1.
     * 
     * @return
     */
    public static IAtomicClosure notEqual() {
        return notEqual;
    }

    /**
     * The returned closure takes the two supplied closures and executes the
     * left closure. If the left closure, once executed leaves a false
     * IBooleanType on the stack then an IBooleanType is placed on the stack
     * with value False. Otherwise, the result is the result of the right
     * closure.
     * 
     * @param left
     *            The left hand side of the and statement to be executed. This
     *            is always executed.
     * @param right
     *            The right hand side of the and statement. This is only
     *            executed if the result of executing the left hand side is
     *            true.
     * 
     * @return
     */
    public static IAtomicClosure and(final IAtomicClosure left,
            final IAtomicClosure right) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {

                ClosureMiscHelper.executeClosureInScope(getClosureScope(), left, sm);
                IBooleanType leftResult = (IBooleanType) sm.pop();

                if (leftResult.getValue().booleanValue()) {
                    ClosureMiscHelper.executeClosureInScope(getClosureScope(), right, sm);
                } else
                    sm.push(leftResult);

            }

            public String toString() {
                return "and";
            }
        };

    }

    /**
     * The returned closure takes the two supplied closures and executes the
     * left closure. If the left closure leaves an IBooleanType on the stack and
     * that have value false then the stack will contain the result of the right
     * closure. Otherwise it will contain an IBooleanType with value true.
     * 
     * @param left
     *            The left hand side of the and statement to be executed. This
     *            is always executed.
     * @param right
     *            The right hand side of the and statement. This is only
     *            executed if the result of executing the left hand side is
     *            false.
     * 
     * @return
     */
    public static IAtomicClosure or(final IAtomicClosure left,
            final IAtomicClosure right) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {

                ClosureMiscHelper.executeClosureInScope(getClosureScope(), left, sm);

                IBooleanType leftResult = (IBooleanType) sm.pop();
                if (leftResult.getValue().booleanValue()) {
                    sm.push(leftResult);
                } else {
                    ClosureMiscHelper.executeClosureInScope(getClosureScope(), right, sm);
                }
            }

            public String toString() {
                return "or";
            }
        };
    }

    /**
     * The returned closure takes the top two items on the stack which must both
     * be IBooleanType instance and places an IBooleanType instance on the stack
     * representing the result of the operation of (arg0 ^ arg1) where arg0 is
     * the item on the top of the stack and arg1 is the item immediately beneath
     * arg0.
     * 
     * @return
     */
    public static IAtomicClosure xor() {
        return xor;
    }

    /**
     * The returned closure takes the top item on the stack which must be an
     * IBooleanType instance and places an IBooleanType instance on the stack
     * representing the result of the operation of (! arg0) where arg0 is the
     * item on the top of the stack.
     * 
     * @return
     */
    public static IAtomicClosure not() {
        return not;
    }
}