/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 07-Jul-2004
 */
package org.wellquite.kenya.stackMachine.ops;

import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure;
import org.wellquite.kenya.stackMachine.types.DoubleType;
import org.wellquite.kenya.stackMachine.types.IntType;
import org.wellquite.kenya.stackMachine.types.PrimitiveTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.ICharType;
import org.wellquite.kenya.stackMachine.types.interfaces.INumericType;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * This class contains static methods which return Closures which do numericy
 * things. Unless otherwise stated, all methods remove from the stack any
 * arguements they require.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class NumericOpsFactory {

    private static final int INTRESULT = 0;

    private static final int DOUBLERESULT = 1;

    private static final int CHARRESULT = 2;

    private NumericOpsFactory() {
    }

    private static final IAtomicClosure add = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            int resultType = INTRESULT;
            int resultI = 0;
            double resultD = 0;

            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    resultI += ((INumericType) arg0).getValue().intValue();
                    resultD += ((INumericType) arg0).getValue().intValue();
                } else if (arg0 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg0).getValue().doubleValue();
                }
            } else if (arg0 instanceof ICharType) {
                resultType = CHARRESULT;
                resultI += ((ICharType) arg0).getValue().charValue();
                resultD += ((ICharType) arg0).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform addition on arg0 from stack: " + arg0);
            }

            if (arg1 instanceof INumericType) {
                if (arg1 instanceof IntType) {
                    resultI += ((INumericType) arg1).getValue().intValue();
                    resultD += ((INumericType) arg1).getValue().intValue();
                } else if (arg1 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg1).getValue().doubleValue();
                }
            } else if (arg1 instanceof ICharType) {
                resultI += ((ICharType) arg1).getValue().charValue();
                resultD += ((ICharType) arg1).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform addition on arg1 from stack: " + arg1);
            }

            if (resultType == INTRESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultI));
            } else if (resultType == DOUBLERESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultD));
            } else {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType((char) resultI));
            }
        }

        public String toString() {
            return "add";
        }
    };

    private static final IAtomicClosure subtract = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            int resultType = INTRESULT;
            int resultI = 0;
            double resultD = 0;

            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    resultI += ((INumericType) arg0).getValue().intValue();
                    resultD += ((INumericType) arg0).getValue().intValue();
                } else if (arg0 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg0).getValue().doubleValue();
                }
            } else if (arg0 instanceof ICharType) {
                resultType = CHARRESULT;
                resultI += ((ICharType) arg0).getValue().charValue();
                resultD += ((ICharType) arg0).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform subtraction on arg0 from stack: " + arg0);
            }

            if (arg1 instanceof INumericType) {
                if (arg1 instanceof IntType) {
                    resultI -= ((INumericType) arg1).getValue().intValue();
                    resultD -= ((INumericType) arg1).getValue().intValue();
                } else if (arg1 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD -= ((INumericType) arg1).getValue().doubleValue();
                }
            } else if (arg1 instanceof ICharType) {
                resultI -= ((ICharType) arg1).getValue().charValue();
                resultD -= ((ICharType) arg1).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform subtraction on arg1 from stack: " + arg1);
            }

            if (resultType == INTRESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultI));
            } else if (resultType == DOUBLERESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultD));
            } else {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType((char) resultI));
            }
        }

        public String toString() {
            return "subtract";
        }
    };

    private static final IAtomicClosure multiply = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            int resultType = INTRESULT;
            int resultI = 0;
            double resultD = 0;

            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    resultI += ((INumericType) arg0).getValue().intValue();
                    resultD += ((INumericType) arg0).getValue().intValue();
                } else if (arg0 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg0).getValue().doubleValue();
                }
            } else if (arg0 instanceof ICharType) {
                resultType = CHARRESULT;
                resultI += ((ICharType) arg0).getValue().charValue();
                resultD += ((ICharType) arg0).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform multiplication on arg0 from stack: " + arg0);
            }

            if (arg1 instanceof INumericType) {
                if (arg1 instanceof IntType) {
                    resultI *= ((INumericType) arg1).getValue().intValue();
                    resultD *= ((INumericType) arg1).getValue().intValue();
                } else if (arg1 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD *= ((INumericType) arg1).getValue().doubleValue();
                }
            } else if (arg1 instanceof ICharType) {
                resultI *= ((ICharType) arg1).getValue().charValue();
                resultD *= ((ICharType) arg1).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform multiplication on arg1 from stack: " + arg1);
            }

            if (resultType == INTRESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultI));
            } else if (resultType == DOUBLERESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultD));
            } else {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType((char) resultI));
            }
        }

        public String toString() {
            return "multiply";
        }
    };

    private static final IAtomicClosure divide = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            int resultType = INTRESULT;
            int resultI = 0;
            double resultD = 0;

            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    resultI += ((INumericType) arg0).getValue().intValue();
                    resultD += ((INumericType) arg0).getValue().intValue();
                } else if (arg0 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg0).getValue().doubleValue();
                }
            } else if (arg0 instanceof ICharType) {
                resultType = CHARRESULT;
                resultI += ((ICharType) arg0).getValue().charValue();
                resultD += ((ICharType) arg0).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform division on arg0 from stack: " + arg0);
            }

            if (arg1 instanceof INumericType) {
                if (arg1 instanceof IntType) {
                    resultI /= ((INumericType) arg1).getValue().intValue();
                    resultD /= ((INumericType) arg1).getValue().intValue();
                } else if (arg1 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD /= ((INumericType) arg1).getValue().doubleValue();
                }
            } else if (arg1 instanceof ICharType) {
                resultI /= ((ICharType) arg1).getValue().charValue();
                resultD /= ((ICharType) arg1).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform division on arg1 from stack: " + arg1);
            }

            if (resultType == INTRESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultI));
            } else if (resultType == DOUBLERESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultD));
            } else {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType((char) resultI));
            }
        }

        public String toString() {
            return "divide";
        }
    };

    private static final IAtomicClosure modulus = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            int resultType = INTRESULT;
            int resultI = 0;
            double resultD = 0;

            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    resultI += ((INumericType) arg0).getValue().intValue();
                    resultD += ((INumericType) arg0).getValue().intValue();
                } else if (arg0 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg0).getValue().doubleValue();
                }
            } else if (arg0 instanceof ICharType) {
                resultType = CHARRESULT;
                resultI += ((ICharType) arg0).getValue().charValue();
                resultD += ((ICharType) arg0).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform modulus on arg0 from stack: " + arg0);
            }

            if (arg1 instanceof INumericType) {
                if (arg1 instanceof IntType) {
                    resultI %= ((INumericType) arg1).getValue().intValue();
                    resultD %= ((INumericType) arg1).getValue().intValue();
                } else if (arg1 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD %= ((INumericType) arg1).getValue().doubleValue();
                }
            } else if (arg1 instanceof ICharType) {
                resultI %= ((ICharType) arg1).getValue().charValue();
                resultD %= ((ICharType) arg1).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform modulus on arg1 from stack: " + arg1);
            }

            if (resultType == INTRESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultI));
            } else if (resultType == DOUBLERESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultD));
            } else {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType((char) resultI));
            }
        }

        public String toString() {
            return "modulus";
        }
    };

    private static final IAtomicClosure inc = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    sm.push(PrimitiveTypeFactory
                            .createPrimitiveType(((INumericType) arg0)
                                    .getValue().intValue() + 1));
                } else {
                    sm.push(PrimitiveTypeFactory
                            .createPrimitiveType(((INumericType) arg0)
                                    .getValue().doubleValue() + 1));
                }
            } else if (arg0 instanceof ICharType) {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType(((ICharType) arg0)
                                .getValue().charValue() + 1));
            } else {
                throw new IllegalArgumentException(
                        "Unable to perform inc operation on arg: " + arg0);
            }
        }

        public String toString() {
            return "inc";
        }
    };

    private static final IAtomicClosure dec = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            IType arg0 = sm.pop();
            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    sm.push(PrimitiveTypeFactory
                            .createPrimitiveType(((INumericType) arg0)
                                    .getValue().intValue() - 1));
                } else {
                    sm.push(PrimitiveTypeFactory
                            .createPrimitiveType(((INumericType) arg0)
                                    .getValue().doubleValue() - 1));
                }
            } else if (arg0 instanceof ICharType) {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType(((ICharType) arg0)
                                .getValue().charValue() - 1));
            } else {
                throw new IllegalArgumentException(
                        "Unable to perform dec operation on arg: " + arg0);
            }
        }

        public String toString() {
            return "dec";
        }
    };

    /**
     * The returned closure adds together the top two items on the stack and
     * place the result on the top of the stack.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguements are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure add() {
        return add;
    }

    /**
     * The returned closure subtracts the top two items on the stack and place
     * the result on the top of the stack. If the top item is arg0 and the next
     * item is arg1 then the result is arg0 - arg1.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguements are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure subtract() {
        return subtract;
    }

    /**
     * The returned closure multiplies together the top two items on the stack
     * and place the result on the top of the stack.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguements are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure multiply() {
        return multiply;
    }

    /**
     * The returned closure divides the top two items on the stack and place the
     * result on the top of the stack. If the top item is arg0 and the next item
     * is arg1 then the result is arg0 / arg1.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguements are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure divide() {
        return divide;
    }

    /**
     * The returned closure finds the remainder after division of the top two
     * items on the stack and place the result on the top of the stack. If the
     * top item is arg0 and the next item is arg1 then the result is arg0 %
     * arg1.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguements are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure modulus() {
        return modulus;
    }

    /**
     * The returned closure increments the top item of the stack. The top item
     * must be an INumbericType instance. The resulting type will be the same
     * type as the arguement.
     * 
     * @return
     */
    public static IAtomicClosure inc() {
        return inc;
    }

    /**
     * The returned closure decrements the top item of the stack. The top item
     * must be an INumbericType instance. The resulting type will be the same
     * type as the arguement.
     * 
     * @return
     */
    public static IAtomicClosure dec() {
        return dec;
    }
}