/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004
 */
package org.wellquite.kenya.stackMachine.ops;

import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.types.AbstractAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IClassInstanceType;
import org.wellquite.kenya.stackMachine.types.interfaces.IClassStaticType;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import org.wellquite.kenya.stackMachine.types.interfaces.INullType;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * This class contains methods which return Closures which affect the stack,
 * store, context and scope. Unless otherwise stated, all methods remove from
 * the stack any arguements they require.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class StackOpsFactory {

    private StackOpsFactory() {

    }

    private static final IAtomicClosure pop = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            sm.pop();
        }

        public String toString() {
            return "pop";
        }
    };

    private static final IAtomicClosure print = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            sm.printStackPeek();
        }

        public String toString() {
            return "print";
        }
    };

    private static final IAtomicClosure println = new AbstractAtomicClosure() {

        public void execute(StackMachine sm) {
            sm.printlnStackPeek();
        }

        public String toString() {
            return "println";
        }
    };

    private static final IAtomicClosure startNewVariableScope = new AbstractAtomicClosure() {
        public void execute(StackMachine sm) {
            sm.getMethodScope().startNewVariableClosure();
        }

        public String toString() {
            return "startNewVariableScope";
        }
    };

    private static final IAtomicClosure endNewVariableScope = new AbstractAtomicClosure() {
        public void execute(StackMachine sm) {
            sm.getMethodScope().endNewVariableClosure();
        }

        public String toString() {
            return "endNewVariableScope";
        }
    };

    /**
     * The returned closure, when executed causes a new local variable scope to
     * be created.
     * 
     * @return
     */
    public static IAtomicClosure startNewVariableScope() {
        return startNewVariableScope;
    }

    /**
     * The returned closure, when executed causes the most recently created
     * variable scope to be deleted. All variables within that scope are
     * deleted.
     * 
     * @return
     */
    public static IAtomicClosure endNewVariableScope() {
        return endNewVariableScope;
    }

    /**
     * The returned closure removes the top item from the stack.
     * 
     * @return
     */
    public static IAtomicClosure pop() {
        return pop;
    }

    /**
     * The returned closure peeks at the top of the stack and prints the top
     * item on the stack. Does not alter the stack contents.
     * 
     * @return
     */
    public static IAtomicClosure printStackPeek() {
        return print;
    }

    /**
     * The returned closure peeks at the top of the stack and printlns the top
     * item on the stack. Does not alter the stack contents.
     * 
     * @return
     */
    public static IAtomicClosure printlnStackPeek() {
        return println;
    }

    /**
     * The returned closure pushs the supplied object onto the top of the stack.
     * This object should be created through one of the TypeFactories.
     * 
     * @param object
     *            The object to place on the top of the stack.
     * @return
     */
    public static IAtomicClosure push(final IType object) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                sm.push(object);
            }

            public String toString() {
                return "push: " + object;
            }
        };
    }

    /**
     * The returned closure fetches the variable stored with the supplied name
     * and push it onto the top of the stack. This method will search the
     * current method's variables, the method's class's instance variables and
     * finally the method's class's static variables.
     * 
     * @param varName
     *            The variable name to search for.
     * @return
     */
    public static IAtomicClosure fetch(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                sm.push(sm.getMethodScope().fetchVariable(varName));
            }

            public String toString() {
                return "fetch: " + varName;
            }
        };
    }

    /**
     * The returned closure fetches the variable stored with the supplied name
     * in static context in the IInterpretedClass value of the IClassStaticType
     * instance on the stack.
     * <p>
     * The top of the stack must contain an IClassStaticType instance.
     * 
     * @param varName
     *            The variable name to search for.
     * @return
     */
    public static IAtomicClosure fetchFromClassStatic(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IInterpretedClass theClass = ((IClassStaticType) sm.pop())
                        .getValue();
                sm.push(sm.getMethodScope().fetchVariable(theClass, varName));
            }

            public String toString() {
                return "fetchFromClassStatic: " + varName;
            }
        };
    }

    /**
     * The returned closure fetches the variable stored with the supplied name
     * in class instance context in the IInterpretedClassInstance value of the
     * IClassInstanceType instance on the stack. This method will first search
     * the class instance's variables and then it will search the class's static
     * variables.
     * <p>
     * The top of the stack must contain an IClassInstanceType instance.
     * 
     * @param varName
     *            The variable name to search for.
     * @return
     */
    public static IAtomicClosure fetchFromClassInstance(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IInterpretedClassInstance theClassInstance = ((IClassInstanceType) sm
                        .pop()).getValue();
                sm.push(sm.getMethodScope().fetchVariable(theClassInstance,
                        varName));
            }

            public String toString() {
                return "fetchFromClassInstance: " + varName;
            }
        };
    }

    /**
     * The returned closure stores a new variable in the current method's set of
     * temporary variables.
     * <p>
     * The item at the top of the stack will be taken off the stack and used for
     * the variable's value.
     * <p>
     * This should be used immediately whenever you enter a new method to store
     * the method's arguements. See the pop() method in StackMachine for the
     * reasons why.
     * 
     * @param varName
     *            The name underwhich to store the variable.
     * @param mutable
     *            Indicates whether this variable can be re-stored.
     * @return
     */
    public static IAtomicClosure storeNewVariable(final String varName,
            final boolean mutable) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType value = sm.pop();
                sm.getMethodScope().storeNewVariable(varName, value, mutable);
            }

            public String toString() {
                return "storeNewVariable: " + varName + ", mutable: " + mutable;
            }
        };
    }

    /**
     * The returned closure updates a variable's value. The variable must
     * already have been stored. It will be found under the name supplied and
     * the value will be updated with the item at the top of the stack. Only
     * variables declared as mutable can have their values' changed.
     * <p>
     * The item at the top of the stack will be taken off the stack and used for
     * the variable's new value.
     * 
     * @param varName
     *            The name of the variable to update.
     * @return
     */
    public static IAtomicClosure store(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType value = sm.pop();
                sm.getMethodScope().updateVariable(varName, value);
            }

            public String toString() {
                return "store: " + varName;
            }
        };
    }

    /**
     * The returned closure updates a variable's value. The variable must
     * already have been stored. It will be found under the name supplied in the
     * the supplied static class context and the value will be updated with the
     * item at the top of the stack. Only variables declared as mutable can have
     * their values' changed.
     * <p>
     * The item at the top of the stack will be taken off the stack and used for
     * the variable's new value. The item on the stack immediately below the
     * variable value must be an IClassStaticType instance, the value of which
     * is the class in which the variable will be searched for.
     * 
     * @param varName
     *            The name of the variable to update.
     * @return
     */
    public static IAtomicClosure storeInClassStatic(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType value = sm.pop();
                IInterpretedClass theClass = ((IClassStaticType) sm.pop())
                        .getValue();
                sm.getMethodScope().updateVariable(theClass, varName, value);
            }

            public String toString() {
                return "storeInClassStatic: " + varName;
            }
        };
    }

    /**
     * The returned closure updates a variable's value. The variable must
     * already have been stored. It will be found under the name supplied in the
     * the supplied class instance context and the value will be updated with
     * the item at the top of the stack. Only variables declared as mutable can
     * have their values' changed.
     * <p>
     * The item at the top of the stack will be taken off the stack and used for
     * the variable's new value. The item on the stack immediately below the
     * variable value must be an IClassInstanceType instance, the value of which
     * is the class instance in which the variable will be searched for.
     * 
     * @param varName
     *            The name of the variable to update.
     * @return
     */
    public static IAtomicClosure storeInClassInstance(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType value = sm.pop();
                IType inst = sm.pop();
                if(inst instanceof INullType) {
                	throw new NullPointerException();
                }
                IInterpretedClassInstance theClassInstance
								  = ((IClassInstanceType) inst).getValue();
                sm.getMethodScope().updateVariable(theClassInstance, varName,
                        value);
            }

            public String toString() {
                return "storeInClassInstance: " + varName;
            }
        };
    }

    /**
     * The returned closure declares a static immutable variable. The variable
     * must not be already declared as a static variable. Do not create
     * instances of the class before declaring all static variables.
     * <p>
     * The item at the top of the stack will be taken off the stack and used for
     * the variable's value. The item on the stack immediately below the
     * variable value must be an IClassStaticType, to which the variable is
     * added.
     * 
     * @param varName
     * @return
     */
    public static IAtomicClosure declareStaticImmutable(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType value = sm.pop();
                IInterpretedClass theClass = ((IClassStaticType) sm.pop())
                        .getValue();
                sm.getMethodScope().declareImmutableVariable(theClass, varName,
                        value);
            }

            public String toString() {
                return "declareStaticImmutable: " + varName;
            }
        };
    }

    /**
     * The returned closure declares a class instance immutable variable. The
     * variable must not be already declared as a static variable or as a class
     * instance variable.
     * <p>
     * The item at the top of the stack will be taken off the stack and used for
     * the variable's value. The item on the stack immediately below the
     * variable value must be an IClassInstanceType, to which the variable is
     * added.
     * 
     * @param varName
     * @return
     */
    public static IAtomicClosure declareClassImmutable(final String varName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType value = sm.pop();
                IInterpretedClassInstance theClassInstance = ((IClassInstanceType) sm
                        .pop()).getValue();
                sm.getMethodScope().declareImmutableVariable(theClassInstance,
                        varName, value);
            }

            public String toString() {
                return "declareClassImmutable: " + varName;
            }
        };
    }

    /**
     * The returned closure invokes the method with the supplied name. This
     * method must exist in the current method's class. If the current method is
     * static then the new method must also be static. If the current method is
     * not static then the new method can be either a static or non-static
     * method.
     * 
     * @param methodName
     *            The name of the method to invoke.
     * @return
     */
    public static IAtomicClosure invokeLocalMethod(final String methodName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                sm.invokeMethod(methodName);
            }

            public String toString() {
                return "invokeLocalMethod: " + methodName;
            }
        };
    }

    /**
     * Invokes the method with the supplied name in the supplied class's static
     * context.
     * <p>
     * The top of the stack must contain an IClassStaticType, the value of which
     * will be searched for the supplied method name.
     * 
     * @param methodName
     *            The name of the method to invoke.
     * @return
     */
    public static IAtomicClosure invokeForeignStaticMethod(
            final String methodName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IInterpretedClass theClass = ((IClassStaticType) sm.pop())
                        .getValue();
                sm.invokeMethod(theClass, methodName);
            }

            public String toString() {
                return "invokedForeignStaticMethod: " + methodName;
            }
        };
    }

    /**
     * The returned closure invokes the method with the supplied name in the
     * supplied class instance's context.
     * <p>
     * The top of the stack must contain an IClassInstanceType, the value of
     * which will be searched for the supplied method name. If no method by the
     * supplied name is found in the supplied class instance then the class's
     * static methods will be searched.
     * 
     * @param methodName
     *            The name of the method to invoke.
     * @return
     */
    public static IAtomicClosure invokeForeignInstanceMethod(
            final String methodName) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IInterpretedClassInstance theClassInstance = ((IClassInstanceType) sm
                        .pop()).getValue();
                sm.invokeMethod(theClassInstance, methodName);
            }

            public String toString() {
                return "invokeForeignInstanceMethod: " + methodName;
            }
        };
    }

    /**
     * The returned closure fires positionReachedListeners when this closure is
     * executed. The parameter can be any object at all, which will be handed
     * down to all of the positionReachedListeners.
     * 
     * @param data
     *            The data to pass to the positionReachedListeners when this
     *            closure is executed.
     * @return
     */
    public static IAtomicClosure positionReached(final Object data) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                sm.positionReached(data);
            }

            public String toString() {
                return "positionReached";
            }
        };
    }

    /**
     * The returned closure, when executed, revserses the order of the indicated
     * number of items on the stack. Ie, if you pass the argument 2, then after
     * execution, the item that was on the top of the stack is now second, and
     * the item that was second is now on the top.
     * 
     * @param count
     * @return
     */
    public static IAtomicClosure reverseStackTop(final int count) {
        return new AbstractAtomicClosure() {

            public void execute(StackMachine sm) {
                IType[] ary = new IType[count];
                for (int idx = 0; idx < count; idx++) {
                    ary[idx] = sm.pop();
                }
                for (int idx = 0; idx < count; idx++) {
                    sm.push(ary[idx]);
                    ary[idx] = null;
                }
            }

            public String toString() {
                return "Reverse top " + count + " items on stack";
            }
        };
    }
}