/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package org.wellquite.kenya.stackMachine.scope;

import java.util.Map;

import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * Instances of this class represent method invocations in classes. The scope
 * looks after manipulation of variables.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public interface IMethodScope {


    /**
     * The previous scope before the method in this scope was invoked.
     * 
     * @return The previous scope.
     */
    IMethodScope getPreviousScope();

    /**
     * Set the previous scope. This allows a fully linked chain to be
     * established, providing a return path as methods complete.
     * 
     * @param previousScope
     */
    void setPreviousScope(IMethodScope previousScope);

    /**
     * Switch to the supplied new scope. This will be called immediately prior
     * to the execution of a method.
     * 
     * @param newScope
     */
    void switchToNewScope(IMethodScope newScope, StackMachine sm);

    /**
     * Return to the previous scope. This will be called when a method
     * completes.
     */
    void switchToPreviousScope(StackMachine sm);

    /**
     * Fetch a variable. The order of searching is: 1) the current method's
     * temporary variables (declared with the storeNewVariable() method), 2) the
     * current method's variables, 3) the current method's class's instance
     * variables 4) the current method's class's static variables. Both mutable
     * and immutable variables are searched. If the variable can't be found an
     * error is thrown.
     * 
     * @param name
     *            The name of the variable.
     * @return The IType instance representing the variable.
     */
    IType fetchVariable(String name);

    /**
     * Fetch a static variable from the supplied class. Both mutable and
     * immutable variables are searched. If the variable can't be found then an
     * error is thrown.
     * 
     * @param targetClass
     *            The class to search.
     * @param name
     *            The name of the variable.
     * @return The IType instance representing the variable.
     */
    IType fetchVariable(IInterpretedClass targetClass, String name);

    /**
     * Fetch a instance variable from the supplied class instance. Both mutable
     * and immutable variables are searched. If the variable can't be found in
     * the class instance then the class's static variables are searched. If the
     * variable can't be found then an error is thrown.
     * 
     * @param targetClass
     *            The class to search.
     * @param name
     *            The name of the variable.
     * @return The IType instance representing the variable.
     */
    IType fetchVariable(IInterpretedClassInstance targetClass, String name);

    /**
     * Stores a new variable in the method's temporary variable store. These
     * variables only exist until switchToPrevious() is called on this scope.
     * Note that a variable must be declared through this method before an
     * updateVariable() call for this variable will succeed.
     * 
     * @param name
     *            The name of the variable.
     * @param value
     *            The value of the variable.
     * @param mutable
     *            Whether the variable is mutable or not.
     */
    void storeNewVariable(String name, IType value, boolean mutable);

    void startNewVariableClosure();
    
    void endNewVariableClosure();
    
    /**
     * Update a variable. The order of searching is: 1) the current method's
     * temporary variables (declared with the storeNewVariable() method), 2) the
     * current method's variables, 3) the current method's class's instance
     * variables 4) the current method's class's static variables. The variable
     * to be updated must have been declared to be a mutable variable. If the
     * variable can't be found an error is thrown.
     * 
     * @param name
     *            The name of the variable.
     * @param value
     *            The new value of the variable.
     */
    void updateVariable(String name, IType value);

    /**
     * Update a static variable in the supplied class. The variable must have
     * been declare mutable. If the variable can't be found, an error will be
     * thrown.
     * 
     * @param targetClass
     *            The class containing the static mutable variable to be
     *            updated.
     * @param name
     *            The name of the variable.
     * @param value
     *            The new value of the variable.
     */
    void updateVariable(IInterpretedClass targetClass, String name, IType value);

    /**
     * Update a class instance variable in the supplied class instance. The
     * variable must have been declared as mutable. If the variable can't be
     * found in the class's instance variables then the class's static variables
     * are searched. If the variable can't be found, an error will be thrown.
     * 
     * @param targetClassInstance
     *            The class instance containing the mutable instance variable to
     *            be updated.
     * @param name
     *            The name of the variable.
     * @param value
     *            The new value of the variable.
     */
    void updateVariable(IInterpretedClassInstance targetClassInstance,
            String name, IType value);

    /**
     * Declares an immutable variable in the supplied class. As this is a class
     * rather than a class instance, you are infact declaring an immutable
     * static variable. Do not confuse this method with the storeNewVariable()
     * method - that is for temporary method-wide variables only. If there is
     * already a static variable of that name in the supplied class then an
     * error is thrown.
     * 
     * @param targetClass
     *            The class to add the variable to.
     * @param name
     *            The name of the variable.
     * @param value
     *            The value of the variable.
     */
    void declareImmutableVariable(IInterpretedClass targetClass, String name,
            IType value);

    /**
     * Declares an immutable variable in the supplied class instance. As this is
     * a class instance rather than a class, you are infact declaring an
     * immutable class-instance variable. Do not confuse this method with the
     * storeNewVariable() method - that is for temporary method-wide variables
     * only. If there is already a variable of that name in the supplied class
     * or a static variable of the same name then an error is thrown.
     * 
     * @param targetClassInstance
     *            The class instance to add the variable to.
     * @param name
     *            The name of the variable.
     * @param value
     *            The value of the variable.
     */
    void declareImmutableVariable(
            IInterpretedClassInstance targetClassInstance, String name,
            IType value);

    /**
     * Set the method for this scope.
     * 
     * @param method
     */
    void setMethod(IInterpretedMethod method);

    /**
     * Returns the method set in this scope.
     * 
     * @return
     */
    IInterpretedMethod getCurrentMethod();

    /**
     * Sets the class instance containing the method set in this scope. This
     * will only succeed if the method is not static.
     * <p>
     * Note that newClass.getInterpretedClass() == method.getInterpretedClass()
     * must be true.
     * 
     * @param newClass
     */
    void setClassInstance(IInterpretedClassInstance newClass);

    /**
     * Gets the class instance containing the method set in this scope. If the
     * method in this scope is a static method then this will return null.
     * 
     * @return
     */
    IInterpretedClassInstance getCurrentClassInstance();

    /**
     * Is the method set in this scope a static method?
     * 
     * @return
     */
    boolean isStatic();

    /**
     * Set the class containing the method set in this scope.
     * <p>
     * Note that newClass == method.getInterpretedClass() must be true.
     * 
     * @param newClass
     */
    void setClass(IInterpretedClass newClass);

    /**
     * Get the class which contains the method set in this scope.
     * 
     * @return
     */
    IInterpretedClass getCurrentClass();
    
    Map getLocalMutableVariables();
    
    Map getLocalImmutableVariables();
}