/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004
 */
package org.wellquite.kenya.stackMachine.scope;

import java.util.Map;
import java.util.Stack;
import java.util.TreeMap;

import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * This class represents scope. A scope contains a method and a method must be
 * in a scope before it can be executed. The invokeMethod() calls from
 * StackMachine and the closures returned by StackMachineOps take care of this
 * automatically.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class MethodScope implements IMethodScope {

    private IMethodScope previousScope = null;

    private IInterpretedMethod method = null;

    private final Stack localMutableVariableStack = new Stack();

    private final Stack localImmutableVariableStack = new Stack();

    private volatile Map localMutableVariables = new TreeMap();

    private volatile Map localImmutableVariables = new TreeMap();

    private IInterpretedClassInstance myClass = null;

    public MethodScope() {
    }

    /**
     * Creates a new scope representing the supplied static method in the
     * supplied class. Because you are supplying an IInterpretedClass instance
     * and not an IInterpretedClassInstance instance it is assumed that the
     * method is static. If the method is not static then an error is thrown.
     * <p>
     * Note that the supplied class (targetClass) and the method's
     * getInterpretedClass() method must match. Ie no transplanting methods
     * around between classes.
     * 
     * @param targetMethod
     *            The method this scope is to represent.
     * @param targetClass
     *            The class in the context of which this method is to be
     *            executed.
     */
    public MethodScope(IInterpretedMethod targetMethod,
            IInterpretedClass targetClass) {
        if (targetMethod.isStatic()
                && targetMethod.getInterpretedClass().equals(targetClass)) {
            method = targetMethod;
            myClass = null;
        } else {
            throw new RuntimeException(
                    "Attempt made to create scope with non static method and static class: "
                            + targetMethod + ", " + targetClass);
        }
    }

    /**
     * Creates a new scope representing the supplied method in the supplied
     * class instance. Because you are supplying an IInterpretedClassInstance,
     * it is assumed that the method is non-static. However, if the method is
     * static then this is coped with silently.
     * <p>
     * Note that the supplied class instance's class
     * (targetClass.getInterpretedClass()) and the method's
     * getInterpretedClass() method results must match. Ie no transplanting
     * methods around between classes.
     * 
     * @param targetMethod
     *            The method this scope is to represent.
     * @param targetClassInstance
     *            The class instance in the context of which this method is to
     *            be executed.
     */
    public MethodScope(IInterpretedMethod targetMethod,
            IInterpretedClassInstance targetClassInstance) {
        method = targetMethod;
        if (method.isStatic()) {
            myClass = null;
        } else {
            myClass = targetClassInstance;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#getPreviousScope()
     */
    public synchronized IMethodScope getPreviousScope() {
        return previousScope;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#setPreviousScope(org.wellquite.kenya.stackMachine.scope.IScope)
     */
    public synchronized void setPreviousScope(IMethodScope previousScope) {
        this.previousScope = previousScope;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#setMethod(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod)
     */
    public synchronized void setMethod(IInterpretedMethod method) {
        if (this.method == null) {
            this.method = method;
        } else {
            throw new RuntimeException("Attempt made to set scope method to '"
                    + method + "' when scope method already set to '"
                    + this.method + "'");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#getMethod()
     */
    public synchronized IInterpretedMethod getCurrentMethod() {
        return method;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#storeNewVariable(java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType, boolean)
     */
    public synchronized void storeNewVariable(String name, IType value,
            boolean mutable) {
        if (mutable) {
            localMutableVariables.put(name, value);
        } else {
            localImmutableVariables.put(name, value);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#updateVariable(java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void updateVariable(String name, IType value) {
        Map found = null;

        if (localMutableVariables.containsKey(name)) {
            localMutableVariables.put(name, value);
            return;
        } else {

            for (int idx = localMutableVariableStack.size() - 1; idx >= 0; idx--) {
                Map variableMap = (Map) localMutableVariableStack.get(idx);
                if (variableMap.containsKey(name)) {
                    variableMap.put(name, value);
                    return;
                }
            }

            if (method.getMutableMethodVariables().containsKey(name)) {
                found = method.getMutableMethodVariables();
            } else if (!method.isStatic()) {
                if (myClass.getMutableInstanceVariables().containsKey(name)) {
                    found = myClass.getMutableInstanceVariables();
                }
            }
        }

        if (found == null) {
            if (method.getInterpretedClass().getMutableStaticVariables()
                    .containsKey(name)) {
                found = method.getInterpretedClass()
                        .getMutableStaticVariables();
            }
        }

        if (found == null) {
            noSuchVariable(name);
        }

        found.put(name, value);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#updateVariable(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass,
     *      java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void updateVariable(IInterpretedClass targetClass,
            String name, IType value) {
        if (targetClass.getMutableStaticVariables().containsKey(name)) {
            targetClass.getMutableStaticVariables().put(name, value);
        } else {
            noSuchVariable(targetClass, name);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#updateVariable(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance,
     *      java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void updateVariable(
            IInterpretedClassInstance targetClassInstance, String name,
            IType value) {
        if (targetClassInstance.getMutableInstanceVariables().containsKey(name)) {
            targetClassInstance.getMutableInstanceVariables().put(name, value);
        } else {
            updateVariable(targetClassInstance.getInterpretedClass(), name,
                    value);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#fetch(java.lang.String)
     */
    public synchronized IType fetchVariable(String name) {

        if (localMutableVariables.containsKey(name)) {
            return (IType) localMutableVariables.get(name);
        }

        for (int idx = localMutableVariableStack.size() - 1; idx >= 0; idx--) {
            Map variableMap = (Map) localMutableVariableStack.get(idx);
            if (variableMap.containsKey(name)) {
                return (IType) variableMap.get(name);
            }
        }

        if (localImmutableVariables.containsKey(name)) {
            return (IType) localImmutableVariables.get(name);
        }

        for (int idx = localImmutableVariableStack.size() - 1; idx >= 0; idx--) {
            Map variableMap = (Map) localImmutableVariableStack.get(idx);
            if (variableMap.containsKey(name)) {
                return (IType) variableMap.get(name);
            }
        }

        if (method.isStatic()) {
            return fetchVariable(method.getInterpretedClass(), name);
        } else {
            return fetchVariable(myClass, name);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#fetchVariable(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass,
     *      java.lang.String)
     */
    public synchronized IType fetchVariable(IInterpretedClass targetClass,
            String name) {
        if (targetClass.getMutableStaticVariables().containsKey(name)) {
            return (IType) targetClass.getMutableStaticVariables().get(name);
        } else if (targetClass.getImmutableStaticVariables().containsKey(name)) {
            return (IType) targetClass.getImmutableStaticVariables().get(name);
        }

        noSuchVariable(targetClass, name);
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#fetchVariable(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance,
     *      java.lang.String)
     */
    public synchronized IType fetchVariable(
            IInterpretedClassInstance targetClass, String name) {
        if (targetClass.getMutableInstanceVariables().containsKey(name)) {
            return (IType) targetClass.getMutableInstanceVariables().get(name);
        } else if (targetClass.getImmutableInstanceVariables()
                .containsKey(name)) {
            return (IType) targetClass.getImmutableInstanceVariables()
                    .get(name);
        } else {
            return fetchVariable(targetClass.getInterpretedClass(), name);
        }
    }

    private void noSuchVariable(String name) {
        throw new RuntimeException("Unable to find variable '" + name
                + "' in method " + method);
    }

    private void noSuchVariable(IInterpretedClass targetClass, String name) {
        throw new RuntimeException("Unable to find variable '" + name
                + "' in class " + targetClass);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#setClassInstance(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClassInstance)
     */
    public synchronized void setClassInstance(IInterpretedClassInstance newClass) {
        if (!method.isStatic())
            myClass = newClass;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#getCurrentClassInstance()
     */
    public synchronized IInterpretedClassInstance getCurrentClassInstance() {
        return myClass;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#isStatic()
     */
    public synchronized boolean isStatic() {
        if (method == null)
            return true;
        return method.isStatic();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#setClass(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass)
     */
    public synchronized void setClass(IInterpretedClass newClass) {
        if (method.isStatic())
            myClass = null;
        else
            throw new RuntimeException(
                    "Attempt made to set scope to static class context '"
                            + newClass + "' but method is not static.");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.scope.IScope#getCurrentClass()
     */
    public synchronized IInterpretedClass getCurrentClass() {
        if (method == null)
            return null;
        return method.getInterpretedClass();
    }

    public synchronized void switchToNewScope(IMethodScope newScope,
            StackMachine sm) {
        newScope.setPreviousScope(sm.getMethodScope());
        sm.setMethodScope(newScope);
    }

    public synchronized void switchToPreviousScope(StackMachine sm) {
        if (sm.getMethodScope() == null) {
            throw new RuntimeException(
                    "Unable to switch to non-existant previous scope.");
        }
        sm.setMethodScope(getPreviousScope());
        method = null;
        previousScope = null;
        myClass = null;
        localMutableVariables.clear();
        localImmutableVariables.clear();
        localMutableVariableStack.clear();
        localImmutableVariableStack.clear();
    }

    public synchronized void declareImmutableVariable(
            IInterpretedClass targetClass, String name, IType value) {
        if (targetClass.getImmutableStaticVariables().containsKey(name)
                || targetClass.getMutableStaticVariables().containsKey(name))
            throw new RuntimeException("Unable to declare immutable variable "
                    + name + " in class " + targetClass
                    + " as variable already exists.");
        else
            targetClass.addImmutableStaticVariable(name, value);
    }

    public synchronized void declareImmutableVariable(
            IInterpretedClassInstance targetClassInstance, String name,
            IType value) {
        if (targetClassInstance.getImmutableInstanceVariables().containsKey(
                name)
                || targetClassInstance.getMutableInstanceVariables()
                        .containsKey(name)
                || targetClassInstance.getInterpretedClass()
                        .getImmutableStaticVariables().containsKey(name)
                || targetClassInstance.getInterpretedClass()
                        .getMutableStaticVariables().containsKey(name))
            throw new RuntimeException("Unable to declare immutable variable "
                    + name + " in class instance " + targetClassInstance
                    + " as variable already exists.");
        else
            targetClassInstance.getInterpretedClass()
                    .addImmutableInstanceVariable(name, value);
    }

    public synchronized Map getLocalMutableVariables() {
        Map variableMap = new TreeMap(localMutableVariables);
        for (int idx = localMutableVariableStack.size() - 1; idx >= 0; idx--) {
            variableMap.putAll((Map) localMutableVariableStack.get(idx));
        }
        return variableMap;
    }

    public synchronized Map getLocalImmutableVariables() {
        Map variableMap = new TreeMap(localImmutableVariables);
        for (int idx = localImmutableVariableStack.size() - 1; idx >= 0; idx--) {
            variableMap.putAll((Map) localImmutableVariableStack.get(idx));
        }
        return variableMap;
    }

    public synchronized void startNewVariableClosure() {
        localImmutableVariableStack.push(localImmutableVariables);
        localImmutableVariables = new TreeMap();

        localMutableVariableStack.push(localMutableVariables);
        localMutableVariables = new TreeMap();
    }

    public synchronized void endNewVariableClosure() {
        localMutableVariables.clear();
        localImmutableVariables.clear();
        localMutableVariables = (Map) localMutableVariableStack.pop();
        localImmutableVariables = (Map) localImmutableVariableStack.pop();
    }

}