/*
 * Created on 26-Feb-2005
 */
package kenya.eclipse.ast;

import java.util.Iterator;

import kenya.eclipse.ast.bindings.BuiltinMethodBinding;
import kenya.eclipse.ast.bindings.ClassBinding;
import kenya.eclipse.ast.bindings.ClassVariableBinding;
import kenya.eclipse.ast.bindings.ConstantBinding;
import kenya.eclipse.ast.bindings.EnumBinding;
import kenya.eclipse.ast.bindings.EnumConstantBinding;
import kenya.eclipse.ast.bindings.IInnerVariableBinding;
import kenya.eclipse.ast.bindings.MethodBinding;
import kenya.eclipse.ast.bindings.VariableBinding;
import kenya.eclipse.multieditor.kenya.util.SourceCodeLocationComparator;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.interfaces.IVariable;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AClassDecDeclaration;
import minijava.node.ACommaEnumList;
import minijava.node.AConstDecDeclaration;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AEnumList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.AFunctionApplication;
import minijava.node.AQualifiedNameName;
import minijava.node.ASimpleName;
import minijava.node.ASimpleNameName;
import minijava.node.ATypeName;
import minijava.node.AVarDecInnerDeclaration;
import minijava.node.Node;
import minijava.node.PActualParamList;
import minijava.node.PCommaEnumList;
import minijava.node.PFormalParamList;
import minijava.node.PName;
import minijava.node.Start;
import minijava.node.TIdentifier;

/**
 * @author Thomas Timbul
 */
public class DeclarationFinder {
	
	DeclarationFinder() {
		
	}
	
	/**
	 * returns the signature of the identifier's declaration as a String.
	 * examples:
	 * 
	 * 'void main()'
	 * 'String get(String s)'
	 * '<T>boolean find(String s,boolean b)'
	 * 'boolean b'
	 * 'class Point'
	 * 'enum Types'
	 * 
	 * @param t
	 * @param s
	 * @return
	 */
	static IBinding resolveBinding(TIdentifier t, Start s) {
		Helper f = new Helper(t);
		return f.getSignature(s);
	}
	
	/**
	 * returns the binding for t that matches
	 * the function specified. Null is returned if t cannot possibly
	 * be bound to that function
	 * @param t
	 * @param f
	 * @return
	 */
	static IBinding resolveFunction(TIdentifier t, IFunction fun) {
		if(fun!=null) {
			Start s = (Start)NodeTools.getParent(fun.getDeclarationNode(), Integer.MAX_VALUE);
			if(s==null) return null;
			Helper f = new Helper(t, fun);
			return f.getSignature(s);
		} else {
			//builtin function
			return new BuiltinMethodBinding(t);
		}
	}
	
	/**
	 * @param name
	 * @param f
	 * @return
	 */
	static IBinding resolveLocal(PName name, IFunction f) {
		if(name instanceof ASimpleNameName) {
			ASimpleName n = (ASimpleName)((ASimpleNameName)name).getSimpleName();
			return resolveLocal(n.getIdentifier(), f);
		}
		return null;
	}
	
	static IBinding resolveLocal(TIdentifier name, IFunction f) {
		LocalVariableHelper lf = new LocalVariableHelper(name, f);
		return lf.getSignature();
	}
	
	/**
	 * @param variable
	 * @param code
	 * @return
	 */
	static IBinding resolveConstant(IVariable variable, ICheckedCode code) {
		Start s = (Start)NodeTools.getParent(code.getFunctions()[0].getDeclarationNode(), Integer.MAX_VALUE);
		ConstantHelper cf = new ConstantHelper(variable);
		return cf.getSignature(s);
	}

	/**
	 * @param id
	 * @return
	 */
	static IBinding resolveReferenceType(TIdentifier id, IClass c) {
		Start s = (Start)NodeTools.getParent(id, Integer.MAX_VALUE);
		ClassFieldOrEnumConstantHelper rf = new ClassFieldOrEnumConstantHelper(id, c);
		return rf.getSignature(s);
	}

	/**
	 * @param identifier
	 * @return
	 */
	static IBinding resolveClassOrEnum(TIdentifier id) {
		Start s = (Start)NodeTools.getParent(id, Integer.MAX_VALUE);
		ClassOrEnumHelper cef = new ClassOrEnumHelper(id);
		return cef.getSignature(s);
	}
}

class Helper extends DepthFirstAdapter {
	
	private TIdentifier f;
	private IFunction func;
	private IBinding dec;
	
	Helper(TIdentifier t) {
		f = t;
	}
	
	Helper(TIdentifier t, IFunction fun) {
		this(t);
		func = fun;
	}
	
	public IBinding getSignature(Start start) {
		start.apply(this);
		return dec;
	}
	
	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
		if(func!=null && node!=func.getDeclarationNode()) return;
		if(f!=null && node.getIdentifier().toString().equals(f.toString())) {
			if(f==node.getIdentifier()) {
				//direct match: same node
				String s = NodeToString.toString(node);
				String rp = node.getRParenthese().toString();
				String decl = s.substring(0, s.indexOf(rp)+1);
				
				dec = new MethodBinding(f, node, node.getType(), decl);
			} else {
				// the selected identifier is part of
				//  a fun call 'targetting' this declaration
				
				AFunctionApplication app
				  = (AFunctionApplication)NodeTools.getParent(f, AFunctionApplication.class);
				if(app==null) {
					//maybe not...
					return;
				}
				
				//to make sure its the correct method (if at all)
				// have to compare the number of parameters and their types
				
				PActualParamList l1 = app.getActualParamList();
				PFormalParamList l2 = node.getFormalParamList();
				String a = NodeToString.toString(l1);
				String b = NodeToString.toString(l2);
				
				//in the first place, the length has to be correct.
				int length = a.split(",").length;
				
				int length2 = b.split(",").length;
				if(length!=length2) {
					//not the one
					return;
				}
				
				//now compare the types of the parameters for overloaded methods
				
				//FIXME: not comparing parameter types? stupidly assuming that same no. of parameters is correct
				String s = NodeToString.toString(node);
				String rp = node.getRParenthese().toString();
				String decl = s.substring(0, s.indexOf(rp)+1);
				
				dec = new MethodBinding(f, node, node.getType(), decl);
				
			}
		} else {
			//one of parameters?
			if(node.getFormalParamList()!=null) {
				ATypeName tn = Resolver.findParameter(
						NodeTools.getLocation(f), node.getFormalParamList());
				if(tn!=null) {
					dec = new VariableBinding(f, tn, tn.getType(), NodeToString.toString(tn));
					return;
				}
				
			}
			
		}
	}
	
	public void caseAClassDecDeclaration(AClassDecDeclaration node) {
		//dec = DeclarationFinder.resolveClassOrEnum(f);
	}
	
	public void caseAConstDecDeclaration(AConstDecDeclaration node) {
		//check that identifier doesnt actually refer to a method
		if(NodeTools.getParent(f, 1) instanceof AFuncDecDeclaration
				|| NodeTools.getParent(node, 1) instanceof AFunctionApplication) {
			return;
		}
		if(node.getIdentifier().toString().equals(f.toString())) {
			String decl = NodeToString.toString(node);
			String cons_ = node.getConst().toString();
			String assign = NodeToString.toString(node.getInitialiser());
			int kg = decl.indexOf(assign)-1;
			decl = decl.substring(cons_.length(), kg);
			
			dec = new ConstantBinding(f, node, node.getType(), decl);
			
		}
	}
	
	public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
		//dec = DeclarationFinder.resolveClassOrEnum(f);
	}
	
	public void caseASimpleNameName(ASimpleNameName node) {
		return; //there are no declarations using PName
	}
	
	public void caseAQualifiedNameName(AQualifiedNameName node) {
		return; //there are no declarations using PName
	}
	
	
}

class LocalVariableHelper extends DepthFirstAdapter {

	private TIdentifier name;
	private IFunction func;
	private IBinding dec;
	
	LocalVariableHelper(TIdentifier n, IFunction f) {
		func = f;
		name = n;
	}
	
	public IBinding getSignature() {
		func.getDeclarationNode().apply(this);
		return dec;
	}
	
	public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
		if(!occursBefore(node.getIdentifier(), name)) {
			return;
		}
		if(node.getIdentifier().toString().equals(name.toString())) {
			String d = NodeToString.toString(node);
			if(node.getArrayInitialiser()!=null) {
				String init = NodeToString.toString(node.getArrayInitialiser());
				d = d.substring(0, d.indexOf(init)-1);
			}
			dec = new VariableBinding(name, node, node.getType(), d);
		}
	}
	public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
		if(!occursBefore(node.getIdentifier(), name)) {
			return;
		}
		if(node.getIdentifier().toString().equals(name.toString())) {
			String d = NodeToString.toString(node);
			if(node.getInitialiser()!=null) {
				String init = NodeToString.toString(node.getInitialiser());
				d = d.substring(0, d.indexOf(init)-1);
			}
			dec = new VariableBinding(name, node, node.getType(), d);
		}
	}
	
	private boolean occursBefore(Node n1, Node n2) {
		ISourceCodeLocation l1 = NodeTools.getLocation(n1);
		ISourceCodeLocation l2 = NodeTools.getLocation(n2);
		
		return SourceCodeLocationComparator.getInstance().compare(l1, l2)<=0;
		
	}
	
}

class ConstantHelper extends DepthFirstAdapter {

	private IVariable fVar;
	private IBinding dec;
	
	ConstantHelper(IVariable var) {
		fVar = var;
	}
	
	public IBinding getSignature(Start start) {
		start.apply(this);
		return dec;
	}
	
	public void caseAConstDecDeclaration(AConstDecDeclaration node) {
		if(node.getIdentifier().getText().trim().equals(fVar.getName())) {
			String decl = NodeToString.toString(node);
			String cons_ = node.getConst().toString();
			String assign = NodeToString.toString(node.getInitialiser());
			int kg = decl.indexOf(assign)-1;
			decl = decl.substring(cons_.length(), kg);
			
			dec = new ConstantBinding(node.getIdentifier(), node, node.getType(), decl);
		}
	}
	
}

class ClassFieldOrEnumConstantHelper extends DepthFirstAdapter {
	
	private IClass cl;
	private TIdentifier id;
	private IInnerVariableBinding dec;
	
	ClassFieldOrEnumConstantHelper(TIdentifier t, IClass c) {
		id = t;
		cl = c;
	}
	
	public IInnerVariableBinding getSignature(Start start) {
		start.apply(this);
		return dec;
	}
	
	public void caseAClassDecDeclaration(AClassDecDeclaration node) {
		inAClassDecDeclaration(node);
		if(NodeToString.toString(node.getIdentifier()).equals(cl.getName())) {
			for(Iterator it = node.getClassInnerDeclaration().iterator(); it.hasNext();) {
				Node n = (Node)it.next();
				n.apply(this);
				if(dec!=null) {
					return;
				}
			}
		} else {
			//?
		}
		outAClassDecDeclaration(node);
	}
	
	public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
		if(node.getIdentifier().toString().equals(id.toString())) {
			String d = NodeToString.toString(node);
			if(node.getArrayInitialiser()!=null) {
				String init = NodeToString.toString(node.getArrayInitialiser());
				d = d.substring(0, d.indexOf(init)-1);
			}
			dec = new ClassVariableBinding(id, cl, node, node.getType(), d);
		}
	}
	public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
		if(node.getIdentifier().toString().equals(id.toString())) {
			String d = NodeToString.toString(node);
			if(node.getInitialiser()!=null) {
				String init = NodeToString.toString(node.getInitialiser());
				d = d.substring(0, d.indexOf(init)-1);
			}
			dec = new ClassVariableBinding(id, cl, node, node.getType(), d);
		}
	}
	
	public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
		if(node.getIdentifier()==id ||
				node.getIdentifier().toString().equals(id)) {
			dec = new EnumConstantBinding(id, cl, node, null, NodeToString.toString(node));
			return;
		}
		if(node.getEnumList()!=null) {
			node.getEnumList().apply(this);
		}
	}
	public void caseAEnumList(AEnumList node) {
		if(node.getIdentifier()==id ||
				node.getIdentifier().toString().equals(id)) {
			dec = new EnumConstantBinding(id, cl, node, null, NodeToString.toString(node));
			return;
		}
		if(node.getCommaEnumList()!=null) {
			for(Iterator it = node.getCommaEnumList().iterator(); it.hasNext();) {
				PCommaEnumList list = (PCommaEnumList)it.next();
				list.apply(this);
			}
		}
	}
	public void caseACommaEnumList(ACommaEnumList node) {
		if(node.getIdentifier()==id ||
				node.getIdentifier().toString().equals(id)) {
			dec = new EnumConstantBinding(id, cl, node, null, NodeToString.toString(node));
		}
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAFuncDecDeclaration(minijava.node.AFuncDecDeclaration)
	 */
	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
	}
	
}

class ClassOrEnumHelper extends DepthFirstAdapter {
	
	private TIdentifier id;
	private IBinding fBinding;
	
	ClassOrEnumHelper(TIdentifier i) {
		id = i;
	}
	
	IBinding getSignature(Start s) {
		s.apply(this);
		return fBinding;
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAEnumDecDeclaration(minijava.node.AEnumDecDeclaration)
	 */
	public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
		TIdentifier nid = node.getIdentifier();
		if(nid==id || nid.toString().equals(id.toString())) {
			fBinding = new EnumBinding(id, node, null, NodeToString.toString(node));
		}
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAClassDecDeclaration(minijava.node.AClassDecDeclaration)
	 */
	public void caseAClassDecDeclaration(AClassDecDeclaration node) {
		TIdentifier nid = node.getIdentifier();
		if(nid==id || nid.toString().equals(id.toString())) {
			fBinding = new ClassBinding(id, node, null, NodeToString.toString(node));
		}
	}
	
}


