/*
 * Created on 26-Feb-2005
 */
package kenya.eclipse.ast;

import java.util.Iterator;
import java.util.LinkedList;

import kenya.eclipse.ast.bindings.ClassBinding;
import kenya.eclipse.ast.bindings.EnumBinding;
import kenya.eclipse.ast.bindings.VariableBinding;
import kenya.eclipse.multieditor.kenya.util.SourceCodeLocationComparator;
import kenya.passes.Util;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AArrayFieldAccess;
import minijava.node.ABasicTypeType;
import minijava.node.ABlock;
import minijava.node.AClassDecDeclaration;
import minijava.node.AClassTypeReferenceType;
import minijava.node.ACommaTypeName;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AFormalParamList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.AFunctionApplication;
import minijava.node.AIntBasicType;
import minijava.node.AQualifiedName;
import minijava.node.AQualifiedNameName;
import minijava.node.AReferenceTypeType;
import minijava.node.AScalarFieldAccess;
import minijava.node.ASimpleName;
import minijava.node.ASimpleNameName;
import minijava.node.ATypeName;
import minijava.node.Node;
import minijava.node.PActualParamList;
import minijava.node.PFieldAccess;
import minijava.node.PFormalParamList;
import minijava.node.PName;
import minijava.node.PType;
import minijava.node.Start;
import minijava.node.TIdentifier;

/**
 * @author Thomas Timbul
 */
public class Resolver extends DepthFirstAdapter {
	
	/**
	 * 
	 */
	public Resolver() {
		super();
	}
	
	public static IBinding resolve(PName name, ICheckedCode code) {
		if(name instanceof ASimpleNameName) {
			return resolve((ASimpleNameName)name, code);
		} else if(name instanceof AQualifiedNameName) {
			return resolve((AQualifiedNameName)name, code);
		} else {
			return null;
		}
	}
	
	public static IBinding resolve(ASimpleNameName name, ICheckedCode code) {
		return resolve((ASimpleName)name.getSimpleName(), code);
	}
	
	public static IBinding resolve(AQualifiedNameName name, ICheckedCode code) {
		return resolve((AQualifiedName)name.getQualifiedName(), code);
	}
	
	public static IBinding resolve(AQualifiedName name, ICheckedCode code) {
		return resolve(name.getIdentifier(), code);
	}
	
	public static IBinding resolve(ASimpleName name, ICheckedCode code) {
		return resolve(name.getIdentifier(), code);
	}
	
	public static IBinding resolve(TIdentifier name, ICheckedCode code) {
		ResHelper w = new ResHelper(code);
		name.apply(w);
		return w.getResolve();
	}
	
	static ATypeName findParameter(ISourceCodeLocation loc, PFormalParamList list) {
		AFormalParamList l = (AFormalParamList)list;
		int commasize = l.getCommaTypeName()!=null?l.getCommaTypeName().size():0;
		ATypeName[] names = new ATypeName[commasize+1];
		names[0] = (ATypeName)l.getTypeName();
		LinkedList ll = l.getCommaTypeName();
		int i = 1;
		for(Iterator iter = ll.iterator(); iter.hasNext(); ++i) {
			names[i] = (ATypeName)((ACommaTypeName)iter.next()).getTypeName();
		}
		
		for(int j = 0; j < names.length; j++) {
			ATypeName n = names[j];
			if(NodeTools.getLocation(n.getType()).equals(loc)
					|| NodeTools.getLocation(n.getIdentifier()).equals(loc)) {
				return n;
			}
		}
		
		return null;
	}
	
	public static IFunction getContainingMethod(ISourceCodeLocation loc, ICheckedCode code) {
		if(code.isErroredCode()) {
			return null;
		}
		IFunction[] fs = code.getFunctions();
		for(int i = 0; i < fs.length; i++) {
			IFunction f = fs[i];
			ABlock block = (ABlock)f.getDeclarationNode().getBlock();
			ISourceCodeLocation start = NodeTools.getLocation(block.getLBrace());
			ISourceCodeLocation end = NodeTools.getLocation(block.getRBrace());
			
			boolean inside =
				  SourceCodeLocationComparator.getInstance().compare(start, loc)<=0
				&&SourceCodeLocationComparator.getInstance().compare( loc , end)<=0;
			
			if(inside) {
				return f;
			}
		}
		return null;
	}
	
	public static IClass getClassDefinedAt(ISourceCodeLocation loc, ICheckedCode code) {
		int classes = code.getClasses().length;
		int enums = code.getEnums().length;
		IClass[] fs = new IClass[classes+enums];
		
		System.arraycopy(code.getClasses(), 0, fs, 0, classes);
		System.arraycopy(code.getEnums(), 0, fs, classes, enums);
		
		for(int i = 0; i < fs.length; i++) {
			IClass f = fs[i];
			if(SourceCodeLocationComparator.getInstance().compare(f.getPosition(), loc)==0) {
				return f;
			}
		}
		return null;
	}
}

class ResHelper extends DepthFirstAdapter {
	
	private ICheckedCode fCode;
	private Start aa;
	
	private IBinding fBinding;
	
	public ResHelper(ICheckedCode code) {
		fCode = code;
	}
	
	public IBinding getResolve() {
		return fBinding;
	}
	
	public void caseTIdentifier(TIdentifier node) {
		aa = (Start)NodeTools.getParent(node, Integer.MAX_VALUE);
		
		PName name = (PName)NodeTools.getParent(node, PName.class);
		if(name!=null) {
			name.apply(this);
		} else {
			if(NodeTools.getParent(node, 1) instanceof AClassTypeReferenceType) {
				//the type name of the variable or array
				fBinding = DeclarationFinder.resolveClassOrEnum(node);
				return;
			}
			ISourceCodeLocation loc = NodeTools.getLocation(node);
			IFunction f = Resolver.getContainingMethod(loc, fCode);
			if(f!=null) {
				fBinding = DeclarationFinder.resolveLocal(node, f);
			} else {
				loc = null;
				AClassDecDeclaration cdec = (AClassDecDeclaration)NodeTools.getParent(node, AClassDecDeclaration.class);
				if(cdec!=null) {
					int[] decloc = Util.getFirstIdent(cdec.getIdentifier());
					loc = new SourceCodeLocation(decloc[0], decloc[1], decloc[2]);
				}
				if(loc==null) {
					AEnumDecDeclaration edec = (AEnumDecDeclaration)NodeTools.getParent(node, AEnumDecDeclaration.class);
					if(edec!=null) {
						int[] decloc = Util.getFirstIdent(edec.getIdentifier());
						loc = new SourceCodeLocation(decloc[0], decloc[1], decloc[2]);
					}
				}
				if(loc!=null) {
					IClass cl = Resolver.getClassDefinedAt(loc, fCode);
					int asdf = 0;
					fBinding = DeclarationFinder.resolveReferenceType(node, cl);
				}
				
				if(fBinding==null) {
					//maybe a classname
					fBinding = DeclarationFinder.resolveClassOrEnum(node);
				}
				
				if(fBinding==null) {
					//ok, whatever
					fBinding = DeclarationFinder.resolveBinding(node, aa);
				}
			}
		}
//		if(fBinding!=null) //debugging
//			System.out.println(fBinding.toString());
	}
	
	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
		String name = NodeToString.toString(node);
		IFunction[] funs = fCode.getFunctions();
		IFunction match = null;
		for(int i = 0; i < funs.length; i++) {
			IFunction f = funs[i];
			if(f.getName().equals(name)) {
				if(match!=null) {
					
					//to make sure its the correct method (if at all)
					// have to compare the number of parameters and their types
					
					PFormalParamList l1 = node.getFormalParamList();
					String a = NodeToString.toString(l1);
					
					//in the first place, the length has to be correct.
					int length = a.split(",").length;
					
					int length2 = f.getArguments().length;
					if(length!=length2) {
						//not the one
						continue;
					}
					
					int matchlen = match.getArguments().length;
					if(matchlen==length) {
						//we cannot handle this particular case
						return;
					}
					match = f;
				} else {
					match = f;
				}
			}
		}
		fBinding = DeclarationFinder.resolveFunction(node.getIdentifier(), match);
		return;
	}
	public void caseAFunctionApplication(AFunctionApplication node) {
		String name = NodeToString.toString(node.getName());
		ASimpleNameName snn = (ASimpleNameName)node.getName();
		ASimpleName sn = (ASimpleName)snn.getSimpleName();
		IFunction[] funs = fCode.getFunctions();
		IFunction match = null;
		for(int i = 0; i < funs.length; i++) {
			IFunction f = funs[i];
			if(f.getName().equals(name)) {
				if(match!=null) {
					
					//to make sure its the correct method (if at all)
					// have to compare the number of parameters and their types
					
					PActualParamList l1 = node.getActualParamList();
					String a = NodeToString.toString(l1);
					
					//in the first place, the length has to be correct.
					int length = a.split(",").length;
					
					int length2 = f.getArguments().length;
					if(length!=length2) {
						//not the one
						continue;
					}
					
					int matchlen = match.getArguments().length;
					if(matchlen==length) {
						//we cannot handle this particular case
						return;
					}
					match = f;
				} else {
					match = f;
				}
			}
		}
		fBinding = DeclarationFinder.resolveFunction(sn.getIdentifier(), match);
		return;
	}
	
	public void caseAQualifiedName(AQualifiedName node) {
		PFieldAccess access = node.getFieldAccess();
		TIdentifier id = node.getIdentifier();
		access.apply(this);
		
		if(fBinding!=null) {
			PType ptype = fBinding.getType();
			if(!(ptype instanceof AReferenceTypeType)) {
				//array?
				if(NodeToString.toString(id).equals("length")) {
					ABasicTypeType t = new ABasicTypeType(new AIntBasicType());
					fBinding = new VariableBinding(id, fBinding.getDeclaringNode(), t, "");
					return;
				}
				return; //only reference type can have fields
			}
			
			AClassTypeReferenceType ctrt
			  = (AClassTypeReferenceType)((AReferenceTypeType)ptype).getReferenceType();
			
			IBinding ceBinding
			  = DeclarationFinder.resolveClassOrEnum(ctrt.getIdentifier());
			String name = NodeToString.toString(ctrt.getIdentifier());
			
			IClass[] classes;
			if(ceBinding instanceof EnumBinding) {
				classes = fCode.getEnums();
			} else if(ceBinding instanceof ClassBinding) {
				classes = fCode.getClasses();
			} else {
				classes = new IClass[0];
			}
			
			for(int i = 0; i < classes.length; i++) {
				IClass c = classes[i];
				if(!c.getName().equals(name)) continue;
				fBinding = DeclarationFinder.resolveReferenceType(id, c);
				if(fBinding!=null) {
					return;
				}
			}
		}
		
	}
	
	public void caseASimpleName(ASimpleName node) {
		Node wholeName = NodeTools.getTopMostName((PName)node.parent());
		Node immediateParent = NodeTools.getParent(wholeName, 1); 
		if(immediateParent instanceof AFunctionApplication ||
				immediateParent instanceof PFieldAccess) {
			immediateParent.apply(this);
		}
		
	}
	
	// field access
	public void caseAArrayFieldAccess(AArrayFieldAccess node) {

		if(node.getName() instanceof AQualifiedNameName) {
			//resolve nested classes
			node.getName().apply(this);
			return;
		}
		
		//constant or local variable
		ISourceCodeLocation loc = NodeTools.getLocation(node);
		IFunction f = Resolver.getContainingMethod(loc, fCode);
		if(f!=null) {
			//check for argument
			IVariable[] vars = f.getArguments();
			for(int j = 0; j < vars.length; j++) {
				IVariable variable = vars[j];
				if(variable.getName().equals(NodeToString.toString(node))) {
					ATypeName name = Resolver.findParameter(
							variable.getPosition(),
							f.getDeclarationNode().getFormalParamList()
						);
					if(name!=null) {
						fBinding = new VariableBinding(node, name, name.getType(), NodeToString.toString(name));
						return;
					}
				}
			}
			
			//locally declared? //do this before constant resolution to respect scope!
			fBinding = DeclarationFinder.resolveLocal(node.getName(), f);
			if(fBinding!=null) return;
			
			//constant
			vars = fCode.getConstants();
			for(int j = 0; j < vars.length; j++) {
				IVariable variable = vars[j];
				if(variable.getName().equals(NodeToString.toString(node))) {
					fBinding = DeclarationFinder.resolveConstant(variable, fCode);
					break;
				}
			}
		}
	}
	public void caseAScalarFieldAccess(AScalarFieldAccess node) {
		
		if(node.getName() instanceof AQualifiedNameName) {
			//resolve nested classes
			node.getName().apply(this);
			return;
		}
		
		//constant or local variable
		ISourceCodeLocation loc = NodeTools.getLocation(node);
		IFunction f = Resolver.getContainingMethod(loc, fCode);
		if(f!=null) { //if inside a method
			//check for argument
			IVariable[] vars = f.getArguments();
			for(int j = 0; j < vars.length; j++) {
				IVariable variable = vars[j];
				if(variable.getName().equals(NodeToString.toString(node))) {
					ATypeName name = Resolver.findParameter(
							variable.getPosition(),
							f.getDeclarationNode().getFormalParamList()
						);
					if(name!=null) {
						fBinding = new VariableBinding(node, name, name.getType(), NodeToString.toString(name));
						return;
					}
				}
			}
			
			//locally declared? //do this before constant resolution to respect scope!
			fBinding = DeclarationFinder.resolveLocal(node.getName(), f);
			if(fBinding!=null) return;
			
			//constant
			vars = fCode.getConstants();
			for(int j = 0; j < vars.length; j++) {
				IVariable variable = vars[j];
				if(variable.getName().equals(NodeToString.toString(node))) {
					fBinding = DeclarationFinder.resolveConstant(variable, fCode);
					break;
				}
			}
		}
	}
	
}
