/*
 * Created on 13-Dec-2004
 */
package kenya.eclipse.debug;

import java.text.DateFormat;
import java.util.Date;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.debug.bridge.ICommunicationBridge;
import kenya.eclipse.debug.launcher.IKenyaLaunchConfigurationConstants;
import kenya.eclipse.debug.model.KenyaDebugModel;
import kenya.eclipse.debug.model.KenyaDebugTarget;
import kenya.eclipse.debug.model.VKDebugProcess;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.IProcess;

/**
 * @author Thomas Timbul
 */
public class KenyaRunner {
	
	/**
	 * empty constructor
	 */
	public KenyaRunner() {
		setKenyaNIOPath();
	}
	
	private void setKenyaNIOPath() {
		//NOTE: this is quite a hack to get javacode to run with
		// the *cough* 'stupid' JavaCode.execute implementation,
		// which expects the classpath for execution to be in
		// the system property java.class.path
		
		//add KenyaNIO.jar and KenyaCore.jar to the classpath
		//this is so stuff (Kenya programs) executes properly
		String classpath = System.getProperty("java.class.path", ".");
		String classpathSeparator = System.getProperty("path.separator");
		
		String cutoff = "update@/";
		
		String loc = KenyaPlugin.getDefault().getBundle().getLocation();
		loc = loc.substring(cutoff.length());
		String nioPath = (loc + "lib/KenyaNIO.jar");
		
		String commonPath = (loc + "lib/KenyaCore.jar");
		
		classpath += classpathSeparator + nioPath
		           + classpathSeparator + commonPath;
		System.setProperty("java.class.path", classpath);
		
	}
	
	/**
	 * run and return a new IProcess based on the configuration supplied
	 * @param config the KenyaRunnerConfiguration to configure the launch
	 * @param launch the launch to represent the launch of the process
	 * @param monitor progressmonitor to indicate progress
	 * @return IProcess object representing the run of the configuration
	 * @throws CoreException if an internal error occurs or the run has to abort otherwise
	 */
	public IProcess run(KenyaRunnerConfiguration config, ILaunch launch, IProgressMonitor monitor) throws CoreException {
		IProcess p = null;
		String mode = launch.getLaunchMode();
		if(ILaunchManager.DEBUG_MODE.equals(mode)) {
			ICommunicationBridge bridge = new DebugVKBridge();
			
			VKDebugProcess proc = (VKDebugProcess)KenyaDebugModel.newDebugProcess(launch, config.getWorkingDirectory(), config.getCheckedCode(), config.getArguments(), config.getStopInMain());
			p = newProcess(launch, proc, renderLabel(launch, config));
			proc.setBridge(bridge);
			
			KenyaDebugTarget target = new KenyaDebugTarget(launch, proc, config.getClassName(), bridge, config.getStopInMain());
			
			launch.addDebugTarget(target);
			
		} else if(ILaunchManager.RUN_MODE.equals(mode)) {
			VKRunProcess proc = new VKRunProcess(config.getWorkingDirectory(), config.getJavaCode(), config.getClassName(), config.getArguments());
			p = newProcess(launch, proc, renderLabel(launch, config));
			launch.addDebugTarget(new AdapterDebugTarget(launch, proc, config.getClassName()));
		} else {
			abort("Unsupported launch operation: "+mode, null, IKenyaLaunchConfigurationConstants.ERR_INTERNAL_ERROR);
		}
		
		return p;
	}
	
	/**
	 * @param launch
	 * @param config
	 * @return
	 */
	private String renderLabel(ILaunch launch, KenyaRunnerConfiguration config) {
//		int dateStyle = DateFormat.DATE_FIELD | DateFormat.MONTH_FIELD | DateFormat.YEAR_FIELD;
//		int timeStyle = DateFormat.HOUR_OF_DAY0_FIELD | DateFormat.MINUTE_FIELD | DateFormat.SECOND_FIELD;
		String projectName = config.getProjectName();
		return projectName + "/" + config.getClassName() + " - "+ DateFormat.getDateTimeInstance().format(new Date());
	}
	
	/**
	 * create a new IProcess based on a system Process
	 * This method directly calls DebugPlugin.newProcess(launch, p, label)
	 * to create the process.
	 * @param launch
	 * @param p
	 * @param label
	 * @return
	 * @throws CoreException if the process could not be created
	 */
	protected IProcess newProcess(ILaunch launch, Process p, String label) throws CoreException {
		IProcess process= DebugPlugin.newProcess(launch, p, label);
		if (process == null) {
			p.destroy();
			abort("KenyaRunner encountered an internal error", null, IKenyaLaunchConfigurationConstants.ERR_INTERNAL_ERROR);
		}
		return process;
	}
	
	/**
	 * Throws a core exception with an error status object built from
	 * the given message, lower level exception, and error code.
	 * 
	 * @param message the status message
	 * @param exception lower level exception associated with the
	 *  error, or <code>null</code> if none
	 * @param code error code
	 * @throws CoreException The exception encapsulating the reason for the abort
	 */
	protected void abort(String message, Throwable exception, int code) throws CoreException {
		throw new CoreException(new Status(IStatus.ERROR, KenyaPlugin.getPluginId(), code, message, exception));
	}
	
}
