/*
 * Created on 23-Dec-2004
 */
package kenya.eclipse.debug.model;

import java.lang.reflect.InvocationTargetException;
import java.util.Map.Entry;

import kenya.eclipse.debug.model.values.KValue;
import kenya.eclipse.debug.model.values.KValueFactory;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IVariable;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IValue;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * @author Thomas Timbul
 */
public class KVariable extends KenyaDebugElement implements org.eclipse.debug.core.model.IVariable {
	
	private String internalName;     //null if noLookup==true
	private String presentationName; //null if noLookup==false
	
	private IVariable fVar;
	private IType fType;   //MUST NOT be null
	private IClass fClass; //fClass must not be null if fVar is not
	
//	private KValue currValue; //current value (since last suspend)
//	private KValue prevValue; //previous value (before last suspend)
	
	private final boolean noLookup; //true if internal name not known
	                       //(ie, cannot look up the underlying variable from it)
	private IStackMachineInformationProvider ismip;
	
	private Exception failure;
	//thrown back in any method that throws DebugException if this is non-null
	//failure signals that fVar and fClass are null. fType is available
	
	/**
	 * create KVariable with given DebugTarget. The Entry passed is assumed
	 * to consist of a Key, Value pair where Key is the name of the variable
	 * that is internal to the StackMachine and the value is an IType.
	 */
	public KVariable(KenyaDebugTarget target, Entry entry) {
		this(target, (IType)entry.getValue(), (String)entry.getKey());
	}
	
	/**
	 * create KVariable with given DebugTarget, the IType containing the underlying value
	 * and the variables name that is internal to the StackMachine 
	 * @param target associated DebugTarget
	 * @param type underlying IType
	 * @param iName the StackMachine internal name of this variable
	 */
	public KVariable(KenyaDebugTarget target, IType type, String iName) {
		this(target, type, iName, false);
	}
	
	/**
	 * create KVariable with given DebugTarget, displayable name and underlying IType
	 * @param target associated DebugTarget
	 * @param eName the name to display to user
	 * @param IType underlying IType
	 */
	public KVariable(KenyaDebugTarget target, String eName, IType type, IVariable var) {
		this(target, type, eName, true);
		fVar = var;
		updateFields();
		updateBeforeResume();
	}
	
	/**
	 * if noLookup is true, name refers to presentation name, otherwise it is the
	 * StackMachine internal name of the underlying variable
	 * @param target
	 * @param type
	 * @param name
	 * @param noLookup
	 */
	private KVariable(KenyaDebugTarget target, IType type, String name, boolean noLookup) {
		super(target);
		fType = type;
		this.noLookup = noLookup;
		if(noLookup) {
			presentationName = name;
		} else {
			internalName = name;
		}
		updateFields();
		
	}
	
	/**
	 * attempts to get the StackMachineInformationProvider from the debug target
	 * if it is null. Otherwise the previously resolved provider is used to
	 * update the fields fVar and fClass.
	 * use in conjunction with checkFailure to ensure proper behaviour
	 */
	private void updateFields() {
		if(!noLookup) {
			try {
				if(ismip==null) {
					ismip = fTarget.getStackMachineInformationProvider();
				}
				fVar = ismip.lookupVariable(internalName);
				if(fVar==null) {
					failure = new NullPointerException("unable to resolve variable "+internalName);
					return;
				}
				failure = null;
			} catch(InvocationTargetException e) {
				failure = e;
				//if this happens, ie, we couldnt get the ismip, then see comment on field >failure<
			}
		}
		if(fVar!=null) {
			fClass = fVar.getType();
//			currValue = KValueFactory.create(fTarget, fVar, fType);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IValue#getReferenceTypeName()
	 */
	public String getReferenceTypeName() throws DebugException {
		checkFailure();
		return fClass.getName();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IVariable#getValue()
	 */
	public IValue getValue() throws DebugException {
//		checkFailure();
//		return currValue;
		return KValueFactory.create(fTarget, fVar, fType);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IVariable#getName()
	 */
	public String getName() throws DebugException {
		if(noLookup) {
			return presentationName;
		} else {
			checkFailure();
			return fVar.getName();
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IVariable#hasValueChanged()
	 */
	public boolean hasValueChanged() throws DebugException {
//		if(currValue!=null && prevValue!=null) {
//			return !currValue.isValueEqual(prevValue);
//		} else if(currValue==null && prevValue==null) {
//			return false;
//		} else {
//			return true;
//		}
		//doesnt work, so just ignore it :p (until someone more capable does it)
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IValueModification#setValue(java.lang.String)
	 */
	public void setValue(String expression) throws DebugException {
		unsupported("changing values is not supported");
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IValueModification#setValue(org.eclipse.debug.core.model.IValue)
	 */
	public void setValue(IValue value) throws DebugException {
		unsupported("changing values is not supported");
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IValueModification#supportsValueModification()
	 */
	public boolean supportsValueModification() {
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IValueModification#verifyValue(java.lang.String)
	 */
	public boolean verifyValue(String expression) throws DebugException {
		unsupported("changing values is not supported");
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IValueModification#verifyValue(org.eclipse.debug.core.model.IValue)
	 */
	public boolean verifyValue(IValue value) throws DebugException {
		unsupported("changing values is not supported");
		return false;
	}
	
	/**
	 * includes a call to updateFields
	 * @throws DebugException if even after calling updateFields failure is not null
	 * or fVar is null
	 */
	private void checkFailure() throws DebugException {
		if(!noLookup) {
			updateFields();
		}
		if(failure!=null || (fVar==null)) {
			abort("variable lookup failed", failure);
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean isValueEqual(KVariable var) {
		try {
			boolean b = ( (KValue)getValue() ).isValueEqual((KValue)var.getValue());
			return b;
		} catch(DebugException e) {
		}
		return false;
	}
	
	/**
	 * re-calculates the prev and curr values, so that the
	 * hasChanged method returns the correct value
	 */
	public synchronized void updateBeforeResume() {
//		prevValue = currValue==null?null:(KValue)currValue.clone();
//		currValue = KValueFactory.create(fTarget, fVar, fType);
	}
	
}
