/*
 * Created on 25-Dec-2004
 */
package kenya.eclipse.debug.model;

import java.lang.reflect.Array;
import java.util.HashMap;

import kenya.eclipse.KenyaPlugin;
import kenya.interpreter.util.InterpreterInspectorListContentProvider;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.wellquite.kenya.stackMachine.IPointListener;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.scope.IMethodScope;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod;

/**
 * @author Thomas Timbul
 */
public class KenyaThread extends KenyaDebugElement implements IThread {
	
	protected InterpreterInspectorListContentProvider fInspector;
	protected StackMachine sm;
	protected IStackMachineInformationProvider ismip;
	
	private HashMap frameMap = new HashMap();
	
	private PositionThing execListener;
	
	/**
	 * 
	 */
	public KenyaThread(KenyaDebugTarget target,
			InterpreterInspectorListContentProvider inspector,
			StackMachine sm,
			IStackMachineInformationProvider ismip) {
		super(target);
		fInspector = inspector;
		this.sm = sm;
		this.ismip = ismip;
		
		execListener = new PositionThing();
		sm.addPositionReachedListener(execListener);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getStackFrames()
	 */
	public IStackFrame[] getStackFrames() throws DebugException {
		Object[] list = fInspector.getElements(sm);
		
		KStackFrame[] r = new KStackFrame[list.length];
		for(int i=0; i<list.length; i++) {
			IMethodScope scope = (IMethodScope)list[i];
			if(frameMap.containsKey(scope)) {
				r[i] = (KStackFrame)frameMap.get(scope);
			} else {
				r[i] = stackFrameFromScope(scope);
				frameMap.put(scope, r[i]);
			}
			if(r[i]==null) {
				abort("Possible shutdown in progress", null);
			}
		}
		return r;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#hasStackFrames()
	 */
	public boolean hasStackFrames() throws DebugException {
		return !isTerminated() && sm.getMethodScope()!=null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getPriority()
	 */
	public int getPriority() throws DebugException {
		return 0;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getTopStackFrame()
	 */
	public IStackFrame getTopStackFrame() throws DebugException {
		IMethodScope scope = sm.getMethodScope();
		if(frameMap.containsKey(scope)) {
			return (KStackFrame)frameMap.get(scope);
		} else {
			KStackFrame frame = stackFrameFromScope(sm.getMethodScope());
			if(frame==null) {
				abort("possible stack machine shutdown", null);
			}
			frameMap.put(scope, frame);
			return frame;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getName()
	 */
	public String getName() {
		return "Main";
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getBreakpoints()
	 */
	public IBreakpoint[] getBreakpoints() {
		return new IBreakpoint[0];
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canResume()
	 */
	public boolean canResume() {
		return sm.canResume();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canSuspend()
	 */
	public boolean canSuspend() {
		return sm!=null && sm.canSuspend();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#isSuspended()
	 */
	public boolean isSuspended() {
		return sm!=null && sm.canResume();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#resume()
	 */
	public void resume() throws DebugException {
		if(sm==null) {
			Status s = new Status(Status.ERROR, KenyaPlugin.getPluginId(), DebugException.TARGET_REQUEST_FAILED, "shutting down", null);
			throw new DebugException(s);
		}
		fireResumeEvent(DebugEvent.EVALUATION);
		sm.resume();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#suspend()
	 */
	public void suspend() throws DebugException {
		suspend(DebugEvent.CLIENT_REQUEST);
	}
	
	protected void suspend(int eventcode) {
		if(canSuspend()) {
			sm.setStepMode(true);
			fireSuspendEvent(eventcode);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepInto()
	 */
	public boolean canStepInto() {
		try {
			return !isTerminated() && getTopStackFrame().canStepInto();
		} catch(DebugException e) {
			return false;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepOver()
	 */
	public boolean canStepOver() {
		try {
			return !isTerminated() && getTopStackFrame().canStepOver();
		} catch(DebugException e) {
			return false;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepReturn()
	 */
	public boolean canStepReturn() {
		try {
			return !isTerminated() && getTopStackFrame().canStepReturn();
		} catch(DebugException e) {
			return false;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#isStepping()
	 */
	public boolean isStepping() {
		return !isTerminated() && sm!=null && sm.getStepMode();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepInto()
	 */
	public void stepInto() throws DebugException {
		if(sm==null) {
			Status s = new Status(Status.ERROR, KenyaPlugin.getPluginId(), DebugException.TARGET_REQUEST_FAILED, "shutting down", null);
			throw new DebugException(s);
		}
		if(canResume()) {
			fireResumeEvent(DebugEvent.STEP_INTO);
//			updateBeforeResume();
			sm.step();
			fireSuspendEvent(DebugEvent.STEP_END);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepOver()
	 */
	public void stepOver() throws DebugException {
		if(sm==null) {
			Status s = new Status(Status.ERROR, KenyaPlugin.getPluginId(), DebugException.TARGET_REQUEST_FAILED, "shutting down", null);
			throw new DebugException(s);
		}
		if(canResume()) {
			KStackFrame kf = (KStackFrame)getTopStackFrame();
			execListener.waitFor(kf.getLineNumber()+2); //line numbers in the frame are behind by 1, plus 1 extra
			execListener.waitFor(kf); //this is for stepping over method calls
			fireResumeEvent(DebugEvent.STEP_OVER);
			sm.resume();
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepReturn()
	 */
	public void stepReturn() throws DebugException {
		if(sm==null) {
			Status s = new Status(Status.ERROR, KenyaPlugin.getPluginId(), DebugException.TARGET_REQUEST_FAILED, "shutting down", null);
			throw new DebugException(s);
		}
		if(isSuspended()) {
			IStackFrame[] frames = getStackFrames();
			if(Array.getLength(frames) > 1) {
				KStackFrame kf = (KStackFrame)frames[1];
				execListener.waitFor(kf);
				fireResumeEvent(DebugEvent.STEP_RETURN);
				resume();
			} else {
				fireResumeEvent(DebugEvent.STEP_RETURN);
				resume();
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		return getDebugTarget().canTerminate();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		return getDebugTarget().isTerminated() || sm==null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#terminate()
	 */
	public void terminate() throws DebugException {
		getDebugTarget().terminate();
	}
	
	private synchronized KStackFrame stackFrameFromScope(IMethodScope scope) {
		String name = "Stackframe";
		ISourceCodeLocation loc = null;
		
		{ //hide this detail
			StringBuffer buf = new StringBuffer();
			
			IInterpretedMethod method = scope.getCurrentMethod();
			if(method==null) {
				return null;
			}
			IFunction function = ismip.lookupFunction(method.getName());
			if(function==null) {
				//must be a predefined method ?
				return new KStackFrame(fTarget, this, scope, method.getName(), null, false);
			}
			loc = function.getPosition();
			
			//get arguments and put them into brackets
			buf.append(function.getName());
			buf.append("( ");
			kenya.sourceCodeInformation.interfaces.IVariable[] vars = function.getArguments();
			for(int idx = 0; idx < vars.length; idx++) {
				buf.append(vars[idx].getType());
				buf.append(" ");
				buf.append(vars[idx].getName());
				buf.append(", ");
			}
			if (buf.toString().endsWith(", ")) {
				buf.delete(buf.length() - ", ".length(), buf.length());
			}
			buf.append(" )");
			//done with that
			name = buf.toString();
		} //
		
		KStackFrame f = new KStackFrame(fTarget, this, scope, name, loc);
		return f;
	}
	
	boolean breakPointAt(ISourceCodeLocation loc) {
		return fTarget.breakpointAt(loc);
	}
	
	class PositionThing implements IPointListener {
		
		private int waitfor_line = -1; //line number to suspend on regardless
		private IStackFrame waitfor_frame = null; //frame to suspend when it becomes the top frame
		
		public void waitFor(int line) {
			waitfor_line = line;
		}
		
		public void waitFor(IStackFrame frame) {
			waitfor_frame = frame;
		}
		
		/* (non-Javadoc)
		 * @see org.wellquite.kenya.stackMachine.IPointListener#pointReached(java.lang.Object)
		 */
		public void pointReached(final Object data) {
			//order matters:
			//   breakpoints
			//   a specific stackframe becomes top frame (step return/step over)
			//   single step
			
			if(data instanceof ISourceCodeLocation) {
				ISourceCodeLocation loc = (ISourceCodeLocation)data;
				
				//update StackFrame positions
				try {
					((KStackFrame)getTopStackFrame()).pointReached(loc);
				} catch(Exception e) {
					//ignore
				}
				
				if(breakPointAt(loc)) {
					suspend(DebugEvent.BREAKPOINT);
					waitfor_line = -1;
					waitfor_frame = null;
				} else if(waitfor_frame!=null) {
					try {
						if(waitfor_frame == getTopStackFrame()) {
						  // == comparison is possible since frames stay the same through their life cycle
							
							if(waitfor_line==-1 || (loc.getLineNumber()>=waitfor_line) ) {
								//either there is no condition on a line, in which case we can just suspend (step return)
								// or there is a specification for a particular line to stop on (step over) which must in
								// that case also be honoured
								
								suspend(DebugEvent.STEP_END);
								waitfor_line = -1;
								waitfor_frame = null;
							}
							
						}
					} catch(DebugException e1) {
						//ignore, probably terminated
					}
				} else if(isStepping()) {
					suspend(DebugEvent.STEP_END); //step into
					waitfor_line = -1;
					waitfor_frame = null;
				} else {
					//?? anything?
				}
			}
			
		}
	}
	
	/**
	 * updates the internal state of all StackFrames if required
	 */
	protected synchronized void updateBeforeResume() {
		try {
			if(hasStackFrames()) {
				KStackFrame[] frames = (KStackFrame[])getStackFrames();
				for(int i=0; i<frames.length; i++) {
					frames[i].updateBeforeResume();
				}
			}
		} catch(DebugException e) {
		}
	}
	
}
