/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package kenya.eclipse.debug.model.breakpoints;

import java.util.List;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.debug.model.KenyaDebugModel;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.ui.actions.IToggleBreakpointsTarget;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.texteditor.IEditorStatusLine;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * Toggles a line breakpoint in a Java editor.
 * 
 * @since 3.0
 */
public class ToggleBreakpointAdapter implements IToggleBreakpointsTarget {
	
	protected void report(String message, IWorkbenchPart part) {
		IEditorStatusLine statusLine= (IEditorStatusLine) part.getAdapter(IEditorStatusLine.class);
		if (statusLine != null) {
			if (message != null) {
				statusLine.setMessage(true, message, null);
			} else {
				statusLine.setMessage(true, null, null);
			}
		}
		if (message != null && KenyaPlugin.getActiveWorkbenchShell() != null) {
			KenyaPlugin.getActiveWorkbenchShell().getDisplay().beep();
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.actions.IToggleBreakpointsTarget#toggleLineBreakpoints(IWorkbenchPart, ISelection)
	 */
	public void toggleLineBreakpoints(IWorkbenchPart part, ISelection selection)
	throws CoreException {
		if (selection instanceof ITextSelection) {
			report(null, part);
			IEditorPart editorPart = (IEditorPart)part;
			ITextSelection textSelection = (ITextSelection)selection;
			
			ICheckedCode code
			  = (ICheckedCode)editorPart.getAdapter(ICheckedCode.class);
			
			if (code == null) {
				report("Setting a breakpoint is currently not possible", part);
				return;
			}
			
			IEditorInput editorInput = editorPart.getEditorInput();
			IDocument document = ( (ITextEditor)editorPart
					.getAdapter(ITextEditor.class) ).getDocumentProvider()
					.getDocument(editorInput);
			
			try {
				IStackMachineInformationProvider smip = code.getBaseStackMachine();
				if (smip != null) {
					
					int lineNumber = textSelection.getStartLine()+1;
					int offset = textSelection.getOffset();
//					int column = document.getLineOffset(lineNumber) - offset + 1;
					
					List positions
					  = smip.getDiscreteExecutionPointMarkersFromTheAST();
					int realOffset = 0;
					ISourceCodeLocation realLoc = null;
					int delta = Integer.MAX_VALUE;
					for(int idx = 0; idx < positions.size(); idx++) {
						ISourceCodeLocation scl
						  = (ISourceCodeLocation)positions.get(idx);
						int thisOffset
						  = document.getLineOffset(scl.getLineNumber()-1)
								    + scl.getColumnNumber()-1;
						
						//can only mark breakpoints _forward_
						int newDelta = thisOffset - offset; //ie, this has to be positive
						if (newDelta>=0 && newDelta < delta) {
							realOffset = thisOffset;
							realLoc = scl;
							delta = newDelta;
						}
					}
					
					try {
						
						IResource resource;
						
						if (editorInput instanceof IFileEditorInput) {
							resource = ( (IFileEditorInput)editorInput ).getFile();
						} else {
							resource = ResourcesPlugin.getWorkspace().getRoot();
						}
						
						KenyaLineBreakpoint existingBreakpoint
						  = KenyaDebugModel.lineBreakpointExists(resource, lineNumber);
						if (existingBreakpoint != null) {
							DebugPlugin.getDefault().getBreakpointManager()
							.removeBreakpoint(existingBreakpoint, true);
							return;
						}
						
						//determine real document offsets
						int l = realLoc.getLineNumber();
						int e = realOffset + realLoc.getTokenLength();
						
						KenyaDebugModel.createLineBreakpoint(
								resource, realLoc, realOffset, e,
								true);
						
					} catch(CoreException ce) {
						return;
					}
					
				}
			} catch(IllegalArgumentException error) {
				//ignore
			} catch(BadLocationException e) {
				KenyaPlugin.log(e);
			}
			
		}
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.debug.ui.actions.IToggleBreakpointsTarget#canToggleLineBreakpoints(IWorkbenchPart,
	 *      ISelection)
	 */
	public boolean canToggleLineBreakpoints(IWorkbenchPart part, ISelection selection) {
		return selection instanceof ITextSelection;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.actions.IToggleBreakpointsTarget#toggleMethodBreakpoints(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void toggleMethodBreakpoints(IWorkbenchPart part, ISelection selection) throws CoreException {
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.actions.IToggleBreakpointsTarget#canToggleMethodBreakpoints(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public boolean canToggleMethodBreakpoints(IWorkbenchPart part, ISelection selection) {
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.actions.IToggleBreakpointsTarget#toggleWatchpoints(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void toggleWatchpoints(IWorkbenchPart part, ISelection selection) throws CoreException {
		
	}
	
	protected static IResource getResource(IEditorPart editor) {
		IResource resource;
		IEditorInput editorInput = editor.getEditorInput();
		if (editorInput instanceof IFileEditorInput) {
			resource= ((IFileEditorInput)editorInput).getFile();
		} else {
			resource= ResourcesPlugin.getWorkspace().getRoot();
		}
		return resource;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.actions.IToggleBreakpointsTarget#canToggleWatchpoints(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public boolean canToggleWatchpoints(IWorkbenchPart part, ISelection selection) {
		return false;
	}
	
	
}
