/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package kenya.eclipse.debug.ui.launcher;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.debug.KenyaTypeSearchEngine;
import kenya.eclipse.debug.launcher.IKenyaLaunchConfigurationConstants;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.ILaunchShortcut;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

/**
 * Performs single click launching for local Java applications.
 */
public class KenyaApplicationLaunchShortcut implements ILaunchShortcut {
	
	public static final String NO_MAIN_IN_EDITOR = "The active editor does not contain a Kenya application.";
	public static final String NO_MAIN_IN_SELECTION = "The selection does not contain a Kenya application.";
	public static final String LAUNCH_FAILED = "Launch failed...";
	public static final String LAUNCH_ERROR = "An error occurred while launching...";
	public static final String SELECT_DEBUG = "Select the Kenya type to debug:";
	public static final String SELECT_RUN = "Select the Kenya type to run:";
	
	/**
	 * @param search the java elements to search for a main type
	 * @param mode the mode to launch in
	 * @param editor activated on an editor (or from a selection in a viewer)
	 */
	public void searchAndLaunch(IResource[] search, String mode, boolean editor) {
		IPath[] types = null;
		if (search != null) {
			try {
				KenyaTypeSearchEngine engine = new KenyaTypeSearchEngine();
				types = engine.search(search);
			} catch(CoreException e) {
				MessageDialog.openError(getShell(), "Launch failed...", e.getMessage());
			}
			IPath type = null;
			if (types.length == 0) {
				String message = null;
				if (editor) {
					message = NO_MAIN_IN_EDITOR;
				} else {
					message = NO_MAIN_IN_SELECTION;
				}
				MessageDialog.openError(getShell(), LAUNCH_FAILED, message);
			} else if (types.length > 1) {
				type = chooseType(types, mode);
			} else {
				type = types[0];
			}
			if (type != null) {
				launch(type, mode);
			}
		}

	}	
	
	/**
	 * Prompts the user to select a type
	 * 
	 * @return the selected type or <code>null</code> if none.
	 */
	protected IPath chooseType(IPath[] types, String mode) {
		TypeSelectionDialog dialog= new TypeSelectionDialog(getShell(), types);		
		if (mode.equals(ILaunchManager.DEBUG_MODE)) {
			dialog.setTitle(SELECT_DEBUG);
		} else {
			dialog.setTitle(SELECT_RUN);
		}
		dialog.setMultipleSelection(false);
		if (dialog.open() == Window.OK) {
			return (IPath)dialog.getFirstResult();
		}
		return null;
	}
	
	/**
	 * Launches a configuration for the given type
	 */
	protected void launch(IPath type, String mode) {
		ILaunchConfiguration config = findLaunchConfiguration(type, mode);
		if (config != null) {
			DebugUITools.launch(config, mode);
		}			
	}
	
	/**
	 * Locate a configuration to relaunch for the given type.  If one cannot be found, create one.
	 * 
	 * @return a re-useable config or <code>null</code> if none
	 */
	protected ILaunchConfiguration findLaunchConfiguration(IPath type, String mode) {
		ILaunchConfigurationType configType = getKenyaLaunchConfigType();
		List candidateConfigs = Collections.EMPTY_LIST;
		try {
			ILaunchConfiguration[] configs = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations(configType);
			candidateConfigs = new ArrayList(configs.length);
			for (int i = 0; i < configs.length; i++) {
				ILaunchConfiguration config = configs[i];
				if (config.getAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, "").equals(type.makeAbsolute().toOSString())) {
					if (config.getAttribute(IKenyaLaunchConfigurationConstants.ATTR_PROJECT_NAME, "").equals(type.makeRelative().uptoSegment(1).toString())) {
						candidateConfigs.add(config);
					}
				}
			}
		} catch (CoreException e) {
			KenyaPlugin.log(e);
		}
		
		// If there are no existing configs associated with the IPath, create one.
		// If there is exactly one config associated with the IPath, return it.
		// Otherwise, if there is more than one config associated with the IPath, prompt the
		// user to choose one.
		int candidateCount = candidateConfigs.size();
		if (candidateCount < 1) {
			return createConfiguration(type);
		} else if (candidateCount == 1) {
			return (ILaunchConfiguration) candidateConfigs.get(0);
		} else {
			// Prompt the user to choose a config.  A null result means the user
			// cancelled the dialog, in which case this method returns null,
			// since cancelling the dialog should also cancel launching anything.
			ILaunchConfiguration config = chooseConfiguration(candidateConfigs, mode);
			if (config != null) {
				return config;
			}
		}
		
		return null;
	}
	
	/**
	 * Show a selection dialog that allows the user to choose one of the specified
	 * launch configurations.  Return the chosen config, or <code>null</code> if the
	 * user cancelled the dialog.
	 */
	protected ILaunchConfiguration chooseConfiguration(List configList, String mode) {
		IDebugModelPresentation labelProvider = DebugUITools.newDebugModelPresentation();
		ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), labelProvider);
		dialog.setElements(configList.toArray());
		dialog.setTitle("Select a launch configuration");
		if (mode.equals(ILaunchManager.DEBUG_MODE)) {
			dialog.setMessage(SELECT_DEBUG);
		} else {
			dialog.setMessage(SELECT_RUN);
		}
		dialog.setMultipleSelection(false);
		int result = dialog.open();
		labelProvider.dispose();
		if (result == Window.OK) {
			return (ILaunchConfiguration) dialog.getFirstResult();
		}
		return null;		
	}
	
	/**
	 * Create & return a new configuration based on the specified <code>IPath</code>.
	 */
	protected ILaunchConfiguration createConfiguration(IPath type) {
		ILaunchConfiguration config = null;
		ILaunchConfigurationWorkingCopy wc = null;
		try {
			ILaunchConfigurationType configType = getKenyaLaunchConfigType();
			wc = configType.newInstance(null, getLaunchManager().generateUniqueLaunchConfigurationNameFrom(type.removeFileExtension().lastSegment()));
		} catch (CoreException exception) {
			reportCreatingConfiguration(exception);
			return null;
		}
		
		wc.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, type.makeAbsolute().toOSString());
		wc.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_PROJECT_NAME, type.makeRelative().uptoSegment(1).toString());
		try {
			config = wc.doSave();
		} catch (CoreException exception) {
			reportCreatingConfiguration(exception);
		}
		return config;
	}
	
	protected void reportCreatingConfiguration(final CoreException exception) {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				ErrorDialog.openError(getShell(), LAUNCH_ERROR, "Exception in KenyaApplicationLaunchShortcut", exception.getStatus());
			}
		});
	}
	
	/**
	 * Returns the local java launch config type
	 */
	protected ILaunchConfigurationType getKenyaLaunchConfigType() {
		return getLaunchManager().getLaunchConfigurationType(IKenyaLaunchConfigurationConstants.ID_KENYA_APPLICATION);		
	}
	
	protected ILaunchManager getLaunchManager() {
		return DebugPlugin.getDefault().getLaunchManager();
	}
	
	/**
	 * Convenience method to get the window that owns this action's Shell.
	 */
	protected Shell getShell() {
		return KenyaPlugin.getActiveWorkbenchShell();
	}
	
	/**
	 * @see ILaunchShortcut#launch(IEditorPart, String)
	 */
	public void launch(IEditorPart editor, String mode) {
		IEditorInput input = editor.getEditorInput();
		IFile r = null;
		if(input instanceof IFileEditorInput) {
			r = ((IFileEditorInput)input).getFile();
		}
		if (r != null) {
			searchAndLaunch(new IResource[] {r}, mode, true);
		} else {
			MessageDialog.openError(getShell(), LAUNCH_FAILED, NO_MAIN_IN_EDITOR);
		}
		
	}
	
	/**
	 * @see ILaunchShortcut#launch(ISelection, String)
	 */
	public void launch(ISelection selection, String mode) {
		if (selection instanceof IStructuredSelection) {
			searchAndLaunch((IResource[])((IStructuredSelection)selection).toList().toArray(new IResource[0]), mode, false);
		} else {
			MessageDialog.openError(getShell(), LAUNCH_FAILED, NO_MAIN_IN_SELECTION);
		}
	}

}
