package kenya.eclipse.multieditor;


import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Iterator;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.buildext.AbstractKenyaPostBuildAnalyser;
import kenya.eclipse.buildext.IKenyaPostBuildAnalyserFactory;
import kenya.eclipse.multieditor.java.JavaEditor;
import kenya.eclipse.multieditor.kenya.KenyaEditor;
import kenya.eclipse.multieditor.util.GeneratedJava;
import kenya.eclipse.multieditor.util.GeneratedJavaInput;
import kenya.eclipse.multieditor.util.KenyaHelperThread;
import kenya.ui.UIUtils;
import mediator.ICheckedCode;
import mediator.IJavaCode;
import mediator.subscription.ISubscriber;
import mediator.subscription.ISubscriptionData;
import mediator.subscription.MediationService;
import mediator.subscription.MediationSubscription;
import mediator.subscription.MediationTerms;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FontDialog;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * An example showing how to create a multi-page editor.
 * This example has 3 pages:
 * <ul>
 * <li>page 0 contains a Kenya text editor.
 * <li>page 1 contains a Java text viewer.
 * </ul>
 * Code typed in page 1 is translated to Java, which is then shown on page 1.
 */
public class EditingWindow extends MultiPageEditorPart implements IResourceChangeListener, ISubscriber {
	
	/** The text editor used in page 0 (Kenya). */
	private KenyaEditor kenyaEditor;
	
	/** The text editor used in page 1 (Java). */
	private JavaEditor javaEditor;
	
	/** The font chosen in page 1. */
	private Font font;

	/** The text widget used in page 2. */
	private StyledText text;
	
	private KenyaHelperThread kht;
	private String subscriptionID;
	private GeneratedJava javaStorage;
	
	/**
	 * Creates the Kenya Editing Window with separate Kenya and Java editors
	 */
	public EditingWindow() {
		super();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}
	
	/**
	 * Creates page 0 of the multi-page editor,
	 * which contains the KenyaEditor.
	 */
	private void createKenyaPage() {
		try {
			kenyaEditor = new KenyaEditor();
			
			int index = addPage(kenyaEditor, getEditorInput());
			setPageText(index, kenyaEditor.getTitle());
			
			kenyaEditor.addPropertyListener(new IPropertyListener() {
				public void propertyChanged(Object source, int propId) {
					if(propId==PROP_DIRTY) {
						kht.getDocumentListener().documentChanged(null);
					}
				}
			});
			
		} catch (PartInitException e) {
			ErrorDialog.openError(
				getSite().getShell(),
				"Error creating nested text editor",
				null,
				e.getStatus());
		}
	}
	/**
	 * Creates page 1 of the multi-page editor,
	 * which contains the JavaEditor.
	 * It also subscribes to the subscription service to provide
	 * the Java editor with constant parse updates
	 */
	private void createJavaPage() {
		try {
			javaEditor = new JavaEditor();
			
			javaStorage = new GeneratedJava(getFileName());
			GeneratedJavaInput jInput = new GeneratedJavaInput(javaStorage);
			
			MediationTerms terms
				= new MediationTerms(kenyaEditor, kenyaEditor.getKenyaCodeManager(), javaStorage);
			MediationSubscription sub = new MediationSubscription(this, terms);
			
			//subscribe to the mediator service
			subscriptionID = MediationService.getInstance().addSubscription(sub);
			
			//init the Java editor
			int index = addPage(javaEditor, jInput);
			setPageText(index, javaEditor.getTitle());
			//disable the editor
			javaEditor.setEnabled(false);
		} catch (PartInitException e) {
			ErrorDialog.openError(
				getSite().getShell(),
				"Error creating nested text editor",
				null,
				e.getStatus());
//			e.printStackTrace();
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.part.MultiPageEditorPart#createPages()
	 */
	protected void createPages() {
		createKenyaPage();
		createJavaPage();
		//start the helper for this editor
		kht = new KenyaHelperThread("Kenya AutoBuild", subscriptionID);
		kht.schedule();
		kenyaEditor.getDocumentProvider().getDocument(getEditorInput())
			.addDocumentListener(kht.getDocumentListener());
		MediationService.getInstance().provideASyncSubscriptionUpdate(subscriptionID);
	}
	/**
	 * The <code>MultiPageEditorPart</code> implementation of this 
	 * <code>IWorkbenchPart</code> method disposes all nested editors.
	 * Subclasses may extend.
	 */
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		if(kht!=null) {
			kht.stoppit();
		}
		MediationService.getInstance().cancelSubscription(subscriptionID);
		
		super.dispose();
	}
	/**
	 * Saves the multi-page editor's document.
	 */
	public void doSave(IProgressMonitor monitor) {
		kenyaEditor.doSave(monitor);
	}
	/**
	 * Saves the multi-page editor's document as another file.
	 * Also updates the text for page 0's tab, and updates this multi-page editor's input
	 * to correspond to the nested editor's.
	 */
	public void doSaveAs() {
		IEditorPart editor = getEditor(0);
		editor.doSaveAs();
		setPageText(0, editor.getTitle());
		setInput(editor.getEditorInput());
	}
	/* (non-Javadoc)
	 * Method declared on IEditorPart
	 */
	public void gotoMarker(IMarker marker) {
		setActivePage(0);
		IDE.gotoMarker(getEditor(0), marker);
	}
	
	/**
	 * checks that the input is an instance of <code>IFileEditorInput</code>.
	 */
	public void init(IEditorSite site, IEditorInput editorInput)
		throws PartInitException {
		super.init(site, editorInput);
		setPartName(editorInput.getName());
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.MultiPageEditorPart#createSite(org.eclipse.ui.IEditorPart)
	 */
	protected IEditorSite createSite(IEditorPart editor) {
		return new AdaptedMultiPageEditorSite(this, editor);
	}
	
	/* (non-Javadoc)
	 * Method declared on IEditorPart.
	 */
	public boolean isSaveAsAllowed() {
		return true;
	}
	
	/**
	 * Closes all project files on project close.
	 */
	public void resourceChanged(final IResourceChangeEvent event){
		if(event.getType() == IResourceChangeEvent.PRE_CLOSE){
			Display.getDefault().asyncExec(new Runnable(){
				public void run(){
					IWorkbenchPage[] pages = getSite().getWorkbenchWindow().getPages();
					for (int i = 0; i<pages.length; i++){
						if(((FileEditorInput)kenyaEditor.getEditorInput()).getFile().getProject().equals(event.getResource())){
							IEditorPart editorPart = pages[i].findEditor(kenyaEditor.getEditorInput());
							pages[i].closeEditor(editorPart,true);
						}
					}
				}            
			});
		}
		
	}
	/**
	 * Sets the font related data to be applied to the text in page 2.
	 */
	void setFont() {
		FontDialog fontDialog = new FontDialog(getSite().getShell());
		fontDialog.setFontList(text.getFont().getFontData());
		FontData fontData = fontDialog.open();
		if (fontData != null) {
			if (font != null)
				font.dispose();
			font = new Font(text.getDisplay(), fontData);
			text.setFont(font);
		}
	}
	
	/* (non-Javadoc)
	 * @see mediator.subscription.ISubscriber#subscriptionUpdate(mediator.subscription.ISubscriptionData)
	 */
	public void subscriptionUpdate(ISubscriptionData data) {
		if(javaStorage!=null && javaStorage.hasChanged()) {
			UIUtils.getStandardDisplay().asyncExec(new Runnable() {
				public void run() {
					javaEditor.doRevertToSaved();
				}
			});
			
		}
		
		if(data!=null && Array.getLength(data.getContents())>0) {
			ICheckedCode cc = null;
			IJavaCode jc = null;
			if(!(data.getContents()[0] instanceof Exception)) {
				cc = (ICheckedCode)data.getContents()[0];
				jc = (IJavaCode)data.getContents()[1];
			}
			
			//buildext: get all post build analysers..
			Collection ans = KenyaPlugin.getDefault().getPostBuildAnalyserFactories();
			
			//create and call each registered analyser
			for(Iterator iter = ans.iterator(); iter.hasNext();) {
				IKenyaPostBuildAnalyserFactory element = (IKenyaPostBuildAnalyserFactory)iter.next();
				//create configured with current file
				AbstractKenyaPostBuildAnalyser rec
				 = element.createAnalyser(( (IFileEditorInput)getEditorInput() ).getFile());
				
				if(rec!=null) {
					if(cc!=null) {
						rec.setCheckedCode(cc);
					}
					if(jc!=null) {
						rec.setJavaCode(jc);
					}
				}
			}
			
		}
		
	}
	
	/**
	 * returns the current input's name up to and excluding the first dot.
	 * For "foo.bar" this will return "foo" only.
	 * @return
	 */
	protected String getFileName() {
		return getEditorInput().getName().split("\\.")[0];
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		if(adapter.isAssignableFrom(ITextEditor.class)) {
			return kenyaEditor;
		}
		Object o = kenyaEditor.getAdapter(adapter);
		return o==null?super.getAdapter(adapter):o;
	}
}
