/*
 * Created on 22-Oct-2004
 */
package kenya.eclipse.multieditor.kenya;


import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Stack;

import kenya.eclipse.KenyaConstants;
import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.ast.NodeFinder;
import kenya.eclipse.ast.OccurrencesFinder;
import kenya.eclipse.buildext.AbstractKenyaPostBuildAnalyser;
import kenya.eclipse.buildext.util.MarkerPropertySetter;
import kenya.eclipse.multieditor.kenya.correction.KenyaCorrectionAssistant;
import kenya.eclipse.multieditor.kenya.occurrences.ISelectionListenerWithAST;
import kenya.eclipse.multieditor.kenya.occurrences.MethodExitsFinder;
import kenya.eclipse.multieditor.kenya.occurrences.SelectionListenerWithASTManager;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.eclipse.multieditor.text.CharacterPairMatcher;
import kenya.eclipse.preferences.PreferenceConstants;
import kenya.eclipse.ui.actions.AddBlockCommentAction;
import kenya.eclipse.ui.actions.CompositeActionGroup;
import kenya.eclipse.ui.actions.GenerateActionGroup;
import kenya.eclipse.ui.actions.IndentAction;
import kenya.eclipse.ui.actions.RefactorActionGroup;
import kenya.eclipse.ui.actions.RemoveBlockCommentAction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeInformation;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import mediator.IJavaCode;
import mediator.subscription.MediatorInput;
import minijava.node.Node;
import minijava.node.PName;
import minijava.node.Start;
import minijava.node.TIdentifier;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.BadPositionCategoryException;
import org.eclipse.jface.text.DocumentCommand;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentExtension;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.IPositionUpdater;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ISelectionValidator;
import org.eclipse.jface.text.ISynchronizable;
import org.eclipse.jface.text.ITextInputListener;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.ITextViewerExtension;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.jface.text.IWidgetTokenKeeper;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.TextUtilities;
import org.eclipse.jface.text.contentassist.IContentAssistant;
import org.eclipse.jface.text.link.ILinkedModeListener;
import org.eclipse.jface.text.link.LinkedModeModel;
import org.eclipse.jface.text.link.LinkedModeUI;
import org.eclipse.jface.text.link.LinkedPosition;
import org.eclipse.jface.text.link.LinkedPositionGroup;
import org.eclipse.jface.text.link.LinkedModeUI.ExitFlags;
import org.eclipse.jface.text.link.LinkedModeUI.IExitPolicy;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationRulerColumn;
import org.eclipse.jface.text.source.CompositeRuler;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModelExtension;
import org.eclipse.jface.text.source.IOverviewRuler;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.IVerticalRuler;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.jface.text.source.projection.ProjectionSupport;
import org.eclipse.jface.text.source.projection.ProjectionViewer;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.search.ui.IContextMenuConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.VerifyKeyListener;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPartService;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.ide.IGotoMarker;
import org.eclipse.ui.texteditor.AbstractDecoratedTextEditor;
import org.eclipse.ui.texteditor.ContentAssistAction;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.IEditorStatusLine;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;
import org.eclipse.ui.texteditor.SourceViewerDecorationSupport;
import org.eclipse.ui.texteditor.TextOperationAction;
import org.eclipse.ui.texteditor.link.EditorLinkedModeUI;

/**
 * @author Thomas Timbul
 */
public class KenyaEditor extends AbstractDecoratedTextEditor
              implements MediatorInput, IGotoMarker {
	
	/*
	 * takes care of smart inserting brackets and quotes.
	 * can be configured to insert closing brackets and closing quotes
	 * the bracketinserter is installed in createPartControl
	 * @author IBM
	 */
	private class BracketInserter implements VerifyKeyListener, ILinkedModeListener {
		private final String CATEGORY= toString();
		private Stack fBracketLevelStack= new Stack();
		
		private boolean fCloseBrackets= true;
		private boolean fCloseStrings= true;
		private IPositionUpdater fUpdater= new ExclusivePositionUpdater(CATEGORY);

		private boolean hasCharacterToTheRight(IDocument document, int offset, char character) {
			try {
				int end= offset;
				IRegion endLine= document.getLineInformationOfOffset(end);
				int maxEnd= endLine.getOffset() + endLine.getLength();
				while (end != maxEnd && Character.isWhitespace(document.getChar(end)))
					++end;
				
				return end != maxEnd && document.getChar(end) == character;


			} catch (BadLocationException e) {
				// be conservative
				return true;
			}			
		}

		private boolean hasIdentifierToTheLeft(IDocument document, int offset) {
			try {
				int start= offset;
				IRegion startLine= document.getLineInformationOfOffset(start);
				int minStart= startLine.getOffset();
				while (start != minStart && Character.isWhitespace(document.getChar(start - 1)))
					--start;
				
				return start != minStart && Character.isJavaIdentifierPart(document.getChar(start - 1));

			} catch (BadLocationException e) {
				return true;
			}			
		}

		private boolean hasIdentifierToTheRight(IDocument document, int offset) {
			try {
				int end= offset;
				IRegion endLine= document.getLineInformationOfOffset(end);
				int maxEnd= endLine.getOffset() + endLine.getLength();
				while (end != maxEnd && Character.isWhitespace(document.getChar(end)))
					++end;

				return end != maxEnd && Character.isJavaIdentifierPart(document.getChar(end));

			} catch (BadLocationException e) {
				// be conservative
				return true;
			}
		}
		
		/*
		 * @see org.eclipse.jface.text.link.ILinkedModeListener#left(org.eclipse.jface.text.link.LinkedModeModel, int)
		 */
		public void left(LinkedModeModel environment, int flags) {
			
			final BracketLevel level= (BracketLevel) fBracketLevelStack.pop();

			if (flags != ILinkedModeListener.EXTERNAL_MODIFICATION)
				return;

			// remove brackets
			final ISourceViewer sourceViewer= getSourceViewer();
			final IDocument document= sourceViewer.getDocument();
			if (document instanceof IDocumentExtension) {
				IDocumentExtension extension= (IDocumentExtension) document;
				extension.registerPostNotificationReplace(null, new IDocumentExtension.IReplace() {
				
					public void perform(IDocument d, IDocumentListener owner) {
						if ((level.fFirstPosition.isDeleted || level.fFirstPosition.length == 0) && !level.fSecondPosition.isDeleted && level.fSecondPosition.offset == level.fFirstPosition.offset) {
							try {
								document.replace(level.fSecondPosition.offset, level.fSecondPosition.length, null);
							} catch (BadLocationException e) {
							}
						}
						
						if (fBracketLevelStack.size() == 0) {
							document.removePositionUpdater(fUpdater);
							try {
								document.removePositionCategory(CATEGORY);
							} catch (BadPositionCategoryException e) {
							}
						}
					}
	
				});
			}

		}

		/*
		 * @see org.eclipse.jface.text.link.ILinkedModeListener#resume(org.eclipse.jface.text.link.LinkedModeModel, int)
		 */
		public void resume(LinkedModeModel environment, int flags) {
		}

		public void setCloseBracketsEnabled(boolean enabled) {
			fCloseBrackets= enabled;
		}

		public void setCloseStringsEnabled(boolean enabled) {
			fCloseStrings= enabled;
		}

		/*
		 * @see org.eclipse.jface.text.link.ILinkedModeListener#suspend(org.eclipse.jface.text.link.LinkedModeModel)
		 */
		public void suspend(LinkedModeModel environment) {
		}
		
		/*
		 * @see org.eclipse.swt.custom.VerifyKeyListener#verifyKey(org.eclipse.swt.events.VerifyEvent)
		 */
		public void verifyKey(VerifyEvent event) {			

			if (!event.doit || getInsertMode() != SMART_INSERT)
				return;
				
			final ISourceViewer sourceViewer= getSourceViewer();
			IDocument document= sourceViewer.getDocument();

			final Point selection= sourceViewer.getSelectedRange();
			final int offset= selection.x;
			final int length= selection.y;

			switch (event.character) {
			case '(':
				if (hasCharacterToTheRight(document, offset + length, '('))
					return;

				// fall through

			case '[':
					if (!fCloseBrackets)
						return;
					if (hasIdentifierToTheRight(document, offset + length))
						return;
			
				// fall through
			
			case '\'':
				if (event.character == '\'') {
					if (!fCloseStrings)
						return;
					if (hasIdentifierToTheLeft(document, offset) || hasIdentifierToTheRight(document, offset + length))
						return;
				}
				
				// fall through

			case '"':
				if (event.character == '"') {
					if (!fCloseStrings)
						return;
					if (hasIdentifierToTheLeft(document, offset) || hasIdentifierToTheRight(document, offset + length))
						return;
				}
				
				try {		
					ITypedRegion partition= TextUtilities.getPartition(document, KenyaConstants.KENYA_PARTITIONING, offset, true);
//					if (! IDocument.DEFAULT_CONTENT_TYPE.equals(partition.getType()) && partition.getOffset() != offset)
					if (! IDocument.DEFAULT_CONTENT_TYPE.equals(partition.getType()))
						return;
						
					if (!validateEditorInputState())
						return;

					final char character= event.character;
					final char closingCharacter= getPeerCharacter(character);
					final StringBuffer buffer= new StringBuffer();
					buffer.append(character);
					buffer.append(closingCharacter);

					document.replace(offset, length, buffer.toString());
					
					
					BracketLevel level= new BracketLevel();
					fBracketLevelStack.push(level);
					
					LinkedPositionGroup group= new LinkedPositionGroup(); 
					group.addPosition(new LinkedPosition(document, offset + 1, 0, LinkedPositionGroup.NO_STOP));

					LinkedModeModel model= new LinkedModeModel();
					model.addLinkingListener(this);
					model.addGroup(group);
					model.forceInstall();
					
					level.fOffset= offset;
					level.fLength= 2;
			
					// set up position tracking for our magic peers
					if (fBracketLevelStack.size() == 1) {
						document.addPositionCategory(CATEGORY);
						document.addPositionUpdater(fUpdater);
					}
					level.fFirstPosition= new Position(offset, 1);
					level.fSecondPosition= new Position(offset + 1, 1);
					document.addPosition(CATEGORY, level.fFirstPosition);
					document.addPosition(CATEGORY, level.fSecondPosition);
					
					level.fUI= new EditorLinkedModeUI(model, sourceViewer);
					level.fUI.setSimpleMode(true);
					level.fUI.setExitPolicy(new ExitPolicy(closingCharacter, getEscapeCharacter(closingCharacter), fBracketLevelStack));
					level.fUI.setExitPosition(sourceViewer, offset + 2, 0, Integer.MAX_VALUE);
					level.fUI.setCyclingMode(LinkedModeUI.CYCLE_NEVER);
					level.fUI.enter();
					
					
					IRegion newSelection= level.fUI.getSelectedRegion();
					sourceViewer.setSelectedRange(newSelection.getOffset(), newSelection.getLength());
	
					event.doit= false;

				} catch (BadLocationException e) {
					KenyaPlugin.log(e);
				} catch (BadPositionCategoryException e) {
					KenyaPlugin.log(e);
				}
				break;	
			}
		}
	}
	
	/*
	 * used by BracketInserter
	 * @author IBM
	 */
	private static class BracketLevel {
		Position fFirstPosition;
		int fLength;
		int fOffset;
		Position fSecondPosition;
		LinkedModeUI fUI;
	}
	
	/*
	 * Position updater that takes any changes at the borders of
	 * a position to not belong to the position.
	 * @author IBM
	 */
	private static class ExclusivePositionUpdater implements IPositionUpdater {

		/** The position category. */
		private final String fCategory;

		/**
		 * Creates a new updater for the given <code>category</code>.
		 * 
		 * @param category the new category.
		 */
		public ExclusivePositionUpdater(String category) {
			fCategory= category;
		}

		/**
		 * Returns the position category.
		 * 
		 * @return the position category
		 */
		public String getCategory() {
			return fCategory;
		}

		/*
		 * @see org.eclipse.jface.text.IPositionUpdater#update(org.eclipse.jface.text.DocumentEvent)
		 */
		public void update(DocumentEvent event) {

			int eventOffset= event.getOffset();
			int eventOldLength= event.getLength();
			int eventNewLength= event.getText() == null ? 0 : event.getText().length();
			int deltaLength= eventNewLength - eventOldLength;

			try {
				Position[] positions= event.getDocument().getPositions(fCategory);

				for (int i= 0; i != positions.length; i++) {

					Position position= positions[i];

					if (position.isDeleted())
						continue;

					int offset= position.getOffset();
					int length= position.getLength();
					int end= offset + length;

					if (offset >= eventOffset + eventOldLength) 
						// position comes
						// after change - shift
						position.setOffset(offset + deltaLength);
					else if (end <= eventOffset) {
						// position comes way before change -
						// leave alone
					} else if (offset <= eventOffset && end >= eventOffset + eventOldLength) {
						// event completely internal to the position - adjust length
						position.setLength(length + deltaLength);
					} else if (offset < eventOffset) {
						// event extends over end of position - adjust length
						int newEnd= eventOffset;
						position.setLength(newEnd - offset);
					} else if (end > eventOffset + eventOldLength) {
						// event extends from before position into it - adjust offset
						// and length
						// offset becomes end of event, length ajusted acordingly
						int newOffset= eventOffset + eventNewLength;
						position.setOffset(newOffset);
						position.setLength(end - newOffset);
					} else {
						// event consumes the position - delete it
						position.delete();
					}
				}
			} catch (BadPositionCategoryException e) {
				// ignore and return
			}
		}

	}
	
	/*
	 * used by BracketInserter
	 * @author IBM
	 */
	private class ExitPolicy implements IExitPolicy {
		final char fEscapeCharacter;
		
		final char fExitCharacter;
		final int fSize;
		final Stack fStack;
		
		public ExitPolicy(char exitCharacter, char escapeCharacter, Stack stack) {
			fExitCharacter= exitCharacter;
			fEscapeCharacter= escapeCharacter;
			fStack= stack;
			fSize= fStack.size();
		}

		/*
		 * @see org.eclipse.jdt.internal.ui.text.link.LinkedPositionUI.ExitPolicy#doExit(org.eclipse.jdt.internal.ui.text.link.LinkedPositionManager, org.eclipse.swt.events.VerifyEvent, int, int)
		 */
		public ExitFlags doExit(LinkedModeModel model, VerifyEvent event, int offset, int length) {
			
			if (event.character == fExitCharacter) {
				
				if (fSize == fStack.size() && !isMasked(offset)) {
					BracketLevel level= (BracketLevel) fStack.peek();
					if (level.fFirstPosition.offset > offset || level.fSecondPosition.offset < offset)
						return null;
					if (level.fSecondPosition.offset == offset && length == 0)
						// don't enter the character if if its the closing peer
						return new ExitFlags(ILinkedModeListener.UPDATE_CARET, false);
				}
			}
			return null;
		}
			
		private boolean isMasked(int offset) {
			IDocument document= getSourceViewer().getDocument();
			try {
				return fEscapeCharacter == document.getChar(offset - 1);
			} catch (BadLocationException e) {
			}
			return false;
		}
	}
	
	interface ITextConverter {
		void customizeDocumentCommand(IDocument document, DocumentCommand command);
	}
	
	public class KenyaCodeManager extends AbstractKenyaPostBuildAnalyser {
		
		private ICheckedCode recentCode;
		private ICheckedCode recentValidCode;
		
		public KenyaCodeManager() {}
		
		/**
		 * returns the most recent ICheckedCode
		 * @return the most recent ICheckedCode
		 */
		public synchronized ICheckedCode getLastCheckedCode() {
			return recentCode;
		}
		
		/**
		 * returns the most recent valid ICheckedCode
		 * @return the most recent valid ICheckedCode
		 */
		public synchronized ICheckedCode getLastValidCode() {
			return recentValidCode;
		}
		
		/*
		 *  (non-Javadoc)
		 * @see mediator.subscription.ICheckedCodeReceiver#setCheckedCode(mediator.ICheckedCode)
		 */
		public void setCheckedCode(ICheckedCode code) {
			try {
				recentCode = code;
				if(!code.isErroredCode()) {
					recentValidCode = code;
				}
				//do error analysis and add markers
				
				//get the resource
				final IFile resource = (IFile)getEditorInput().getAdapter(IFile.class);
				
				if(resource==null) return; //no file = no annotations
				
				//clear the problem markers (but not its children)
				resource.deleteMarkers(IMarker.PROBLEM, false, IResource.DEPTH_ZERO);
				//we could put DEPTH_INFINITE. It doesnt make a difference since
				//  this is a file and thus has no members anyway
				
				final List allInfo = new ArrayList();
				allInfo.addAll(code.getInfos());
				if(code.isErroredCode()) {
					allInfo.addAll(code.getErrors());
				}
				
				IWorkspaceRunnable r = new IWorkspaceRunnable() {
					public void run(IProgressMonitor monitor) throws CoreException {
						//loop through this list of infos and create the markers/annotations
						for(final Iterator it = allInfo.iterator(); it.hasNext();) {
							ISourceCodeInformation info = (ISourceCodeInformation)it.next();
							
							IDocument doc = getDocumentProvider().getDocument(getEditorInput());
							
							//create marker on resource
							IMarker m = resource.createMarker(IMarker.PROBLEM);
							MarkerPropertySetter.setProps(m, info);
							
							ISourceCodeLocation loc = info.getLocation();
							Position pos = LocationUtils.convert(loc, doc);
							
							MarkerPropertySetter.setLocation(m, pos);
						}
					}
				};
				
				resource.getWorkspace().run(r, null, IWorkspace.AVOID_UPDATE, null);
				
			} catch(CoreException e) {
				//something went wrong when creating or deleting markers
				//this could indicate problems with the resource itself
				//probably something more fundamental is wrong
				//can only ignore this as most likely we don't need annotations
				//if the file itself is broken
			}
		}
		
		/* (non-Javadoc)
		 * @see kenya.eclipse.buildext.IKenyaPostBuildAnalyser#setJavaCode(mediator.IJavaCode)
		 */
		public void setJavaCode(IJavaCode code) {
			//not dealing with this
		}
		
	}
	
	class KenyaSourceViewer extends ProjectionViewer {
		private KenyaCorrectionAssistant fCorrectionAssistant;
		private boolean fIgnoreTextConverters= false;
		
		private List fTextConverters;
		
		public KenyaSourceViewer(Composite parent, IVerticalRuler ruler,
				IOverviewRuler overviewRuler, boolean showsAnnotationOverview,
				int styles) {
			super(parent, ruler, overviewRuler, showsAnnotationOverview, styles);
		}
		
		public void addTextConverter(ITextConverter textConverter) {
			if (fTextConverters == null) {
				fTextConverters= new ArrayList(1);
				fTextConverters.add(textConverter);
			} else if (!fTextConverters.contains(textConverter))
				fTextConverters.add(textConverter);
		}
		
		/*
		 * @see ITextOperationTarget#canDoOperation(int)
		 */
		public boolean canDoOperation(int operation) {
			if (operation == CORRECTIONASSIST_PROPOSALS)
				return isEditable();
			
			return super.canDoOperation(operation);
		}

		/*
		 * @see org.eclipse.jface.text.source.ISourceViewer#configure(org.eclipse.jface.text.source.SourceViewerConfiguration)
		 */
		public void configure(SourceViewerConfiguration configuration) {
			super.configure(configuration);
			fCorrectionAssistant= new KenyaCorrectionAssistant(KenyaEditor.this);
			fCorrectionAssistant.install(this);
//			IAutoEditStrategy smartSemi= new SmartSemicolonAutoEditStrategy(IJavaPartitions.JAVA_PARTITIONING);
//			prependAutoEditStrategy(smartSemi, IDocument.DEFAULT_CONTENT_TYPE);
//			prependAutoEditStrategy(smartSemi, IKenyaPartitions.JAVA_STRING);
//			prependAutoEditStrategy(smartSemi, IJavaPartitions.JAVA_CHARACTER);
		}
		
		/*
		 * @see TextViewer#customizeDocumentCommand(DocumentCommand)
		 */
		protected void customizeDocumentCommand(DocumentCommand command) {
			super.customizeDocumentCommand(command);
			if (!fIgnoreTextConverters && fTextConverters != null) {
				for (Iterator e = fTextConverters.iterator(); e.hasNext();)
					((ITextConverter) e.next()).customizeDocumentCommand(getDocument(), command);
			}
		}
		
		/*
		 * @see ITextOperationTarget#doOperation(int)
		 */
		public void doOperation(int operation) {
		
			if (getTextWidget() == null)
				return;
			
			switch (operation) {
				case CONTENTASSIST_PROPOSALS:
					String msg= fContentAssistant.showPossibleCompletions();
					setStatusLineErrorMessage(msg);
					return;
				case CORRECTIONASSIST_PROPOSALS:
					msg= fCorrectionAssistant.showPossibleCompletions();
					setStatusLineErrorMessage(msg);
					return;
				case UNDO:
					fIgnoreTextConverters= true;
					super.doOperation(operation);
					fIgnoreTextConverters= false;
					return;
				case REDO:
					fIgnoreTextConverters= true;
					super.doOperation(operation);
					fIgnoreTextConverters= false;
					return;
			}
			
			super.doOperation(operation);
		}
		
		public IContentAssistant getContentAssistant() {
			return fContentAssistant;
		}
		
		public void insertTextConverter(ITextConverter textConverter, int index) {
			throw new UnsupportedOperationException();
		}
		
		public void removeTextConverter(ITextConverter textConverter) {
			if (fTextConverters != null) {
				fTextConverters.remove(textConverter);
				if (fTextConverters.size() == 0)
					fTextConverters= null;
			}
		}
		
		/*
		 * @see IWidgetTokenOwner#requestWidgetToken(IWidgetTokenKeeper)
		 */
		public boolean requestWidgetToken(IWidgetTokenKeeper requester) {
			if (WorkbenchHelp.isContextHelpDisplayed())
				return false;
			return super.requestWidgetToken(requester);
		}
		
		/*
		 * @see IWidgetTokenOwnerExtension#requestWidgetToken(IWidgetTokenKeeper, int)
		 * @since 3.0
		 */
		public boolean requestWidgetToken(IWidgetTokenKeeper requester, int priority) {
			if (WorkbenchHelp.isContextHelpDisplayed())
				return false;
			return super.requestWidgetToken(requester, priority);
		}
		
		/*
		 * @see org.eclipse.jface.text.source.ISourceViewerExtension2#unconfigure()
		 * @since 3.0
		 */
		public void unconfigure() {
			if (fCorrectionAssistant != null) {
				fCorrectionAssistant.uninstall();
				fCorrectionAssistant= null;
			}
			super.unconfigure();
		}
		
		// http://dev.eclipse.org/bugs/show_bug.cgi?id=19270
		public void updateIndentationPrefixes() {
			SourceViewerConfiguration configuration= getSourceViewerConfiguration();
			String[] types= configuration.getConfiguredContentTypes(this);
			for (int i= 0; i < types.length; i++) {
				String[] prefixes= configuration.getIndentPrefixes(this, types[i]);
				if (prefixes != null && prefixes.length > 0)
					setIndentPrefixes(prefixes, types[i]);
			}
		}
		
	}
	
	/**
	 * Holds the current occurrence annotations.
	 * @since 3.0
	 */
	private Annotation[] fOccurrenceAnnotations= null;
	
	/**
	 * Returns the lock object for the given annotation model.
	 * 
	 * @param annotationModel the annotation model
	 * @return the annotation model's lock object
	 * @since JDT 3.0
	 */
	private Object getLockObject(IAnnotationModel annotationModel) { 
		if (annotationModel instanceof ISynchronizable)
			return ((ISynchronizable)annotationModel).getLockObject();
		else
			return annotationModel;
	}
	
	/**
	 * Finds and marks occurrence annotations.
	 * 
	 * @since 3.0
	 */
	class OccurrencesFinderJob extends Job {
		private boolean fCanceled= false;
		
		private IDocument fDocument;
		private Position[] fPositions;
		private ISelectionValidator fPostSelectionValidator;
		private IProgressMonitor fProgressMonitor;
		private ISelection fSelection;
		
		public OccurrencesFinderJob(IDocument document, Position[] positions, ISelection selection) {
			super(KenyaEditorMessages.getString("KenyaEditor.markOccurrences.job.name")); //$NON-NLS-1$
			fDocument= document;
			fSelection= selection;
			fPositions= positions;
			
			if (getSelectionProvider() instanceof ISelectionValidator)
				fPostSelectionValidator= (ISelectionValidator)getSelectionProvider(); 
		}
		
		// cannot use cancel() because it is declared final
		void doCancel() {
			fCanceled= true;
			cancel();
		}
		
		private boolean isCanceled() {
			return fCanceled || fProgressMonitor.isCanceled()
				||  fPostSelectionValidator != null && !(fPostSelectionValidator.isValid(fSelection) || fForcedMarkOccurrencesSelection == fSelection)
				|| LinkedModeModel.hasInstalledModel(fDocument);
		}
		
		/*
		 * @see Job#run(org.eclipse.core.runtime.IProgressMonitor)
		 */
		public IStatus run(IProgressMonitor progressMonitor) {
			
			fProgressMonitor= progressMonitor;
			
			if (isCanceled())
				return Status.CANCEL_STATUS;
			
			ITextViewer textViewer= getViewer(); 
			if (textViewer == null)
				return Status.CANCEL_STATUS;
			
			IDocument document= textViewer.getDocument();
			if (document == null)
				return Status.CANCEL_STATUS;
			
			IDocumentProvider documentProvider= getDocumentProvider();
			if (documentProvider == null)
				return Status.CANCEL_STATUS;
		
			IAnnotationModel annotationModel= documentProvider.getAnnotationModel(getEditorInput());
			if (annotationModel == null)
				return Status.CANCEL_STATUS;
			
			// Add occurrence annotations
			int length= fPositions.length;
			Map annotationMap= new HashMap(length);
			for (int i= 0; i < length; i++) {
				
				if (isCanceled())
					return Status.CANCEL_STATUS; 
				
				String message;
				Position position= fPositions[i];
				
				// Create & add annotation
				try {
					message= document.get(position.offset, position.length);
				} catch (BadLocationException ex) {
					// Skip this match
					continue;
				}
				annotationMap.put(
						new Annotation("kenya.eclipse.ui.occurrences", false, message), //$NON-NLS-1$
						position);
			}
			
			if (isCanceled())
				return Status.CANCEL_STATUS;
			synchronized (getLockObject(annotationModel)) {
				if (annotationModel instanceof IAnnotationModelExtension) {
					((IAnnotationModelExtension)annotationModel).replaceAnnotations(fOccurrenceAnnotations, annotationMap);
				} else {
					removeOccurrenceAnnotations();
					Iterator iter= annotationMap.entrySet().iterator();
					while (iter.hasNext()) {
						Map.Entry mapEntry= (Map.Entry)iter.next(); 
						annotationModel.addAnnotation((Annotation)mapEntry.getKey(), (Position)mapEntry.getValue());
					}
				}
				fOccurrenceAnnotations= (Annotation[])annotationMap.keySet().toArray(new Annotation[annotationMap.keySet().size()]);
			}
			
			return Status.OK_STATUS;
		}
	}	
	
	/**
	 * Cancels the occurrences finder job upon document changes.
	 * 
	 * @since 3.0
	 */
	class OccurrencesFinderJobCanceler implements IDocumentListener, ITextInputListener {
				

		/*
		 * @see org.eclipse.jface.text.IDocumentListener#documentAboutToBeChanged(org.eclipse.jface.text.DocumentEvent)
		 */
		public void documentAboutToBeChanged(DocumentEvent event) {
			if (fOccurrencesFinderJob != null)
				fOccurrencesFinderJob.doCancel();
		}

		/*
		 * @see org.eclipse.jface.text.IDocumentListener#documentChanged(org.eclipse.jface.text.DocumentEvent)
		 */
		public void documentChanged(DocumentEvent event) {
		}

		/*
		 * @see org.eclipse.jface.text.ITextInputListener#inputDocumentAboutToBeChanged(org.eclipse.jface.text.IDocument, org.eclipse.jface.text.IDocument)
		 */
		public void inputDocumentAboutToBeChanged(IDocument oldInput, IDocument newInput) {
			if (oldInput == null)
				return;

			oldInput.removeDocumentListener(this);
		}

		/*
		 * @see org.eclipse.jface.text.ITextInputListener#inputDocumentChanged(org.eclipse.jface.text.IDocument, org.eclipse.jface.text.IDocument)
		 */
		public void inputDocumentChanged(IDocument oldInput, IDocument newInput) {
			if (newInput == null)
				return;
			newInput.addDocumentListener(this);
		}

		public void install() {
			ISourceViewer sourceViewer= getSourceViewer();
			if (sourceViewer == null)
				return;
				
			StyledText text= sourceViewer.getTextWidget();			
			if (text == null || text.isDisposed())
				return;

			sourceViewer.addTextInputListener(this);
			
			IDocument document= sourceViewer.getDocument();
			if (document != null)
				document.addDocumentListener(this);			
		}
		
		public void uninstall() {
			ISourceViewer sourceViewer= getSourceViewer();
			if (sourceViewer != null)
				sourceViewer.removeTextInputListener(this);

			IDocumentProvider documentProvider= getDocumentProvider();
			if (documentProvider != null) {
				IDocument document= documentProvider.getDocument(getEditorInput());
				if (document != null)
					document.removeDocumentListener(this);
			}
		}
	}
	
	/**
	 * Internal activation listener.
	 * @since 3.0
	 */
	private class ActivationListener extends ShellAdapter {
		/*
		 * @see org.eclipse.swt.events.ShellAdapter#shellActivated(org.eclipse.swt.events.ShellEvent)
		 */
		public void shellActivated(ShellEvent e) {
			if (fMarkOccurrenceAnnotations && isActivePart()) {
				fForcedMarkOccurrencesSelection= getSelectionProvider().getSelection();
				SelectionListenerWithASTManager.getDefault().forceSelectionChange(KenyaEditor.this, (ITextSelection)fForcedMarkOccurrencesSelection);
			}
		}
		
		/*
		 * @see org.eclipse.swt.events.ShellAdapter#shellDeactivated(org.eclipse.swt.events.ShellEvent)
		 */
		public void shellDeactivated(ShellEvent e) {
			removeOccurrenceAnnotations();
		}
	}
	
	protected boolean isActivePart() {
		IWorkbenchPart part= getActivePart();
		return part != null && part.equals(this);
	}
	
	private IWorkbenchPart getActivePart() {
		IWorkbenchWindow window= getSite().getWorkbenchWindow();
		IPartService service= window.getPartService();
		IWorkbenchPart part= service.getActivePart();
		return part;
	}
	
	/*
	 * array of brackets. This is used to highlight the matching brace
	 * or bracket.
	 */
	protected final static char[] BRACKETS= { '{', '}', '(', ')', '[', ']' };
	
	/** 
	 * Text operation code for requesting correction assist to show correction
	 * proposals for the current position. 
	 */
	public static final int CORRECTIONASSIST_PROPOSALS= 50;
	
	private static char getEscapeCharacter(char character) {
		switch (character) {
			case '"':
			case '\'':
				return '\\';
			default:
				return 0;
		}
	}
	
	private static char getPeerCharacter(char character) {
		switch (character) {
			case '(':
				return ')';
				
			case ')':
				return '(';
				
			case '[':
				return ']';

			case ']':
				return '[';
				
			case '"':
				return character;
				
			case '\'':
				return character;
			
			default:
				throw new IllegalArgumentException();
		}					
	}
	
	protected CompositeActionGroup fActionGroups;
	/** The bracket inserter. */
	private BracketInserter fBracketInserter= new BracketInserter();
	/** The editor's bracket matcher */
	protected CharacterPairMatcher fBracketMatcher = new CharacterPairMatcher(BRACKETS);
	private CompositeActionGroup fContextMenuGroup;
	/** The standard action groups added to the menu */
	private GenerateActionGroup fGenerateActionGroup;
	
	/**
	 * Tells whether all occurrences of the element at the
	 * current caret location are automatically marked in
	 * this editor.
	 * @since 3.0
	 */
	private boolean fMarkOccurrenceAnnotations;
	/**
	 * Tells whether the occurrence annotations are sticky
	 * i.e. whether they stay even if there's no valid Java
	 * element at the current caret position.
	 * Only valid if {@link #fMarkOccurrenceAnnotations} is <code>true</code>.
	 * @since 3.0
	 */
	private boolean fStickyOccurrenceAnnotations;
	/**
	 * Tells whether to mark type occurrences in this editor.
	 * Only valid if {@link #fMarkOccurrenceAnnotations} is <code>true</code>.
	 * @since 3.0
	 */
	private boolean fMarkTypeOccurrences;
	/**
	 * Tells whether to mark method exits in this editor.
	 * Only valid if {@link #fMarkOccurrenceAnnotations} is <code>true</code>.
	 * @since 3.0
	 */
	private boolean fMarkMethodExitPoints;
	
	private ISelection fForcedMarkOccurrencesSelection;
	/**
	 * The internal shell activation listener for updating occurrences.
	 * @since 3.0
	 */
	private ActivationListener fActivationListener= new ActivationListener();
	private ISelectionListenerWithAST fPostSelectionListenerWithAST;
	private OccurrencesFinderJob fOccurrencesFinderJob;
	/** The occcurrences finder job canceler */
	private OccurrencesFinderJobCanceler fOccurrencesFinderJobCanceler;
	/** The projection support */
	private ProjectionSupport fProjectionSupport;
	
	
	private KenyaCodeManager fCodeManager;
	
	/**
	 * 
	 */
	public KenyaEditor() {
		super();
		this.setRulerContextMenuId("#KenyaRulerContext");
		
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#configureSourceViewerDecorationSupport(org.eclipse.ui.texteditor.SourceViewerDecorationSupport)
	 */
	protected void configureSourceViewerDecorationSupport(SourceViewerDecorationSupport support) {
		support.setCharacterPairMatcher(fBracketMatcher);
		support.setMatchingCharacterPainterPreferenceKeys(PreferenceConstants.EDITOR_MATCHING_BRACKETS, PreferenceConstants.EDITOR_MATCHING_BRACKETS_COLOR);
		super.configureSourceViewerDecorationSupport(support);
		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#createActions()
	 */
	protected void createActions() {
		super.createActions();
		
		ResourceBundle bundle = KenyaEditorMessages.getResourceBundle();
		
		fActionGroups= new CompositeActionGroup();
		fContextMenuGroup= new CompositeActionGroup();
		
		// add annotation actions
		Action action= new KenyaSelectMarkerRulerAction2(bundle, "KenyaEditor.RulerAnnotationSelection.", this);
		setAction("AnnotationAction", action);
		
		action= new TextOperationAction(bundle, "CorrectionAssistProposal.", this, CORRECTIONASSIST_PROPOSALS); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.CORRECTION_ASSIST_PROPOSALS);		
		setAction("CorrectionAssistProposal", action); //$NON-NLS-1$
		markAsStateDependentAction("CorrectionAssistProposal", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.QUICK_FIX_ACTION);
		
		action= new ContentAssistAction(bundle, "ContentAssistProposal.", this); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.CONTENT_ASSIST_PROPOSALS);		
		setAction("ContentAssistProposal", action); //$NON-NLS-1$
		markAsStateDependentAction("ContentAssistProposal", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.CONTENT_ASSIST_ACTION);
		
		action= new TextOperationAction(bundle, "ContentAssistContextInformation.", this, ISourceViewer.CONTENTASSIST_CONTEXT_INFORMATION);	//$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.CONTENT_ASSIST_CONTEXT_INFORMATION);		
		setAction("ContentAssistContextInformation", action); //$NON-NLS-1$
		markAsStateDependentAction("ContentAssistContextInformation", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.PARAMETER_HINTS_ACTION);
		
		action= new TextOperationAction(bundle, "Comment.", this, ITextOperationTarget.PREFIX); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.COMMENT);		
		setAction("Comment", action); //$NON-NLS-1$
		markAsStateDependentAction("Comment", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.COMMENT_ACTION);
		
		action= new TextOperationAction(bundle, "Uncomment.", this, ITextOperationTarget.STRIP_PREFIX); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.UNCOMMENT);		
		setAction("Uncomment", action); //$NON-NLS-1$
		markAsStateDependentAction("Uncomment", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.UNCOMMENT_ACTION);
		
		action= new ToggleCommentAction(bundle, "ToggleComment.", this); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.TOGGLE_COMMENT);		
		setAction("ToggleComment", action); //$NON-NLS-1$
		markAsStateDependentAction("ToggleComment", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.TOGGLE_COMMENT_ACTION);
		configureToggleCommentAction();
		
		action= new TextOperationAction(bundle, "Format.", this, ISourceViewer.FORMAT); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.FORMAT);		
		setAction("Format", action); //$NON-NLS-1$
		markAsStateDependentAction("Format", true); //$NON-NLS-1$
		markAsSelectionDependentAction("Format", true); //$NON-NLS-1$		
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.FORMAT_ACTION);
		
		action= new AddBlockCommentAction(bundle, "AddBlockComment.", this);  //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.ADD_BLOCK_COMMENT);		
		setAction("AddBlockComment", action); //$NON-NLS-1$
		markAsStateDependentAction("AddBlockComment", true); //$NON-NLS-1$
		markAsSelectionDependentAction("AddBlockComment", true); //$NON-NLS-1$		
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.ADD_BLOCK_COMMENT_ACTION);
		
		action= new RemoveBlockCommentAction(bundle, "RemoveBlockComment.", this);  //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.REMOVE_BLOCK_COMMENT);		
		setAction("RemoveBlockComment", action); //$NON-NLS-1$
		markAsStateDependentAction("RemoveBlockComment", true); //$NON-NLS-1$
		markAsSelectionDependentAction("RemoveBlockComment", true); //$NON-NLS-1$		
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.REMOVE_BLOCK_COMMENT_ACTION);
		
		action= new IndentAction(bundle, "Indent.", this, false); //$NON-NLS-1$
		action.setActionDefinitionId(IKenyaEditorActionDefinitionIds.INDENT);		
		setAction("Indent", action); //$NON-NLS-1$
		markAsStateDependentAction("Indent", true); //$NON-NLS-1$
		markAsSelectionDependentAction("Indent", true); //$NON-NLS-1$
//		WorkbenchHelp.setHelp(action, IKenyaHelpContextIds.INDENT_ACTION);
		
		action= new IndentAction(bundle, "Indent.", this, true); //$NON-NLS-1$
		setAction("IndentOnTab", action); //$NON-NLS-1$
		markAsStateDependentAction("IndentOnTab", true); //$NON-NLS-1$
		markAsSelectionDependentAction("IndentOnTab", true); //$NON-NLS-1$
		
//		action= new RemoveOccurrenceAnnotations(this);
//		action.setActionDefinitionId(IJavaEditorActionDefinitionIds.REMOVE_OCCURRENCE_ANNOTATIONS);
//		setAction("RemoveOccurrenceAnnotations", action); //$NON-NLS-1$
		
		if (getPreferenceStore().getBoolean(PreferenceConstants.EDITOR_SMART_TAB)) {
			// don't replace Shift Right - have to make sure their enablement is mutually exclusive
			removeActionActivationCode(ITextEditorActionConstants.SHIFT_RIGHT);
			setActionActivationCode("IndentOnTab", '\t', -1, SWT.NONE); //$NON-NLS-1$
		}
		
		fGenerateActionGroup= new GenerateActionGroup(this, ITextEditorActionConstants.GROUP_EDIT);
		ActionGroup rg= new RefactorActionGroup(this, ITextEditorActionConstants.GROUP_EDIT);
		
		fActionGroups.addGroup(rg);
		fActionGroups.addGroup(fGenerateActionGroup);
		
		// We have to keep the context menu group separate to have better control over positioning
		fContextMenuGroup= new CompositeActionGroup(new ActionGroup[] {
			fGenerateActionGroup, 
			rg});
		
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#createCompositeRuler()
	 */
	protected CompositeRuler createCompositeRuler() {
		//composite ruler for line numbers, annotations, etc
		CompositeRuler ruler = new CompositeRuler();
		
		AnnotationRulerColumn column = new AnnotationRulerColumn(VERTICAL_RULER_WIDTH, getAnnotationAccess());
		//this hover makes it possible to select a particular annotation when hovering, rather than
		// having them grouped together.
		
//		column.setHover(new KenyaExpandHover(ruler, getAnnotationAccess(), new IDoubleClickListener() {
//				
//				public void doubleClick(DoubleClickEvent event) {
//					// for now: just invoke ruler double click action
//					triggerAction(ITextEditorActionConstants.RULER_DOUBLE_CLICK);
//				}
//				
//				private void triggerAction(String actionID) {
//					IAction action= getAction(actionID);
//					if (action != null) {
//						if (action instanceof IUpdate)
//							((IUpdate) action).update();
//						// hack to propagate line change
//						if (action instanceof ISelectionListener) {
//							((ISelectionListener)action).selectionChanged(null, null);
//						}
//						if (action.isEnabled())
//							action.run();
//					}
//				}
//				
//			}));
		ruler.addDecorator(0, column);
		
		// create line number ruler
		ruler.addDecorator(1, createLineNumberRulerColumn());
		
		return ruler;
	}
	
	/*
	 * @see org.eclipse.ui.texteditor.ExtendedTextEditor#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		super.createPartControl(parent);
		ProjectionViewer viewer= (ProjectionViewer) getSourceViewer();
		fProjectionSupport= new ProjectionSupport(viewer, getAnnotationAccess(), getSharedColors());
		//add the annotation types as summarisable for when we collapse code
//		fProjectionSupport.addSummarizableAnnotationType("org.eclipse.ui.workbench.texteditor.error");
//		fProjectionSupport.addSummarizableAnnotationType("org.eclipse.ui.workbench.texteditor.warning");
		fProjectionSupport.install();
		viewer.doOperation(ProjectionViewer.TOGGLE);
		
//		for bracket completion
//		copied from JDT***
		
		IPreferenceStore preferenceStore= getPreferenceStore();
		boolean closeBrackets= preferenceStore.getBoolean(PreferenceConstants.EDITOR_CLOSE_BRACKETS);
		boolean closeStrings= preferenceStore.getBoolean(PreferenceConstants.EDITOR_CLOSE_STRINGS);
		
		fBracketInserter.setCloseBracketsEnabled(closeBrackets);
		fBracketInserter.setCloseStringsEnabled(closeStrings);
		
		ISourceViewer sourceViewer= getSourceViewer();
		if (sourceViewer instanceof ITextViewerExtension)
			((ITextViewerExtension) sourceViewer).prependVerifyKeyListener(fBracketInserter);
		
		// ***
		
		if (fMarkOccurrenceAnnotations)
			installOccurrencesFinder();
		
		getEditorSite().getShell().addShellListener(fActivationListener);
	}
	
	/*
	 * @see AbstractTextEditor#createSourceViewer(Composite, IVerticalRuler, int)
	 */
	protected final ISourceViewer createSourceViewer(Composite parent, IVerticalRuler ruler, int styles) {
		
		fAnnotationAccess= createAnnotationAccess();
		fOverviewRuler= createOverviewRuler(getSharedColors());
		
		ISourceViewer viewer= new KenyaSourceViewer(parent, ruler, getOverviewRuler(), true, styles);
		// ensure decoration support has been created and configured.
		getSourceViewerDecorationSupport(viewer);
		
		return viewer;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#dispose()
	 */
	public void dispose() {
		//cancel possible running computation
		fMarkOccurrenceAnnotations= false;
		uninstallOccurrencesFinder();
		
		if (fActivationListener != null) {
			Shell shell= getEditorSite().getShell();
			if (shell != null && !shell.isDisposed())
				shell.removeShellListener(fActivationListener);
			fActivationListener= null;
		}
		
		super.dispose();
	}
	
	/*
	 * @see AbstractTextEditor#editorContextMenuAboutToShow(IMenuManager)
	 */
	public void editorContextMenuAboutToShow(IMenuManager menu) {
		super.editorContextMenuAboutToShow(menu);		
		menu.appendToGroup(ITextEditorActionConstants.GROUP_UNDO, new Separator(IContextMenuConstants.GROUP_OPEN));	
		menu.insertAfter(IContextMenuConstants.GROUP_OPEN, new GroupMarker(IContextMenuConstants.GROUP_SHOW));	
		
		ActionContext context= new ActionContext(getSelectionProvider().getSelection());
		fContextMenuGroup.setContext(context);
		fContextMenuGroup.fillContextMenu(menu);
		fContextMenuGroup.setContext(null);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		if(ICheckedCode.class==adapter) {
			return getKenyaCodeManager().getLastCheckedCode();
		} else if(Reader.class==adapter) {
			return getReader();
		}
		return super.getAdapter(adapter);
	}
	
	protected StyledText getContents() {
		return getSourceViewer().getTextWidget();
	}
	
	public KenyaCodeManager getKenyaCodeManager() {
		if(fCodeManager==null) {
			fCodeManager = new KenyaCodeManager();
		}
		return fCodeManager;
		
	}
	
	/* (non-Javadoc)
	 * @see mediator.subscription.MediatorInput#getReader()
	 */
	public synchronized Reader getReader() {
		try {
			return new StringReader(getSourceViewer().getDocument().get());
		} catch(Exception e) {
			//ignore, shutting down?
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#getTitle()
	 */
	public String getTitle() {
		return "Kenya";
	}
	
	public final ISourceViewer getViewer() {
		return getSourceViewer();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#handlePreferenceStoreChanged(org.eclipse.jface.util.PropertyChangeEvent)
	 */
	protected void handlePreferenceStoreChanged(PropertyChangeEvent event) {
		try {			
			
			ISourceViewer sourceViewer= getSourceViewer();
			if (sourceViewer == null)
				return;
				
			String property= event.getProperty();	
			
			if (PreferenceConstants.EDITOR_MARK_OCCURRENCES.equals(property)) {
				if (event.getNewValue() instanceof Boolean) {
					boolean markOccurrenceAnnotations= ((Boolean)event.getNewValue()).booleanValue();
					if (markOccurrenceAnnotations != fMarkOccurrenceAnnotations) {
						fMarkOccurrenceAnnotations= markOccurrenceAnnotations;
						if (!fMarkOccurrenceAnnotations)
							uninstallOccurrencesFinder();
						else
							installOccurrencesFinder();
					}
				}
				return;
			}
			
			if (PreferenceConstants.EDITOR_MARK_TYPE_OCCURRENCES.equals(property)) {
				if (event.getNewValue() instanceof Boolean)
					fMarkTypeOccurrences= ((Boolean)event.getNewValue()).booleanValue();
				return;
			}
			if (PreferenceConstants.EDITOR_MARK_METHOD_EXIT_POINTS.equals(property)) {
				if (event.getNewValue() instanceof Boolean)
					fMarkMethodExitPoints= ((Boolean)event.getNewValue()).booleanValue();
				return;
			}
			if (PreferenceConstants.EDITOR_STICKY_OCCURRENCES.equals(property)) {
				if (event.getNewValue() instanceof Boolean)
					fStickyOccurrenceAnnotations= ((Boolean)event.getNewValue()).booleanValue();
				return;
			}
//			if (PreferenceConstants.EDITOR_SEMANTIC_HIGHLIGHTING_ENABLED.equals(property)) {
//				if (isSemanticHighlightingEnabled())
//					installSemanticHighlighting();
//				else
//					uninstallSemanticHighlighting();
//				return;
//			}
			
		} finally {
			super.handlePreferenceStoreChanged(event);
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#initializeEditor()
	 */
	protected void initializeEditor() {
		super.initializeEditor();
		setSourceViewerConfiguration(new KenyaSourceViewerConfiguration());
		IPreferenceStore store = KenyaPlugin.getDefault().getCombinedPreferenceStore();
		setPreferenceStore(store);
		fMarkOccurrenceAnnotations= store.getBoolean(PreferenceConstants.EDITOR_MARK_OCCURRENCES);
		fStickyOccurrenceAnnotations= store.getBoolean(PreferenceConstants.EDITOR_STICKY_OCCURRENCES);
		fMarkTypeOccurrences= store.getBoolean(PreferenceConstants.EDITOR_MARK_TYPE_OCCURRENCES);
		fMarkMethodExitPoints= store.getBoolean(PreferenceConstants.EDITOR_MARK_METHOD_EXIT_POINTS);
	}
	
	/*
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#initializeKeyBindingScopes()
	 */
	protected void initializeKeyBindingScopes() {
		setKeyBindingScopes(new String[] { "kenya.eclipse.multieditor.kenyaEditorScope" });  //$NON-NLS-1$
	}
	
	protected void installOccurrencesFinder() {
		
		fMarkOccurrenceAnnotations= true;
		fPostSelectionListenerWithAST= new ISelectionListenerWithAST() {
			public void selectionChanged(IEditorPart part, ITextSelection selection, Start astRoot) {
				updateOccurrenceAnnotations(selection, astRoot);
			}
		};
		SelectionListenerWithASTManager.getDefault().addListener(this, fPostSelectionListenerWithAST);
		if (getSelectionProvider() != null) {
			fForcedMarkOccurrencesSelection= getSelectionProvider().getSelection();
			SelectionListenerWithASTManager.getDefault().forceSelectionChange(this, (ITextSelection)fForcedMarkOccurrencesSelection);
		}
		
		if (fOccurrencesFinderJobCanceler == null) {
			fOccurrencesFinderJobCanceler= new OccurrencesFinderJobCanceler();
			fOccurrencesFinderJobCanceler.install();
		}
	}
	protected boolean isMarkingOccurrences() {
		return fMarkOccurrenceAnnotations;
	}
	
	void removeOccurrenceAnnotations() {
		IDocumentProvider documentProvider= getDocumentProvider();
		if (documentProvider == null)
			return;
		IAnnotationModel annotationModel= documentProvider.getAnnotationModel(getEditorInput());
		if (annotationModel == null || fOccurrenceAnnotations == null)
			return;
		
		synchronized (getLockObject(annotationModel)) {
			if (annotationModel instanceof IAnnotationModelExtension) {
				((IAnnotationModelExtension)annotationModel).replaceAnnotations(fOccurrenceAnnotations, null);
			} else {
				for (int i= 0, length= fOccurrenceAnnotations.length; i < length; i++)
					annotationModel.removeAnnotation(fOccurrenceAnnotations[i]);
			}
			fOccurrenceAnnotations= null;
		}
	}
	
	/**
	 * Sets the given message as error message to this editor's status line.
	 * 
	 * @param msg message to be set
	 */
	protected void setStatusLineErrorMessage(String msg) {
		IEditorStatusLine statusLine= (IEditorStatusLine) getAdapter(IEditorStatusLine.class);
		if (statusLine != null)
			statusLine.setMessage(true, msg, null);	
	}
	
	/**
	 * Sets the given message as message to this editor's status line.
	 * 
	 * @param msg message to be set
	 * @since 3.0
	 */
	protected void setStatusLineMessage(String msg) {
		IEditorStatusLine statusLine= (IEditorStatusLine) getAdapter(IEditorStatusLine.class);
		if (statusLine != null)
			statusLine.setMessage(false, msg, null);	
	}
	
	protected void uninstallOccurrencesFinder() {
		fMarkOccurrenceAnnotations= false;
		
		if (fOccurrencesFinderJob != null) {
			fOccurrencesFinderJob.cancel();
			fOccurrencesFinderJob= null;
		}
		
		if (fOccurrencesFinderJobCanceler != null) {
			fOccurrencesFinderJobCanceler.uninstall();
			fOccurrencesFinderJobCanceler= null;
		}
		
		if (fPostSelectionListenerWithAST != null) {
			SelectionListenerWithASTManager.getDefault().removeListener(this, fPostSelectionListenerWithAST);
			fPostSelectionListenerWithAST= null;
		}
		
		removeOccurrenceAnnotations();
	}
	
	/**
	 * Updates the occurrences annotations based
	 * on the current selection.
	 * 
	 * @param selection the text selection
	 * @param astRoot the compilation unit AST
	 * @since 3.0
	 */
	protected void updateOccurrenceAnnotations(ITextSelection selection, Start astRoot) {
		
		if (fOccurrencesFinderJob != null)
			fOccurrencesFinderJob.cancel();
		
		if (!fMarkOccurrenceAnnotations)
			return;
		if (fCodeManager.recentCode!=fCodeManager.recentValidCode) {
			removeOccurrenceAnnotations();
			return;
		}
		if (astRoot == null || selection == null)
			return;
		
		IDocument document= getSourceViewer().getDocument();
		if (document == null)
			return;
		
		List matches= new ArrayList();
		String message;
		if (matches.size() == 0) {
			ISourceCodeLocation selectedLocation
			  = LocationUtils.convert(selection.getOffset(), selection.getLength() , document);
			if (fMarkMethodExitPoints || fMarkTypeOccurrences) {
				MethodExitsFinder finder = new MethodExitsFinder();
				Node node = NodeFinder.perform(astRoot, document, selectedLocation);
				message = finder.initialise(node);
				if (message == null) {
					matches = finder.perform();
				}
			}
			if (matches.size() == 0) {
				Node node= NodeFinder.perform(astRoot, document, selectedLocation);
				if (!(node instanceof TIdentifier || node instanceof PName)) {
					if (!fStickyOccurrenceAnnotations)
						removeOccurrenceAnnotations();
					return;
				}
				
				// Find the matches && extract positions so we can forget the AST
				OccurrencesFinder finder = new OccurrencesFinder();
				message= finder.initialize(astRoot, fCodeManager.recentValidCode, document, selectedLocation);
				if (message == null)
					matches= finder.perform();
			} else if (!fMarkMethodExitPoints) {
				if (!fStickyOccurrenceAnnotations)
					removeOccurrenceAnnotations();
				return;
			}
		}
		
		Position[] positions= new Position[matches.size()];
		int i= 0;
		for (Iterator each= matches.iterator(); each.hasNext();) {
			Node currentNode= (Node)each.next();
			ISourceCodeLocation loc = AdvancedPositionFinder.getFullLocation(currentNode, document);
			positions[i++] = LocationUtils.convert(loc, document);
		}
		
		fOccurrencesFinderJob= new OccurrencesFinderJob(document, positions, selection);
		//fOccurrencesFinderJob.setPriority(Job.DECORATE);
		//fOccurrencesFinderJob.setSystem(true);
		//fOccurrencesFinderJob.schedule();
		fOccurrencesFinderJob.run(new NullProgressMonitor());
	}
	
	/**
	 * Configures the toggle comment action
	 * 
	 * @since 3.0
	 */
	private void configureToggleCommentAction() {
		IAction action= getAction("ToggleComment"); //$NON-NLS-1$
		if (action instanceof ToggleCommentAction) {
			ISourceViewer sourceViewer = getSourceViewer();
			SourceViewerConfiguration configuration= getSourceViewerConfiguration();
			((ToggleCommentAction)action).configure(sourceViewer, configuration);
		}
	}
}
