/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Thomas Timbul - re-implementation for Kenya
 *******************************************************************************/
package kenya.eclipse.multieditor.kenya.completion;


import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import kenya.builtIns.BuiltInMethodsLoader;
import kenya.builtIns.IBuiltInMethod;
import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.ast.IBinding;
import kenya.eclipse.ast.NodeFinder;
import kenya.eclipse.ast.NodeFinder2;
import kenya.eclipse.ast.NodeToString;
import kenya.eclipse.ast.NodeTools;
import kenya.eclipse.ast.Resolver;
import kenya.eclipse.ast.SimpleASTProvider;
import kenya.eclipse.ast.bindings.ClassBinding;
import kenya.eclipse.ast.bindings.EnumBinding;
import kenya.eclipse.ast.bindings.VariableBinding;
import kenya.eclipse.multieditor.kenya.KenyaEditor;
import kenya.eclipse.multieditor.kenya.util.GroupedArrayList;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.eclipse.ui.KenyaImages;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.types.KType;
import mediator.ICheckedCode;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AClassDecDeclaration;
import minijava.node.AClassInnerDeclaration;
import minijava.node.AClassTypeReferenceType;
import minijava.node.AEnumDecDeclaration;
import minijava.node.AEnumList;
import minijava.node.AReferenceTypeType;
import minijava.node.AVarDecInnerDeclaration;
import minijava.node.Node;
import minijava.node.Start;
import minijava.node.TIdentifier;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.contentassist.CompletionProposal;
import org.eclipse.jface.text.contentassist.ContextInformation;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContentAssistProcessor;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.text.contentassist.IContextInformationValidator;
import org.eclipse.swt.graphics.Image;

//TODO: Completion does not cope with returns from method calls (array or not)

/**
 * Example Java completion processor.
 */
public class KenyaCompletionProcessor implements IContentAssistProcessor {
	
	private static final String FIELDS = "_fields"; //group name for fields and variables
	private static final String FUNCTIONS = "_functions"; //group name for methods
	private static final String BUILTINS = "_built_ins"; //group name for builtin methods
	
	private final static String ID = "\\w+";
	private final static String STATEMENT = "(.*;)";
	private final static String METH = "(("+ID+")" + "\\s*\\()";
	private final static String QUAL = "(\\s*\\.\\s*"+ID+"(\\[.*\\])*)";
	private final static String VAR_ACCESS = "("+METH+"*("+ID+"(\\[.*\\])*"+QUAL+"*))"; //accessing a variable
	
	private static final Pattern fieldAccess = Pattern.compile(VAR_ACCESS+"!?\\s*\\.\\s*("+ID+"(\\[.*\\])*)?"); //field access
	private static final Pattern methCall = Pattern.compile("\\s*"+METH+"\\s*"); //call, but not declaration
	private static final Pattern assign   = Pattern.compile(".*\\s*=\\s*"); //assignment
	private static final Pattern nothing  = Pattern.compile("\\s*(.*\\s+)?(\\w+([\\s!]*\\()?)?[\\s!]*"); //nothing special, might be method call, but last word is an identifier 
	private static final String objecttype = "\\@#\\d+"; // generic object type in builtin methods
	
	protected IContextInformationValidator fValidator;
	
	/**
	 * returns String with the most significant part extracted from argument.
	 * lastMethCall is specified to preserve the last methodcall.
	 * For example:
	 * <code><pre>
	 *  input       output     lastMethCall
	 *  &quot;a(b(&quot;         &quot;b(&quot;           &quot;&quot;
	 *  &quot;a=b; a.&quot;    &quot;a.&quot;          &quot;a.&quot;
	 *  &quot;a( );  b.&quot;    &quot;b.&quot;          &quot;b.&quot;
	 * </pre></code>
	 * @param text
	 * @return
	 */
	protected String shortenToLast(String text, boolean lastMethCall) {
		int semi = text.lastIndexOf(';') + 1;
		if(semi>0) {
			text = text.substring(semi);
		}
		int eq = text.lastIndexOf('=')+1;
		if(eq>0) {
			text = text.substring(eq);
		}
		int brace = text.lastIndexOf('(');
		if(brace>0 && lastMethCall) {
			brace = text.substring(0, brace).lastIndexOf('(');
		}
		if(brace>=0) { //nested meth call
			text = text.substring(brace+1);
		}
		int exclam = text.lastIndexOf('!')+1;
		if(exclam>0) {
			text = text.substring(exclam);
		}
		return text;
	}
	
	protected KenyaEditor getKenyaEditor(ITextViewer viewer) {
		
		//get the code, which should be found in the current editor
		KenyaEditor editor = (KenyaEditor)KenyaPlugin.getActivePage()
				.getActiveEditor().getAdapter(KenyaEditor.class);
		
		//if the editor is null, then this thing does not apply
		if(editor==null || editor.getViewer()!=viewer) {
			return null;
		}
		
		return editor;
		//the code is guaranteed to be valid, so don't need to check
		
	}
	
	/* (non-Javadoc)
	 * Method declared on IContentAssistProcessor
	 */
	public ICompletionProposal[] computeCompletionProposals(ITextViewer viewer, int documentOffset) {
		
		GroupedArrayList list = new GroupedArrayList(3);
		list.addGroup(FIELDS);
		list.addGroup(FUNCTIONS);
		list.addGroup(BUILTINS);
		
		KenyaEditor editor = getKenyaEditor(viewer);
		if(editor==null) {
			return new ICompletionProposal[0];
		}
		ICheckedCode code = editor.getKenyaCodeManager().getLastValidCode();
		if(code==null) {
			code = editor.getKenyaCodeManager().getLastCheckedCode();
		}
		try {
			//what is the variable before the current offset...
			IDocument doc = viewer.getDocument();
			IRegion lineinfo = doc.getLineInformationOfOffset(documentOffset);
			final int start = lineinfo.getOffset();
			
			//the line text
			String text = doc.get(start, documentOffset-start);
			int diff = 0;
			String ntext = shortenToLast(text, false);
			if(!text.equals(ntext)) {
				diff = text.length() - ntext.length();
			}
			
			//obtain an AST (required for field access stuff)
			Start ast = SimpleASTProvider.getAST(editor.getReader());
			if(ast==null) {
				//this one will work, unless no function is declared
				//usually there is at least the 'main' method
				ast = SimpleASTProvider.getAST(code);
			}
			
			Matcher m = fieldAccess.matcher(text);
			if(m.find(0) && m.end()==text.length()) {
				String last = m.group(4);
				int lastindex = start+m.start(4);
				int lastdot = last.lastIndexOf('.')+1;
				if(lastdot>0) last = last.substring(lastdot);
				int abcde = m.groupCount();
				String match = (m.start(9)>0)?m.group(9):"";
				String lastNoAccess = last.indexOf('[')>0?last.substring(0, last.indexOf('[')):last;
				Position pos = new Position(lastindex, lastNoAccess.length());
//				System.out.println(doc.get(pos.offset, pos.length));
				Node n = NodeFinder.perform(ast, doc, pos);
				if(!(n instanceof TIdentifier)) {
					if(n==null) {
						IFunction f = Resolver.getContainingMethod(LocationUtils.convert(pos, doc), code);
						if(f!=null) {
							n = f.getDeclarationNode();
						}
					}
					if(n!=null)
						n = NodeFinder2.perform(n, lastNoAccess);
				}
				if(n instanceof TIdentifier) {
					ArrayList l = new ArrayList();
					computeFieldAccessProposals((TIdentifier)n, code, l, documentOffset, match, !lastNoAccess.equals(last));
					ICompletionProposal[] pps
					  = (ICompletionProposal[])l.toArray(new ICompletionProposal[l.size()]);
					return sort(pps);
				}
			}
			
//			m = methCall.matcher(text);
//			if(m.find() && m.end()==text.length()) {
//				System.out.println("looks like method call");
//			}
//			
//			m = assign.matcher(text);
//			if(m.find() && m.end()==text.length()) {
//				System.out.println("looks like assignment");
//				text = text.substring(text.indexOf('='));
//			}
			
			m = nothing.matcher(ntext);
			
			if(m.find()) {
				int s = m.start(2);
				int e = m.end();
				String match = (s>=0 /*&& e!=text.length()*/)
				  ?m.group(2)
				  :"";
				
				ArrayList l = new ArrayList();
				ISourceCodeLocation loc = LocationUtils.convert(documentOffset, 0, doc);
				IFunction fun = Resolver.getContainingMethod(loc, code);
				if(fun==null) throw new BadLocationException(); //means of returning
				
				computeVariableProposals(code, loc, l, documentOffset, shortenToLast(match, false));
				ICompletionProposal[] vars
				  = sort((ICompletionProposal[])l.toArray(new ICompletionProposal[l.size()]));
				list.addAllToGroup(FIELDS, vars);
				l.clear();
				computeFunctionCallProposals(code, l, documentOffset, match);
				ICompletionProposal[] funs
			    = sort((ICompletionProposal[])l.toArray(new ICompletionProposal[l.size()]));
				list.addAllToGroup(FUNCTIONS, funs);
				l.clear();
				computeBuiltinMethodCallProposals(l, documentOffset, match);
				ICompletionProposal[] bims
		      = sort((ICompletionProposal[])l.toArray(new ICompletionProposal[l.size()]));
				list.addAllToGroup(BUILTINS, bims);
				
			}
			
			return (ICompletionProposal[])list.toArray(new ICompletionProposal[list.size()]);
			
		} catch(BadLocationException e) {
		}
		
		return new ICompletionProposal[0];
	}
	
	/**
	 * computes field access proposals on the variable referenced by tid (gets the fields of tid)
	 * and adds them to list. The computation occurs for the given code at given offset. Only
	 * completions that match the specified prefix are added.
	 * isArrayAccess is specified to be true if the proposals should be
	 * resolved against the contents of an array rather than the array itself.
	 */
	protected void computeFieldAccessProposals(TIdentifier tid, ICheckedCode code, List list, int offset, String matchPrefix, boolean isArrayAccess) {
		IBinding b = Resolver.resolve(tid, code);
		if(b!=null) {
			if(b instanceof VariableBinding && ((VariableBinding)b).isArray() && !isArrayAccess) {
				//for arrays (without access)
				//the only possibility is the .length field..
				String name = "length";
				String insert = name.substring(matchPrefix.length());
				String[] format = new String[] {name, "int"};
				String display = MessageFormat.format("{0}  {1}", format);
				
				ICompletionProposal comp = new CompletionProposal(insert, offset, 0, insert.length(), KenyaImages.getImage(KenyaImages.IMG_OBJ_FIELD_PUBLIC), display, null, "");
				list.add(comp);
				return;
			} else if(b.getType()!=null) {
				if(b.getType() instanceof AReferenceTypeType) {
					AClassTypeReferenceType type = (AClassTypeReferenceType)( (AReferenceTypeType)b.getType() ).getReferenceType();
					TIdentifier classname = type.getIdentifier();
					IBinding bb = Resolver.resolve(classname, code);
					if(bb!=null) {
						String id;
						List ll;
						if(bb instanceof ClassBinding) {
							//for a class, iterate through the inner declaration and return a list of all fields with their types
							AClassDecDeclaration n = (AClassDecDeclaration)bb.getDeclaringNode();
							id = NodeToString.toString(n.getIdentifier());
							ll = n.getClassInnerDeclaration();
							
						} else if(bb instanceof EnumBinding) {
							//for an enumeration, iterate through the inner declaration and return a list of all the enumerated types
							AEnumDecDeclaration n = (AEnumDecDeclaration)bb.getDeclaringNode();
							id = NodeToString.toString(n.getIdentifier());
							
							AEnumList el = (AEnumList)n.getEnumList();
							ll = el.getCommaEnumList();
						} else {
							return;
						}
						
						computeClassInnerCompletions(id, ll.iterator(), list, offset, matchPrefix);
						
					}
				}
			}
		}
	}
	
	/**
	 * computes variable/constant proposals and adds them to list.
	 * The computation occurs for the given code at given offset. Only
	 * completions that match the specified prefix are added.
	 * loc is given as a limit: only variables declared BEFORE this
	 * location are added (to respect scope).
	 */
	protected void computeVariableProposals(ICheckedCode code, ISourceCodeLocation loc, List list, int offset, String matchPrefix) {
		IFunction fun = Resolver.getContainingMethod(loc, code);
		if(fun==null) return;
		VariableDeclarationFinder finder = new VariableDeclarationFinder(fun, loc);
		List l = finder.perform();
		for(Iterator it = l.iterator(); it.hasNext();) {
			TypeIdentifierPair pair = (TypeIdentifierPair)it.next();
			String insert = pair.getIdString();
			ISourceCodeLocation location = NodeTools.getLocation(pair.getId());
			if(!insert.startsWith(matchPrefix) || !LocationUtils.occursBefore(location, loc)) {
				continue;
			}
			String display = insert + "  " + pair.getTypeString();
			
			IContextInformation info = null;//new ContextInformation(insert, "");
			insert = insert.substring(matchPrefix.length());
			Image image = KenyaImages.getImage(
					(pair.isLocal())
					  ?KenyaImages.IMG_OBJ_VAR_LOCAL
					  :KenyaImages.IMG_OBJ_FIELD_PUBLIC
					);
			list.add( new CompletionProposal(insert, offset, 0, insert.length(), image, display, info, "") );
			
			
		}
	}
	
	/**
	 * computes function call proposals and adds them to list. 
	 * The computation occurs for the given code at given offset. Only
	 * completions that match the specified prefix are added.
	 * ContextInformation is created for the parameters of each method.
	 */
	protected void computeFunctionCallProposals(ICheckedCode code, List list, int offset, String matchPrefix) {
		IFunction[] funs = code.getFunctions();
		for (int i= 0; i < funs.length; i++) {
			IFunction f = funs[i];
			String insert = f.getName()+"()";
			if(!insert.startsWith(matchPrefix)) {
				continue;
			}
			String display = NodeToString.toString(f.getDeclarationNode());
			String block = NodeToString.toString(f.getDeclarationNode().getBlock());
			int firstspace = display.indexOf(' ');
			int decl = display.indexOf(block);
			String type = display.substring(0, firstspace);
			display = display.substring(firstspace+1, decl-1);
			
			display += "  " + f.getReturnType().getName();
			
			StringBuffer params = new StringBuffer();
			for(int j = 0; j < f.getArguments().length; j++) {
				IVariable var = f.getArguments()[j];
				params.append(var.getType().getName()).append(' ').append(var.getName()).append(", ");
			}
			//the new cursor position.. inside brackets if there are args, after if not
			params.setLength(Math.max(0, params.length()-2)); //cut off last comma
			IContextInformation info = params.length()>0?new ContextInformation(insert, params.toString()):null;
			insert = insert.substring(matchPrefix.length());
			int cursorPos = insert.length()-(f.getArguments().length>0?1:0);
			
			list.add( new CompletionProposal(insert, offset, 0, cursorPos, KenyaImages.getImage(KenyaImages.IMG_OBJ_METH_PUBLIC), display, info, MessageFormat.format("Method call: {0}", new Object[] { funs[i]})) );
		}
		
	}
	
	/**
	 * computes builtin method call proposals and adds them to list. 
	 * The computation occurs for  given offset. Only
	 * completions that match the specified prefix are added.
	 * ContextInformation is created for the parameters of each method.
	 */
	protected void computeBuiltinMethodCallProposals(List list, int offset, String matchPrefix) {
		//add all built in methods
		
		boolean noBrackets = false;
		int bracket = matchPrefix.indexOf('('); 
		if(bracket>0) {
			noBrackets = true;
			matchPrefix = matchPrefix.substring(0, bracket);
		}
		
		Set builtIns = BuiltInMethodsLoader.getBuiltInMethods();
		for(Iterator it = builtIns.iterator(); it.hasNext();) {
			IBuiltInMethod bim = (IBuiltInMethod)it.next();
			String name = bim.getName();
			if(!name.startsWith(matchPrefix)) {
				continue;
			}
			String params = ktypeParametersToString(bim.getMethodParameters());
			if(params.indexOf('#')>0 || name.indexOf('#')>0) {
				//generic 'Object' parameters. Replace with 'Anything' as type (so user understands)
				params = params.replaceAll(objecttype, "Anything");
			}
			String insert = name;
			if(!noBrackets) insert += "()";
			
			IContextInformation info = params.length()>0?new ContextInformation(insert, params.toString()):null;
			insert = insert.substring(matchPrefix.length());
			int cursorPos = Math.max(0, insert.length()-(bim.getMethodParameters().size()>0?1:0));
			
			String display = name+'('+params+')' + "  " + bim.getReturnType().getName();
			
			list.add( new CompletionProposal(insert, offset, 0, cursorPos, KenyaImages.getImage(KenyaImages.IMG_OBJ_METH_PUBLIC), display, info, MessageFormat.format("Method call: {0}", new Object[] { name })) );
		}
	}
	
	/**
	 * computes the CompletionProposals based on the contents of the iterator, which
	 * MUST contain AClassInnerDeclarations ONLY. The results are added to the given
	 * list in the order they are returned by it.next(). Only items that match the
	 * given prefix are added and that prefix removed, so that the inserted String
	 * COMPLETES the prefix; to add all inner declarations, the empty string should be passed here.
	 * Offset specifies the offset of insertion that has to be supplied as argument
	 * to the constructor of ContextInformation.
	 * The completions are shown with their name, type and containgType, which must not be null.
	 */
	protected void computeClassInnerCompletions(String containingType, Iterator it, List list, int offset, String matchPrefix) {
		for(; it.hasNext();) {
			AClassInnerDeclaration inner = (AClassInnerDeclaration)it.next();
			
			String name = null;
			String type = null;
			
			if(inner.getInnerDeclaration() instanceof AVarDecInnerDeclaration) {
				AVarDecInnerDeclaration v = (AVarDecInnerDeclaration)inner.getInnerDeclaration();
				name = NodeToString.toString(v.getIdentifier());
				type = NodeToString.toString(v.getType());
			} else if(inner.getInnerDeclaration() instanceof AArrayDecInnerDeclaration) {
				AArrayDecInnerDeclaration v = (AArrayDecInnerDeclaration)inner.getInnerDeclaration();
				name = NodeToString.toString(v.getIdentifier());
				type = NodeToString.toString(v.getType());
			}
			
			if(name==null || !name.startsWith(matchPrefix)) {
				continue;
			}
//			String info = MessageFormat.format("", new String[] {});
			IContextInformation information = null;
//			  = new ContextInformation(name, info);
			
			String insert = name.substring(matchPrefix.length());
			String[] format = new String[] {name, type, containingType};
			String display = MessageFormat.format("{0}  {1} - {2}", format);
			
			ICompletionProposal comp = new CompletionProposal(insert, offset, 0, insert.length(), KenyaImages.getImage(KenyaImages.IMG_OBJ_FIELD_PUBLIC), display, information, "");
			list.add(comp);
			
		}
	}
	
	/**
	 * builds a String of comma separated type variable pairs
	 * from all KTypes in list
	 * @param list List of KTypes
	 * @return String representation of the list
	 */
	protected String ktypeParametersToString(List list) {
		StringBuffer buf = new StringBuffer();
		for(Iterator it = list.iterator(); it.hasNext();) {
			KType t = (KType)it.next();
			buf.append(t.getName()); //parameter 'type'
			buf.append(' ');
			buf.append('x'); //parameter 'name' (which is not specified, but we can just put sth)
			if(it.hasNext()) {
				buf.append(", ");
			}
		}
		return buf.toString();
	}
	
	/**
	 * sorts the given array by order of alphabet of the actual completions
	 * @param cps
	 * @return
	 */
	protected ICompletionProposal[] sort(ICompletionProposal[] cps) {
		
		Comparator c = new Comparator() {
			public int compare(Object o1, Object o2) {
				if(o1 instanceof ICompletionProposal && o2 instanceof ICompletionProposal) {
					return compare((ICompletionProposal)o1, (ICompletionProposal)o2);
				}
				return 0;
			}
			public int compare(ICompletionProposal a, ICompletionProposal b) {
				return a.getDisplayString().compareTo(b.getDisplayString());
			}
		};
		
		Arrays.sort(cps, c);
		
		return cps;
	}
	
	/* (non-Javadoc)
	 * Method declared on IContentAssistProcessor
	 */
	public IContextInformation[] computeContextInformation(ITextViewer viewer, int documentOffset) {
		
		ArrayList list = new ArrayList();
		
		KenyaEditor editor = getKenyaEditor(viewer);
		if(editor==null) {
			return new IContextInformation[0];
		}
		ICheckedCode code = editor.getKenyaCodeManager().getLastValidCode();
		
		try {
			//what is the variable before the current offset...
			IDocument doc = viewer.getDocument();
			IRegion lineinfo = doc.getLineInformationOfOffset(documentOffset);
			int start = lineinfo.getOffset();
			
			//the line text
			String text = doc.get(start, documentOffset-start);
			text = shortenToLast(text, true);
			
			Matcher m = methCall.matcher(text);
			if(m.find() && m.start()==0) {
				
				String meth = m.group(2);
				
				IFunction[] funs = code.getFunctions();
				for(int i=0; i<funs.length; i++) {
					IFunction f = funs[i];
					String insert = f.getName();
					if(!insert.equals(meth)) {
						continue;
					}
					
					StringBuffer params = new StringBuffer();
					for(int j = 0; j < f.getArguments().length; j++) {
						IVariable var = f.getArguments()[j];
						params.append(var.getType().getName()).append(' ').append(var.getName()).append(", ");
					}
					
					params.setLength(Math.max(0, params.length()-2)); //cut off last comma
					insert += "("+ params.toString()+")";
					IContextInformation info = params.length()>0?new ContextInformation(insert, params.toString()):null;
					if(info!=null) list.add(info);
				}
				
				if(list.size()==0) {
					Set builtIns = BuiltInMethodsLoader.getBuiltInMethods();
					for(Iterator it = builtIns.iterator(); it.hasNext();) {
						IBuiltInMethod bim = (IBuiltInMethod)it.next();
						String name = bim.getName();
						if(!name.startsWith(meth)) {
							continue;
						}
						String params = ktypeParametersToString(bim.getMethodParameters());
						if(params.indexOf('#')>0 || name.indexOf('#')>0) {
							//generic 'Object' parameters. Replace with 'Anything' as type (so user understands)
							params = params.replaceAll(objecttype, "Anything");
						}
						String insert = name +"("+ params.toString()+")";
						IContextInformation info = params.length()>0?new ContextInformation(insert, params.toString()):null;
						
						if(info!=null) list.add(info);
					}
				}
				
			}
			
		} catch(BadLocationException e) {
		}
		
		return (IContextInformation[])list.toArray(new IContextInformation[list.size()]);
	}
	
	/* (non-Javadoc)
	 * Method declared on IContentAssistProcessor
	 */
	public char[] getCompletionProposalAutoActivationCharacters() {
		return new char[] { '.', '(' };
	}
	
	/* (non-Javadoc)
	 * Method declared on IContentAssistProcessor
	 */
	public char[] getContextInformationAutoActivationCharacters() {
		return new char[] { '#' };
	}
	
	/* (non-Javadoc)
	 * Method declared on IContentAssistProcessor
	 */
	public IContextInformationValidator getContextInformationValidator() {
		if (fValidator == null)
			fValidator= new JavaParameterListValidator();
		return fValidator;
	}
	
	/* (non-Javadoc)
	 * Method declared on IContentAssistProcessor
	 */
	public String getErrorMessage() {
		return null;
	}
}
