/*
 * Created on 11-Jan-2005
 */
package kenya.eclipse.style.checks.bool;

import kenya.eclipse.ast.NodeTools;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AAndBoolTerm;
import minijava.node.ABoolexpExpression;
import minijava.node.ABoolliteralFactor;
import minijava.node.AEqEquality;
import minijava.node.AEqualityBoolTerm;
import minijava.node.AExpressionFactor;
import minijava.node.AFactorPostfixExp;
import minijava.node.AFalseBooleanliteral;
import minijava.node.AGtRelational;
import minijava.node.AGteqRelational;
import minijava.node.ALtRelational;
import minijava.node.ALteqRelational;
import minijava.node.AMathRelational;
import minijava.node.ANegateUnaryExp;
import minijava.node.ANeqEquality;
import minijava.node.AOrBoolExpression;
import minijava.node.APostfixExpUnaryExp;
import minijava.node.ARelationalEquality;
import minijava.node.ATermBoolExpression;
import minijava.node.ATermMathExpression;
import minijava.node.ATrueBooleanliteral;
import minijava.node.AUnaryTerm;
import minijava.node.AXorBoolExpression;
import minijava.node.Node;
import minijava.node.PBoolExpression;
import minijava.node.PBoolTerm;
import minijava.node.PBooleanliteral;
import minijava.node.PEquality;
import minijava.node.PFactor;
import minijava.node.PMathExpression;
import minijava.node.PPostfixExp;
import minijava.node.PRelational;
import minijava.node.PTerm;
import minijava.node.PUnaryExp;
import minijava.node.TEqual;
import minijava.node.TFalse;
import minijava.node.TGreater;
import minijava.node.TGreaterequal;
import minijava.node.TIdentifier;
import minijava.node.TLParenthese;
import minijava.node.TLess;
import minijava.node.TLessequal;
import minijava.node.TNot;
import minijava.node.TNotequal;
import minijava.node.TRParenthese;
import minijava.node.TTrue;

/**
 * provides facility to examine and manipulate boolean related nodes
 * 
 * 
 * @author Thomas Timbul
 */

public class BoolNodeTools extends NodeTools {
	
	protected BoolNodeTools() {
		super();
	}
	
	static class LargestSearch extends DepthFirstAdapter {
		private Node largest;
		
		private final Node toSearch;
		
		private LargestSearch(Node n) {
			toSearch = n;
		}
		
		protected Node getLargestBoolNode() {
			if(largest==null) {
				toSearch.apply(this);
			}
			return largest==null?toSearch:largest;
		}
		
		public void caseABoolexpExpression(ABoolexpExpression node) {
			//largest = node;
			super.caseABoolexpExpression(node);
		}
		public void caseAAndBoolTerm(AAndBoolTerm node) {
			largest = node;
		}
		public void caseABoolliteralFactor(ABoolliteralFactor node) {
			largest = node;
			super.caseABoolliteralFactor(node);
		}
		public void caseAXorBoolExpression(AXorBoolExpression node) {
			largest = node;
		}
		public void caseAEqEquality(AEqEquality node) {
			largest = node;
		}
		public void caseAEqualityBoolTerm(AEqualityBoolTerm node) {
			largest = node;
			super.caseAEqualityBoolTerm(node);
		}
		public void caseAFalseBooleanliteral(AFalseBooleanliteral node) {
			largest = node;
		}
		public void caseANegateUnaryExp(ANegateUnaryExp node) {
			largest = node;
		}
		public void caseANeqEquality(ANeqEquality node) {
			largest = node;
		}
		public void caseAOrBoolExpression(AOrBoolExpression node) {
			largest = node;
		}
		public void caseALtRelational(ALtRelational node) {
			largest = node;
		}
		public void caseALteqRelational(ALteqRelational node) {
			largest = node;
		}
		public void caseAGtRelational(AGtRelational node) {
			largest = node;
		}
		public void caseAGteqRelational(AGteqRelational node) {
			largest = node;
		}
		public void caseATrueBooleanliteral(ATrueBooleanliteral node) {
			largest = node;
		}
		
		public void caseTIdentifier(TIdentifier node) {
			//the ultimate case
			largest = node;
		}
	}
	
	/**
	 * find the largest Boolean component in
	 * a particular node.
	 * 
	 * for example for the expression
	 * (a && b) || c
	 * the result would be a OrBoolTerm with toString value (a && b) || c
	 * apply recursively might give the following set:
	 * 
	 * OrBoolTerm            (a && b) || c
	 *   AndBoolTerm         (a && b)
	 *     EqualityBoolTerm  a
	 *     EqualityBoolTerm  b
	 *   EqualityBoolTerm    c
	 * 
	 * if this node itself is not a 'boolean' node, the last such occurrence will be returned
	 * @param n
	 * @return
	 */
	public static Node getLargestBoolNode(Node n) {
		LargestSearch g = new LargestSearch(n);
		Node nana = g.getLargestBoolNode();
		return nana;
	}
	
	public static boolean isAnd(Node n) {
		if(n==null) return false;
		Node l = getLargestBoolNode(n);
		if(l instanceof AAndBoolTerm) return true;
		else {
			if(l instanceof ATermBoolExpression) {
				l = ((ATermBoolExpression)l).getBoolTerm();
				return isAnd(l);
			}
			return false;
		}
	}
	
	public static boolean isOr(Node n) {
		if(n==null) return false;
		Node l = getLargestBoolNode(n);
		if(l instanceof AOrBoolExpression) return true;
		else {
			if(l instanceof ATermBoolExpression) {
				l = ((ATermBoolExpression)l).getBoolTerm();
				return isOr(l);
			}
			if(l instanceof AEqualityBoolTerm) {
				l = ((AEqualityBoolTerm)l).getEquality();
				return isOr(l);
			}
			return false;
		}
	}
	
	public static boolean isXor(Node n) {
		return n!=null && getLargestBoolNode(n) instanceof AXorBoolExpression;
	}
	
	public static boolean isNot(Node n) {
		return n!=null && getLargestBoolNode(n) instanceof ANegateUnaryExp;
	}
	
	public static boolean isTrue(Node n) {
		return n!=null && getLargestBoolNode(n) instanceof ATrueBooleanliteral;
	}
	
	public static boolean isFalse(Node n) {
		return n!=null && getLargestBoolNode(n) instanceof AFalseBooleanliteral;
	}
	
	public static boolean isComposite(Node n) {
		return isAnd(n) || isOr(n) || isXor(n);
	}
	
	//the negated nodes are always clones and never have parents
	
	public static ANeqEquality negate(AEqEquality node) {
		ISourceCodeLocation loc = getLocation(node.getEqual());
		AEqEquality clone = (AEqEquality)node.clone();
		ANeqEquality neq = new ANeqEquality(clone.getEquality(), new TNotequal(loc.getLineNumber(), loc.getColumnNumber()), clone.getRelational());
		return neq;
	}
	
	public static AEqEquality negate(ANeqEquality node) {
		ISourceCodeLocation loc = getLocation(node.getNotequal());
		ANeqEquality clone = (ANeqEquality)node.clone();
		AEqEquality eq = new AEqEquality(clone.getEquality(), new TEqual(loc.getLineNumber(), loc.getColumnNumber()), clone.getRelational());
		return eq;
	}
	
	public static PEquality negate(PEquality node) {
		
		if(node instanceof AEqEquality) {
			return negate((AEqEquality)node);
		} else if(node instanceof ANeqEquality) {
			return negate((ANeqEquality)node);
		} else {
			//RelationalEquality
			ARelationalEquality clone = (ARelationalEquality)node.clone();
			
			//establish a negated tree...
			PEquality peq = clone;
			PBoolTerm pbt = new AEqualityBoolTerm(peq);
			PBoolExpression pbe = new ATermBoolExpression(pbt);
			PFactor pf = new AExpressionFactor(new TLParenthese(), pbe, new TRParenthese());
			PPostfixExp ppe = new AFactorPostfixExp(pf);
			PUnaryExp pue = new APostfixExpUnaryExp(ppe);
			PUnaryExp npue = new ANegateUnaryExp(new TNot(), pue);
			PTerm pt = new AUnaryTerm(npue);
			PMathExpression pme = new ATermMathExpression(pt);
			PRelational pr = new AMathRelational(pme);
			PEquality pe = new ARelationalEquality(pr);
			
//			PRelational pr = negate(clone.getRelational());
//			clone.getRelational().replaceBy(pr);
			return pe;
		}
	}
	
	public static PRelational negate(PRelational node) {
		if (node instanceof ALtRelational) {
			ALtRelational n = (ALtRelational)node.clone();
			TLess l = n.getLess();
			TGreaterequal g = new TGreaterequal(l.getLine(), l.getPos());
			AGteqRelational r = new AGteqRelational(n.getRelational(), g, n.getMathExpression());
			return r;
		} else if (node instanceof ALteqRelational) {
			ALteqRelational n = (ALteqRelational)node.clone();
			TLessequal l = n.getLessequal();
			TGreater g = new TGreater(l.getLine(), l.getPos());
			AGtRelational r = new AGtRelational(n.getRelational(), g, n.getMathExpression());
			return r;
		} else if (node instanceof AGtRelational) {
			AGtRelational n = (AGtRelational)node.clone();
			TGreater g = n.getGreater();
			TLessequal l = new TLessequal(g.getLine(), g.getPos());
			ALteqRelational r = new ALteqRelational(n.getRelational(), l, n.getMathExpression());
			return r;
		} else if (node instanceof AGteqRelational) {
			AGteqRelational n = (AGteqRelational)node.clone();
			TGreaterequal g = n.getGreaterequal();
			TLess l = new TLess(g.getLine(), g.getPos());
			ALtRelational r = new ALtRelational(n.getRelational(), l, n.getMathExpression());
			return r;
		} else if (node instanceof AMathRelational) {
			AMathRelational n = (AMathRelational)node.clone();
			return n;
		} else {
			return node;
		}
	}
	
	public static PUnaryExp negate(APostfixExpUnaryExp node) {
		ISourceCodeLocation loc = getLocation(node);
		APostfixExpUnaryExp clone = (APostfixExpUnaryExp)node.clone();
		ANegateUnaryExp n = new ANegateUnaryExp(new TNot(loc.getLineNumber(), loc.getColumnNumber()), clone);
		return n;
	}
	
	public static PUnaryExp negate(ANegateUnaryExp node) {
		return (PUnaryExp)node.getUnaryExp().clone();
	}
	
	public static PUnaryExp negate(PUnaryExp node) {
		if(node instanceof APostfixExpUnaryExp) {
			return negate((APostfixExpUnaryExp)node);
		} else if(node instanceof ANegateUnaryExp) {
			return negate((ANegateUnaryExp)node);
		} else {
			return null;
		}
	}
	
	public static PBooleanliteral negate(PBooleanliteral node) {
		Node clone = (Node)node.clone();
		if(isTrue(clone)) {
			ATrueBooleanliteral t = (ATrueBooleanliteral)clone;
			TTrue tt = t.getTrue();
			ISourceCodeLocation loc = NodeTools.getLocation(tt);
			AFalseBooleanliteral f = new AFalseBooleanliteral(
					new TFalse(loc.getLineNumber(), loc.getColumnNumber())
				);
			return f;
		} else {
			AFalseBooleanliteral f = (AFalseBooleanliteral)clone;
			TFalse tf = f.getFalse();
			ISourceCodeLocation loc = NodeTools.getLocation(tf);
			ATrueBooleanliteral t = new ATrueBooleanliteral(
					new TTrue(loc.getLineNumber(), loc.getColumnNumber())
				);
			return t;
		}
		
	}
	
}