/*
 * Created on 08-Jan-2005
 */
package kenya.eclipse.style.checks.bool;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.ast.NodeToString;
import kenya.eclipse.ast.NodeTools;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentModificationOperation;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.eclipse.style.checks.IdentifierSearch;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AAndBoolTerm;
import minijava.node.ABoolexpExpression;
import minijava.node.ABoolliteralFactor;
import minijava.node.AEqEquality;
import minijava.node.AEqualityBoolTerm;
import minijava.node.AExpressionFactor;
import minijava.node.AFactorPostfixExp;
import minijava.node.AFalseBooleanliteral;
import minijava.node.AGtRelational;
import minijava.node.AGteqRelational;
import minijava.node.ALtRelational;
import minijava.node.ALteqRelational;
import minijava.node.AMathRelational;
import minijava.node.ANegateUnaryExp;
import minijava.node.ANeqEquality;
import minijava.node.AOrBoolExpression;
import minijava.node.APostfixExpUnaryExp;
import minijava.node.ARelationalEquality;
import minijava.node.ATermBoolExpression;
import minijava.node.ATermMathExpression;
import minijava.node.ATrueBooleanliteral;
import minijava.node.AUnaryTerm;
import minijava.node.AXorBoolExpression;
import minijava.node.Node;
import minijava.node.PBoolExpression;
import minijava.node.PBoolTerm;
import minijava.node.PBooleanliteral;
import minijava.node.PEquality;
import minijava.node.PRelational;
import minijava.node.PUnaryExp;
import minijava.node.TFalse;
import minijava.node.TLParenthese;
import minijava.node.TNot;
import minijava.node.TRParenthese;
import minijava.node.TTrue;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.IDocument;

/**
 * @author Thomas Timbul
 */
public class BooleanExpressionChecker extends AbstractStyleChecker {
	
	private static final String MESSAGE = "The boolean expression \"{0}\" can be simplified. It can be rewritten as \"{1}\"";
	
	private ICheckedCode fCode;
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, final IFile file) {
		fCode = code;
		IFunction[] functions = code.getFunctions();
		
		//accumulator
		final HashMap warnings = new HashMap();
		
		for(int i = 0; i < functions.length; i++) {
			IFunction fun = functions[i];
			
			//clone the entire tree. This is necessary so we can perform reductions without
			// modifying actual contents
			Node n = (Node)fun.getDeclarationNode().getBlock().clone();
			
			List l = new BooleanSearch(n).getBooleans();
			
			for(Iterator iter = l.iterator(); iter.hasNext();) {
				Node node = (Node)iter.next();
				
				if(AbstractStyleChecker.overlapsWithError(node, code)) {
					//we can't deal with errors reliably, so skip
					continue;
				}
				
				Node temp = (Node)node.clone();
				
				ReductionSearch rs = new ReductionSearch(node);
				try {
					boolean _ret = rs.isReducible();
					Node _reduced = rs.getReducedNode();
					
					if(_ret) {
						warnings.put(temp, _reduced);
					}
				} catch(Exception e) {
					//do not throw back at user: log quietly and continue with next item
					KenyaPlugin.log(e);
				}
			}
		}
		
		//now go through the warnings that we found and create markers for it
		
		IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				
				Set s = warnings.entrySet();
				
				for(Iterator it = s.iterator(); it.hasNext();) {
					Entry pair = (Entry)it.next();
					Node b4 = (Node)pair.getKey();
					Node after = (Node)pair.getValue();
					
					IDocument doc = getDocument(file);
					ISourceCodeLocation loc
					  = AdvancedPositionFinder.getFullLocation(b4, doc);
					String msg = MessageFormat.format(MESSAGE,
					    new String[] {NodeToString.toString(b4), NodeToString.toString(after)});
					
					IMarker m = createKenyaStyleMarker(file, loc, msg);
					
					final String label = new String(NodeToString.toString(after));
					
					if(fResolutionMap==null) {
						fResolutionMap = new HashMap(s.size());
					}
					
					ReductionMarkerResolution resolution
					  = new ReductionMarkerResolution(label, m, label);
					
					fResolutionMap.put(m, new ReductionMarkerResolution[] {resolution});
					
					
				}
				
			}
		};
		
		runMarkerUpdate(runnable);
		
	}
	
	/**
	 * can be used to find all boolean related nodes inside of another node
	 * Does not report nested boolean nodes, ie only the largest
	 * boolean expression is in the result set
	 */
	private class BooleanSearch extends DepthFirstAdapter {
		
		private final Node toSearch;
		private ArrayList list;
		
		public BooleanSearch(Node n) {
			toSearch = n;
		}
		
		public List getBooleans() {
			if(list==null) {
				list = new ArrayList();
				toSearch.apply(this);
			}
			return list;
		}
		
		//these are the largest reducible boolean expressions:
		// && (and), || (or), ^ (xor), ! (not), == (equals)
		//whenever we encounter one, add it to the list of found nodes, but do
		// not apply to subnodes
		
		public void caseAAndBoolTerm(AAndBoolTerm node) {
			inAAndBoolTerm(node);
			list.add(node);
			outAAndBoolTerm(node);
		}
		public void caseAOrBoolExpression(AOrBoolExpression node) {
			inAOrBoolExpression(node);
			list.add(node);
			outAOrBoolExpression(node);
		}
		public void caseANegateUnaryExp(ANegateUnaryExp node) {
			inANegateUnaryExp(node);
			list.add(node);
			outANegateUnaryExp(node);
		}
		public void caseAEqualityBoolTerm(AEqualityBoolTerm node) {
			inAEqualityBoolTerm(node);
			list.add(node);
			outAEqualityBoolTerm(node);
		}
		public void caseAXorBoolExpression(AXorBoolExpression node) {
			inAXorBoolExpression(node);
			list.add(node);
			outAXorBoolExpression(node);
		}
	}
	
	/**
	 * does a reduction search on a given boolean related node.
	 * For example (true && false) reduces to false.
	 * 
	 * NOTE: the given node is modified during the application of this
	 * search. Operate on clones only!!!
	 */
	private class ReductionSearch extends DepthFirstAdapter {
		
		//implementation notice:
		// the ORDER in the if/else constructs matters in almost all cases. Do not mess around!
		
		private boolean _ret = false;
		private boolean tried = false;
		private Node _reduced;
		
		private final Node toSearch;
		
		public ReductionSearch(Node n) {
			toSearch = n;
		}
		
		public boolean isReducible() {
			if(!tried) {
				toSearch.apply(this);
				tried = true;
			}
			return _ret;
		}
		
		public Node getReducedNode() {
			return (isReducible())
					?_reduced:toSearch;
		}
		
		public void caseAAndBoolTerm(AAndBoolTerm node) {
			inAAndBoolTerm(node);
			
			PBoolTerm term = node.getBoolTerm();
			PEquality eq = node.getEquality();
			final Class boolterm = PBoolTerm.class; // due to a bug with PBoolTerm.class mutating to PEquality.class
			final Class peq = PEquality.class; //due to a bug with PEquality.class mutating to something else
			ReductionSearch s1 = new ReductionSearch((Node)term.clone());
			ReductionSearch s2 = new ReductionSearch((Node)eq.clone());
			
			Node n1 = BoolNodeTools.getLargestBoolNode(s1.getReducedNode());
			Node n2 = BoolNodeTools.getLargestBoolNode(s2.getReducedNode());
			
			//return value is true if either of them was reduced
			_ret = s1.isReducible() || s2.isReducible();
			
			if(BoolNodeTools.isTrue(n1) || BoolNodeTools.isFalse(n2)) {
				//n2
				//need to wrap a EqualityBoolTerm around it
				PEquality rep = (PEquality)BoolNodeTools.getParent(n2, peq);
				AEqualityBoolTerm ebt = new AEqualityBoolTerm(rep);
				node.replaceBy(ebt);
				_reduced = ebt;
			} else if(BoolNodeTools.isTrue(n2) || BoolNodeTools.isFalse(n1)
					|| BoolNodeTools.limitedAreEqual(n1, n2)) {
				//n1
				Node rep = BoolNodeTools.getParent(n1, boolterm);
				node.replaceBy(rep);
				_reduced = rep;
			}	else if(_ret) {
				//replace with reduced nodes
				
				term.replaceBy((PBoolTerm)BoolNodeTools.getParent(n1, boolterm));
				eq.replaceBy((PEquality)BoolNodeTools.getParent(n2, peq));
				_reduced = node;
			} else {
				
				_reduced = node;
				return;
			}
			
			if(_reduced instanceof AAndBoolTerm) {
				AAndBoolTerm nn = (AAndBoolTerm)_reduced;
				if(BoolNodeTools.limitedAreEqual(nn.getBoolTerm(), nn.getEquality())) {
					_reduced.replaceBy(nn.getBoolTerm());
				}
			}
			
			_ret = true;
			outAAndBoolTerm(node);
		}
		
		public void caseAOrBoolExpression(AOrBoolExpression node) {
			inAOrBoolExpression(node);
			
			PBoolExpression exp = node.getBoolExpression();
			PBoolTerm term = node.getBoolTerm();
			
			ReductionSearch s1 = new ReductionSearch((Node)exp.clone());
			ReductionSearch s2 = new ReductionSearch((Node)term.clone());
			Node n1 = BoolNodeTools.getLargestBoolNode(s1.getReducedNode());
			Node n2 = BoolNodeTools.getLargestBoolNode(s2.getReducedNode());
			
			//return value is true if either of them was reduced
			_ret = s1.isReducible() || s2.isReducible();
			
			if(BoolNodeTools.isTrue(n1) || BoolNodeTools.isFalse(n2)
					|| BoolNodeTools.limitedAreEqual(n1, n2)) {
				//n1
				Node rep = BoolNodeTools.getParent(n1, PBoolExpression.class);
				node.replaceBy(rep);
				_reduced = rep;
			} else if(BoolNodeTools.isTrue(n2) || BoolNodeTools.isFalse(n1)) {
				//n2
				//wrap with ATermBoolExpression
				ATermBoolExpression tbe = new ATermBoolExpression((PBoolTerm)BoolNodeTools.getParent(n2, PBoolTerm.class));
				node.replaceBy(tbe);
				_reduced = tbe;
			} else if(_ret) {
//			 replace the reduced values (we may not need to, but do it anyway)
				exp.replaceBy(BoolNodeTools.getParent(n1, PBoolExpression.class));
				term.replaceBy((PBoolTerm)BoolNodeTools.getParent(n2, PBoolTerm.class));
				_reduced = node;
				
			} else {
				_reduced = node;
				return;
			}
			
			if(_reduced instanceof AOrBoolExpression) {
				AOrBoolExpression nn = (AOrBoolExpression)_reduced;
				if(BoolNodeTools.limitedAreEqual(nn.getBoolTerm(), nn.getBoolExpression())) {
					_reduced.replaceBy(node.getBoolTerm());
				}
			}
			
			_ret = true;
			
			outAOrBoolExpression(node);
		}
		
		public void caseABoolexpExpression(ABoolexpExpression node) {
			inABoolexpExpression(node);
			//just apply to the contained expression
			node.getBoolExpression().apply(this);
			outABoolexpExpression(node);
		}
		
		public void caseANegateUnaryExp(ANegateUnaryExp node) {
			inANegateUnaryExp(node);
			
			PUnaryExp unary = node.getUnaryExp();
			
			
			ReductionSearch s1 = new ReductionSearch((Node)unary.clone());
			Node n1 = BoolNodeTools.getLargestBoolNode(s1.getReducedNode());
			
			_ret = s1.isReducible();
			
			if(BoolNodeTools.isTrue(n1) || BoolNodeTools.isFalse(n1)) {
				// !true or !false
				PBooleanliteral l = BoolNodeTools.negate((PBooleanliteral)BoolNodeTools.getParent(n1, PBooleanliteral.class));
				ABoolliteralFactor f = new ABoolliteralFactor(l);
				AFactorPostfixExp p = new AFactorPostfixExp(f);
				APostfixExpUnaryExp e = new APostfixExpUnaryExp(p);
				
				node.replaceBy(e);
				_reduced = e;
				_ret = true;
			} else if(_ret) {
				//the inside should have reduced itself
				
				PUnaryExp rep = (PUnaryExp)BoolNodeTools.getParent(n1, PUnaryExp.class);
				
				node.getUnaryExp().replaceBy(rep);
				_reduced = node;
				_ret = true;
				
			} else if(BoolNodeTools.isNot(n1)) {
				//!(!a) -> a
				ANegateUnaryExp u = (ANegateUnaryExp)BoolNodeTools.getLargestBoolNode(n1);
				Node rep = u.getUnaryExp();
				node.replaceBy(rep);
				_reduced = rep;
				_ret = true;
			} else {
				//dont have to do anything
				_reduced = node;
			}
			
			outANegateUnaryExp(node);
		}
		
		public void caseAEqEquality(AEqEquality node) {
			inAEqEquality(node);
			
			PEquality eq = node.getEquality();
			PRelational rel = node.getRelational();
			
			ReductionSearch s1 = new ReductionSearch((Node)eq.clone());
			ReductionSearch s2 = new ReductionSearch((Node)rel.clone());
			Node n1 = BoolNodeTools.getLargestBoolNode(s1.getReducedNode());
			Node n2 = BoolNodeTools.getLargestBoolNode(s2.getReducedNode());
			
			//return value is true if either of them was reduced
			_ret = s1.isReducible() || s2.isReducible();
			
			if(BoolNodeTools.isFalse(n1) && BoolNodeTools.isFalse(n2)
					|| BoolNodeTools.isTrue(n1) && BoolNodeTools.isTrue(n2)
				  || BoolNodeTools.limitedAreEqual(n1, n2)) {
				//true == true
				//false == false
				//a == a
				
				//we have True
				
				ARelationalEquality releq = new ARelationalEquality();
				
				{
					ISourceCodeLocation loc = NodeTools.getLocation(node);
					TTrue tt = new TTrue(loc.getLineNumber(), loc.getColumnNumber());
					ATrueBooleanliteral literal = new ATrueBooleanliteral(tt);
					ABoolliteralFactor lfac      = new ABoolliteralFactor(literal);
					AFactorPostfixExp pexp       = new AFactorPostfixExp(lfac);
					APostfixExpUnaryExp unaryexp = new APostfixExpUnaryExp(pexp);
					AUnaryTerm uterm             = new AUnaryTerm(unaryexp);
					ATermMathExpression mexp     = new ATermMathExpression(uterm);
					AMathRelational mrel         = new AMathRelational(mexp);
					
					releq.setRelational(mrel);
				}
				
				node.replaceBy(releq);
				_reduced = releq;
				
			} else if(BoolNodeTools.isTrue(n2) && !BoolNodeTools.isFalse(n1)) {
				//a == true
				//n1
				Node n = BoolNodeTools.getParent(n1, PEquality.class);
				node.replaceBy(n);
				_reduced = n;
			} else if(BoolNodeTools.isTrue(n1) && !BoolNodeTools.isFalse(n2)) {
				//true == a
				//n2 - is part of some relational
				PRelational mrel = (PRelational)BoolNodeTools.getParent(n2, PRelational.class);
				ARelationalEquality releq    = new ARelationalEquality(mrel);
				
				node.replaceBy(releq);
				_reduced = releq;
			} else if(BoolNodeTools.isFalse(n2) && !BoolNodeTools.isTrue(n1)) {
				// a == false
				//!n1
				//n1 is part of a PEquality
				PEquality neg = BoolNodeTools.negate((PEquality)BoolNodeTools.getParent(n1, PEquality.class));
				node.replaceBy(neg);
				_reduced = neg;
				
			} else if(BoolNodeTools.isFalse(n1) && !BoolNodeTools.isTrue(n2)) {
				//false == a
				//!n2
				//n2 is part of some relational
				PRelational mrel = (PRelational)BoolNodeTools.getParent(n2, PRelational.class);
				ARelationalEquality releq = new ARelationalEquality(mrel);
				
				PEquality neg = BoolNodeTools.negate(releq);
				node.replaceBy(neg);
				_reduced = neg;
				
			} else if(
					//this case is matched, if neither are identifiers and they are not literally equal
					//if BOTH WERE identifiers, then we couldn't tell!
						SingleVariableExpressionChecker.isSingleVariableExpression(n1)
						&& SingleVariableExpressionChecker.isSingleVariableExpression(n2)
						&& !(IdentifierSearch.isIdentifier(n1) || IdentifierSearch.isIdentifier(n2))
						&& !BoolNodeTools.limitedAreEqual(n1, n2)
					) {
				
				//true == false
				//we have False
				
				ARelationalEquality releq = new ARelationalEquality();
				{
					ISourceCodeLocation loc = NodeTools.getLocation(node);
					TFalse tf = new TFalse(loc.getLineNumber(), loc.getColumnNumber());
					AFalseBooleanliteral literal = new AFalseBooleanliteral(tf);
					ABoolliteralFactor lfac      = new ABoolliteralFactor(literal);
					AFactorPostfixExp pexp       = new AFactorPostfixExp(lfac);
					APostfixExpUnaryExp unaryexp = new APostfixExpUnaryExp(pexp);
					AUnaryTerm uterm             = new AUnaryTerm(unaryexp);
					ATermMathExpression mexp     = new ATermMathExpression(uterm);
					AMathRelational mrel         = new AMathRelational(mexp);
					
					releq.setRelational(mrel);
				}
				node.replaceBy(releq);
				_reduced = releq;
				
			} else {
				_reduced = node;
				return;
			}
			
			_ret = true;
			
			outAEqEquality(node);
		}
		
		public void caseANeqEquality(ANeqEquality node) {
			inANeqEquality(node);

			PEquality eq = node.getEquality();
			PRelational rel = node.getRelational();
			
			ReductionSearch s1 = new ReductionSearch((Node)eq.clone());
			ReductionSearch s2 = new ReductionSearch((Node)rel.clone());
			Node n1 = BoolNodeTools.getLargestBoolNode(s1.getReducedNode());
			Node n2 = BoolNodeTools.getLargestBoolNode(s2.getReducedNode());
			
			//return value is true if either of them was reduced
			_ret = s1.isReducible() || s2.isReducible();
			
			
			if(BoolNodeTools.isFalse(n1) && BoolNodeTools.isFalse(n2)
					|| BoolNodeTools.isTrue(n1) && BoolNodeTools.isTrue(n2)
				  || BoolNodeTools.limitedAreEqual(n1, n2)) {
				//we have False
				
				ARelationalEquality releq = new ARelationalEquality();
				{
					ISourceCodeLocation loc = NodeTools.getLocation(node);
					TFalse tf = new TFalse(loc.getLineNumber(), loc.getColumnNumber());
					AFalseBooleanliteral literal = new AFalseBooleanliteral(tf);
					ABoolliteralFactor lfac      = new ABoolliteralFactor(literal);
					AFactorPostfixExp pexp       = new AFactorPostfixExp(lfac);
					APostfixExpUnaryExp unaryexp = new APostfixExpUnaryExp(pexp);
					AUnaryTerm uterm             = new AUnaryTerm(unaryexp);
					ATermMathExpression mexp     = new ATermMathExpression(uterm);
					AMathRelational mrel         = new AMathRelational(mexp);
					
					releq.setRelational(mrel);
				}
				node.replaceBy(releq);
				_reduced = releq;
				
			} else if(BoolNodeTools.isFalse(n2)) {
				//n1
				Node n = BoolNodeTools.getParent(n1, PEquality.class);
				node.replaceBy(n);
				_reduced = n;
			} else if(BoolNodeTools.isFalse(n1)) {
				//n2 - is part of some relational
				PRelational mrel = (PRelational)BoolNodeTools.getParent(n2, PRelational.class);
				ARelationalEquality releq    = new ARelationalEquality(mrel);
				
				node.replaceBy(releq);
				_reduced = releq;
			} else if(BoolNodeTools.isTrue(n2)) {
				//!n1
				//n1 is part of a PEquality
				PEquality neg = BoolNodeTools.negate((PEquality)BoolNodeTools.getParent(n1, PEquality.class));
				node.replaceBy(neg);
				_reduced = neg;
				
			} else if(BoolNodeTools.isTrue(n1)) {
				//!n2
				//n2 is part of some relational
				PRelational mrel = (PRelational)BoolNodeTools.getParent(n2, PRelational.class);
				ARelationalEquality releq = new ARelationalEquality(mrel);
				
				PEquality neg = BoolNodeTools.negate(releq);
				node.replaceBy(neg);
				_reduced = neg;
				
			} else {
				//compare the sides for equality
				_reduced = node;
				return;
			}
			
			_ret = true;
			
			outANeqEquality(node);
		}
		
		public void caseAFalseBooleanliteral(AFalseBooleanliteral node) {
			inAFalseBooleanliteral(node);
			//stop, cant reduce this thing
			outAFalseBooleanliteral(node);
		}
		
		public void caseATrueBooleanliteral(ATrueBooleanliteral node) {
			inATrueBooleanliteral(node);
			//stop, cant reduce this thing
			outATrueBooleanliteral(node);
		}
		
		public void caseAXorBoolExpression(AXorBoolExpression node) {
			inAXorBoolExpression(node);
			
			PBoolExpression exp = node.getBoolExpression();
			PBoolTerm term = node.getBoolTerm();
			
			ReductionSearch s1 = new ReductionSearch((Node)exp.clone());
			ReductionSearch s2 = new ReductionSearch((Node)term.clone());
			Node n1 = BoolNodeTools.getLargestBoolNode(s1.getReducedNode());
			Node n2 = BoolNodeTools.getLargestBoolNode(s2.getReducedNode());
			
			//return value is true if either of them was reduced
			_ret = s1.isReducible() || s2.isReducible();
			
			if(BoolNodeTools.isFalse(n1)) {
				//n2
				//wrap with ATermBoolExpression
				ATermBoolExpression tbe = new ATermBoolExpression(term);
				node.replaceBy(tbe);
				_reduced = tbe;
			} else if(BoolNodeTools.isFalse(n2)) {
				//n1
				node.replaceBy(exp);
				_reduced = exp;
			} else if(BoolNodeTools.isTrue(n1) && BoolNodeTools.isTrue(n2)
					      || BoolNodeTools.isFalse(n1) && BoolNodeTools.isFalse(n2)
					      || BoolNodeTools.limitedAreEqual(n1, n2)) {
				//now we have false...
				//create an ATermBoolExpression containing 'False'
				ATermBoolExpression tbe = new ATermBoolExpression();
				{
					ISourceCodeLocation loc = NodeTools.getLocation(node);
					TFalse tf = new TFalse(loc.getLineNumber(), loc.getColumnNumber());
					AFalseBooleanliteral literal = new AFalseBooleanliteral(tf);
					ABoolliteralFactor lfac      = new ABoolliteralFactor(literal);
					AFactorPostfixExp pexp       = new AFactorPostfixExp(lfac);
					APostfixExpUnaryExp unaryexp = new APostfixExpUnaryExp(pexp);
					AUnaryTerm uterm             = new AUnaryTerm(unaryexp);
					ATermMathExpression mexp     = new ATermMathExpression(uterm);
					AMathRelational mrel         = new AMathRelational(mexp);
					ARelationalEquality releq    = new ARelationalEquality(mrel);
					AEqualityBoolTerm ebt        = new AEqualityBoolTerm(releq);
					tbe.setBoolTerm(ebt);
				}
				
				node.replaceBy(tbe);
				_reduced = tbe;
			} else if(BoolNodeTools.isTrue(n1) || BoolNodeTools.isTrue(n2)) {
				// !(the respective other)
				
				boolean negated = false;
				
				//repl1 is what we will wrap into a not soon
				PBoolExpression repl1;
				
				if(BoolNodeTools.isTrue(n1)) {
					// !n2
					
					if(BoolNodeTools.isNot(n2)) {
						PUnaryExp p = BoolNodeTools.negate((PUnaryExp)BoolNodeTools.getParent(n2, PUnaryExp.class));
						
						n2.replaceBy(p);
						n2 = p;
						negated = true;
					}
					
					PBoolTerm pbt = (PBoolTerm) BoolNodeTools.getParent(n2, PBoolTerm.class);
					repl1 = new ATermBoolExpression(pbt);
				} else {
					// !n1
					
					if(BoolNodeTools.isNot(n1)) {
						PUnaryExp p = BoolNodeTools.negate((PUnaryExp)BoolNodeTools.getParent(n1, PUnaryExp.class));
						n1.replaceBy(p);
						n1 = p;
						negated = true;
					}
					
					repl1 = (PBoolExpression) BoolNodeTools.getParent(n1, PBoolExpression.class);
				}
				
				//now wrap it in a not (if we havent already negated)
				if (!negated) {
					
					//repl is what we will replace node with
					ATermBoolExpression repl = new ATermBoolExpression();
					
					{
						ISourceCodeLocation loc = BoolNodeTools.getLocation(repl1);
						int line = loc.getLineNumber();
						int col = loc.getColumnNumber();
						TLParenthese lparen = new TLParenthese(line, col + 1); //+1 for the 'not'
						TRParenthese rparen = new TRParenthese(line, col + 1 + loc.getTokenLength()); //+1 for ! and +length for length of repl1
						AExpressionFactor aef = new AExpressionFactor(lparen, repl1, rparen);

						AFactorPostfixExp afpe = new AFactorPostfixExp(aef);
						APostfixExpUnaryExp apeue = new APostfixExpUnaryExp(afpe);

						TNot not = new TNot(line, col);
						ANegateUnaryExp anue = new ANegateUnaryExp(not, apeue);
						AUnaryTerm aut = new AUnaryTerm(anue);
						ATermMathExpression atme = new ATermMathExpression(aut);
						AMathRelational amr = new AMathRelational(atme);
						ARelationalEquality are = new ARelationalEquality(amr);
						AEqualityBoolTerm aebt = new AEqualityBoolTerm(are);

						//phew, after all that work:
						repl.setBoolTerm(aebt);
					}

					node.replaceBy(repl);
					_reduced = repl;
					
				} else {
					node.replaceBy(repl1);
					_reduced = repl1;
				}
			} else if(BoolNodeTools.isNot(n1) || BoolNodeTools.isNot(n2)) {
				//check if they are negations of each other. Then we would have True
				
				boolean thisIsTrue = false;
				
				if(BoolNodeTools.isNot(n1)) {
					PUnaryExp neg = (PUnaryExp) BoolNodeTools.getParent(n1, PUnaryExp.class);
					neg = BoolNodeTools.negate(neg);
					//now if they are equal...
					thisIsTrue = BoolNodeTools.limitedAreEqual(neg, n2);
					
				} else {
					PUnaryExp neg = (PUnaryExp) BoolNodeTools.getParent(n2, PUnaryExp.class);
					neg = BoolNodeTools.negate(neg);
					//now if they are equal...
					thisIsTrue = BoolNodeTools.limitedAreEqual(n1, neg);
					
				}
				
				if(thisIsTrue) {
					
					//create an ATermBoolExpression containing 'True'
					ATermBoolExpression tbe = new ATermBoolExpression();
					{
						ISourceCodeLocation loc = NodeTools.getLocation(node);
						TTrue tt = new TTrue(loc.getLineNumber(), loc.getColumnNumber());
						ATrueBooleanliteral literal  = new ATrueBooleanliteral(tt);
						ABoolliteralFactor lfac      = new ABoolliteralFactor(literal);
						AFactorPostfixExp pexp       = new AFactorPostfixExp(lfac);
						APostfixExpUnaryExp unaryexp = new APostfixExpUnaryExp(pexp);
						AUnaryTerm uterm             = new AUnaryTerm(unaryexp);
						ATermMathExpression mexp     = new ATermMathExpression(uterm);
						AMathRelational mrel         = new AMathRelational(mexp);
						ARelationalEquality releq    = new ARelationalEquality(mrel);
						AEqualityBoolTerm ebt        = new AEqualityBoolTerm(releq);
						tbe.setBoolTerm(ebt);
					}
					
					node.replaceBy(tbe);
					_reduced = tbe;
					
				} else {
					_ret = false;
					return;
				}
				
			} else if(_ret) {
//			 replace the reduced values (we may not need to, but do it anyway)
				exp.replaceBy(BoolNodeTools.getParent(n1, PBoolExpression.class));
				term.replaceBy((PBoolTerm)BoolNodeTools.getParent(n2, PBoolTerm.class));
				_reduced = node;
				
			} else {
				_reduced = node;
				return;
			}
			
			_ret = true;
			
			outAXorBoolExpression(node);
		}
		
		public void caseAExpressionFactor(AExpressionFactor node) {
			inAExpressionFactor(node);
			
			//strips brackets from a bool if they are not required
			
			node.getBoolExpression().apply(this);
			
			//Thomas:
			// I tried here previously to remove superfluous brackets.
			// it turns out that this is usually not required because of the
			// replacement algorithm in the other methods.
			
			//in future someone might handle an expression like
			// (a && b) to reduce to a && b, given that this is not a subterm of a larger expression
			
			outAExpressionFactor(node);
		}
		
		public void caseALtRelational(ALtRelational node) {
			_reduced = node;
		}
		public void caseALteqRelational(ALteqRelational node) {
			_reduced = node;
		}
		public void caseAGtRelational(AGtRelational node) {
			_reduced = node;
		}
		public void caseAGteqRelational(AGteqRelational node) {
			_reduced = node;
		}
		
	}
	
	class ReductionMarkerResolution extends StyleWarningResolution {
		
		public ReductionMarkerResolution(String label, IMarker marker, String replacement) {
			super("replace with "+label, marker, new DocumentModificationOperation());
			
			fDescription = "<p><b>Explanation:</b></p>" +
				"<p>The given expression is complicated and might obscure the actual " +
				"purpose of the statment. If it is simplified, the code becomes " +
				"clearer and much easier to read and understand.</p>";
			
			int offset = marker.getAttribute(IMarker.CHAR_START, -1);
			int length = marker.getAttribute(IMarker.CHAR_END, -1) - offset;
			
			if(offset>=0) {
				fOperation.addOperation(
						DocumentTextOperation.newTextReplacement(offset, length, replacement)
					);
			}
			
		}
		
	}
}
