/*
 * Created on 21-Jan-2005
 */
package kenya.eclipse.style.checks.swit;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import kenya.eclipse.KenyaConstants;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentModificationOperation;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.ABlock;
import minijava.node.ACasePossibleCase;
import minijava.node.ADefaultPossibleCase;
import minijava.node.AForStatement;
import minijava.node.ASwitchBlock;
import minijava.node.ASwitchStatement;
import minijava.node.AWhileStatement;
import minijava.node.Node;
import minijava.node.PPossibleCase;
import minijava.node.TBreak;
import minijava.node.TReturn;
import minijava.node.Token;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;

/**
 * @author Thomas Timbul
 */
public class BreakOmissionChecker extends AbstractStyleChecker {
	
	private static final String MESSAGE = "This case does not have a break; statement. What may happen is that cases 'fall through'.";
	
	public static final int CREATE_BREAK_STATEMENT = 0;
	public static final int CREATE_FALL_THROUGH_COMMENT = 1;
	
	protected IMarker m;
	
	protected static Position calculatePosition(ISourceCodeLocation loc, IDocument doc) {
		return AbstractStyleChecker.calculatePosition(loc, doc);
	}
	
	public BreakOmissionChecker() {
		super();
	}

	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, final IFile file) {
		IFunction[] functions = code.getFunctions();
		
		List bigList = new ArrayList(1); //1 since there won't be many
		
		for(int i = 0; i < functions.length; i++) {
			Node node = (Node)functions[i].getDeclarationNode().clone();
			bigList.addAll(SwitchSearch.getSwitchStatements(node));
		}
		
		//accumulator for offending cases
		final List caseList = new ArrayList(bigList.size()*2);
		
		for(Iterator it = bigList.iterator(); it.hasNext();) {
			ASwitchStatement s = (ASwitchStatement)it.next();
			
			ASwitchBlock block = (ASwitchBlock)s.getSwitchBlock();
			LinkedList cases = (LinkedList)block.getPossibleCase();
			
			for(Iterator iter = cases.listIterator(); iter.hasNext();) {
				PPossibleCase element = (PPossibleCase)iter.next();
				
				if(!BreakSearch.containsBreak(element)
				    && !FallThroughCommentSearch.containsFallThroughComment(element, getDocument(file))) {
					caseList.add(element);
				}
			}
		}
		
		//caseList contains only PPossibleCases without break
		
		IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				
				for(Iterator it = caseList.iterator(); it.hasNext();) {
					Object o = it.next();
					Token t;
					
					ABlock caseBlock;
					
					if(o instanceof ADefaultPossibleCase) {
						ADefaultPossibleCase c = (ADefaultPossibleCase)o;
						t = c.getDefault();
						caseBlock = (ABlock)c.getBlock();
					} else if(o instanceof ACasePossibleCase) {
						ACasePossibleCase c = (ACasePossibleCase)o;
						t = c.getCase();
						caseBlock = (ABlock)c.getBlock();
					} else {
						continue;
					}
					SourceCodeLocation loc
					  = new SourceCodeLocation(t.getLine(), t.getPos(), t.getText().length());
					
					m = createKenyaStyleMarker(file, loc, MESSAGE);
					
					Token brace = caseBlock.getRBrace();
					int line = brace.getLine()-1; //reduce because of indexing
					
					int fixOffset = -1;
					try {
						fixOffset = getDocument(file).getLineOffset(line)+brace.getPos()-1;
					} catch(BadLocationException e) {
						return;
					}
					m.setAttribute(FIX_OFFSET, fixOffset);
					
					BreakOmissionMarkerResolution r0
					  = new BreakOmissionMarkerResolution(m, CREATE_BREAK_STATEMENT);
					BreakOmissionMarkerResolution r1
					  = new BreakOmissionMarkerResolution(m, CREATE_FALL_THROUGH_COMMENT);
					
					if (fResolutionMap == null) {
						fResolutionMap = new HashMap(caseList.size());
					}
					
					fResolutionMap.put(m, new StyleWarningResolution[] {r0, r1});
					
				}
				
			}
		};
		
		runMarkerUpdate(runnable);
	}
	
	class BreakOmissionMarkerResolution extends StyleWarningResolution {
		
		public BreakOmissionMarkerResolution(IMarker marker, int type) {
			super(null, marker, new DocumentModificationOperation());
			initLabel(type);
			initDescription(type);
			init(marker, type);
		}
		
		private void initDescription(int type) {
			switch(type) {
				case CREATE_BREAK_STATEMENT:
					fDescription = "<p><b>Explanation:</b></p>" +
							"<p>A <b>break statement</b> marks the end of a 'case' in a " +
							"switch block. This prevents cases from falling through, which " +
							"means that the <b>next</b> case is <b>also</b> executed.</p>" +
							"<p><b>Note:</b> If the case is intended to fall through, you " +
							"should create a <em>fall-through comment</em> instead.</p>";
					break;
				case CREATE_FALL_THROUGH_COMMENT:
					fDescription = "<p><b>Explanation:</b></p>" +
							"<p>An <b>explicit fall-through comment</b> helps to distinguish " +
							"accidental fall-through situations from ones that were created on purpose.<br>" +
							"In larger software projects it is important to signal the intentions behind " +
							"code in case another person works on the same bit of code so as to " +
							"prevent mistakes.</p>" +
							"<p><b>Note:</b> If the case should not fall through, you should " +
							"create a <i>break statement</i> instead.</p>";
					break;
				default:
					fDescription = "Something went wrong when creating the automatic solution, " +
							"so just ignore this warning. If this error persists, contact a qualified helper.";
					break;
			}
		}
		
		private void initLabel(int type) {
			switch(type) {
				case CREATE_BREAK_STATEMENT:
					fLabel = "Create break statement";
					break;
				case CREATE_FALL_THROUGH_COMMENT:
					fLabel = "Create fall-through comment";
					break;
				default:
					fLabel = "invalid operation, see description..";
					break;
			}
		}
		
		private void init(IMarker marker, int type) {
			IDocument doc = getDocument((IFile)marker.getResource());
			if(doc==null) {
				return;
			}
			
			//find the place where we are supposed to insert our generation
			int offset = marker.getAttribute(FIX_OFFSET, -1);
			
			if(offset<0) {
				return; //no such place defined, then we can't continue
			}
			
			String prefix; //a prefix to the comment or whatever
			String postfix; //a postfix to the comment or whatever
			
			try {
				int line = doc.getLineOfOffset(offset); //line number
				String delim = doc.getLineDelimiter(line); //line delimiter
				int lineOffset = doc.getLineOffset(line); //line offset
				int length = offset - lineOffset;
				
				//the line up to our insert position
				String lineContent = doc.get(lineOffset, length);
				
				//find the indent of the given line
				if("".equals(lineContent.trim())) {
					//then the indent is that of the previous line! (which will be ours +1 unit)
					String indent = lineContent;
					if(indent.charAt(0)=='\t') {
						prefix = "\t";
					} else {
						//add tabwidth many spaces
						prefix = "";
						for(int i =0; i<KenyaConstants.EDITOR_TAB_WIDTH; i++) {
							prefix += ' ';
						}
					}
					if(type==CREATE_FALL_THROUGH_COMMENT) {
						prefix += "//";
					}
					postfix = delim + indent;
				} else if(type==CREATE_FALL_THROUGH_COMMENT){
					//the closing bracket is not on its own line, put it on the same line
					// as a 'multiline' comment
					prefix = " /*";
					postfix = "*/ ";
				} else {
					postfix = prefix = " ";
				}
				
			} catch(BadLocationException e) {
				return;
			}
			
			String insertion = prefix + getGeneration(type) + postfix;
			fOperation.addOperation(DocumentTextOperation.newTextInsertion(offset, insertion));
		}
		private String getGeneration(int type) {
			switch(type) {
				case CREATE_BREAK_STATEMENT:
					return "break;";
				case CREATE_FALL_THROUGH_COMMENT:
					return SwitchConstants.FALL_THROUGH_COMMENT;
				default:
					return "";
			}
		}
		
	}
	
}

class BreakSearch extends DepthFirstAdapter {
	protected BreakSearch() {}
	
	public static boolean containsBreak(PPossibleCase n) {
		BreakSearch bs = new BreakSearch();
		n.apply(bs);
		return bs._ret;
	}
	
	protected boolean _ret;
	
	public void caseTBreak(TBreak node) {
		_ret = true;
	}
	
	public void caseASwitchStatement(ASwitchStatement node) {
		//ignore, do not increase the depth
	}
	
	public void caseAForStatement(AForStatement node) {
		//ignore
	}
	
	public void caseAWhileStatement(AWhileStatement node) {
		//ignore
	}
	
	public void caseTReturn(TReturn node) {
		_ret = true; //return is equivalent to break
	}
	
}

class FallThroughCommentSearch extends BreakSearch {
	
	static boolean containsFallThroughComment(PPossibleCase n, IDocument d) {
		FallThroughCommentSearch s = new FallThroughCommentSearch(d);
		n.apply(s);
		return s._ret;
	}
	
	protected FallThroughCommentSearch(IDocument d) {
		doc = d;
	}
	
	private final IDocument doc;
	private boolean _ret;
	
	public void caseACasePossibleCase(ACasePossibleCase node) {
		ABlock block = (ABlock)node.getBlock();
		considerBlock(block);
	}
	public void caseADefaultPossibleCase(ADefaultPossibleCase node) {
		ABlock block = (ABlock)node.getBlock();
		considerBlock(block);
	}
	
	private void considerBlock(ABlock block) {
		Token begin = block.getLBrace();
		Token t = block.getRBrace();
		try {
			String txt = "";
			Position bp = LocationUtils.convert(LocationUtils.getLocation(begin), doc);
			Position tp = LocationUtils.convert(LocationUtils.getLocation(  t  ), doc);
			
			//obtain the String form
			String s = doc.get(bp.getOffset(), tp.getOffset()-bp.getOffset());
			if(s.charAt(s.length()-1)=='}') {
				//cut off the trailing brace if exists
				s = s.substring(0, s.length()-1);
			}
			
			//we only examine the last part of the block - the last 'instruction'
			int lastblock = s.lastIndexOf('}');
			if(lastblock>=0) {
				s = s.substring(lastblock+1);
			}
			
			//now a simple indexOf will do
			_ret = s.indexOf(SwitchConstants.FALL_THROUGH_COMMENT)>=0;
			
		} catch(BadLocationException e) {
			//ignore
		}
	}
}
