package kenya.eclipse.ui.wizards;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.operation.*;
import java.lang.reflect.InvocationTargetException;
import java.text.DateFormat;
import java.util.Date;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import java.io.*;

import kenya.eclipse.KenyaConstants;
import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.preferences.PreferenceConstants;

import org.eclipse.ui.*;
import org.eclipse.ui.ide.IDE;

/**
 * This is a sample new wizard. Its role is to create a new file 
 * resource in the provided container. If the container resource
 * (a folder or a project) is selected in the workspace 
 * when the wizard is opened, it will accept it as the target
 * container. The wizard creates one file with the extension
 * "k". If a sample multi-page editor (also available
 * as a template) is registered for the same extension, it will
 * be able to open it.
 */

public class NewKFileWizard extends Wizard implements INewWizard {
	private NewKFileWizardPage page;
	private ISelection selection;

	/**
	 * Constructor for NewKFileWizard.
	 */
	public NewKFileWizard() {
		super();
		setNeedsProgressMonitor(true);
		setWindowTitle("New Kenya File");
	}
	
	/**
	 * Adding the page to the wizard.
	 */

	public void addPages() {
		page = new NewKFileWizardPage(selection);
		addPage(page);
	}

	/**
	 * This method is called when 'Finish' button is pressed in
	 * the wizard. We will create an operation and run it
	 * using wizard as execution context.
	 */
	public boolean performFinish() {
		final String containerName = page.getContainerName();
		final String fileName = page.getFileName()+KenyaConstants.SUFFIX_STRING_k;
		IRunnableWithProgress op = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException {
				try {
					doFinish(containerName, fileName, monitor);
				} catch (CoreException e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			}
		};
		try {
			getContainer().run(true, false, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			Throwable realException = e.getTargetException();
			MessageDialog.openError(getShell(), "Error", realException.getMessage());
			return false;
		}
		return true;
	}
	
	/**
	 * The worker method. It will find the container, create the
	 * file if missing or just replace its contents, and open
	 * the editor on the newly created file.
	 */

	private void doFinish(
		String containerName,
		String fileName,
		IProgressMonitor monitor)
		throws CoreException {
		// create a sample file
		monitor.beginTask("Creating " + fileName, 2);
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource resource = root.findMember(new Path(containerName));
		if (!resource.exists() || !(resource instanceof IContainer)) {
			throwCoreException("Container \"" + containerName + "\" does not exist.");
		}
		IContainer container = (IContainer) resource;
		final IFile file = container.getFile(new Path(fileName));
		InputStream stream = openContentStream();
		try {
			if (file.exists()) {
				file.setContents(stream, true, true, monitor);
			} else {
				file.create(stream, true, monitor);
			}
		} finally {
			try {
				stream.close();
			} catch(IOException e) {
				KenyaPlugin.log(e);
			}
		}
		monitor.worked(1);
		monitor.setTaskName("Opening file for editing...");
		getShell().getDisplay().asyncExec(new Runnable() {
			public void run() {
				IWorkbenchPage page =
					PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
				try {
					IDE.openEditor(page, file, true);
				} catch (PartInitException e) {
				}
			}
		});
		monitor.worked(1);
	}
	
	/**
	 * We will initialize file contents depending on what
	 * main method creation method the user chose.
	 */
	private InputStream openContentStream() {
		boolean generateDate = true;
		boolean generateAuthor = true;
		
		StringBuffer contents = new StringBuffer();
		String nl = System.getProperty("line.separator");
		
		String indent = "\t";
		if(KenyaPlugin.getDefault().getPluginPreferences().getBoolean(PreferenceConstants.EDITOR_SPACES_FOR_TABS)) {
			indent = "";
			for(int i=0; i<KenyaPlugin.getDefault().getPluginPreferences().getInt(PreferenceConstants.EDITOR_TAB_WIDTH); i++) {
				indent+=" ";
			}
		}
		
		if(generateDate || generateAuthor) {
			contents.append("/*");
			if(generateDate) {
				contents.append(nl).append(" * ").append("Created on "+DateFormat.getDateInstance().format(new Date()));
			}
			if(generateAuthor) {
				contents.append(nl).append(" * ").append("Author: "+System.getProperty("user.name"));
			}
			contents.append(nl).append(" */");
		}
		
		contents.append(nl).append(nl);
		
		switch(page.getMainMethodCreationType()) {
			case NewKFileWizardPage.NO_MAIN:
				break;
			case NewKFileWizardPage.CREATE_MAIN:
				contents.append("void main() {");
				contents.append(nl).append(indent);
				contents.append(nl).append("}");
				break;
			case NewKFileWizardPage.CREATE_MAIN_ARGS:
				contents.append("void main(String[] args) {");
				contents.append(nl).append(indent);
				contents.append(nl).append("}");
				break;
			default:
				break;
		}
		return new ByteArrayInputStream(contents.toString().getBytes());
	}
	
	private void throwCoreException(String message) throws CoreException {
		IStatus status =
			new Status(IStatus.ERROR, KenyaPlugin.getPluginId(), IStatus.OK, message, null);
		throw new CoreException(status);
	}

	/**
	 * We will accept the selection in the workbench to see if
	 * we can initialize from it.
	 * @see IWorkbenchWizard#init(IWorkbench, IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.selection = selection;
	}
}