/*
 * Created on 27-Nov-2004
 */
package kenya.eclipse;

import java.util.Arrays;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

/**
 * defines conventions for Kenya such as language keywords
 * This class cannot be instantiated or subclassed. It may however
 * be freely referenced.
 * @author Thomas Timbul
 */
public final class KenyaConventions implements KenyaConstants {
	
	public static final String[] KENYA_KEYWORDS = { "class", "const", "if", "else",
			"while", "return", "switch", "case", "break", "default", "for",
			"assert", "new", "enum" };
	
	public static final String[] KENYA_TYPES = { "boolean", "char", "int",
      "double", "String", "void" };
	
	public static final String[] KENYA_CONSTANTS = { "false", "null", "true" };
	
	private static final String[] jkKeyWords; //all the Kenya AND Java reserved words
	
	static { //initialise jkKeyWords
		
		final String[] javaReserved = {
				   //in addition to the Kenya ones
				   //note that some Kenya words are not in Java, but this is not used
				   //for Java validation and should thus be ok.
				"abstract", "byte", "catch", "continue", "do", "extends",
				"final", "finally", "float", "implements", "import", "instanceof",
				"interface", "long", "native", "package", "private", "protected",
				"public", "short", "static", "super", "synchronized",
				"this", "throw", "throws", "transient", "try", "volatile"
			};
		int length = javaReserved.length + KENYA_KEYWORDS.length + KENYA_TYPES.length + KENYA_CONSTANTS.length;
		String[] reserved = new String[length];
		
		int offset = 0;
		
		//is this way fast enough?
		System.arraycopy(javaReserved,    0, reserved, offset, javaReserved.length   );
		offset += javaReserved.length;
		System.arraycopy(KENYA_KEYWORDS,  0, reserved, offset, KENYA_KEYWORDS.length );
		offset += KENYA_KEYWORDS.length;
		System.arraycopy(KENYA_TYPES,     0, reserved, offset, KENYA_TYPES.length    );
		offset += KENYA_TYPES.length;
		System.arraycopy(KENYA_CONSTANTS, 0, reserved, offset, KENYA_CONSTANTS.length);
		
		Arrays.sort(reserved);
		
		jkKeyWords = reserved;
	}
	
	private KenyaConventions() {
		//prevent instantiation
	}
	
	/**
	 * Returns true if id clashes with
	 * any of the keywords defined in Kenya/Java and false otherwise
	 * @param id the word to check
	 * @return true if id clashes with a Kenya/Java keyword, false otherwise
	 */
	public static boolean clashesWithReservedWord(String id) {
		//find a violation using binary search (faster than for loop)
		int s = Arrays.binarySearch(jkKeyWords, id);
		return s>=0;
	}
	
	/*
	 * returns true if the String contains anything but [a-zA-Z_0-9] or $
	 */
	private static boolean containsIllegalCharacters(String id) {
		return !id.matches("[a-zA-Z_$][a-zA-Z_0-9$]*");
	}
	
	/**
	 * Validate the given (simple) Kenya (Java) type name.
	 * For example, <code>"Object"</code>, or <code>"TrashCan"</code>.
	 * <p>
	 *
	 * @param name the name of a type
	 * @return a status object with code <code>IStatus.OK</code> if
	 *		the given name is valid as a Kenya (Java) type name, 
	 *      a status with code <code>IStatus.WARNING</code>
	 *		indicating why the given name is discouraged, 
	 *      otherwise a status object indicating what is wrong with 
	 *      the name
	 * @see org.eclipse.jdt.core.JavaConventions#validateJavaTypeName(java.lang.String)
	 */
	public static IStatus validateKenyaTypeName(String name) {
		if (name == null || name.length()==0) {
			return new Status(IStatus.ERROR, K_PLUGIN_ID, -1,
					"Type name is empty", null);
		}
		String trimmed = name.trim();
		if (!name.equals(trimmed)) {
			return new Status(IStatus.ERROR, K_PLUGIN_ID, -1,
					"Type name is not valid." +
					" A Kenya/Java type name must not start or end with a blank", null);
		}
		int index = name.lastIndexOf('.');
		if (index == -1) {
			if(clashesWithReservedWord(name)) {
				return new Status(IStatus.ERROR, K_PLUGIN_ID, -1,
						"Type name is not valid." +
						" The type name '"+name+"' clashes with a" +
						" reserved keyword or operator" +
						" in either Kenya or Java", null);
			} else if(containsIllegalCharacters(name)) {
				return new Status(IStatus.ERROR, K_PLUGIN_ID, -1,
						"Type name is not valid." +
						" The type name '"+name+"' is not a valid Kenya/Java identifier", null);
			} else if(Character.isDigit(name.charAt(0))) {
				return new Status(IStatus.ERROR, K_PLUGIN_ID, -1,
						"Type name is not valid." +
						" Identifiers cannot start with a digit.", null);
			}
		} else {
			// 'qualified' name
			return new Status(IStatus.ERROR, K_PLUGIN_ID, -1,
					"There is no need to specify a file extension." +
					" Kenya type names also must not contain any dots", null);
		}
		
		IStatus status = ResourcesPlugin.getWorkspace().validateName(name, IResource.FILE);
		if (!status.isOK()) {
			return status;
		}
		if (name.indexOf('$')>=0) {
			return new Status(IStatus.WARNING, K_PLUGIN_ID, -1,
					"Type name is discouraged." +
					" By convention, Kenya/Java type names usually" +
					" don't contain the $ character", null);
		}
		if ((name.length() > 0 && Character.isLowerCase(name.charAt(0)))) {
			return new Status(IStatus.WARNING, K_PLUGIN_ID, -1,
					"Type name is discouraged." +
					" By convention, Kenya/Java type names usually" +
					" start with an uppercase letter", null);
		}
		return new Status(IStatus.OK, K_PLUGIN_ID, IStatus.OK, "", null);
	}
	
}
